#!/usr/bin/python3
"""

Copyright (C) 2024 Efinix Inc. All rights reserved.

No portion of this code may be reused, modified or
distributed in any way without the expressed written
consent of Efinix Inc.

Created on Nov 5, 2024

@author: kylen
"""

import os
import sys
import re
import argparse
import logging
from configparser import ConfigParser


def main():
    parser = argparse.ArgumentParser()
    parser.add_argument('-s', '--silent', action='store_true')
    args = parser.parse_args()

    if args.silent:
        logging.basicConfig(format='%(message)s', level=logging.CRITICAL, stream=sys.stdout)
    else:
        logging.basicConfig(format='%(message)s', level=logging.INFO, stream=sys.stdout)
    logger = logging.getLogger(__name__)

    EFINITY_HOME = os.environ.get("EFINITY_HOME")
    if EFINITY_HOME is None:
        logger.error("could not find EFINITY_HOME")
        sys.exit(1)

    main_manifest = os.path.join(EFINITY_HOME, "MANIFEST")
    patch_manifest = os.path.join(
        os.path.dirname(os.path.abspath(__file__)), "PATCH_MANIFEST"
    )
    cp = ConfigParser()
    cp.read(main_manifest)

    if "build_info" not in cp.sections():
        logger.error("ERROR: unable to parse MANIFEST file, may be corrupted, aborting!")
        sys.exit(1)

    # example patch_verison = 2024.2.284.3.6 => <main.base.ver>.<patch_number>.<patch_build_number>
    main_platform = cp.get("build_info", "build_platform")
    main_base_ver = cp.get("build_info", "base_version")

    patch_ver_regex = re.compile(r"(\d+.\d+.\d+).(\d+).(\d+)")

    curr_patch_base_ver, curr_patch_num, curr_patch_build_num = None, None, None
    curr_patch_str = None
    if "patch_info" in cp.sections():
        matches = patch_ver_regex.match(cp.get("patch_info", "patch_version"))
        if matches is not None:
            curr_patch_base_ver, curr_patch_num, curr_patch_build_num = matches.groups()
            curr_patch_str = matches.string

    cp.read(patch_manifest)
    new_patch_platform = cp.get("patch_info", "build_platform")

    matches = patch_ver_regex.match(cp.get("patch_info", "patch_version"))
    if matches is not None:
        new_patch_base_ver, new_patch_num, new_patch_build_num = matches.groups()
        new_patch_str = matches.string
    else:
        new_patch_base_ver, new_patch_num, new_patch_build_num = None, None, None
        new_patch_str = None

    # Validate patch is compatible with target Efinity installation
    error_code = 0
    error_msg = ""
    if new_patch_platform != main_platform:
        error_msg = (
            f"ERROR: Patch is meant for {new_patch_platform} but target Efinity installation is for {main_platform}"
        )
        error_code = 3

    if main_base_ver != new_patch_base_ver:
        error_msg = (
            "ERROR: Base version of patch does not match base version of target Efinity installation!\n"
            f"{'Current base version:' : <21} {main_base_ver}\n"
            f"{'Patch base version:' : <21} {new_patch_base_ver}"
        )
        error_code = 4

    if curr_patch_num is not None and new_patch_num is not None:
        if int(new_patch_num) < int(curr_patch_num):
            error_msg = (
                "ERROR: Patch version is older than current verison of target Efinity installation!\n"
                f"{'Current version:' : <17} {curr_patch_str}\n"
                f"{'Patch version:' : <17} {new_patch_str}"
            )
            error_code = 5
    
    if error_code != 0:
        if not args.silent:
            logger.error(error_msg)
            ignore_err = input("Would you still like to continue? (Y/[N]) ")
            if ignore_err.lower() != 'y':
                sys.exit(error_code)
        else:
            sys.exit(error_code)

    with open(main_manifest, "w") as manifest:
        cp.write(manifest)
    sys.exit(0)


if __name__ == "__main__":
    main()
