'''
Copyright (C) 2017-2018 Efinix Inc. All rights reserved.

No portion of this code may be reused, modified or
distributed in any way without the expressed written
consent of Efinix Inc.

Created on May 2, 2018

@author: maryam
'''

import os
import sys
import datetime

import xml.etree.ElementTree as et
from prettytable import PrettyTable
from decimal import Decimal

from typing import Dict, Tuple, List, Union

from util.singleton_logger import Logger
import util.gen_util as pt_util
import util.excp as app_excp

import device.db_interface as device_dbi
import device.timing_model as tmodel
import device.excp as dev_excp

import writer.excp as writer_excp
# import writer.timing.gpio as tgpio
# import writer.timing.pll as tpll
# import writer.timing.osc as tosc
import design.db as des_db

from common_device.writer import BlockTiming, BlockTimingWithArcs

import common_device.rule as base_rule

import common_device.gpio.writer.timing as gpio_timing
import common_device.osc.writer.timing as osc_timing
import common_device.lvds.writer.tx_timing as lvds_tx_timing
import common_device.lvds.writer.rx_timing as lvds_rx_timing
import common_device.jtag.writer.timing as jtag_timing
import common_device.ctrl.writer.timing as ctrl_timing
import common_device.mipi.writer.rx_tx_timing as mipi_timing
import common_device.h264.writer.timing as h264_timing
import common_device.ddr.writer.timing as ddr_timing
import common_device.hposc.writer.timing as hposc_timing
import common_device.spi_flash.writer.timing as spiflash_timing
import common_device.hyper_ram.writer.timing as hram_timing
import common_device.hsio.writer.timing as hsio_timing
import common_device.mipi_hard_dphy.writer.timing as mipi_hdphy_timing

import an08_device.pll.writer.timing as an08_pll_timing
import an20_device.pll.writer.timing as an20_pll_timing

import tx60_device.osc.writer.timing as tx60_osc_timing
import tx60_device.pll.writer.timing as tx60_pll_timing
import tx60_device.gpio.writer.timing as tx60_gpio_timing
from tx60_device.clock_mux.writer.timing import ClockMuxTimingAdvance

import tx180_device.pll.writer.timing as tx180_pll_timing
import tx180_device.mipi_dphy.writer.rx_tx_timing as tx180_mdphy_timing
import tx180_device.ddr.writer.timing as tx180_ddr_timing
from tx375_device.fpll.writer.timing import EfxFpllV1Timing
from tx375_device.quad_pcie.writer.timing import QuadPCIETiming
from tx375_device.soc.writer.timing import SOCTimingWriter
from tx375_device.quad.writer.timing import QuadTiming

from device.core_clock_delay import DeviceCoreClockDelay

from _version import __version__


sys.path.append(os.path.join(os.path.dirname(__file__), '..'))


class TimingReport(object):
    '''
    Class for generating a timing report summary.
    '''
    xml_ns = "{http://www.efinixinc.com/peri_device_db}"

    def __init__(self, device_db, design_db,
                 dir_name, project_name):
        '''
        constructor
        '''
        self.logger = Logger

        # it needs to access the design configuration
        # in order to write out the file.
        self.__design = design_db

        # contains the device db info
        self.__device = device_db

        # Get the project name from design
        self.__project_name = project_name

        self.__output_dir = dir_name

        # Check that the directory is valid
        if os.path.isdir(dir_name):
            # The filename to dump out the interace configuration data
            self.__report_file = dir_name + "/" + self.__project_name + ".pt_timing.rpt"
            self.__sdc_file = dir_name + "/" + self.__project_name + ".pt.sdc"
        else:
            raise ValueError("Invalid directory {}".format(dir_name))

        # selected timing model
        self.__selected_model = None

        # Max and min model of the selected combination
        self.__max_model = None
        self.__min_model = None

        # Map of instance to io standard
        self.__ins2iostd_map = {}

        # List of block services required for the device. This is
        # to be able to tell which block to use for writing out interface config.
        # It is a map of block type (defined in DBInterface) to
        # the Block InterfaceConfig (peri_block either in common or device specific)
        self.__blocks_timing = {}

        # The map of timing parameters to the delay that is
        # in common location (die). Map of parameter name
        # to the TimingParameter object
        self.__common_timing = {}

        # A map of the interface name to the location map
        # generated from the interface writer based on the container
        # type: input, clkout interface type
        self._input_to_loc_map = {}
        self._clkout_to_loc_map = {}

        # Keep the list of instance (in resource map) map
        # to bank name for use later with GPIO
        self._ins2bank_name_map: Dict[str, str] = {}

    def set_pre_populated_input_loc_map(self, interface_to_loc_map: Dict[str, Dict[str, Union[Tuple[int,int,int], List[Tuple[int,int,int]]]]]):
        """
        :param interface_to_loc_map: A map of interface type str to
                a map of the name to interface coordinate
            # ["input"] = {input: coordinate tuple}
            # ["clkout"] = {clkout: list of coordinate tuple}
        """
        # Clear the map first
        self._input_to_loc_map.clear()
        self._clkout_to_loc_map.clear()

        if "input" in interface_to_loc_map:
            self._input_to_loc_map = interface_to_loc_map["input"]

        if "clkout" in interface_to_loc_map:
            self._clkout_to_loc_map = interface_to_loc_map["clkout"]

    def _parse_common_timing_parameters(self):
        '''
        Parse the timing parameters that reside in the die
        (common).
        '''
        clk2delay_map = {}

        # Get the network delay from the die and apply scaling
        # TODO: Update this if we need to do scaling on clk network delay
        # There can be a case where a device does not have this common timing
        dev_timing = self.__device.get_common_timing()
        if dev_timing is not None:
            tparam_obj = dev_timing.get_parameter(
                tmodel.CommonTiming.ptype2str_map[tmodel.CommonTiming.ParamType.core_clk])

            if tparam_obj is not None:
                # Lookup in the timing model to get
                # the corresponding min and max delay
                clk2delay_map = self._get_common_delay(
                    dev_timing, tparam_obj)

        return clk2delay_map

    def _get_common_delay(self, common_timing, tparam_obj):
        clk2delay_map = {}

        max_param2delay_map = self._parse_timing_table(
            self.__max_model, "die")
        min_param2delay_map = self._parse_timing_table(
            self.__min_model, "die")

        # Save the delay to the existing common TimingParameter
        pname2obj_map = common_timing.get_all_parameters()

        # Iterate through the map which stores the name to
        # the timing parameter name
        for dname in pname2obj_map.keys():
            tpobj = pname2obj_map[dname]

            if tpobj is not None:
                pname = tpobj.get_value()
                min_delay = 0
                max_delay = 0

                # We apply scaling when we save it here
                if pname in max_param2delay_map:
                    # max_ori_delay = round(decimal.Decimal(max_param2delay_map[pname]), 3)
                    max_ori_delay = max_param2delay_map[pname]
                    max_delay = \
                        max_ori_delay * self.__max_model.get_tscale() / 1000

                if pname in min_param2delay_map:
                    # min_ori_delay = round(decimal.Decimal(min_param2delay_map[pname]), 3)
                    min_ori_delay = min_param2delay_map[pname]
                    min_delay = \
                        min_ori_delay * self.__min_model.get_tscale() / 1000

                tpobj.set_max(max_delay)
                tpobj.set_min(min_delay)

                # Add this parameter to the common_timing
                self.__common_timing[pname] = tpobj

                # Save the clk delay and return if found
                if tpobj == tparam_obj:
                    # The map is type to the final delay
                    clk2delay_map[BlockTiming.DelayType.max] = max_delay
                    clk2delay_map[BlockTiming.DelayType.min] = min_delay
                    self.logger.debug("core clk delay - max: {} min: {} in ns".format(
                        max_delay, min_delay))

        return clk2delay_map

    def _parse_timing_table(self, table_model, blk_name):

        param2delay_map = {}

        try:
            tree = et.ElementTree(file=table_model.get_filename())
            # print("REading file: {}".format(table_model.get_filename()))

            block_tag = ".//" + self.xml_ns + "block"

            blocksec = None
            for elem in tree.iterfind(block_tag):
                # print("{}: {}".format(elem.tag, elem.attrib))

                block_attrib = elem.attrib
                if block_attrib["type"] == blk_name:
                    blocksec = elem
                    break

            if blocksec is not None:
                # This will be the list of names without any
                # variables
                arc_param_tag = ".//" + self.xml_ns + "arc_parameter"

                for elem in blocksec.iterfind(arc_param_tag):
                    arc_attrib = elem.attrib
                    param_count = 0
                    delay_value = None

                    if "name" in arc_attrib and arc_attrib["name"] != "":
                        pname = arc_attrib["name"]
                        # Get the delay with the expectation that there is
                        # no variable parameter
                        if arc_attrib["pcount"] is not None and\
                                int(arc_attrib["pcount"]) > 0:
                            raise ValueError(
                                'Die variable pcount is expected '
                                'to be 0 instead of {}'.format(arc_attrib["pcount"]))

                        # Get the delay (children of arc_parameter)
                        param_tag = ".//" + self.xml_ns + "parameter"

                        for pelem in elem.iterfind(param_tag):
                            param_attrib = pelem.attrib

                            delay_value = float(param_attrib["delay"])
                            param_count += 1

                        if param_count == 1 and pname not in param2delay_map:
                            param2delay_map[pname] = delay_value
                            # self.logger.debug("Saving common parameter {} to {}".format(
                            #    pname, delay_value))

                        elif param_count > 1:
                            raise ValueError(
                                'Die arc_parameter should only have one'
                                ' delay value. Instead found {}'.format(param_count))

                        elif pname in param2delay_map:
                            raise ValueError(
                                'Found duplicated arc_parameter {} in die block'.format(
                                    pname))

                        else:
                            raise ValueError(
                                'No delay value stated for arc_parameter {}'.format(
                                    pname))

        except Exception as excp:
            # self.logger.error("Error with reading the timing common_models: {}".format(excp))
            raise excp

        return param2delay_map

    def get_report_file_name(self):
        '''
        Get full file name
        :return: Generated report filename
        '''
        return self.__report_file

    def get_sdc_file_name(self):
        '''
        Get the sdc filename
        :return: Generated sdc filename
        '''
        return self.__sdc_file

    def _identify_blocks(self):
        '''
        Initiate the list of periphery block LPF writer.
        This is determine by the block type available in the
        device.

        :param block_names: list of block names that are defined
                            for this device
        '''

        # Go through all the block to generate the block
        # pcr definition
        # device_db = device_dbi.DeviceDBService(self.__device)

        # Get the list of block types from the dbi
        for btype, bname in device_dbi.DeviceDBService.block_type2str_map.items():
            block_obj = None

            # Skip if the device does not have this block
            if self.__device.find_block(bname) is None:
                continue

            if btype == device_dbi.DeviceDBService.BlockType.GPIO:
                if self.__design.is_block_supported(des_db.PeriDesign.BlockType.comp_gpio):
                    block_obj = tx60_gpio_timing.GPIOTimingComplex(
                        bname, self.__device, self.__design,
                        self.__report_file, self.__sdc_file,
                        self.__max_model, self.__min_model,
                        self.__ins2iostd_map)

                elif self.__design.is_block_supported(des_db.PeriDesign.BlockType.adv_l2_gpio) or \
                        self.__design.is_block_supported(des_db.PeriDesign.BlockType.gpio):
                    block_obj = gpio_timing.GPIOTiming(
                        bname, self.__device, self.__design,
                        self.__report_file, self.__sdc_file,
                        self.__max_model, self.__min_model,
                        self.__ins2iostd_map)

                else:
                    msg = 'Unable to determine GPIO type'

                    raise dev_excp.BlockDoesNotExistException(
                        msg, app_excp.MsgLevel.error)

            elif btype == device_dbi.DeviceDBService.BlockType.OSC:
                if self.__design.is_block_supported(des_db.PeriDesign.BlockType.adv_osc):
                    block_obj = tx60_osc_timing.OSCTimingAdv(
                        bname, self.__device, self.__design,
                        self.__report_file, self.__sdc_file)

                elif self.__design.is_block_supported(des_db.PeriDesign.BlockType.osc):
                    block_obj = osc_timing.OSCTiming(
                        bname, self.__device, self.__design,
                        self.__report_file, self.__sdc_file)

                else:
                    msg = 'Unable to determine OSC type'
                    raise dev_excp.BlockDoesNotExistException(
                        msg, app_excp.MsgLevel.error)

            elif btype == device_dbi.DeviceDBService.BlockType.HPOSC:
                block_obj = hposc_timing.HPOSCTiming(
                    bname, self.__device, self.__design,
                    self.__report_file, self.__sdc_file)

            elif btype == device_dbi.DeviceDBService.BlockType.PLL:
                # This is going to be device dependent

                if self.__design.is_block_supported(des_db.PeriDesign.BlockType.efx_fpll_v1):
                    block_obj = EfxFpllV1Timing(
                        bname, self.__device, self.__design,
                        self.__report_file, self.__sdc_file,
                        self.__max_model, self.__min_model)

                elif self.__design.is_block_supported(des_db.PeriDesign.BlockType.efx_pll_v3_comp):
                    block_obj = tx180_pll_timing.PLLTimingV3Complex(
                        bname, self.__device, self.__design,
                        self.__report_file, self.__sdc_file)

                elif self.__design.is_block_supported(des_db.PeriDesign.BlockType.comp_pll):
                    block_obj = tx60_pll_timing.PLLTimingComplex(
                        bname, self.__device, self.__design,
                        self.__report_file, self.__sdc_file)

                elif self.__design.is_block_supported(des_db.PeriDesign.BlockType.adv_pll):
                    block_obj = an20_pll_timing.PLLTiming(
                        bname, self.__device, self.__design,
                        self.__report_file, self.__sdc_file)

                elif self.__design.is_block_supported(des_db.PeriDesign.BlockType.pll):
                    block_obj = an08_pll_timing.PLLTiming(
                        bname, self.__device, self.__design,
                        self.__report_file, self.__sdc_file)

                else:
                    msg = 'Unable to determine PLL type'

                    raise dev_excp.BlockDoesNotExistException(
                        msg, app_excp.MsgLevel.error)

            elif btype == device_dbi.DeviceDBService.BlockType.LVDS_TX:
                block_obj = lvds_tx_timing.LVDSTxTiming(
                    bname, self.__device, self.__design,
                    self.__report_file, self.__sdc_file,
                    self.__max_model, self.__min_model,
                    self.__ins2iostd_map)

            elif btype == device_dbi.DeviceDBService.BlockType.LVDS_RX:
                block_obj = lvds_rx_timing.LVDSRxTiming(
                    bname, self.__device, self.__design,
                    self.__report_file, self.__sdc_file,
                    self.__max_model, self.__min_model,
                    self.__ins2iostd_map)

            elif btype == device_dbi.DeviceDBService.BlockType.JTAG:
                block_obj = jtag_timing.JTAGTiming(
                    bname, self.__device, self.__design,
                    self.__report_file, self.__sdc_file,
                    self.__max_model, self.__min_model,
                    self.__ins2iostd_map)

            elif btype == device_dbi.DeviceDBService.BlockType.CONTROL:
                block_obj = ctrl_timing.ControlTiming(
                    bname, self.__device, self.__design,
                    self.__report_file, self.__sdc_file,
                    self.__max_model, self.__min_model)

            elif btype == device_dbi.DeviceDBService.BlockType.MIPI_RX:
                if self.__design.is_block_supported(des_db.PeriDesign.BlockType.mipi_hard_dphy):
                    block_obj = tx180_mdphy_timing.MIPITimingAdvance(
                        bname, self.__device, self.__design,
                        self.__report_file, self.__sdc_file,
                        self.__max_model, self.__min_model, False)

                else:
                    block_obj = mipi_timing.MIPITiming(
                        bname, self.__device, self.__design,
                        self.__report_file, self.__sdc_file,
                        self.__max_model, self.__min_model, False)

            elif btype == device_dbi.DeviceDBService.BlockType.MIPI_TX:
                if self.__design.is_block_supported(des_db.PeriDesign.BlockType.mipi_hard_dphy):
                    block_obj = mipi_hdphy_timing.MIPIDPHYTxTiming(
                        bname, self.__device, self.__design,
                        self.__report_file, self.__sdc_file,
                        self.__max_model, self.__min_model)

                else:
                    block_obj = mipi_timing.MIPITiming(
                        bname, self.__device, self.__design,
                        self.__report_file, self.__sdc_file,
                        self.__max_model, self.__min_model, True)

            elif btype == device_dbi.DeviceDBService.BlockType.H264:
                block_obj = h264_timing.H264Timing(
                    bname, self.__device, self.__design,
                    self.__report_file, self.__sdc_file,
                    self.__max_model, self.__min_model)

            elif btype == device_dbi.DeviceDBService.BlockType.DDR:

                if self.__design.is_block_supported(des_db.PeriDesign.BlockType.adv_ddr):
                    block_obj = tx180_ddr_timing.DDRTimingAdvance(
                        bname, self.__device, self.__design,
                        self.__report_file, self.__sdc_file,
                        self.__max_model, self.__min_model)

                else:
                    block_obj = ddr_timing.DDRTiming(
                        bname, self.__device, self.__design,
                        self.__report_file, self.__sdc_file,
                        self.__max_model, self.__min_model)

            elif btype == device_dbi.DeviceDBService.BlockType.HSIO:
                block_obj = hsio_timing.HSIOTiming(
                    bname, self.__device, self.__design,
                    self.__report_file, self.__sdc_file,
                    self.__max_model, self.__min_model,
                    self.__ins2iostd_map)             

            elif btype == device_dbi.DeviceDBService.BlockType.SPI_FLASH:
                block_obj = spiflash_timing.SPIFlashTiming(
                    bname, self.__device, self.__design,
                    self.__report_file, self.__sdc_file,
                    self.__max_model, self.__min_model,
                    self.__ins2iostd_map)

            elif btype == device_dbi.DeviceDBService.BlockType.HYPER_RAM:
                block_obj = hram_timing.HyperRAMTiming(
                    bname, self.__device, self.__design,
                    self.__report_file, self.__sdc_file,
                    self.__max_model, self.__min_model,
                    self.__ins2iostd_map)

            elif btype == device_dbi.DeviceDBService.BlockType.CLKMUX:
                if self.__device.clkmux_type == device_dbi.DeviceDBService.BlockType.CLKMUX_V4 or \
                        self.__device.clkmux_type == device_dbi.DeviceDBService.BlockType.CLKMUX_COMPLEX or \
                        self.__device.clkmux_type == device_dbi.DeviceDBService.BlockType.CLKMUX_ADV:
                    # Only need the timing writer for titanium clkmux (dynamic clkmux)
                    block_obj = ClockMuxTimingAdvance(
                        bname, self.__device, self.__design,
                        self.__report_file, self.__sdc_file
                    )

            elif btype == device_dbi.DeviceDBService.BlockType.QUAD_PCIE:
                block_obj = QuadPCIETiming(
                    bname, self.__device, self.__design,
                    self.__report_file, self.__sdc_file,
                    self.__max_model, self.__min_model)

            elif btype == device_dbi.DeviceDBService.BlockType.SOC:
                block_obj = SOCTimingWriter(
                    bname, self.__device, self.__design,
                    self.__report_file, self.__sdc_file,
                    self.__max_model, self.__min_model
                )

            elif btype == device_dbi.DeviceDBService.BlockType.QUAD:
                block_obj = QuadTiming(
                    bname, self.__device, self.__design,
                    self.__report_file, self.__sdc_file,
                    self.__max_model, self.__min_model)
                
            # For now we don't need to raise error because other
            # blocks are known not to have timing data
            if block_obj is not None:
                self.__blocks_timing[btype] = block_obj

    def generate_report_header(self, outfile):
        '''
        Generates the header for report file

        :param outfile: file handle that has already been opened
        '''

        now = datetime.datetime.now()

        outfile.write("\nEfinity Interface Designer Timing Report\n")
        outfile.write("Version: {}\n".format(__version__))
        outfile.write("Date: {}\n".format(now.strftime("%Y-%m-%d %H:%M")))

        outfile.write("\n{}\n".format(pt_util.get_copyright_string()))

        # Device and Design Names
        outfile.write("\nDevice: {}\n".format(
            self.__device.get_device_name()))
        outfile.write("Project: {}\n".format(
            self.__project_name))

        # Print the timing model
        # Generate timing model name

        if self.__selected_model is not None:
            self.__selected_model.write_timing_model_name(outfile)

    def generate_sdc_header(self, outfile):
        '''
        Generate the header for the sdc template file

        :param outfile: file handle that has already been opened.
        '''
        now = datetime.datetime.now()

        outfile.write(
            "\n# Auto-generated by Interface Designer\n#"
        )
        outfile.write(
            "\n# WARNING: Any manual changes made to this file will be lost "\
                "when generating constraints.\n")
        outfile.write("\n# Efinity Interface Designer SDC\n")
        outfile.write("# Version: {}\n".format(__version__))
        outfile.write("# Date: {}\n".format(now.strftime("%Y-%m-%d %H:%M")))

        outfile.write("\n# {}\n".format(pt_util.get_copyright_string()))

        # Device and Design Names
        outfile.write("\n# Device: {}\n".format(
            self.__device.get_device_name()))
        outfile.write("# Project: {}\n".format(
            self.__project_name))

        # Print the timing model
        # Generate timing model name

        if self.__selected_model is not None:
            self.__selected_model.write_timing_model_name(outfile, "#")

    def _load_instance_to_io_standard(self):
        '''
        Get the io standard for each io bank. Then, assigned the
        io standard to each GPIO instances.
        '''
        bank2std = {}

        # It can happen if device setting is None, during unit test
        if self.__design.device_setting is not None:
            iob_reg = self.__design.device_setting.iobank_reg

            # Cannot do much if the io bank standard has not been
            # assigned for each io bank
            if iob_reg is not None:

                # Save a mapping of io bank to the selected io std
                for bank in iob_reg.get_all_iobank():
                    io_std = bank.get_raw_io_standard()
                    # Just take the float string
                    str_list = io_std.split(" ")
                    bank2std[bank.name] = str_list[0]

        # If we havent' fill in the bank2std, then we do so
        # now with default io std assignment
        # TODO: Disable this when the device already has
        # the device setting correct in the ui that lists all
        # supported banks. For now, there can be a case where
        # the existing bank2std does not cover all banks

        '''
        if not bank2std:
            io_info = self.__device.get_io_pad()
            if io_info is not None:
                raw_bank2std = io_info.get_bank_to_default_standard()
                for bank in raw_bank2std:
                    io_std = raw_bank2std[bank]
                    str_list = io_std.split(" ")
                    bank2std[bank] = str_list[0]
            else:
                raise ValueError("Device does not have pad info")
        '''
        full_banks = {}
        io_info = self.__device.get_io_pad()
        if io_info is not None:
            raw_bank2std = io_info.get_bank_to_default_standard()
            for bank in raw_bank2std:
                io_std = raw_bank2std[bank]
                str_list = io_std.split(" ")
                full_banks[bank] = str_list[0]

        if bank2std:
            for bank, std_set in full_banks.items():
                # If there is already a setting from user
                # for the bank, then we ignore. Else, take default
                if bank not in bank2std:
                    bank2std[bank] = std_set

        else:
            bank2std = full_banks

        device_db = device_dbi.DeviceDBService(self.__device)
        ins2bank = device_db.get_instance_to_io_bank_names()

        # for bank in bank2std:
        #    print("Bank: {} IO Std: {}".format(bank, bank2std[bank]))

        # Get the Device IO
        for ins in ins2bank:
            # Get the bank name and find the io std
            bank_name = ins2bank[ins]

            if bank_name in bank2std:
                if ins not in self.__ins2iostd_map:
                    self.__ins2iostd_map[ins] = bank2std[bank_name]
                else:
                    # When there are multiple bank associated to instances
                    # TODO: This should be revisit when we support it
                    raise ValueError(
                        "Found instance {} with multiple banks".format(ins))

            else:
                raise ValueError(
                    "Unexpected Error: Instance {} have undefined bank {}".format(
                        ins, bank_name))

        self._ins2bank_name_map = ins2bank

    def _write_report_header(self):
        '''
        Writes out the report file for system timing analysis.
        '''

        # Open the file
        outfile = open(self.__report_file, 'w')

        # Generate file header and TOC
        self.generate_report_header(outfile)

        # Close file
        outfile.close()

    def _write_sdc_header(self):
        '''
        Write the SDC template file for core timing analysis.
        '''
        outfile = open(self.__sdc_file, 'w')

        # Generate file header and TOC
        self.generate_sdc_header(outfile)

        outfile.close()

    def _check_pll_ssc_constraint(self):
        if self.__design.is_block_supported(des_db.PeriDesign.BlockType.pll_ssc):
            # Only allow it to write if it supports
            if self.__design.pll_ssc_reg is not None and\
                    self.__design.pll_ssc_reg.get_inst_count() > 0:
                return True

        return False

    def _get_clock_source_delay_and_clkout_pins(self, ins_to_block_map):
        '''
        Before writing out the delay information, we need to get
        the following information prepared/populated since it will
        be used in the sdc and timing report
        '''
        blk_with_clk_src_latency = [
                device_dbi.DeviceDBService.BlockType.GPIO,
                device_dbi.DeviceDBService.BlockType.LVDS_RX,
                device_dbi.DeviceDBService.BlockType.LVDS_TX,
                device_dbi.DeviceDBService.BlockType.HSIO]

        # Both are map of blk type to map[inst_name: clock/clkout name, max delay, min delay]
        blktype2alt_delay_map = {}
        blktype2clkout_delay_map = {}

        for btype in blk_with_clk_src_latency:
            if btype in self.__blocks_timing:
                blk_writer = self.__blocks_timing[btype]
                if blk_writer is not None:

                    blk_writer.save_source_latency_and_clkout(ins_to_block_map)
                    if blk_writer.ins2_alt_delay_map:
                        blktype2alt_delay_map[btype] = blk_writer.ins2_alt_delay_map

                    if blk_writer.ins2_clkout_delay_map:
                        blktype2clkout_delay_map[btype] = blk_writer.ins2_clkout_delay_map

        return blktype2alt_delay_map, blktype2clkout_delay_map
    
    def _generate_files(self, ins_to_block_map, clk2delay_map, core_delay_obj: DeviceCoreClockDelay):
        '''
        Calls the writer for each block type that has
        a valid timing writer object. Write the data
        into the sdc and the timing report file.

        :param ins_to_block_map: A map of instance to the block type
        :param clk2delay_map: Mapping of BlockTiming.DelayType
                to the delay of the core clock
        '''
        blktype2alt_delay_map, blktype2clkout_delay_map = \
            self._get_clock_source_delay_and_clkout_pins(ins_to_block_map)
        warning_msg_list: List[str] = []

        index = 1
        # We should use a customize order instead of using the std one
        # because the order matters. We need to print out osc,pll then only
        # gpio or lvds because the SDC constraint for gpio might depends
        # on the osc/pll clock definition
        # We do PLL and OSC first
        clk_src_blks = [device_dbi.DeviceDBService.BlockType.OSC,
                        device_dbi.DeviceDBService.BlockType.OSC_ADV,
                        device_dbi.DeviceDBService.BlockType.HPOSC]                        

        for btype in clk_src_blks:
            if btype in self.__blocks_timing:
                blk_writer = self.__blocks_timing[btype]
                if blk_writer is not None:

                    if blk_writer.write(index, ins_to_block_map, clk2delay_map):
                        index += 1

        # For PLL, we need some additional information
        design_clock_latency_all = []

        pll_writer = self.__blocks_timing.get(device_dbi.DeviceDBService.BlockType.PLL, None)
        if pll_writer is not None:
            ins_exists, clock_latency_all, warning_msg_list = pll_writer.write_with_delay(index, ins_to_block_map, 
                                        core_delay_obj, self._input_to_loc_map,
                                        blktype2alt_delay_map, blktype2clkout_delay_map)
            if ins_exists:
                index += 1

            if clock_latency_all is not None and len(clock_latency_all) > 0:
                design_clock_latency_all = clock_latency_all

        # PT-1813: Cleaning up loose-ends. If it was MIPI Hard DPHY Tx
        # with PLL_SSC configured, then we need to define the create_clock
        # constraint earlier since it is a clock source. But, we don't want to write
        # out the non PLL SSC instances constraint yet
        if self._check_pll_ssc_constraint():
            # Get the block writer
            mipi_tx_adv_writer = self.__blocks_timing.get(
                device_dbi.DeviceDBService.BlockType.MIPI_TX, None)
            if mipi_tx_adv_writer is not None:
                if mipi_tx_adv_writer.write_pll_clocks(index, ins_to_block_map, clk2delay_map):
                    index += 1

        skip_blk = clk_src_blks
        skip_blk.append(device_dbi.DeviceDBService.BlockType.PLL)

        for btype in sorted(device_dbi.DeviceDBService.block_type2str_map.keys()):
            # Skip the block that we've iterated earlier
            if btype in self.__blocks_timing and btype not in skip_blk:
                blk_writer = self.__blocks_timing[btype]

                if blk_writer is not None:

                    if blk_writer.write(index, ins_to_block_map, clk2delay_map):
                        index += 1

        # Write out the GPIO Input and pll clock latency into the sdc at the end
        # since different block can still have create_clock and we want to write
        # out latency after clocks are defined
        if design_clock_latency_all or blktype2alt_delay_map:
            self.print_clock_latency_sdc(design_clock_latency_all, blktype2alt_delay_map)

        # Write out the last section in the timing report which is the
        # Core clock network delay
        self.write_all_clkout_delay(index, core_delay_obj)

        return warning_msg_list
    
    def print_clock_latency_sdc(self, design_clock_latency_all,
                                blktype2alt_delay_map):
        
        write_successful = False
        sdcfile = None

        try:
            # Nothing to be done if no latency to be printed
            if len(design_clock_latency_all) == 0 and \
                len(blktype2alt_delay_map) == 0:
                return
            
            sdcfile = open(self.__sdc_file, 'a')

            # Flag to tell whether we should write the title.
            # Not placing outside in case the contents of the
            # container might not be needed to be printed out
            is_title_written = False
            title_header = "\n# Clock Latency Constraints\n############################\n"

            for entry in design_clock_latency_all:
                if not is_title_written:
                    sdcfile.write(title_header)
                    is_title_written = True
                    
                # The last entry in tuply is the command to print
                _, _, sdc_command = entry

                sdcfile.write(sdc_command)

            for btype in blktype2alt_delay_map:
                ins2info_map = blktype2alt_delay_map[btype]

                # This list includes IO of all alternate connection type.
                # But for clock latency, we only want to write out
                # if it was gclk/rclk. Exclude other alternate type
                for ins_name, tup_info in ins2info_map.items():
                    clock_name, max_delay, min_delay, is_clk = tup_info

                    # Don't write out clock latency for non clk signals
                    if is_clk:
                        if not is_title_written:
                            sdcfile.write(title_header)
                            is_title_written = True

                        formatted_max_delay = "{0:.3f}".format(max_delay)
                        formatted_min_delay = "{0:.3f}".format(min_delay)

                        # Remove the '+' sign if the value of the latency is negative
                        if max_delay >= 0:
                            sdcfile.write("# set_clock_latency -source -setup <board_max + {}> [get_ports {{{}}}]\n".format(
                                formatted_max_delay, clock_name))
                        else:
                            sdcfile.write("# set_clock_latency -source -setup <board_max {}> [get_ports {{{}}}]\n".format(
                                formatted_max_delay, clock_name))
                            
                        if min_delay >= 0:                        
                            sdcfile.write("# set_clock_latency -source -hold <board_min + {}> [get_ports {{{}}}]\n".format(
                                formatted_min_delay, clock_name))                    
                        else:
                            sdcfile.write("# set_clock_latency -source -hold <board_min {}> [get_ports {{{}}}]\n".format(
                                formatted_min_delay, clock_name))                    

            sdcfile.close()

            write_successful = True

        except Exception as excp:
            if not write_successful and sdcfile is not None:
                # Close the file that has been opened
                sdcfile.close()
            raise excp
        
    def write(self):
        '''
        Writes out the report file
        '''

        write_successful = False
        excp_msg = ""

        is_register = False
        issue_reg = None
        if self.__design is not None and self.__design.issue_reg is not None:
            issue_reg = self.__design.issue_reg
            is_register = True

        try:

            # Get the timing model. If the timing model isn't ready, we just
            # return successful
            dev_timing = self.__device.get_timing_model()
            if dev_timing is None:
                self.logger.warning(
                    "Device does not have timing model associated. Nothing to write.")
                return True

            # Get the selected model
            selected_timing = dev_timing.get_selected_model()
            if selected_timing is None:
                self.logger.error("Undefined selected timing model")
                return write_successful

            # Set the selected model here
            self.__selected_model = selected_timing

            # Check if the device has the core clock network delay table populated
            # Get the handle to the clock network delay table
            core_delay_obj: DeviceCoreClockDelay = self.__device.populate_core_clock_network_delay_table()
            
            # Load the model
            self.__max_model = self.__selected_model.get_model(
                tmodel.Model.AnalysisType.max)
            self.__min_model = self.__selected_model.get_model(
                tmodel.Model.AnalysisType.min)

            # load the instance to bank
            self._load_instance_to_io_standard()

            self._identify_blocks()

            device_svc = device_dbi.DeviceDBService(self.__device)

            # Get the list of device instance names mapped
            # to the block name. This includes flatten LVDS
            # instances (with mode attached)
            ins_to_block_map = device_svc.get_configured_instances(
                self.__design)

            # Parse the die timing first since it may be used
            # across multiple blocks. This has to be done after
            # the max and min model have been identified
            # UPDATE: This is no longer useful since it's always 0 here
            # We can cleanup by deleting this later
            clk2delay_map = self._parse_common_timing_parameters()

            print("Writing out timing related file")

            # We need to print out 2 files:
            # 1) Timing report file for board level analysis use
            # 2) SDC template file for efinity's use

            self._write_report_header()
            self._write_sdc_header()
            warning_msg_list = self._generate_files(ins_to_block_map, clk2delay_map, core_delay_obj)

            write_successful = True

            if warning_msg_list:
                if is_register:
                    for wmsg in warning_msg_list:
                        self.logger.warning("{}".format(wmsg))
                        # TODO: show the warning in UI or create a new DRC
                        #issue_reg.append_issue_by_tuple(
                        #        ("timing", "report generation",
                        #        base_rule.Rule.SeverityType.warning,
                        #        "timing_sdc", wmsg))

        except IOError as excp:
            excp_msg = "{}".format(excp)
            if is_register:
                issue_reg.append_issue_by_tuple(
                    ("timing", "report generation",
                     base_rule.Rule.SeverityType.error,
                     "timing_rule_io_exception", excp_msg))

        except Exception as excp:
            excp_msg = "{}".format(excp)
            if is_register:
                issue_reg.append_issue_by_tuple(
                    ("timing", "report generation",
                     base_rule.Rule.SeverityType.error,
                     "timing_rule_exception", excp_msg))

        if not write_successful:

            if os.path.exists(self.__report_file):
                os.remove(self.__report_file)

            if os.path.exists(self.__sdc_file):
                os.remove(self.__sdc_file)

            raise writer_excp.GenerateReportException(
                excp_msg, app_excp.MsgLevel.error)

        return write_successful

    def writer_pre_requisite_fail(self):
        '''
        This is just to flag message to indicate
        that timing wasn't run due to pre-requisite (that
        it depends on.. interface info) failed.
        '''
        is_register = False
        issue_reg = None
        excp_msg = "Unable to write timing data due to failure in interface file generation"

        if self.__design is not None and self.__design.issue_reg is not None:
            issue_reg = self.__design.issue_reg
            is_register = True
        
        if is_register:
            issue_reg.append_issue_by_tuple(
                ("timing", "report generation",
                    base_rule.Rule.SeverityType.error,
                    "timing_rule_exception", excp_msg))
            
        raise writer_excp.GenerateReportException(
                excp_msg, app_excp.MsgLevel.error)
    
    def determine_core_clock_network_delay(self, core_delay_obj: DeviceCoreClockDelay):
        
        # Map of the clock name mapped to max and min delay
        clock_max_min_delay_map: Dict[str, Tuple[float, float]] = {}
        internal_clk: List[str] = []  
        
        if core_delay_obj and self._input_to_loc_map:          
            for clock_input_name in sorted(self._clkout_to_loc_map):
                loc_list= self._clkout_to_loc_map[clock_input_name]

                if clock_input_name in self._input_to_loc_map:
                    gbuf_coord_tup = self._input_to_loc_map[clock_input_name]

                    gbufx, gbufy, _ = gbuf_coord_tup

                    core_max_delay = None
                    core_min_delay = None                    

                    # Get any existing numbers, if already used in other blocks
                    if clock_input_name in clock_max_min_delay_map:
                        core_max_delay, core_min_delay = clock_max_min_delay_map[clock_input_name]

                    for loc_tup in loc_list:
                        clkoutx, clkouty, _ = loc_tup

                        core_max, core_min = core_delay_obj.determine_clk_network_delay(
                            "", gbufx, gbufy, clkoutx, clkouty)
                        
                        # We only take the max of max and min of min
                        if core_max_delay is not None:
                            core_max_delay = max(core_max_delay, core_max)
                        else:
                            core_max_delay = core_max

                        if core_min_delay is not None:                            
                            core_min_delay = min(core_min_delay, core_min)
                        else:
                            core_min_delay = core_min

                    clock_max_min_delay_map[clock_input_name] = (core_max_delay, core_min_delay)

                else:
                    # This could be an internally generated clock and we can skip
                    internal_clk.append(clock_input_name)

        return clock_max_min_delay_map, internal_clk

    def write_all_clkout_delay(self, index, core_delay_obj):
        '''
        Iterate through the clkout interface and write out the delay associated to it
        provided that the clock is sourced from periphery (input defined)
        '''
        is_write = False
        
        if core_delay_obj is not None and self._clkout_to_loc_map:
            clock_max_min_delay_map, internal_clk = self.determine_core_clock_network_delay(
                core_delay_obj)

            if internal_clk:
                self.logger.warning("Unable to determine {} core clock network delay for non-periphery defined clocks: {}".format(
                    len(internal_clk),",".join(internal_clk)))
                                        
            if clock_max_min_delay_map:

                write_successful = None

                try:
                    rptfile = open(self.get_report_file_name(), 'a')

                    write_successful = False

                    rptfile.write(
                        "\n---------- {}. Clock Network Delay Report (begin) ----------\n".format(index))

                    # Write out the info in table
                    table = PrettyTable(["Clock Pin", "Max (ns)", "Min (ns)"])

                    for clk_name in sorted(clock_max_min_delay_map):
                        delay_tuple = clock_max_min_delay_map[clk_name]
                        max_delay, min_delay = delay_tuple

                        row_list = []

                        row_list.append(clk_name)
                        row_list.append("{}".format(round(Decimal(max_delay), 3)))
                        row_list.append("{}".format(round(Decimal(min_delay), 3)))
                        
                        table.add_row(row_list)

                    # Write out the table
                    rptfile.write("\n{}\n".format(table.get_string()))

                    rptfile.write(
                        "\n---------- Clock Network Delay Report (end) ----------\n")

                    rptfile.close()

                    write_successful = True
                    is_write = True

                except Exception as excp:
                    if write_successful is not None and\
                            not write_successful:
                        rptfile.close()

                    raise excp
                
        return is_write
