'''
Copyright (C) 2017-2018 Efinix Inc. All rights reserved.

No portion of this code may be reused, modified or
distributed in any way without the expressed written
consent of Efinix Inc.

Created on Jan 3, 2018

@author: maryam
'''
import os
import sys
import datetime

# 3rd party library to display pretty table
from prettytable import PrettyTable

import xml.etree.ElementTree as et
from xml.dom import expatbuilder
from typing import Dict

from util.singleton_logger import Logger
import util.gen_util as pt_util
import util.excp as app_excp

import device.db_interface as device_dbi
import device.package as package
import device.excp as dev_excp
import design.db as des_db

import common_device.rule as base_rule
import common_device.excp as comm_excp

import common_device.osc.writer.summary as osc_sum
import common_device.gpio.writer.summary as gpio_sum
from common_device.mipi.mipi_design import MIPI
import common_device.mipi.writer.rx_summary as mipi_rx_sum
import common_device.mipi.writer.tx_summary as mipi_tx_sum
import common_device.lvds.writer.rx_summary as lvds_rx_sum
import common_device.lvds.writer.tx_summary as lvds_tx_sum
import common_device.clock_mux.writer.summary as clk_sum
import common_device.ctrl.writer.summary as ctrl_sum
import common_device.jtag.writer.summary as jtag_sum
import common_device.ddr.writer.summary as ddr_sum
import common_device.h264.writer.summary as h264_sum
import common_device.seu.writer.summary as seu_sum
import common_device.hposc.writer.summary as hposc_sum
import common_device.spi_flash.writer.summary as sflash_sum
import common_device.hyper_ram.writer.summary as hram_sum
import common_device.rclock_mux.writer.summary as rclkmux_sum

import common_device.lvds.lvds_utility as lvds_util

import common_device.hsio.writer.lvds_summary as hsio_lvds_sum
import common_device.hsio.writer.mipi_dphy_summary as hsio_mdphy_sum
from common_device.ext_flash.writer.summary import ExtFlashControllerSummary
from common_device.quad.res_service import QuadResService, QuadType

import an08_device.pll.writer.summary as an08_pll_sum
import an20_device.pll.writer.summary as an20_pll_sum

import tx60_device.osc.writer.summary as tx60_osc_sum
import tx60_device.gpio.writer.summary as tx60_gpio_sum
import tx60_device.clock_mux.writer.summary as tx60_clk_sum
import tx60_device.pll.writer.summary as tx60_pll_sum

import tx180_device.pll.writer.summary as tx180_pll_sum
import tx180_device.ddr.writer.summary as tx180_ddr_sum
import tx180_device.mipi_dphy.writer.rx_summary as tx180_mdphy_rx_sum
import tx180_device.mipi_dphy.writer.tx_summary as tx180_mdphy_tx_sum
import tx180_device.pll_ssc.writer.summary as tx180_mdphy_pll_ssc_sum

from tx375_device.quad_pcie.writer.summary import QuadPCIESummary
from tx375_device.soc.writer.summary import SOCSummaryWriter
from tx375_device.fpll.writer.summary import EfxFpllV1Summary
from tx375_device.pma_clock_mux.writer.summary import PMAClockMuxSummary
from tx375_device.rclock_mux_4to2.writer.summary import RClockMux4to2Summary
from tx375_device.rclock_mux_8to2.writer.summary import RClockMux8to2Summary
from tx375_device.lane10g.writer.summary import Lane10GSummary
from tx375_device.lane1g.writer.summary import Lane1GSummary
from tx375_device.raw_serdes.writer.summary import LaneRawSerdesSummary

import writer.excp as writer_excp

from _version import __version__


sys.path.append(os.path.join(os.path.dirname(__file__), '..'))


class SummaryReport(object):
    '''
    Class for generating summary report.
    '''

    # Create a map of block names to the offical names used in documentation
    # This only contains block that should be shown to user.
    xmlreport_block_type2username_map = {
        device_dbi.DeviceDBService.BlockType.GPIO: "GPIO",
        device_dbi.DeviceDBService.BlockType.PLL: "PLL",
        device_dbi.DeviceDBService.BlockType.OSC: "Oscillator",
        device_dbi.DeviceDBService.BlockType.LVDS_RX: "LVDS RX",
        device_dbi.DeviceDBService.BlockType.LVDS_TX: "LVDS TX",
        device_dbi.DeviceDBService.BlockType.MIPI_RX: "MIPI RX",
        device_dbi.DeviceDBService.BlockType.MIPI_TX: "MIPI TX",
        device_dbi.DeviceDBService.BlockType.JTAG: "JTAG User TAP",
        device_dbi.DeviceDBService.BlockType.DDR: "DDR",
        device_dbi.DeviceDBService.BlockType.H264: "H.264 Encoder",
        device_dbi.DeviceDBService.BlockType.HSIO: "HSIO",
        device_dbi.DeviceDBService.BlockType.HPOSC: "Oscillator",
        device_dbi.DeviceDBService.BlockType.SPI_FLASH: "SPI Flash",
        device_dbi.DeviceDBService.BlockType.HYPER_RAM: "HyperRAM",
        device_dbi.DeviceDBService.BlockType.QUAD_PCIE: "PCI Express",
        device_dbi.DeviceDBService.BlockType.SOC: "Quad-Core Risc-V Processor",
        device_dbi.DeviceDBService.BlockType.QUAD: "Quad"
    }

    # Instead of using the enum order in DeviceDBService, we use
    # the following list to fix the order. Only list out the
    # blocks that should appear in the summary.
    # NOTE: They should also appear in the DeviceDBService.block_type2str_map
    blk_type_order = \
        [
            device_dbi.DeviceDBService.BlockType.GPIO,
            device_dbi.DeviceDBService.BlockType.PLL,
            device_dbi.DeviceDBService.BlockType.OSC,
            device_dbi.DeviceDBService.BlockType.HPOSC,
            device_dbi.DeviceDBService.BlockType.LVDS_RX,
            device_dbi.DeviceDBService.BlockType.LVDS_TX,
            device_dbi.DeviceDBService.BlockType.LVDS_BG,
            device_dbi.DeviceDBService.BlockType.MIPI_RX,
            device_dbi.DeviceDBService.BlockType.MIPI_TX,
            device_dbi.DeviceDBService.BlockType.CLKMUX,
            device_dbi.DeviceDBService.BlockType.CONTROL,
            device_dbi.DeviceDBService.BlockType.SEU,
            device_dbi.DeviceDBService.BlockType.JTAG,
            device_dbi.DeviceDBService.BlockType.DDR,
            device_dbi.DeviceDBService.BlockType.H264,
            device_dbi.DeviceDBService.BlockType.HSIO,
            device_dbi.DeviceDBService.BlockType.HSIO_BG,
            device_dbi.DeviceDBService.BlockType.HVIO_POC,
            device_dbi.DeviceDBService.BlockType.SPI_FLASH,
            device_dbi.DeviceDBService.BlockType.HYPER_RAM,
            device_dbi.DeviceDBService.BlockType.QUAD_PCIE,
            device_dbi.DeviceDBService.BlockType.SOC,
            device_dbi.DeviceDBService.BlockType.QUAD,
            # We put static clockmux at the end since if there's
            # unrouted nets, it will raise exception and stop
            # writing out report further to allow user sees the
            # report summary on the clkmux usage
            device_dbi.DeviceDBService.BlockType.RCLKMUX,
            device_dbi.DeviceDBService.BlockType.PMA_CLKMUX,
            device_dbi.DeviceDBService.BlockType.RCLKMUX_V2_21,
            device_dbi.DeviceDBService.BlockType.RCLKMUX_V2_41,
        ]

    def __init__(self, device_db, design_db: des_db.PeriDesign,
                 dir_name, project_name):
        '''
        constructor
        '''
        self.logger = Logger

        # it needs to access the design configuration
        # in order to write out the file.
        self.__design = design_db

        # contains the device db info
        self.__device = device_db

        # Get the project name from design
        self.__project_name = project_name

        self.__output_dir = dir_name

        # List of block types that are applicable. super set
        # of the block_section/block_summary
        self.__all_block_types = []

        # Check that the directory is valid
        if os.path.isdir(dir_name):
            # The filename to dump out the interace configuration data
            self.__filename = dir_name + "/" + self.__project_name + ".pt.rpt"

            # summary report file in xml used for Efinity to display resources
            # info
            self.__summary_xml_file = dir_name + "/" + self.__project_name + ".pt.rpt.xml"

        else:
            raise ValueError("Invalid directory {}".format(dir_name))

        # List of block services required for the device. This is
        # to be able to tell which block to use for writing out interface config.
        # It is a map of block type (defined in DBInterface) to
        # the Block InterfaceConfig (peri_block either in common or device
        # specific)
        self.__blocks_summary = {}

        # A map of block type (that exists in the device) mapped to
        # the section number
        self.__blocks_section = {}

        # The index in TOC for printing out design issues. None if there's no
        # issue
        self.__design_issue_index = None

        # The index for the ext flash controller
        self._ext_flash_ctrl_index = None

    def get_file_name(self):
        """
        Get full file name
        :return: Generated report filename
        """
        return self.__filename

    def get_xml_file_name(self):
        """
        Get the xml summary report filename
        :return: Generated simple summary report filename for Efinity usage
        """
        return self.__summary_xml_file

    def _generate_xml_tree(self, blk_to_usage_map):
        '''
        Generate the XML Tree containing the required data

        :param blk_to_usage_map: a map of the block name to he usage string to be printed
        :return ElementTree root

        '''
        root = et.Element("efx:tool_report")

        # Add a comment that indicates the file header
        now = datetime.datetime.now()

        date_str = now.strftime("%Y-%m-%d %H:%M")

        # Add the required attributes
        root.attrib["xmlns:efx"] = "http://www.efinixinc.com/tool_report"
        root.attrib["date"] = date_str
        root.attrib["project"] = self.__project_name
        root.attrib["xmlns:xsi"] = "http://www.w3.org/2001/XMLSchema-instance"
        root.attrib["xsi:schemaLocation"] = "http://www.efinixinc.com/tool_report tool_report.xsd"

        # Create the group first
        group_elem = et.SubElement(
            root, "efx:group", name="Periphery Resource")

        for blk_name in sorted(blk_to_usage_map.keys()):
            et.SubElement(group_elem, "efx:group_data",
                          name=blk_name, severity="info",
                          value=blk_to_usage_map[blk_name])

        return root

    def _identify_blocks(self):
        '''
        Initiate the list of periphery block LPF writer.
        This is determine by the block type available in the
        device.

        :param block_names: list of block names that are defined
                            for this device
        '''

        # Go through all the block to generate the block
        # pcr definition
        # device_db = device_dbi.DeviceDBService(self.__device)

        # Get the list of block types from the dbi
        section = 7
        # TODO: Clean up once the HPOSC design is ready. Only writer out 1 osc

        is_quad_read = False
        # for btype in
        # sorted(device_dbi.DeviceDBService.block_type2str_map.keys()):
        for btype in self.blk_type_order:
            if btype not in device_dbi.DeviceDBService.block_type2str_map:
                self.logger.debug(
                    "Invalid type captured in the blk_type_order: {}".format(btype))
                continue

            bname = device_dbi.DeviceDBService.block_type2str_map[btype]

            block_obj = None

            # Skip if the device does not have this block and also
            # if the block type is in the device but not listed
            # in the resource map
            if self.__device.find_block(bname) is None:
                continue
            ins_names, _ = self.__device.get_resources_by_type(
                device_dbi.DeviceDBService.block_type2str_map[btype])
            # We let gpio pass through since there are some device where
            # it has no dedicated GPIO but has shared GPIO resource (i.e. HSIO).
            # So, we don't want to miss that out in summary
            if not ins_names and bname != "gpio":
                continue

            self.__all_block_types.append(btype)

            if btype == device_dbi.DeviceDBService.BlockType.GPIO:
                if self.__design.is_block_supported(des_db.PeriDesign.BlockType.comp_gpio):
                    block_obj = tx60_gpio_sum.GPIOSummaryComplex(
                        self.__device, bname, section)
                elif self.__design.is_block_supported(des_db.PeriDesign.BlockType.adv_l2_gpio) or \
                        self.__design.is_block_supported(des_db.PeriDesign.BlockType.gpio):
                    block_obj = gpio_sum.GPIOSummary(
                        self.__device, bname, section)
                else:
                    msg = 'Unable to determine GPIO type'

                    raise dev_excp.BlockDoesNotExistException(
                        msg, app_excp.MsgLevel.error)

            elif btype == device_dbi.DeviceDBService.BlockType.OSC:
                if self.__design.is_block_supported(des_db.PeriDesign.BlockType.adv_osc):
                    if self.__device.osc_type != device_dbi.DeviceDBService.BlockType.HPOSC:
                        block_obj = tx60_osc_sum.OscillatorSummaryAdv(
                            self.__device, bname, section)
                    else:
                        continue

                elif self.__design.is_block_supported(des_db.PeriDesign.BlockType.osc):
                    block_obj = osc_sum.OscillatorSummary(
                        self.__device, bname, section)

                else:
                    msg = 'Unable to determine OSC type'
                    raise dev_excp.BlockDoesNotExistException(
                        msg, app_excp.MsgLevel.error)

            elif btype == device_dbi.DeviceDBService.BlockType.HPOSC:
                block_obj = hposc_sum.HPOSCSummary(
                    self.__device, bname, section)

            elif btype == device_dbi.DeviceDBService.BlockType.PLL:
                # This is going to be device dependent
                if self.__design.is_block_supported(des_db.PeriDesign.BlockType.efx_fpll_v1):
                    block_obj = EfxFpllV1Summary(
                        self.__device, bname, section)
                elif self.__design.is_block_supported(des_db.PeriDesign.BlockType.efx_pll_v3_comp):
                    block_obj = tx180_pll_sum.PLLSummaryV3Complex(
                        self.__device, bname, section)
                elif self.__design.is_block_supported(des_db.PeriDesign.BlockType.comp_pll):
                    block_obj = tx60_pll_sum.PLLSummaryComplex(
                        self.__device, bname, section)
                elif self.__design.is_block_supported(des_db.PeriDesign.BlockType.adv_pll):
                    block_obj = an20_pll_sum.PLLSummary(
                        self.__device, bname, section)
                # elif isinstance(pll_reg, pll_des.PLLRegistry):
                elif self.__design.is_block_supported(des_db.PeriDesign.BlockType.pll):
                    block_obj = an08_pll_sum.PLLSummary(
                        self.__device, bname, section)
                else:
                    msg = 'Unable to determine PLL type'

                    raise dev_excp.BlockDoesNotExistException(
                        msg, app_excp.MsgLevel.error)

            elif btype == device_dbi.DeviceDBService.BlockType.MIPI_RX:
                if self.__design.is_block_supported(des_db.PeriDesign.BlockType.mipi_hard_dphy):
                    block_obj = tx180_mdphy_rx_sum.MIPIRxSummaryAdvance(
                        self.__device, bname, section)

                else:
                    block_obj = mipi_rx_sum.MIPIRxSummary(
                        self.__device, bname, section)

            elif btype == device_dbi.DeviceDBService.BlockType.MIPI_TX:
                if self.__design.is_block_supported(des_db.PeriDesign.BlockType.mipi_hard_dphy):
                    self.__blocks_section[section] = device_dbi.DeviceDBService.BlockType.MIPI_TX_ADV
                    self.__blocks_summary[device_dbi.DeviceDBService.BlockType.MIPI_TX_ADV] = tx180_mdphy_tx_sum.MIPITxSummaryAdvance(
                        self.__device, bname, section)
                    section += 1

                    self.__blocks_section[section] = device_dbi.DeviceDBService.BlockType.MIPI_TX_ADV_PLL_SSC
                    self.__blocks_summary[device_dbi.DeviceDBService.BlockType.MIPI_TX_ADV_PLL_SSC] = tx180_mdphy_pll_ssc_sum.PLLSSCSummaryWriter(
                        self.__device, bname, section)
                    section += 1

                    # Reset so nothing new is added at end of loop
                    block_obj = None
                else:
                    block_obj = mipi_tx_sum.MIPITxSummary(
                        self.__device, bname, section)

            elif btype == device_dbi.DeviceDBService.BlockType.LVDS_TX:
                block_obj = lvds_tx_sum.LVDSTxSummary(
                    self.__device, bname, section)
            elif btype == device_dbi.DeviceDBService.BlockType.LVDS_RX:
                block_obj = lvds_rx_sum.LVDSRxSummary(
                    self.__device, bname, section)
            elif btype == device_dbi.DeviceDBService.BlockType.CLKMUX:
                if self.__device.clkmux_type == device_dbi.DeviceDBService.BlockType.CLKMUX:
                    block_obj = clk_sum.ClockMuxSummary(
                        self.__device, bname, section)
                else:
                    block_obj = tx60_clk_sum.ClockMuxSummaryAdvance(
                        self.__device, bname, section)

            elif btype == device_dbi.DeviceDBService.BlockType.CONTROL:
                block_obj = ctrl_sum.ControlSummary(
                    self.__device, bname, section)
            elif btype == device_dbi.DeviceDBService.BlockType.JTAG:
                block_obj = jtag_sum.JTAGSummary(self.__device, bname, section)
            elif btype == device_dbi.DeviceDBService.BlockType.DDR:
                if self.__design.is_block_supported(des_db.PeriDesign.BlockType.adv_ddr):
                    block_obj = tx180_ddr_sum.DDRSummaryAdvance(self.__device, bname, section)
                else:
                    block_obj = ddr_sum.DDRSummary(self.__device, bname, section)
            elif btype == device_dbi.DeviceDBService.BlockType.H264:
                block_obj = h264_sum.H264Summary(self.__device, bname, section)

            elif btype == device_dbi.DeviceDBService.BlockType.HSIO:
                # Don't need to check for HSIO_MAX since we will use HSIO
                # and have them combined.
                # Create all 5 types of summary services
                type_list = [device_dbi.DeviceDBService.BlockType.HSIO_LVDS_RX,
                             device_dbi.DeviceDBService.BlockType.HSIO_LVDS_TX,
                             device_dbi.DeviceDBService.BlockType.HSIO_LVDS_BIDIR,
                             device_dbi.DeviceDBService.BlockType.HSIO_MIPI_DPHY_RX,
                             device_dbi.DeviceDBService.BlockType.HSIO_MIPI_DPHY_TX]

                for hsio_type in type_list:
                    if hsio_type == device_dbi.DeviceDBService.BlockType.HSIO_LVDS_RX:
                        block_obj = hsio_lvds_sum.HSIOLVDSRxSummary(
                            self.__device, bname, section)
                    elif hsio_type == device_dbi.DeviceDBService.BlockType.HSIO_LVDS_TX:
                        block_obj = hsio_lvds_sum.HSIOLVDSTxSummary(
                            self.__device, bname, section)
                    elif hsio_type == device_dbi.DeviceDBService.BlockType.HSIO_LVDS_BIDIR:
                        block_obj = hsio_lvds_sum.HSIOLVDSBidirSummary(
                            self.__device, bname, section)
                    elif hsio_type == device_dbi.DeviceDBService.BlockType.HSIO_MIPI_DPHY_RX:
                        block_obj = hsio_mdphy_sum.HSIOMDPHYRxSummary(
                            self.__device, bname, section)
                    else:
                        block_obj = hsio_mdphy_sum.HSIOMDPHYTxSummary(
                            self.__device, bname, section)

                    if block_obj is not None:
                        self.__blocks_section[section] = hsio_type
                        self.__blocks_summary[hsio_type] = block_obj
                        section += 1

                # Reset so nothing new is added at end of loop
                block_obj = None

            elif btype == device_dbi.DeviceDBService.BlockType.SEU:
                block_obj = seu_sum.SEUSummary(self.__device, bname, section)

            elif btype == device_dbi.DeviceDBService.BlockType.SPI_FLASH:
                block_obj = sflash_sum.SPIFlashSummary(
                    self.__device, bname, section)

            elif btype == device_dbi.DeviceDBService.BlockType.HYPER_RAM:
                block_obj = hram_sum.HyperRAMSummary(
                    self.__device, bname, section)

            elif btype == device_dbi.DeviceDBService.BlockType.RCLKMUX:
                block_obj = rclkmux_sum.RClockMuxSummary(
                    self.__device, bname, section)

            elif btype == device_dbi.DeviceDBService.BlockType.QUAD_PCIE or\
                btype == device_dbi.DeviceDBService.BlockType.QUAD:
                type_list = [device_dbi.DeviceDBService.BlockType.QUAD_PCIE,
                             device_dbi.DeviceDBService.BlockType.QUAD_LANE_10G,
                             device_dbi.DeviceDBService.BlockType.QUAD_LANE_1G,
                             device_dbi.DeviceDBService.BlockType.QUAD_LANE_RAW_SERDES]

                if is_quad_read:
                    continue

                for quad_type in type_list:
                    quad_blk_obj = None
                    if quad_type == device_dbi.DeviceDBService.BlockType.QUAD_PCIE and\
                            self.__design.quad_pcie_reg is not None:
                        quad_blk_obj = QuadPCIESummary(
                            self.__device, bname, section)
                    elif quad_type == device_dbi.DeviceDBService.BlockType.QUAD_LANE_10G and\
                            self.__design.lane_10g_reg is not None:                    
                        quad_blk_obj = Lane10GSummary(
                            self.__device, bname, section)
                    elif quad_type == device_dbi.DeviceDBService.BlockType.QUAD_LANE_1G and\
                            self.__design.lane_1g_reg is not None:                    
                        quad_blk_obj = Lane1GSummary(
                            self.__device, bname, section)
                    elif quad_type == device_dbi.DeviceDBService.BlockType.QUAD_LANE_RAW_SERDES and\
                            self.__design.raw_serdes_reg is not None:
                        quad_blk_obj = LaneRawSerdesSummary(
                            self.__device, bname, section)

                    if quad_blk_obj is not None and \
                        quad_type not in self.__blocks_summary:
                        self.__blocks_section[section] = quad_type
                        self.__blocks_summary[quad_type] = quad_blk_obj
                        section += 1

                # Reset so nothing new is added at end of loop
                block_obj = None
                is_quad_read = True


            elif btype == device_dbi.DeviceDBService.BlockType.SOC:
                block_obj = SOCSummaryWriter(self.__device, bname, section)

            elif btype == device_dbi.DeviceDBService.BlockType.PMA_CLKMUX:
                block_obj = PMAClockMuxSummary(self.__device, bname, section)

            elif btype == device_dbi.DeviceDBService.BlockType.RCLKMUX_V2_21:
                block_obj = RClockMux4to2Summary(self.__device, bname, section)

            elif btype == device_dbi.DeviceDBService.BlockType.RCLKMUX_V2_41:
                block_obj = RClockMux8to2Summary(self.__device, bname, section)

            elif btype != device_dbi.DeviceDBService.BlockType.LVDS_BG and \
                    btype != device_dbi.DeviceDBService.BlockType.HSIO_BG and \
                    btype != device_dbi.DeviceDBService.BlockType.HVIO_POC and \
                    btype != device_dbi.DeviceDBService.BlockType.QUAD and \
                    btype != device_dbi.DeviceDBService.BlockType.SERDES_WRAP and\
                    btype != device_dbi.DeviceDBService.BlockType.TSD and \
                    btype != device_dbi.DeviceDBService.BlockType.TSD_CB:

                # Undefined. Throw exception
                msg = 'Unable to find block {}'.format(bname)

                raise dev_excp.BlockDoesNotExistException(
                    msg, app_excp.MsgLevel.error)

            if block_obj is not None:
                self.__blocks_section[section] = btype
                self.__blocks_summary[btype] = block_obj
                section += 1


    def generate_table_of_contents(self, outfile):
        '''
        Generate the report TOC

        :param outfile: file handle that has already been opened
        '''

        outfile.write("\n---------- Table of Contents (begin) ----------\n")

        outfile.write("   1. Periphery Usage Summary\n")
        outfile.write("   2. Generated Output Files\n")
        outfile.write("   3. I/O Banks Summary\n")
        outfile.write("   4. Global Connection Summary\n")
        outfile.write("   5. Clock Region Usage Summary\n")
        outfile.write("   6. Dual-Function Configuration Pin Usage\n")

        # The table contents depends on which device is being run to indicate
        # the periphery block that exists
        # for btype in sorted(device_dbi.DeviceDBService.block_type2str_map.keys()):
        # Sort based on the section numbers

        for section in sorted(self.__blocks_section.keys()):
            btype = self.__blocks_section[section]

            # bname = device_dbi.DeviceDBService.block_type2str_map[btype]

            # Skip if the device does not have this block
            # if self.__device.find_block(bname) is None:
            #    continue

            # Skip if the block is not defined in the device or no
            # resource of the block type exists in device
            if btype not in self.__blocks_summary:
                # raise ValueError(
                #    "Internal Error with blk_type {} not existed in section map".format(
                #        btype))
                continue

            #index = self.__blocks_section[btype]
            index = section

            if btype == device_dbi.DeviceDBService.BlockType.GPIO:
                outfile.write("   {}. GPIO Usage Summary\n".format(index))
            elif btype == device_dbi.DeviceDBService.BlockType.PLL:
                outfile.write("   {}. PLL Usage Summary\n".format(index))
            elif btype == device_dbi.DeviceDBService.BlockType.OSC or\
                    btype == device_dbi.DeviceDBService.BlockType.HPOSC:
                outfile.write(
                    "   {}. Oscillator Usage Summary\n".format(index))
            elif btype == device_dbi.DeviceDBService.BlockType.MIPI_RX:
                outfile.write("   {}. MIPI Rx Usage Summary\n".format(index))
            elif btype == device_dbi.DeviceDBService.BlockType.MIPI_TX or \
                    btype == device_dbi.DeviceDBService.BlockType.MIPI_TX_ADV:
                outfile.write("   {}. MIPI Tx Usage Summary\n".format(index))
            elif btype == device_dbi.DeviceDBService.BlockType.MIPI_TX_ADV_PLL_SSC:
                outfile.write("   {}. PLL SSC Usage Summary\n".format(index))
            elif btype == device_dbi.DeviceDBService.BlockType.LVDS_RX:
                outfile.write("   {}. LVDS Rx Usage Summary\n".format(index))
            elif btype == device_dbi.DeviceDBService.BlockType.LVDS_TX:
                outfile.write("   {}. LVDS Tx Usage Summary\n".format(index))
            elif btype == device_dbi.DeviceDBService.BlockType.CLKMUX:
                outfile.write("   {}. Clock Mux Usage Summary\n".format(index))
            elif btype == device_dbi.DeviceDBService.BlockType.CONTROL:
                outfile.write(
                    "   {}. Configuration Control Usage Summary\n".format(index))
            elif btype == device_dbi.DeviceDBService.BlockType.JTAG:
                outfile.write("   {}. JTAG Usage Summary\n".format(index))
            elif btype == device_dbi.DeviceDBService.BlockType.DDR:
                outfile.write("   {}. DDR Usage Summary\n".format(index))
            elif btype == device_dbi.DeviceDBService.BlockType.H264:
                outfile.write("   {}. H.264 Usage Summary\n".format(index))
            elif btype == device_dbi.DeviceDBService.BlockType.SEU:
                outfile.write(
                    "   {}. Configuration SEU Detection Usage Summary\n".format(index))
            elif btype == device_dbi.DeviceDBService.BlockType.HSIO_LVDS_RX:
                outfile.write("   {}. LVDS Rx Usage Summary\n".format(index))
            elif btype == device_dbi.DeviceDBService.BlockType.HSIO_LVDS_TX:
                outfile.write("   {}. LVDS Tx Usage Summary\n".format(index))
            elif btype == device_dbi.DeviceDBService.BlockType.HSIO_LVDS_BIDIR:
                outfile.write(
                    "   {}. Bidirectional LVDS Usage Summary\n".format(index))
            elif btype == device_dbi.DeviceDBService.BlockType.HSIO_MIPI_DPHY_RX:
                outfile.write(
                    "   {}. MIPI RX Lane Usage Summary\n".format(index))
            elif btype == device_dbi.DeviceDBService.BlockType.HSIO_MIPI_DPHY_TX:
                outfile.write(
                    "   {}. MIPI TX Lane Usage Summary\n".format(index))
            elif btype == device_dbi.DeviceDBService.BlockType.SPI_FLASH:
                outfile.write(
                    "   {}. SPI Flash Usage Summary\n".format(index))
            elif btype == device_dbi.DeviceDBService.BlockType.HYPER_RAM:
                outfile.write(
                    "   {}. HyperRAM Usage Summary\n".format(index))
            elif btype == device_dbi.DeviceDBService.BlockType.RCLKMUX:
                outfile.write(
                    "   {}. Regional Clock Mux Usage Summary\n".format(index))
            elif btype == device_dbi.DeviceDBService.BlockType.QUAD_PCIE:
                outfile.write(
                    "   {}. PCI Express Usage Summary\n".format(index))
            elif btype == device_dbi.DeviceDBService.BlockType.SOC:
                outfile.write(
                    "   {}. Quad-Core Risc-V Processor Usage Summary\n".format(index)
                )
            elif btype == device_dbi.DeviceDBService.BlockType.QUAD_LANE_10G:
                outfile.write(
                    "   {}. Ethernet XGMII Usage Summary\n".format(index))
            elif btype == device_dbi.DeviceDBService.BlockType.PMA_CLKMUX:
                outfile.write(
                    "   {}. PMA Clock Mux Usage Summary\n".format(index))
            elif btype == device_dbi.DeviceDBService.BlockType.RCLKMUX_V2_21:
                outfile.write(
                    "   {}. Regional Clock Mux (2-To-1) Usage Summary\n".format(index))
            elif btype == device_dbi.DeviceDBService.BlockType.RCLKMUX_V2_41:
                outfile.write(
                    "   {}. Regional Clock Mux (4-To-1) Usage Summary\n".format(index))
            elif btype == device_dbi.DeviceDBService.BlockType.QUAD_LANE_RAW_SERDES:
                outfile.write(
                    "   {}. PMA Direct Usage Summary\n".format(index))
            elif btype == device_dbi.DeviceDBService.BlockType.QUAD_LANE_1G:
                outfile.write(
                    "   {}. Ethernet SGMII Usage Summary\n".format(index))


        # Add the ext flash controller section close to the end
        if self.is_ext_flash_controller_section_exists():
            index += 1
            self._ext_flash_ctrl_index = index
            outfile.write("   {}. External Flash Control Summary\n".format(
                self._ext_flash_ctrl_index))

        # Add the messages in the issue registry if non-empty
        if self.__design.issue_reg is not None and \
                self.__design.issue_reg.get_issue_count() > 0:
            self.__design_issue_index = index + 1
            outfile.write("   {}. Design Issues\n".format(
                self.__design_issue_index))

        outfile.write("---------- Table of Contents (end) ------------\n")

    def generate_file_header(self, outfile):
        '''
        Generates the file header

        :param outfile: file handle that has already been opened
        '''

        now = datetime.datetime.now()

        outfile.write("\nEfinity Interface Designer Report\n")
        outfile.write("Version: {}\n".format(__version__))
        outfile.write("Date: {}\n".format(now.strftime("%Y-%m-%d %H:%M")))

        outfile.write("\n{}\n".format(pt_util.get_copyright_string()))

        # Device and Design Names
        outfile.write("\nDevice: {}\n".format(
            self.__device.get_device_name()))
        outfile.write("Project: {}\n".format(
            self.__project_name))

        # Package Name
        dev_pkg = self.__device.get_package()
        pkg_status = dev_pkg.get_status_str()
        outfile.write("\nPackage: {} ({})\n".format(
            dev_pkg.get_package_name(),
            pkg_status))

        if dev_pkg.get_status() != package.DevicePackage.PackageStatus.final:
            outfile.write("         NOTE: The package data is not final.\n")

        # Generate timing model name
        timing_model = self.__device.get_timing_model()
        if timing_model is not None:
            timing_model.write_selected_timing_model_name(outfile)

        # PT-519: Write out the configuration mode selected
        config_mode_str = self.__device.get_configuration_mode_name()
        if config_mode_str != "":
            outfile.write(
                "Configuration Mode: {}\n".format(
                    config_mode_str))

        # Table of Contents
        self.generate_table_of_contents(outfile)

    def _get_lvds_usage(self, ins_name, blk_type, blk_name, lvds_rx_count, lvds_tx_count):
        '''
        Count the usage for LVDS which also can be configured as GPIO
        :return:
        '''
        #self.logger.debug("Reading ins_name {}: blk {}".format(
        #    ins_name, blk_name))

        capacity, lvds_type = lvds_util.get_instance_lvds_capacity_type(
            ins_name, self.__device)
        #self.logger.debug("Capacity : {}".format(capacity))

        key_name = None
        if lvds_type:
            key_name = "lvds"
        else:
            key_name = "gpio"

        # Keep track of LVDS usage based on type
        if blk_type == device_dbi.DeviceDBService.BlockType.LVDS_RX:
            if key_name in lvds_rx_count:
                cur_count = lvds_rx_count[key_name]
                lvds_rx_count[key_name] = cur_count + 1
            else:
                lvds_rx_count[key_name] = 1

        else:
            if key_name in lvds_tx_count:
                cur_count = lvds_tx_count[key_name]
                lvds_tx_count[key_name] = cur_count + 1
            else:
                lvds_tx_count[key_name] = 1

        return capacity

    def _get_hsio_usage(self, ins_name, blk_type, blk_name, hsio_count):

        #self.logger.debug("Reading ins_name {}: blk {}".format(
        #    ins_name, blk_name))

        dbi = device_dbi.DeviceDBService(self.__design.device_db)
        hsio_dev_svc = dbi.get_block_service(
            device_dbi.DeviceDBService.BlockType.HSIO)

        capacity, non_gpio_type = hsio_dev_svc.get_instance_capacity_type(
            ins_name)

        #self.logger.debug("Capacity : {}".format(capacity))

        key_name = None
        if non_gpio_type:
            key_name = "lvds or mipi lane"
        else:
            key_name = "gpio"

        if key_name in hsio_count:
            cur_count = hsio_count[key_name]
            hsio_count[key_name] = cur_count + 1
        else:
            hsio_count[key_name] = 1

        return capacity

    def _get_quad_usage(self, ins_name: str, quad_count: Dict[str, int]):
        capacity = 1

        res_svc = QuadResService()
        res_svc.build(self.__design)
        assert res_svc.lane10g_devsvc is not None

        capacity, ins_type = res_svc.lane10g_devsvc.get_instance_capacity_type(ins_name)

        if capacity < 0 or ins_type == "":
            return capacity

        inst_list = res_svc.find_resource_user(ins_name)

        if len(inst_list) <= 0:
            return capacity

        # Instance should have the same protocol
        param = "quad_type"
        inst = inst_list[0]

        quad_type = getattr(inst, param)
        display_name = ""

        match quad_type:
            case QuadType.quad_pcie:
                display_name = "PCI Express"
            case QuadType.lane_10g:
                display_name = "Ethernet XGMII"
            case QuadType.lane_1g:
                display_name = "Ethernet SGMII"
            case QuadType.raw_serdes:
                display_name = "PMA Direct"

        if display_name == "":
            return capacity

        cur_count = quad_count.get(display_name, 0)
        quad_count[display_name] = cur_count + 1

        return capacity

    def _get_mipi_hard_dphy_tx_usage(self, ins_name: str, mipi_tx_count: Dict[str, int]):
        '''
        :param ins_name: Device instance name
        :param mipi_tx_count: To be filled, key of string to print mapped
                    to the number of design configured instance count
        '''
        mipi_dphy_tx_reg = self.__design.mipi_hard_dphy_reg
        pll_ssc_reg = self.__design.pll_ssc_reg

        key_name = ""
        capacity = 1

        if mipi_dphy_tx_reg is not None:

            des_ins = mipi_dphy_tx_reg.get_inst_by_device_name(ins_name)
            if des_ins is not None and des_ins.ops_type == MIPI.OpsType.op_tx:
                key_name = "MIPI DPHY Tx"

        if pll_ssc_reg is not None:
            if pll_ssc_reg.is_device_used(ins_name):
                key_name = "PLL SSC"

        if key_name != "":
            if key_name in mipi_tx_count:
                cur_count = mipi_tx_count[key_name]
                mipi_tx_count[key_name] = cur_count + 1
            else:
                mipi_tx_count[key_name] = 1

        return capacity

    def generate_usage_summary(self, outfile, ins_to_blk_map):
        '''
        Generate the summary of periphery block usage.

        :param outfile: file handle that has already been opened
        :param ins_to_blk_map: A map of instance name the block that
                        it instantiates. The name includes flatten
                        names
        :return a map of the block name to the usage string
        '''

        # Map of block name to the count that it has
        # been instantiated
        blk_ins_count = {}
        ins_names_read = []

        # map of raw block name to the user name from xmlreport_block_type2username_map
        # of the block that is only required
        raw2user_names_map = {}

        # Map fo block name to the usage string
        blk_to_usage_str_map = {}

        lvds_rx_count = {}
        lvds_tx_count = {}
        hsio_count = {}
        mipi_tx_count = {}
        quad_count = {}
        quad_pcie_count = {}

        for ins_name, blk_name in ins_to_blk_map.items():

            if ins_name in ins_names_read:
                continue
            else:
                ins_names_read.append(ins_name)

            # If this is lvds instance, then we need
            # to consider the fact that it has 2 instances
            # that makes up 1 count if in lvttl mode
            if blk_name in device_dbi.DeviceDBService.block_str2type_map:
                blk_type = device_dbi.DeviceDBService.block_str2type_map[blk_name]

                if blk_type == device_dbi.DeviceDBService.BlockType.LVDS_RX or\
                        blk_type == device_dbi.DeviceDBService.BlockType.LVDS_TX:

                    capacity = self._get_lvds_usage(ins_name, blk_type, blk_name, lvds_rx_count,
                                                    lvds_tx_count)

                elif blk_type == device_dbi.DeviceDBService.BlockType.HSIO or \
                    blk_type == device_dbi.DeviceDBService.BlockType.HSIO_MAX:

                    capacity = self._get_hsio_usage(
                        ins_name, blk_type, blk_name, hsio_count)

                    # In user report, we name hsio_max as hsio (not differntiating)
                    blk_name = device_dbi.DeviceDBService.block_type2str_map[
                    device_dbi.DeviceDBService.BlockType.HSIO]

                elif blk_type == device_dbi.DeviceDBService.BlockType.MIPI_TX and\
                    (self.__design.mipi_hard_dphy_reg is not None or \
                     self.__design.pll_ssc_reg is not None):

                    # This is only applicable for MIPI Hard DPHY
                    capacity = self._get_mipi_hard_dphy_tx_usage(
                        ins_name, mipi_tx_count)

                elif blk_type == device_dbi.DeviceDBService.BlockType.QUAD:
                    capacity = self._get_quad_usage(ins_name, quad_count)
                elif blk_type == device_dbi.DeviceDBService.BlockType.QUAD_PCIE:
                    capacity = self._get_quad_usage(ins_name, quad_pcie_count)
                else:
                    capacity = 1

                # Count usage capacity for use with percentage usage
                if blk_name not in blk_ins_count:
                    blk_ins_count[blk_name] = capacity
                else:
                    cur_count = blk_ins_count[blk_name]
                    blk_ins_count[blk_name] = cur_count + capacity

        outfile.write(
            "\n---------- 1. Periphery Usage Summary (begin) ----------\n")

        # Iterate through the list to add blocks which does not have
        # any instance used at all. We're using blocks_section as a reference
        # since it will also include blocks that does not have a dedicated
        # summary (i.e. LVDS_BG)
        for bname, btype in device_dbi.DeviceDBService.block_str2type_map.items():

            if btype in self.xmlreport_block_type2username_map:
                raw2user_names_map[bname] = self.xmlreport_block_type2username_map[btype]

            if bname not in blk_ins_count and\
                    btype in self.__all_block_types:
                blk_ins_count[bname] = 0

        # Some of the block name
        sorted_block_order = {}
        for block_name in blk_ins_count.keys():
            printed_bname = block_name

            if block_name == device_dbi.DeviceDBService.block_type2str_map[
                    device_dbi.DeviceDBService.BlockType.HPOSC]:

                # Use the osc name instead of hposc
                printed_bname = device_dbi.DeviceDBService.block_type2str_map[
                    device_dbi.DeviceDBService.BlockType.OSC]

            sorted_block_order[printed_bname] = block_name

        # Write out the count
        for printed_bname in sorted(sorted_block_order.keys(),
                                    key=pt_util.natural_sort_key_for_list):
            block_name = sorted_block_order[printed_bname]

            used_ins = blk_ins_count[block_name]

            total_blk_count = self.__device.get_num_block_instances(
                block_name)

            # Combine HSIO and HSIO_MAX count for device that supports both
            if block_name == device_dbi.DeviceDBService.block_type2str_map[
                    device_dbi.DeviceDBService.BlockType.HSIO] and \
                        self.__device.find_block(device_dbi.DeviceDBService.block_type2str_map[
                    device_dbi.DeviceDBService.BlockType.HSIO_MAX]) is not None:
                hsio_max_blk_count = self.__device.get_num_block_instances(
                    device_dbi.DeviceDBService.block_type2str_map[
                    device_dbi.DeviceDBService.BlockType.HSIO_MAX])

                total_blk_count += hsio_max_blk_count

            if total_blk_count > 0:
                percentage = (used_ins / total_blk_count) * 100
            else:
                # This should not happen
                self.logger.debug(
                    "Invalid total blk count 0 for {}".format(block_name))
                percentage = 0

            percentage = float("{0:.2f}".format(percentage))

            outfile.write(
                "{}: {} / {} ({}%)\n".format(
                    printed_bname, used_ins,
                    total_blk_count, percentage))

            # Translate the block name first
            if block_name in raw2user_names_map:
                translate_blk_name = raw2user_names_map[block_name]

                if translate_blk_name not in blk_to_usage_str_map:
                    blk_to_usage_str_map[translate_blk_name] = "{} / {}".format(
                        used_ins, total_blk_count)

            # For LVDS, we need to further differentiate
            # between LVDS and GPIO mode count
            if lvds_tx_count and block_name == \
                    device_dbi.DeviceDBService.block_type2str_map[device_dbi.DeviceDBService.BlockType.LVDS_TX]:
                for type_name in sorted(lvds_tx_count):
                    outfile.write(
                        "\t{}: {}\n".format(type_name, lvds_tx_count[type_name]))

            elif lvds_rx_count and block_name == \
                    device_dbi.DeviceDBService.block_type2str_map[device_dbi.DeviceDBService.BlockType.LVDS_RX]:
                for type_name in sorted(lvds_rx_count):
                    outfile.write(
                        "\t{}: {}\n".format(type_name, lvds_rx_count[type_name]))

            elif hsio_count and block_name == device_dbi.DeviceDBService.block_type2str_map[device_dbi.DeviceDBService.BlockType.HSIO]:
                for type_name in sorted(hsio_count):
                    outfile.write(
                        "\t{}: {}\n".format(type_name, hsio_count[type_name]))

            elif quad_count and block_name == device_dbi.DeviceDBService.block_type2str_map[device_dbi.DeviceDBService.BlockType.QUAD]:
                for type_name in sorted(quad_count):
                    outfile.write(
                        "\t{}: {}\n".format(type_name, quad_count[type_name]))

            elif quad_pcie_count and block_name == device_dbi.DeviceDBService.block_type2str_map[device_dbi.DeviceDBService.BlockType.QUAD_PCIE]:
                for type_name in sorted(quad_pcie_count.keys(),
                                        key=lambda name: (0, name) if "PCI" in name else (1, name)):
                    outfile.write(
                        "\t{}: {}\n".format(type_name, quad_pcie_count[type_name]))

            elif mipi_tx_count and block_name == device_dbi.DeviceDBService.block_type2str_map[device_dbi.DeviceDBService.BlockType.MIPI_TX]:
                for type_name in sorted(mipi_tx_count):
                    outfile.write(
                        "\t{}: {}\n".format(type_name, mipi_tx_count[type_name]))

        outfile.write(
            "---------- Periphery Usage Summary (end) ----------\n")

        return blk_to_usage_str_map

    def generate_output_files_info(self, outfile):
        '''
        Generate the list of available output filenames.

        :param outfile: file handle that has already been opened
        '''

        outfile.write(
            "\n---------- 2. Generated Output Files (begin) ----------\n")

        filename = self.__project_name + ".interface.csv"
        outfile.write("Interface Configuration: {}\n".format(filename))

        filename = self.__project_name + ".lpf"
        outfile.write("Peripheral Block Configuration: {}\n".format(filename))

        filename = self.__project_name + ".pinout.rpt"
        outfile.write("Pinout Report: {}\n".format(filename))

        filename = self.__project_name + ".pinout.csv"
        outfile.write("Pinout CSV: {}\n".format(filename))

        # The timing files depend on if there's a timing model associated to a
        # device
        if self.__device.get_timing_model() is not None:
            filename = self.__project_name + ".pt_timing.rpt"
            outfile.write("Timing Report: {}\n".format(filename))

            filename = self.__project_name + ".pt.sdc"
            outfile.write("Timing SDC Template: {}\n".format(filename))

        filename = self.__project_name + "_template.v"
        outfile.write("Verilog Template: {}\n".format(filename))

        # The OR file depends on whether there's any control block
        # configuration to write out
        if self.__is_write_option_register():
            filename = self.__project_name + "_or.ini"
            outfile.write("Option Register File: {}\n".format(filename))

        outfile.write("---------- Generated Output Files (end) ----------\n")

    def __is_write_option_register(self):
        '''
        Checks if it is valid to write out the option register file.
        It depends on whether, the following registry exists:
        1) control: remote update
        2) seu: part of configuration
        3) oscillator not controlled by PCRs

        :return: True if the device needs to write out the option_register
        '''
        is_write = False

        if self.__design is not None:
            osc_reg = self.__design.get_block_reg(
                des_db.PeriDesign.BlockType.osc)

            if self.__design.is_block_supported(des_db.PeriDesign.BlockType.adv_osc):
                is_write = True
            elif self.__design.device_setting is not None:
                # Check if there's any configuration
                dev_setting = self.__design.device_setting
                if dev_setting.ctrl_reg is not None or \
                        dev_setting.seu_reg is not None:
                    is_write = True

        self.logger.debug(
            "summary_report: option register file should be written: {}".format(is_write))

        return is_write

    def generate_io_bank_summary(self, outfile):
        '''
        Generate a summary of the io standard assignments on
        io bank.

        :param outfile: file handle that has already been opened
        '''

        outfile.write(
            "\n---------- 3. I/O Banks Summary (begin) ----------\n")

        if self.__design.device_setting is not None:
            iob_reg = self.__design.device_setting.iobank_reg

            if iob_reg is not None:
                io_banks = iob_reg.get_all_iobank()

                def get_name(bank_obj):
                    return bank_obj.name

                row_list = []
                table = PrettyTable(["I/O Bank", "I/O Voltage"])

                for bank in sorted(io_banks, key=get_name):
                    row_list = []

                    row_list.append(bank.name)
                    row_list.append(bank.get_user_io_voltage())

                    table.add_row(row_list)

                outfile.write("\n{}\n\n".format(table.get_string()))
            else:
                outfile.write("No I/O Bank information available.\n")
        else:
            outfile.write("No I/O Bank information available.\n")

        outfile.write("---------- I/O Banks Summary (end) ----------\n")

    def is_global_instance_configured(self, db_global, ins_to_block_map):
        '''
        Check if any of the device global instance was configured.

        :param db_global: Device's GlobalClocks object
        :param ins_to_block_map: Map of instance name to the ref block name.
                    Instance name listed in here are only those that
                    are configured in the design.

        :return True if found any of the instance in the global
                    instance list. False otherwise.
        '''
        found_global = False

        global_ins_set = db_global.get_global_instances()

        for ins_name in global_ins_set:
            # If it is a HSIO resource, we replace PN with P
            if ins_name in ins_to_block_map:
                found_global = True
                break
            elif ins_name.find("_PN_") != -1:
                tmp_ins_name = ins_name
                tmp_ins_name = tmp_ins_name.replace("_PN_", "_P_")

                if tmp_ins_name in ins_to_block_map:
                    found_global = True
                    break

        return found_global

    def generate_global_summary(self, outfile, ins_to_block_map):
        '''
        Writes out the global connection summary (GCLK, GCTRL,
        etc.)
        :param device_db: device DeviceDBService
        :param outfile: file handle that has already been opened
        :param ins_to_block_map: A map of instance name to the
                        referenced block name
        '''
        outfile.write(
            "\n---------- 4. Global Connection Summary (begin) ----------\n")

        # Check if any of the instances specific in the global section
        # are configured
        db_global = self.__device.get_global_clocks()

        if self.is_global_instance_configured(db_global, ins_to_block_map):
            # List of list to be displayed per row. Each entry in the
            # list represents a list for each column.

            table = PrettyTable(["Pin Name", "Resource", "Type"])

            ins2func_map = db_global.get_instance_to_function_map()

            # The order is based on the section ordering
            # for blk_type in sorted(self.__blocks_summary.keys()):
            for section in sorted(self.__blocks_section.keys()):
                blk_type = self.__blocks_section[section]
                block_obj = self.__blocks_summary[blk_type]

                if block_obj is not None:
                    try:
                        block_obj.generate_global_summary(
                            self.__design, db_global,
                            ins2func_map, ins_to_block_map,
                            table)

                    except Exception as excp:
                        pt_util.mark_unused(excp)
                        raise

            table_str = table.get_string(
                sortby="Pin Name",
                sort_key=lambda x: [pt_util.natural_sort_key_str(str(x))])

            outfile.write("\n{}\n".format(table_str))

        else:
            outfile.write("\nNo global connection was found in project.\n")

        outfile.write(
            "\n---------- Global Connection Summary (end) ----------\n")

    def generate_clock_region_summary(self, outfile):
        '''
        :param outfile: File to print out the report
        '''

        from device.block_instance import PeripheryBlockInstance

        outfile.write(
            "\n---------- 5. Clock Region Usage Summary (begin) ----------\n\n")

        # self.__device.print_clock_region_instance_map(outfile)

        # PT-481: Display the following:
        # 1) Display short summary
        # Create the Clock Checker Object but don't need
        # to really do a check. We just want to use its data
        # structure which builds the clk region to clk_name - instance name
        # map
        if self.__design is not None and self.__design.is_tesseract_design():
            from tx60_device.clock.clock_rule_adv import ClockCheckerAdv
            clkchecker = ClockCheckerAdv(self.__design)
        else:
            from common_device.clock.clock_rule import ClockChecker
            clkchecker = ClockChecker(self.__design)

        title_list = ["Clock Region", "Used/Available"]
        short_table = PrettyTable(title_list)

        # This is a map of clock region to another map of
        # user clock name to the design instance name
        region2clkins_map = clkchecker.region2clkins_map

        # We also need to add the clock region that might not
        # have any used resources too
        all_clock_regions = self.__device.get_all_clock_region_names()

        for region in sorted(all_clock_regions,
                             key=pt_util.natural_sort_key_for_list):

            row_list = []
            usage = ""

            # Get the available/limit value
            if region.startswith(
                    PeripheryBlockInstance.side2str_map[PeripheryBlockInstance.DieSideType.left]) or \
                    region.startswith(
                    PeripheryBlockInstance.side2str_map[PeripheryBlockInstance.DieSideType.right]):
                max_capacity = clkchecker.left_right_clock_capacity
            else:
                max_capacity = clkchecker.top_bottom_clock_capacity

            if region in region2clkins_map.keys():

                clk2ins_map = region2clkins_map[region]

                used_capacity = len(clk2ins_map.keys())
                usage = "{}/{}".format(used_capacity, max_capacity)

            else:
                # Print for region that is not used at all
                usage = "0/{}".format(max_capacity)

            row_list.append(region)
            row_list.append(usage)

            short_table.add_row(row_list)

        outfile.write("{}".format(short_table.get_string()))
        outfile.write(
            "\n\n---------- Clock Region Usage Summary (end) ----------\n")

    def generate_config_pin_usage(self, outfile):
        '''
        :param outfile: File to print out the report
        '''
        outfile.write(
            "\n---------- 6. Dual-Function Configuration Pin Usage (begin) ----------\n\n")

        config_model = self.__device.get_config_model()
        not_applicable = True

        if config_model is not None:
            title_list = ["Instance Name", "Function"]
            short_table = PrettyTable(title_list)

            pad_name2function_map = config_model.get_pad_name_function_map()

            gpio_reg = self.__design.gpio_reg
            lvds_reg = self.__design.lvds_reg

            if pad_name2function_map:
                for pad_name in pad_name2function_map.keys():

                    row = []

                    # Get the resource name that is associated to the pad
                    # For lvds instance it will return based on the pin so
                    # GPIOX_RXP if it is associated to RXP pin and also GPIOX_RX
                    # for lvds
                    dev_ins_names = self.__device.find_pad_instances(pad_name)

                    if dev_ins_names:
                        # Find the device instance and then find out if they are
                        # configured. Similar like pinout, we are limited it to
                        # LVDS and GPIO.
                        # TODO: If this assumption is no longer true, it needs to
                        # be updated
                        des_ins = None
                        for dev_ins in dev_ins_names:
                            # Search in GPIO
                            if gpio_reg is not None and \
                                    gpio_reg.is_gpio_device_used(dev_ins):
                                des_ins = gpio_reg.get_gpio_by_device_name(
                                    dev_ins)
                                break
                            elif lvds_reg is not None:
                                des_ins = lvds_reg.get_inst_by_device_name(
                                    dev_ins)
                                if des_ins is not None:
                                    break

                        if des_ins is not None:
                            not_applicable = False

                            row.append(des_ins.name)
                            row.append(pad_name2function_map[pad_name])

                            short_table.add_row(row)

            if not not_applicable:
                table_str = short_table.get_string(
                    sortby="Instance Name",
                    sort_key=lambda x: [pt_util.natural_sort_key_str(str(x))])

                outfile.write("{}".format(table_str))

        if not_applicable:
            outfile.write("No instance using dual-function configuration pin.")

        outfile.write(
            "\n\n---------- Dual-Function Configuration Pin Usage (end) ----------\n")

    def generate_block_summary(self, outfile, ins_to_block_map):
        '''
        :param outfile: File to print out the report
        :param ins_to_block_map: A map of instance to the block
        '''

        device_db = device_dbi.DeviceDBService(self.__device)

        # Order should be based on section
        # for blk_type in sorted(self.__blocks_summary.keys()):
        for section in sorted(self.__blocks_section.keys()):
            blk_type = self.__blocks_section[section]
            block_obj = self.__blocks_summary[blk_type]

            if block_obj is not None:
                try:
                    blk_svc = device_db.get_block_service(blk_type)

                    #print("Generate summary for {} - {}".format(
                    #    blk_type, type(blk_svc)))

                    block_obj.generate_summary(
                        self.__design, outfile,
                        ins_to_block_map, blk_svc)

                except Exception as excp:
                    raise

    def generate_design_issues(self, outfile):
        '''
        Write out the design issues table into the report file.
        :param outfile: File to print out the report
        '''
        if self.__design.issue_reg is not None:
            outfile.write(
                "\n---------- {}. Design Issues (begin) ----------\n\n".format(
                    self.__design_issue_index))

            self.__design.issue_reg.write_pretty_table_to_file(outfile)

            outfile.write(
                "\n---------- Design Issues (end) ----------\n")

    def write(self):
        '''
        Writes out the report file
        '''

        write_successful = False
        outfile = None

        excp_msg = ""
        unrouted_clocks = False

        is_register = False
        issue_reg = None
        if self.__design is not None and self.__design.issue_reg is not None:
            issue_reg = self.__design.issue_reg
            is_register = True

        try:

            self._identify_blocks()

            device_db = device_dbi.DeviceDBService(self.__device)

            # This is a map of device instance name to the block name.
            # It only contains the list of instances that have been
            # configured by users and available in the device.
            ins_to_block_map = device_db.get_configured_instances(
                self.__design)

            print("Writing out summary report file")

            blk_to_usage_map = {}

            # Open the file
            with open(self.__filename, 'w', encoding='UTF-8') as outfile:

                # Generate file header and TOC
                self.generate_file_header(outfile)

                # Write out the Usage summary
                blk_to_usage_map = self.generate_usage_summary(
                    outfile, ins_to_block_map)

                # Write out the List of available generated filenames
                self.generate_output_files_info(outfile)

                # Write the I/O Bank summary
                self.generate_io_bank_summary(outfile)

                # Generate Globals
                self.generate_global_summary(outfile, ins_to_block_map)

                # Write the clock region usage summary
                self.generate_clock_region_summary(outfile)

                # Write the configuration pin usage
                self.generate_config_pin_usage(outfile)

                self.generate_block_summary(
                    outfile, ins_to_block_map)

                # Because the external flash reconfig is not really a block,
                # we write it out here if it is applicable (always the last one)
                # since it is dependent
                self.generate_config_ext_flash_summary(outfile)

                # Write the design issues
                if self.__design_issue_index is not None:
                    self.generate_design_issues(outfile)

            # Write out the xml report
            if blk_to_usage_map:
                try:
                    root = self._generate_xml_tree(blk_to_usage_map)

                    if root is not None:
                        xml_string = et.tostring(root, 'utf-8')

                        reparsed = expatbuilder.parseString(xml_string, False)
                        print_string = reparsed.toprettyxml(indent="  ")

                        if print_string != "":
                            with open(self.__summary_xml_file, 'w', encoding='UTF-8') as xml_outfile:
                                xml_outfile.write(print_string)

                except Exception as excp:
                    # No notification if unable to generate the report xml file so that
                    # Interface Designer will continue to generate required data
                    self.logger.debug("Unable to generate {} file".format(
                        self.__summary_xml_file))

            write_successful = True

        except IOError as excp:
            excp_msg = "{}".format(excp)
            if is_register:
                issue_reg.append_issue_by_tuple(
                    ("summary", "report generation",
                     base_rule.Rule.SeverityType.error,
                     "summary_rule_io_exception", excp_msg))

        except comm_excp.ClockUnrouted as excp:
            excp_msg = "{}".format(excp.msg)
            unrouted_clocks = True

            if is_register:
                issue_reg.append_issue_by_tuple(
                    ("summary", "unrouted clock",
                     base_rule.Rule.SeverityType.error,
                     "unrouted_clk", excp_msg))

        except Exception as excp:
            import traceback
            traceback.print_exc()
            excp_msg = "{}".format(excp)
            if is_register:
                issue_reg.append_issue_by_tuple(
                    ("summary", "report generation",
                     base_rule.Rule.SeverityType.error,
                     "summary_rule_exception", excp_msg))

        if not write_successful:

            if outfile is not None:
                if not unrouted_clocks:
                    os.remove(self.__filename)

            raise writer_excp.GenerateReportException(
                excp_msg, app_excp.MsgLevel.error)

        return write_successful

    def is_ext_flash_controller_section_exists(self):
        if self.__design is not None and self.__design.device_setting is not None:
            if self.__design.device_setting.ext_flash_reg is not None:
                return True

        return False

    def generate_config_ext_flash_summary(self, outfile):
        if self.is_ext_flash_controller_section_exists():

            ext_flash_writer = ExtFlashControllerSummary(self.__device, "ext_flash", self._ext_flash_ctrl_index)
            ext_flash_writer.generate_summary(self.__design, outfile)

