

from enum import Enum
from pathlib import Path
import re
from typing import List, Optional, Set, Tuple

import util.excp as app_excp

from design.db import PeriDesign
from device.block_instance import PeripheryBlockInstance
from device.db import PeripheryDevice
from device.db_interface import DeviceDBService
from device.io_info import DeviceIO, IOBankInfo, IOPadInfo
from device.package import DevicePackage

from writer.pinout import PinoutRecord, PinoutReport

from writer.gen_bsdl import bscan_pin as IBSScanPin
from writer.gen_bsdl import pin as IBSDevicePin
from writer.gen_bsdl import fpga as IBSFpga
from writer.gen_bsdl import bscan as IBScan
from writer.gen_bsdl import bsdl as IBSDL
from writer.gen_bsdl import prepare_bsdl_info, write_bsdl, get_die_name

import writer.excp as writer_excp


class CellType(Enum):
    GPIO = "GPIO"
    LVDS = "LVDS"
    DDR = "DDR"
    HSIO = "HSIO"
    HVIO = "HVIO"

def is_hsio(io_stds_list: List[str]) -> bool:
    matches = ('HSTL', 'SSTL')
    for io_std in io_stds_list:
        for x in matches:
            if x in io_std:
                return True
    return False


class BSDLWriter():
    """
    Adapter to gen_bsdl.py script
    """

    def __init__(self, device_db: PeripheryDevice, design_db: PeriDesign):

        self.__design: PeriDesign = design_db
        self.__device: PeripheryDevice = device_db

    def is_supported(self) -> bool:
        """
        True if the device / design support writing BSDL files
        """
        BLACKLIST: Set[str] = {
            'T8-WB',
            'T8Q100',
            'T4F49',
            'T4F81',
            'T8F49',
            'T20F324TEST',
            'T20F256A',
            'T120FULL',
            'T120F784',
            'T70F324VC',
            'T70F576VC',
            'T8',
            'T120',
            'T35',
            'T20F324VC',
            'T20F256MI',
            'T120F484A',
            'T120F324M1',
            'T120F324MA',
            'T120F324MN',
            'Ti60',
            'Ti60F100',
            'Ti35F100',
            'Ti180TEST',
            'Ti180',
            'Ti180F484',
            # 'Ti90M484',
            'Ti120M484_test',
            # 'Ti180M484',
            'Ti180DUMMY',
            # 'Ti90M361',
            # 'Ti120M361',
            # 'Ti180M361',
            # 'Ti90F529',
            'Ti120F529_test',
            # 'Ti180F529',
            # 'Ti180L484',
            # 'Ti90L484',
            # 'Ti120L484',
            # 'Ti90J361',
            # 'Ti120J361',
            # 'Ti180J361',
            # 'Ti90J484',
            # 'Ti120J484',
            # 'Ti180J484',
            # 'Ti90G529',
            # 'Ti120G529',
            # 'Ti180G529',
            'Ti375test',
            'Ti375',
            'Ti60-WB',
            'Ti180-FCJ',
            'Ti180-FCG',
            'TJ180A484S',
            'Ti180-SIP_test',
            'an08flat',
            'an20slow',
            'an20ctrl',
            'an20dummy',
            'T20',
            'T20-WB',
            'T120TESTPT',
            'TXTESTPT',
            'TX60TEST',
            'TX60F100TEST',
            'TX60',
            'Ti60F225ES',
            'Ti60W64ES',
            'Ti60C64TEST-ES',
            'Ti60C225TEST',
            'Ti60F100ES',
            'Ti60F100TESTDUMMY',
            'Ti375N1156',
            'TJ375N1156X',
            # 'Ti90G400',
            # 'Ti120G400',
            # 'Ti180G400',
            'Ti375C529',
            'TJ375N529',
            'Ti165C529',
            'TJ165N529',
            'Ti240C529',
            'TJ240N529',
            'Ti165N1156',
            'TJ165N1156X',
            'Ti240N1156',
            'TJ240N1156X',
            'Ti60F256',
            'Ti35F256',
            'Ti60F225M',
            'Ti60F100S3F2M',
            'T35F256',
            'Ti375N484',
            'TJ375N484X',
            'Ti240N484',
            'TJ240N484X',
            'Ti165N484',
            'TJ165N484X',
            'Ti135',
            'Ti135N676',
            'TJ135N676X',
            'Ti85N676',
            'TJ85N676X',
            'Tz50F100',
            'Tz50F225',
            'Tz50F256',
            'Tz110J484',
            'Tz170J484',
            'Tz110G400',
            'Tz170G400',
            'Tz110J361',
            'Tz170J361',
            'Ti180J484D1',
            'Tz200C529',
            'Tz325C529',
            'Tz200N484',
            'Tz325N484',
            'Tz75N676',
            'Tz100N676',
            'Ti135N484',
            'Ti85N484',
            'Tz100N484',
            'Tz75N484',
            'TP50F100',
            'TP50F225',
            'TP50F256',
            'TP110G400',
            'TP110J361',
            'TP110J484',
            'TP170G400',
            'TP170J361',
            'TP170J484',
            'TP200N529',
            'TP200N484X',
            'TP325N529',
            'TP325N484X',
            'TP100N676X',
            'TP100N484X',
            'TP75N484X',
            'TP75N676X',
            'TJ135N484X',
            'TJ85N484X',
            'Ti135N676ES',
            'Ti375N900',
            'Ti240N900',
            'Ti165N900',
            'Tz325N900',
            'Tz200N900',
            'TJ375N900X',
            'TJ240N900X',
            'TJ165N900X',
            'TP325N900X',
            'TP200N900X',
        }
        device_io: DeviceIO = self.__device.get_io_pad()
        if device_io is None:
            return False
        if self.__device.get_device_name() in BLACKLIST:
            return False
        return device_io.has_bscan_info()

    def is_titanium(self) -> bool:
        # Include Topaz as Titanium
        return self.__device.get_device_name().startswith(('Ti', 'TX', 'Tz', 'TP', 'TJ'))

    def find_cell_type(self, dev_io: DeviceIO, pad_name: str) -> str:
        instances = dev_io.find_instance(pad_name)
        pad_info_obj:IOPadInfo = dev_io.find_pad(pad_name)
        config_model = self.__device.get_config_model()
        debug = False
        # if pad_name.startswith('CDONE'):
        #     debug = True
        def debug_print(m):
            if debug:
                print(m)
        debug_print(pad_name)
        debug_print(instances)
        debug_print(pad_info_obj)
        debug_print(config_model)
        debug_print(f'is_titanium = {self.is_titanium()}')

        if self.is_titanium():
            bank_name = pad_info_obj.get_bank_name()
            debug_print(f'bank_name = {bank_name} {type(bank_name)}')
            bank_info_obj: IOBankInfo = dev_io.get_bank_info(bank_name)
            debug_print(f'bank_info_obj = {bank_info_obj}')
            io_stds = []
            if bank_info_obj:
                io_stds = bank_info_obj.get_io_standards()
            else:
                die_bank_info = dev_io.get_die_bank_info(bank_name)
                if die_bank_info:
                    io_stds = die_bank_info.get_io_standards()
            debug_print(f'io_stds = {io_stds}')

            if is_hsio(io_stds):
                cell_type = CellType.HSIO
            else:
                cell_type = CellType.HVIO
        else:
            cell_type = CellType.GPIO

        if instances and len(instances) == 1:
            for ins_name in instances:
                ins_obj: Optional[PeripheryBlockInstance] = self.__device.find_instance(ins_name)
                if not ins_obj:
                    continue
                ref_name = ins_obj.get_ref_name()
                debug_print(f"ref_name = {ref_name}")
                if not ref_name:
                    continue
                dev_blk_type = DeviceDBService.block_str2type_map.get(ref_name)
                debug_print(f'dev_blk_type = {dev_blk_type}')
                if dev_blk_type in (DeviceDBService.BlockType.LVDS_TX, DeviceDBService.BlockType.LVDS_RX):
                    cell_type = CellType.LVDS
                    break
                elif dev_blk_type == DeviceDBService.BlockType.DDR:
                    cell_type = CellType.DDR
                    break
                elif dev_blk_type == DeviceDBService.BlockType.HSIO:
                    cell_type = CellType.HSIO
                    break
                elif dev_blk_type == DeviceDBService.BlockType.HVIO:
                    cell_type = CellType.HVIO
                    break

        return cell_type.value

    def build_pre_configuration_pins(self, fpga: IBSFpga) -> List[IBSDevicePin]:
        """
        Construct the pin information for the gen_bsdl writer
        """
        def get_config_type(p: PinoutRecord, function_type: str) -> str:
            config_type = None
            if function_type == 'VCC' or function_type == 'GND' or function_type == 'VDD' or \
                function_type == 'VQPS':
                config_type = 'linkage'
            elif function_type == 'Config':
                if p.direction == 'Input':
                    config_type = 'in'
                elif p.direction == 'Output':
                    config_type = 'out'
                elif p.direction == 'Bidirectional':
                    config_type = 'inout'
                else:
                    raise Exception('cannot determine pin configuration')
            elif function_type == 'DDR':
                if fpga.die == 'tx180':
                    config_type = 'linkage'
                else:
                    if ('CK_N' in p.pad_name) or ('DQS_N' in p.pad_name) or (
                        'VREF' in p.pad_name) or ('ZQ' in p.pad_name):
                        config_type = 'linkage'
                    elif ('DQ' in p.pad_name):
                        config_type = 'inout'
                    else:
                        config_type = 'out'
            elif function_type == 'MIPI':
                config_type = 'linkage'
            elif function_type == 'REF':
                config_type = 'linkage'
            elif function_type == 'LVDS':
                # Remarks: Due to hardware, AN20 LVDS has different pre-configuration setting
                if fpga.die == 'an20':
                    config_type = 'out'
                else:
                    config_type = 'inout'
            elif function_type == 'Flash':
                config_type = 'linkage'
            elif function_type == 'NC':
                config_type = 'linkage'
            else:
                config_type = 'inout'

            return config_type

        device_io: DeviceIO = self.__device.get_io_pad()

        pinout_writer = PinoutReport(self.__device, self.__design, None, self.__design.name)
        pinout_records, _ = pinout_writer.generate_pinout_assignment()
        results: List[IBSDevicePin] = []
        for p in pinout_records:
            function_type = None
            if ('VCC' in p.pad_name) or ('VDD' in p.pad_name) or ('VQPS' in p.pad_name):
                function_type = 'VCC'
            elif 'GND' in p.pad_name:
                function_type = 'GND'
            elif 'GPIO' in p.pad_name:
                if self.is_titanium():
                    pad_info_obj:IOPadInfo = device_io.find_pad(p.pad_name)
                    bank_name = pad_info_obj.get_bank_name()
                    bank_info_obj: IOBankInfo = device_io.get_bank_info(bank_name)
                    # Since this focus on package pin, no need check for io bank in die level
                    io_stds = []
                    if bank_info_obj:
                        io_stds = bank_info_obj.get_io_standards()
                    if is_hsio(io_stds):
                        function_type = 'HSIO'
                    else:
                        function_type = 'HVIO'
                else:
                    matches = ('TX', 'RX', 'CLKN', 'CLKP')
                    if any(x in p.pad_name for x in matches):
                        function_type = 'LVDS'
                    else:
                        function_type = 'GPIO'
            elif 'DDR' in p.pad_name:
                function_type = 'DDR'
            elif 'MIPI' in p.pad_name:
                function_type = 'MIPI'
            elif 'REF' in p.pad_name:
                function_type = 'REF'
            elif 'SPI' in p.pad_name:
                function_type = 'Flash'
            elif 'NC' in p.pad_name:
                function_type = 'NC'
            else:
                function_type = 'Config'

            config_type = get_config_type(p, function_type)

            # need to remove square bracket since items between '[' and ']' might be omitted
            safe_pad_name = p.pad_name.replace('[', '_')
            safe_pad_name = safe_pad_name.replace(']', '')
            results.append(IBSDevicePin(name=safe_pad_name, number=p.pin_name, config=config_type, function=function_type))

        return results

    def build_post_configuration_pins(self, fpga: IBSFpga) -> List[IBSDevicePin]:
        device_io: DeviceIO = self.__device.get_io_pad()
        pinout_writer = PinoutReport(self.__device, self.__design, None, self.__design.name)
        pinout_records, _ = pinout_writer.generate_pinout_assignment()
        def get_config_type(p: PinoutRecord, function_type: str) -> str:
            # Configuration pin
            if function_type == 'Config':
                if p.direction == 'Input':
                    config_type = 'in'
                elif p.direction == 'Output':
                    config_type = 'out'
                elif p.direction == 'Bidirectional':
                    config_type = 'inout'
                else:
                    raise Exception('cannot determine pin configuration')

            elif (p.signal_name and not p.signal_name.isspace()):

                if any(f in function_type for f in ('VCC', 'VDD', 'VQPS', 'GND', 'MIPI', 'REG')):
                    config_type = 'linkage'
                # DDR pin
                elif function_type == 'DDR':
                    if ('CK_N' in p.pad_name) or ('DQS_N' in p.pad_name) or (
                            'VREF' in p.pad_name) or ('ZQ' in p.pad_name):
                        config_type = 'linkage'
                    elif ('DQ' in p.pad_name):
                        config_type = 'inout'
                    else:
                        config_type = 'out'

                # GPIO pin
                elif (p.io_std and not p.io_std.isspace()) or function_type == 'GPIO':
                    if p.direction == 'Input':
                        config_type = 'in'
                    elif p.direction == 'Output':
                        config_type = 'out'
                    elif p.direction == 'Bidirectional':
                        config_type = 'inout'
                    else:
                        raise Exception('cannot determine pin configuration')

                # LVDS pin
                else:
                    if p.signal_name.endswith('.RXN'):
                        config_type = 'lvds_rx_n'
                    elif p.signal_name.endswith('.RXP'):
                        config_type = 'lvds_rx_p'
                    elif p.signal_name.endswith('.TXN'):
                        config_type = 'lvds_tx_n'
                    elif p.signal_name.endswith('.TXP'):
                        config_type = 'lvds_tx_p'
                    else:
                        raise Exception('cannot determine pin configuration')

            # Unused pin
            else:
                config_type = 'linkage'
            return config_type

        def get_config_type_titanium(p: PinoutRecord, function_type: str) -> str:
            # Configuration pin
            if function_type == 'Config':
                if p.direction == 'Input':
                    config_type = 'in'
                elif p.direction == 'Output':
                    config_type = 'out'
                elif p.direction == 'Bidirectional':
                    config_type = 'inout'
                else:
                    raise Exception('cannot determine pin configuration')

            elif (p.signal_name and not p.signal_name.isspace()):

                if any(f in function_type for f in ('VCC', 'VDD', 'VQPS', 'GND', 'MIPI', 'REG')):
                    config_type = 'linkage'
                elif function_type == 'DDR':
                    if ('CK_N' in p.pad_name) or ('DQS_N' in p.pad_name) or (
                            'VREF' in p.pad_name) or ('ZQ' in p.pad_name):
                        config_type = 'linkage'
                    elif ('DQ' in p.pad_name):
                        config_type = 'out'
                    elif ('CAL' in p.pad_name):
                        config_type = 'linkage'
                    else:
                        config_type = 'out'

                elif (p.io_std and not p.io_std.isspace()):
                    # Differential HSTL/SSTL pin
                    if 'Differential' in p.io_std:
                        if p.signal_name.endswith('.P'):
                            if p.direction == 'Input':
                                config_type = 'diff_in_p'
                            elif p.direction == 'Output':
                                config_type = 'diff_out_p'
                            elif p.direction == 'Bidirectional':
                                config_type = 'diff_inout_p'
                            else:
                                raise Exception(
                                    'cannot determine pin configuration')

                        elif p.signal_name.endswith('.N'):
                            if p.direction == 'Input':
                                config_type = 'diff_in_n'
                            elif p.direction == 'Output':
                                config_type = 'diff_out_n'
                            elif p.direction == 'Bidirectional':
                                config_type = 'diff_inout_n'
                            else:
                                raise Exception(
                                    'cannot determine pin configuration')

                    # GPIO pin
                    else:
                        if p.direction == 'Input':
                            config_type = 'in'
                        elif p.direction == 'Output':
                            config_type = 'out'
                        elif p.direction == 'Bidirectional':
                            config_type = 'inout'
                        else:
                            raise Exception('cannot determine pin configuration')

                # LVDS pin
                else:
                    if p.signal_name.endswith('.P'):
                        if p.direction == 'Input':
                            config_type = 'lvds_rx_p'
                        elif p.direction == 'Output':
                            config_type = 'lvds_tx_p'
                        elif p.direction == 'Bidirectional':
                            config_type = 'lvds_bidir_p'
                        else:
                            raise Exception('cannot determine pin configuration')

                    elif p.signal_name.endswith('.N'):
                        if p.direction == 'Input':
                            config_type = 'lvds_rx_n'
                        elif p.direction == 'Output':
                            config_type = 'lvds_tx_n'
                        elif p.direction == 'Bidirectional':
                            config_type = 'lvds_bidir_n'
                        else:
                            raise Exception('cannot determine pin configuration')

            # Unused pin
            else:
                config_type = 'linkage'
            return config_type

        results = []
        for p in pinout_records:
            function_type = None
            if ('VCC' in p.pad_name) or ('VDD' in p.pad_name) or ('VQPS' in p.pad_name):
                function_type = 'VCC'
            elif 'GND' in p.pad_name:
                function_type = 'GND'
            elif 'GPIO' in p.pad_name:
                if self.is_titanium():
                    pad_info_obj:IOPadInfo = device_io.find_pad(p.pad_name)
                    bank_name = pad_info_obj.get_bank_name()
                    bank_info_obj: IOBankInfo = device_io.get_bank_info(bank_name)
                    # Since this focus on package pin, no need check for io bank in die level
                    io_stds = []
                    if bank_info_obj:
                        io_stds = bank_info_obj.get_io_standards()
                    if is_hsio(io_stds):
                        function_type = 'HSIO'
                    else:
                        function_type = 'HVIO'
                else:
                    matches = ('TX', 'RX', 'CLKN', 'CLKP')
                    if any(x in p.pad_name for x in matches):
                        function_type = 'LVDS'
                    else:
                        function_type = 'GPIO'
            elif 'DDR' in p.pad_name:
                function_type = 'DDR'
            elif 'MIPI' in p.pad_name:
                function_type = 'MIPI'
            elif 'REF' in p.pad_name:
                function_type = 'REF'
            elif p.pin_function == 'Configuration':
                function_type = 'Config'
            elif 'SPI' in p.pad_name:
                function_type = 'Flash'
            elif 'NC' in p.pad_name:
                function_type = 'NC'
            else:
                raise NotImplementedError(p.pin_function, p.pad_name, p.pin_name)

            if self.is_titanium():
                config_type = get_config_type_titanium(p, function_type)
            else:
                config_type = get_config_type(p, function_type)

            # need to remove square bracket since items between '[' and ']' might be omitted
            safe_pad_name = p.pad_name.replace('[', '_')
            safe_pad_name = safe_pad_name.replace(']', '')
            results.append(IBSDevicePin(name=safe_pad_name, number=p.pin_name, config=config_type, function=function_type))

        return results

    def build_bscan_pins(self) -> List[IBSScanPin]:
        bscan_pins = []
        device_io: DeviceIO = self.__device.get_io_pad()
        package: DevicePackage = self.__device.get_package()
        default_value = '-'

        io_pads = device_io.get_all_io_pads()
        # IO Pad at DIE level
        for io_pad in io_pads:
            bscan_nums = io_pad.get_bscan_seq_nums()
            if len(bscan_nums) == 0:
                continue

            bscan_seq_0 = bscan_nums[0]
            bscan_seq_1 = bscan_nums[1] if len(bscan_nums) == 2 else default_value

            package_pin_names = package.get_package_pin_names(io_pad.get_pad_name())
            # The IO Pad is bonded out to package level
            if len(package_pin_names) > 0:
                package_pin = package_pin_names[0]
            else:
                package_pin = default_value

            cell_type = self.find_cell_type(device_io, io_pad.get_pad_name())

            bscan_pin = IBSScanPin(bscan_seq_0=bscan_seq_0,
                                   bscan_seq_1=bscan_seq_1,
                                   number=package_pin,
                                   cell_type=cell_type)

            bscan_pins.append(bscan_pin)

        bscan_pins.sort(key=lambda p: p.bscan_seq_0)
        return bscan_pins

    def build_fpga(self) -> IBSFpga:
        package = self.__device.get_package()
        name = package.get_package_name()
        match = re.match(r'(\d+)-ball (FBGA|WLCSP)', name)
        package_name = None
        if match:
            package_type = match.group(2)
            package_name = f'{package_type}{match.group(1)}'

        match = re.match(r'(\d+)-pin (QFP)', name)
        if match:
            package_name = f'LQFP{match.group(1)}'

        family_name = 'Titanium' if self.__device.find_block('hsio') is not None else 'Trion'
        match = re.match(r'(T|Ti)(\d+).*', self.__device.get_device_name())
        product_name = None
        if match:
            product_name = f'{match.group(1)}{match.group(2)}'

        assert product_name is not None
        assert package_name is not None

        device_name = self.__device.get_device_name()
        match = re.match(r'(T|Ti)(\d+)(\w)(\d+).*', device_name)
        assert match is not None
        code_name = match.group(3)
        die_name = get_die_name(device_name, product_name, package_name)

        return IBSFpga(device=self.__device.get_device_name(),
                       family=family_name,
                       package=package_name,
                       product=product_name,
                       code=code_name,
                       die=die_name)

    def build_bsdl_info(self, fpga: IBSFpga, pins_info: List[IBSDevicePin], bscan_pins: List[IBSScanPin]) -> Tuple[IBScan, IBSDL]:
        return prepare_bsdl_info(fpga, pins_info, bscan_pins)

    def write_pre_configure_bsdl(self, project_name: str, out_dir: Path) -> Path:
        print("Writing out pre-configured bscandary scan description file")
        bsdl_file = self.get_file_name(project_name, out_dir, True)
        fpga = self.build_fpga()
        pins_info = self.build_pre_configuration_pins(fpga)
        bscan_pins = self.build_bscan_pins()
        bscan_info, bsdl_info = self.build_bsdl_info(fpga=fpga, pins_info=pins_info, bscan_pins=bscan_pins)
        with open(bsdl_file, mode='w', encoding='utf-8') as f:
            write_bsdl(pre_config=True,
                       fpga=fpga,
                       pins_info=pins_info,
                       bscan_info=bscan_info,
                       bsdl_info=bsdl_info,
                       f=f)

        return bsdl_file

    def write_post_configure_bsdl(self, project_name: str, out_dir: Path) -> Path:
        print("Writing out boundary scan description file")
        bsdl_file = self.get_file_name(project_name, out_dir, False)
        fpga = self.build_fpga()
        pins_info = self.build_post_configuration_pins(fpga)
        bscan_pins = self.build_bscan_pins()
        bscan_info, bsdl_info = self.build_bsdl_info(fpga=fpga, pins_info=pins_info, bscan_pins=bscan_pins)

        with open(bsdl_file, mode='w', encoding='utf-8') as f:
            write_bsdl(pre_config=False,
                       fpga=fpga,
                       pins_info=pins_info,
                       bscan_info=bscan_info,
                       bsdl_info=bsdl_info,
                       f=f)

        return bsdl_file

    def get_file_name(self, project_name: str, out_dir: Path, is_pre: bool = False) -> Path:
        type_str = 'pre' if is_pre else 'post'
        return Path(out_dir) / f'{project_name}-{type_str}.bsd'

    def write(self, project_name: str, out_dir: Path):
        try:
            out_dir_path = Path(out_dir)
            assert out_dir_path.exists()
            assert self.is_supported()

            self.write_post_configure_bsdl(project_name, out_dir)
        except Exception as exc:
            raise writer_excp.GenerateConstraintException(
                str(exc), app_excp.MsgLevel.error)

