'''
Copyright (C) 2017-2018 Efinix Inc. All rights reserved.

No portion of this code may be reused, modified or
distributed in any way without the expressed written
consent of Efinix Inc.

Created on May 7, 2018

@author: maryam
'''

import os
import sys
from typing import List, Dict

from util.singleton_logger import Logger


sys.path.append(os.path.join(os.path.dirname(__file__), '..'))

class VariableOptionTable(object):
    '''
    Class that captures the list of a parameter variable option
    and the list of values-delay pair.
    '''

    def __init__(self, param_name, var_name):
        self.__param = param_name

        self.__variable = var_name

        # A map of the var_value name to delay
        self.__value2delay_map = {}

        # Store the overlap variable with this variable option
        self.__overlap_var = ""

        self.logger = Logger

    def add_delay(self, var_value, delay):
        is_added = False

        if var_value not in self.__value2delay_map:
            self.__value2delay_map[var_value] = delay
            is_added = True
        else:
            self.logger.debug("Unable to add {} delay {}-{}: {}".format(
                self.__param, self.__variable, var_value, delay))

        return is_added

    def set_overlap_variable(self, overlap_var):
        # This will just override without doing any check
        self.__overlap_var = overlap_var

    def get_overlap_variable(self):
        return self.__overlap_var

    def get_var_value_delay(self, var_value):
        # If cannot find, we return 0
        return self.__value2delay_map.get(var_value, 0)


class ArcDelayTable(object):
    '''
    Class that captures the delay table associated to specific
    timing parameter. The information stored here comes from
    the common_models timing_table tag.
    '''

    def __init__(self, block_name, pname, pcount):
        '''
        Constructor
        '''

        # Block name
        self.__block = block_name

        # Parameter name
        self.__pname = pname

        # Parameter count
        self.__pcount = pcount

        # List of variable names that influences the delay
        self.__vars = []

        # This is based on reading the arc_parameter name-delay pair
        # Map of name to the delay (int)
        # Note that when pcount = 0, the name (key) is ""
        self.__name2delay = {}

        # This is based on reading the arc_parameter variable_option
        # Map of the variable name to the variable option table. There can
        # be multiple variable options with the same name, provided that
        # their have unique overlap_var string. THerefore, this is a map
        # of variable name to a list of VariableOptionTable
        self.__var_opt_map = {}

        self.logger = Logger

    def get_variable_names(self):
        return self.__name2delay.keys()
    
    def get_variable_values(self) -> Dict[str, List[str]]:
        '''
        Returns a map of the variable names mapped to the
        value used.
        For example returns {
            bank: [1A, 1B]
            schmitt_trigger: [true, false] }
        '''
        var_name2val_map: Dict[str, List[str]] = {}

        for var_str in self.__name2delay.keys():
            var_assign_list = var_str.split(",")
            for var_assign in var_assign_list:
                assert var_assign.find(":") != -1
                var_name_value = var_assign.split(":")
                assert len(var_name_value) == 2
                var_name = var_name_value[0]
                var_value = var_name_value[1]

                vlist = []

                if var_name in var_name2val_map:
                    vlist = var_name2val_map[var_name]

                vlist.append(var_value)
                var_name2val_map[var_name] = vlist

        return var_name2val_map
    
    def get_parameter_name(self):
        return self.__pname

    def is_variable_option_exists(self):
        if self.__var_opt_map:
            return True

        return False

    def create_variable_table(self, var_name):
        var_table = VariableOptionTable(self.__pname, var_name)
        var_table_list = []
        if var_name in self.__var_opt_map:
            var_table_list = self.__var_opt_map[var_name]

        var_table_list.append(var_table)
        self.__var_opt_map[var_name] = var_table_list

        return var_table

    def get_variable_option_map(self):
        return self.__var_opt_map

    def get_var_name_split(self, var_name):
        '''
        Split the variable name right from the xml into list.

        :param var_name: The concatenated variable name from the file

        :return A list of variable name that's part of name
        '''
        vlist = []

        if ',' in var_name:
            vtmp_list = var_name.split(",")
            for elem in vtmp_list:
                vname = elem.split(":")
                vlist.append(vname[0])
        elif ':' in var_name:
            vname = var_name.split(":")
            vlist.append(vname[0])

        return vlist

    def get_pcount(self):
        '''
        :return the parameter count
        '''
        return self.__pcount

    def add_delay(self, var_name, delay):
        '''
        :param var_name: The name captured in the parameter tag
                        in the xml file. In the case that the
                        table has no parameter count (pcount = 0,
                        the name may be an empty string)
        :param delay: The delay in int

        :return True if it was added to name2delay
        '''
        is_added = False

        if var_name not in self.__name2delay:
            self.__name2delay[var_name] = delay
            is_added = True

            # Update the __vars if it is empty
            if not self.__vars and var_name != "":
                if ',' in var_name:
                    vtmp_list = var_name.split(",")
                    for elem in vtmp_list:
                        vname = elem.split(":")
                        if len(vname) != 2:
                            raise ValueError(
                                "Unexpected format for timing variable {}".format(
                                    elem))

                        self.__vars.append(vname[0])

                elif ':' in var_name:
                    vname = var_name.split(":")
                    if len(vname) != 2:
                        raise ValueError(
                            "Unexpected format for timing variable {}".format(
                                var_name))

                    self.__vars.append(vname[0])

            elif var_name != "":
                # Check that the order of the name matches the var list
                vlist = self.get_var_name_split(var_name)
                if len(vlist) != len(self.__vars):
                    raise ValueError(
                        "The variable name used in table {}.{} is inconsistent in length".format(
                            self.__block, self.__pname))
                else:
                    for i in range(len(vlist)):
                        if vlist[i] != self.__vars[i]:
                            raise ValueError(
                                "Invalid variable name used in {}.{} due to inconsistencies found".format(
                                    self.__block, self.__pname))

        else:
            self.logger.debug("Unable to add delay {}: {}".format(
                var_name, delay))

        return is_added

    def get_variables(self):
        '''
        :return the list of variables associated to this arc
        delay table
        '''

        return self.__vars

    def get_variable_delay(self, vname):
        '''
        Get the delay pointed to by the variable name (concatenated).
        If it cannot find the name, then None is returned.

        :param vname: The variable name read from the file
        :return the delay in integer
        '''
        delay = None

        # If the table has 0 pcount, then vname
        # should be ""
        if vname in self.__name2delay:
            delay = self.__name2delay[vname]

        return delay
