# Utility function for signal / clk calculation
from __future__ import annotations
from decimal import Decimal, ROUND_HALF_UP
from numbers import Number

import util.excp as app_excp

def calc_phase_shifted_time_ps(freq_mhz: Number, ps_degree: Number) -> Decimal:
    """
    Calculate the phase shifted time in ps
    """
    return Decimal(1 / freq_mhz) * (10**6) * Decimal(ps_degree) / 360


def calc_phase_shifted_time_diff_ps(a_freq_mhz: Number, a_ps_degree: Number,
                                    b_freq_mhz: Number, b_ps_degree: Number):
    """
    Calculate the phase shifted time difference between two clocks
    Return unit in ps
    """
    fast_clk_phase_diff = calc_phase_shifted_time_ps(a_freq_mhz, a_ps_degree)
    slow_clk_phase_diff = calc_phase_shifted_time_ps(b_freq_mhz, b_ps_degree)
    return fast_clk_phase_diff - slow_clk_phase_diff, fast_clk_phase_diff, slow_clk_phase_diff


def format_dec(value: Decimal | float | int, precision: int) -> str:
    """
    Utility function to format decimal value to string for print
    """
    return format(Decimal(value), f".{precision}f")


def format_signal_time(value_ps: Decimal, precision: int) -> str:
    """
    Utility function to format time (in ps) to string for print
    """
    unit = 'ps'
    if value_ps > 10 ** 6:
        return f'{format_dec(value_ps / 10**6, precision)} ms'
    elif value_ps > 10 ** 3:
        return f'{format_dec(value_ps / 10**3, precision)} ns'
    else:  # Keep as picoseconds
        return f'{format_dec(value_ps, precision)} {unit}'


def format_frequency(value_mhz: Decimal | float | int, precision: int) -> str:
    """
    Utility function to format frequency (in MHz) to string for print
    """
    return f'{format_dec(value_mhz, precision)} MHz'


def float_to_decimal(value: float | Decimal, precision: int, rounding: str = ROUND_HALF_UP) -> Decimal:
    """
    Utility function to convert float or Decimal to Decimal with given precision
    Precision: number of digit after decimal point
    rounding: rounding method, should be something like ROUND_xxx. Please refer to decimal doc
    """
    value = Decimal(str(value))
    exp = Decimal(str(1 / (10 ** precision)))
    return value.quantize(exp, rounding=rounding)

def calc_period_in_ps(freq_mhz: float | Decimal) -> Decimal:
    """
    Utility to convert frequency in MHz to period in ps. Assumed that
    the frequency is in the MHz unit
    """
    if freq_mhz is not None and freq_mhz > 0.0:
        return Decimal(1 / freq_mhz) * (10**6)        
    
    raise app_excp.PTException('calc_period_in_ps: Frequency is None or 0', app_excp.MsgLevel.error)