"""

Copyright (C) 2017-2018 Efinix Inc. All rights reserved.

No portion of this code may be reused, modified or
distributed in any way without the expressed written
consent of Efinix Inc.

Created on Dec 11, 2017

@author: yasmin
"""

from contextlib import contextmanager
import sys
import os
import re
from typing import Callable, Any
import platform
from pathlib import Path

from PyQt5 import QtGui, QtCore, QtWidgets

sys.path.append(os.path.join(os.path.dirname(__file__), '..'))

def create_subsuper_text(start_text, end_text, subsuper_text, is_subscript=True):
    """
    Create a string with subscript or superscript text

    :param start_text: Normal format start text
    :param end_text: Normal format end text
    :param subsuper_text: Text to be formatted as super/sub text
    :param is_subscript: True, format as subcript, else superscript
    :return: Formatted text
    """

    if is_subscript:
        align_format = "sub"
    else:
        align_format = "super"

    full_text = "<html><head/><body><p>{}<span style = \"vertical-align:{};\">{}</span>{}</p></body></html>"\
        .format(start_text, align_format, subsuper_text, end_text)

    return full_text

def change_widget_bg_colour(widget, new_colour):
    """
    Change a widget background colour

    :param widget: Target widget
    :param new_colour: New colour to set
    """
    palette = widget.palette()
    palette.setColor(widget.backgroundRole(), new_colour)
    widget.setPalette(palette)


def change_widget_fg_colour(widget, new_colour):
    """
    Change a widget foreground colour

    :param widget: Target widget
    :param new_colour: New colour to set
    """
    palette = widget.palette()
    palette.setColor(widget.foregroundRole(), new_colour)
    widget.setPalette(palette)


def get_invalid_config_colour():
    """
    Get colour for widget with invalid configuration

    :return: A colour
    """
    return QtGui.QColor(255, 0, 0, 10)


def get_valid_config_colour():
    """
    Get colour for widget with valid configuration

    :return: A colour
    """
    return QtCore.Qt.white


def set_config_widget(widget, value, tooltip, valid=True):
    """
    For specified widget, set its configuration based on widget type.
    The widget can be changed depending on whether value is valid or invalid.

    :param widget: Target widget
    :param value: Current value to display in widget
    :param tooltip: Tooltip to display when mouse hovers
    :param valid: True, if the value is a valid value, else the value is invalid
    """
    is_set = False
    if isinstance(widget, QtWidgets.QLineEdit):
        widget.setText(value)

    elif isinstance(widget, QtWidgets.QComboBox):
        is_set = True
        if valid is False:
            idx = widget.findData("INVALID")
            if idx == -1:
                widget.addItem(value, userData="INVALID")
                idx = widget.findData("INVALID", role=QtCore.Qt.UserRole)
            else:
                widget.setItemText(idx, value)

            widget.setCurrentIndex(idx)
            widget.setStyleSheet("""
                                 QComboBox { background-color: rgba(220, 53, 69, 1); }
                                 """)
        else:
            idx = widget.findData("INVALID")
            if idx != -1:
                widget.removeItem(idx)
            widget.setCurrentText(value)
            widget.setStyleSheet("")

    elif isinstance(widget, QtWidgets.QSpinBox):
        is_set = True
        widget.setValue(value)
        if valid:
            widget.setSpecialValueText("")
        else:
            widget.setSpecialValueText(str(value))

    elif isinstance(widget, QtWidgets.QDoubleSpinBox):
        is_set = True
        widget.setValue(value)
        if valid:
            widget.setSpecialValueText("")
        else:
            widget.setSpecialValueText(str(value))

    elif isinstance(widget, QtWidgets.QLabel):
        is_set = True
        widget.setText(value)
        widget.setToolTip(tooltip)
        if valid:
            change_widget_fg_colour(widget, QtCore.Qt.black)
        else:
            change_widget_fg_colour(widget, QtCore.Qt.red)

    elif isinstance(widget, QtWidgets.QCheckBox):
        widget.setChecked(value)

    elif isinstance(widget, QtWidgets.QGroupBox):
        widget.setChecked(value)

    if not is_set:
        if tooltip is not None:
            widget.setToolTip(tooltip)

        if valid:
            change_widget_bg_colour(widget, get_valid_config_colour())
        else:
            change_widget_bg_colour(widget, get_invalid_config_colour())


def is_cb_checked(checkbox_widget):
    """
    Check the QCheckBox widget state.

    :param checkbox_widget: Widget to test
    :return: True, if checked, else False
    """
    check_state = checkbox_widget.checkState()
    if check_state == QtCore.Qt.Checked:
        return True

    return False


def is_cb_state_checked(checkbox_widget_state):
    """
    Check the QCheckBox widget state.

    :param checkbox_widget_state: Widget state to test
    :return: True, if checked, else False
    """
    if checkbox_widget_state == QtCore.Qt.Checked:
        return True

    return False


def set_le_enabled(line_edit, is_enabled):
    """
    Enable/disable QLineEdit and change its background colour.
    If enable, the colour is white. Else, it is grey.

    :param line_edit: A QLineEdit widget
    :param is_enabled: True, enable else disable
    """

    if line_edit is None:
        return

    # Qt should do this automatically!
    line_edit.setEnabled(is_enabled)
    if is_enabled:
        change_widget_bg_colour(line_edit, QtCore.Qt.white)
        change_widget_fg_colour(line_edit, QtCore.Qt.black)
    else:
        colour = QtGui.QColor("#C0C0C0")
        change_widget_bg_colour(line_edit, colour)
        change_widget_fg_colour(line_edit, colour)


def enable_case_sensitive(combo_box):
    """
    Make QComboBox to be case-sensitive. By default, it is case-insensitive.

    :param combo_box: QComboBox
    """
    if combo_box != None:
        completer = combo_box.completer()
        if completer != None:
            completer.setCaseSensitivity(QtCore.Qt.CaseSensitive)


def set_child_combo_spinbox_filter(config_object):
    """
    Disable QComboBox, QSpinBox, QDoubleSpinBox scrolling feature

    :param config_object: UI Configuration Object
    """
    for cb_child in config_object.parent.findChildren(QtWidgets.QComboBox):
        cb_child.installEventFilter(config_object)
    for sb_child in config_object.parent.findChildren(QtWidgets.QSpinBox):
        sb_child.installEventFilter(config_object)
    for sb_child in config_object.parent.findChildren(QtWidgets.QDoubleSpinBox):
        sb_child.installEventFilter(config_object)


def natural_order_sort_in_place(name_list):
    """
    Given a list of string, it does in place sorting using natural sort order.
    Adapted from https://blog.codinghorror.com/sorting-for-humans-natural-sort-order/

    :param name_list: A list of name
    """
    convert = lambda text: int(text) if text.isdigit() else text
    alphanum_key = lambda key: [convert(c) for c in re.split('([0-9]+)', key)]
    name_list.sort(key=alphanum_key)

def remove_layout_children(layout):
    """
    Delete all layout's widgets

    :param layout: A layour
    """
    if(isinstance(layout, QtWidgets.QLayout) or
            isinstance(layout, QtWidgets.QVBoxLayout) or
            isinstance(layout, QtWidgets.QHBoxLayout) ):
        while(layout.count() !=0 ):
            child = layout.takeAt(0)
            if(child.layout() is not None):
                remove_layout_children(child.layout())
            elif(child.widget() is not None):
                widget = child.widget()
                widget.setParent(None)
                layout.removeWidget(widget)
                widget.deleteLater()
                widget = None
    else:
        raise TypeError(f'Unexpected type {type(layout)}, expecting Qlayout')

def extract_number(text: str):
    """
    Get a tuple of numbers in the text
    """
    pattern = r"(\d+)"
    result = re.findall(pattern, text)
    return tuple(map(int, result))

def extract_float(text: str):
    """
    Get a tuple of float in the text
    """
    pattern = r"\b(\d+\.\d+?)"
    result = re.findall(pattern, text)
    return tuple(map(float, result))

def extract_text(text: str):
    """
    Get text without number
    """
    name_regex = r"([a-zA-Z][a-zA-Z_]*)+"
    text_list = re.findall(name_regex, text)
    return "".join(text_list)

# Sorting text by checking alphabets and all the numbers
# Return tuple of alphabets, numbers and original text
sort_func: Callable[[str], Any] = lambda text: (
    extract_text(text), # Sort text first
    extract_number(text), # Sort tuple of text number
    text, # default
)

def get_file_size_order(text: str):
    """
    Get the order of file size
    """
    file_size_order = {
        "B": 0,
        "KB": 1,
        "MB": 2,
        "GB": 3,
        "TB": 4,
        "PB": 5,
        "EB": 6,
        "ZB": 7,
        "YB": 8,
    }
    # Get the units of file size
    unit = extract_text(text)
    unit_order = file_size_order.get(unit, 99)

    # Get the number of file size
    numbers = extract_number(text)
    return (unit_order, numbers)

def get_frequency_order(text: str):
    """
    Get the order of frequency
    """
    frequency_order = {
        "Hz": 0,
        "kHz": 1,
        "MHz": 2,
        "GHz": 3,
        "THz": 4,
        "PHz": 5,
        "EHz": 6,
        "ZHz": 7,
        "YHz": 8,
    }
    # Get the units of frequency
    unit = extract_text(text)
    unit_order = frequency_order.get(unit, 99)

    # Get the number of frequency
    number = extract_number(text)[0]
    return (unit_order, number)


class TabWidgetService:
    """
    Provide operation on tab widget to dynamically change tab visibility.

    ..note::
      Starting from Qt5.15, QTabWidget supports setTabVisible().
      Once that version is available, this service can be updated to remove
      custom implementation.

      This service must be stateless, it only has access to the tab widget.
      This way, the service can be instantiated and use anywhere.
    """

    def __init__(self, tab_widget:QtWidgets.QTabWidget):
        self.tab_widget = tab_widget

    def clear_all(self):
        """
        Remove all tabs.

        ..note::
          Refactored from GPIO Bus, EditBusMemberDialog
        """

        # Remove all tabs other than base tab
        # Reference on how to remove tabs:
        # https://stackoverflow.com/questions/33005411/qt-creator-qtabwidget-i-cant-remove-a-tab

        # Work backwards since removeTab will impact the existing
        # indexing of tabs (see link above).
        self.max_index = self.tab_widget.count()
        for rmv_idx in range(self.max_index - 1, -1, -1):
            self.tab_widget.removeTab(rmv_idx)

    def add_tab(self, tab_page_widget:QtWidgets.QWidget, title:str):
        tindex = self.tab_widget.addTab(tab_page_widget, title)

    def remove_tab(self, tab_index:int):
        self.tab_widget.removeTab(tab_index)

    def is_tab_exist(self, tab_name):
        """
        Check if a tab exists

        :param tab_name: The tab label or title
        :return: True, if visible
        """
        for index in range(self.tab_widget.count()):
            if self.tab_widget.tabText(index) == tab_name:
                return True

        return False

    def get_tab_widget(self, tab_name):
        """
        Get tab page widget for speficied tab

        :param tab_name: The tab label or title
        :return: A widget, else None
        """
        for index in range(self.tab_widget.count()):
            if self.tab_widget.tabText(index) == tab_name:
                return self.tab_widget.widget(index)

        return None

    def set_tab_visible(self, tab_name, visible):
        tab = self.get_tab_widget(tab_name)
        if tab:
            idx = self.tab_widget.indexOf(tab)
            if idx != -1 and idx < self.tab_widget.count():
                self.tab_widget.setTabVisible(idx, visible)


def remove_all_tabs(tab_widget: QtWidgets.QTabWidget):
    tab_svc = TabWidgetService(tab_widget)
    tab_svc.clear_all()


@contextmanager
def block_signal(widget: QtWidgets.QWidget):
    """
    PyQt QSignalBlocker in context manager style
    """
    try:
        widget.blockSignals(True)
        yield
    finally:
        widget.blockSignals(False)

class WindowsFilenameValidator():
    WINDOWS_KEYWORD = {'con', 'prn', 'aux', 'nul', 'com1', 'com2', 'com3', 'com4', 'com5', 'com6',
        'com7', 'com8', 'com9', 'lpt1', 'lpt2', 'lpt3', 'lpt4', 'lpt5', 'lpt6',
        'lpt7', 'lpt8', 'lpt9'}

    def check_windows_reserved_keyword(self, name: str):
        """
        Check if the given name is a Windows reserved keyword.
        Args:
            name (str): The filename to check
        Returns:
            bool: True if the name is a Windows reserved keyword, False otherwise
        """
        if platform.system() == 'Windows' and Path(name).stem.lower() in self.WINDOWS_KEYWORD:
            return True
        else:
            return False

if __name__ == "__main__":
    pass
