"""
Copyright (C) 2017-2018 Efinix Inc. All rights reserved.

No portion of this code may be reused, modified or
distributed in any way without the expressed written
consent of Efinix Inc.

Created on Nov 3, 2017

@author: yasmin
"""

import sys
import os
from enum import Enum, auto
from typing import Dict
from configparser import ConfigParser
from dataclasses import dataclass
from typing import List

from util.excp import PTInteralAssertException

EFINITY_HOME = os.environ['EFINITY_HOME']
sys.path.append(os.path.join(EFINITY_HOME, 'scripts'))
from _app_info import _AppInfo

class AppSetting(object):
    """
    Handles application paths based on $EFXPT_HOME

    All paths are normalized based on platforms.
    """

    class PathType(Enum):
        """
        Type for specific document path
        """
        install = auto()
        bin = auto()
        resource = auto()
        schema = auto()
        device_db = auto()
        main_html_help = auto()
        gpio_html_help = auto()
        pll_html_help = auto()
        osc_html_help = auto()
        iobank_html_help = auto()
        gpio_bus_html_help = auto()
        lvds_html_help = auto()
        ctrl_html_help = auto()
        pll_adv_html_help = auto()
        mipi_html_help = auto()
        jtag_html_help = auto()
        h264_html_help = auto()
        ddr_html_help = auto()
        res_asg_html_help = auto()
        pll_clk_calculator_html_help = auto()
        osc_adv_html_help = auto()
        mipi_dphy_html_help = auto()
        pll_comp_html_help = auto()
        lvds_adv_html_help = auto()
        gpio_comp_html_help = auto()
        clockmux_html_help = auto()
        spi_flash_html_help = auto()
        hyper_ram_html_help = auto()
        iobank_adv_html_help = auto()
        mipi_adv_html_help = auto()
        pll_ssc_html_help = auto()
        ddr_adv_html_help = auto()
        quad_pcie_html_help = auto()
        eth_10g_html_help = auto()
        sgmii_html_help = auto()
        raw_serdes_html_help = auto()
        pkg_view_html_help = auto()
        pkg_view_config_html_help = auto()
        pkg_view_pin_browser_html_help = auto()
        qcrv_html_help = auto()


    app_path: Dict[PathType, str] = {}
    '''
    A map of path type to its full path
    '''

    def __init__(self):

        self.app_name = f"{_AppInfo.GUI_COMPANY_NAME} Interface Designer"
        self.output_dir = "outflow"
        self.work_dir = "work_pt"
        self.migration_log_file = "migration.log"

        count = len(self.app_path)
        if count == 0:
            AppSetting.app_path[AppSetting.PathType.install] = os.path.normpath(
                os.getenv("EFXPT_HOME", ""))

            AppSetting.app_path[AppSetting.PathType.bin] = os.path.normpath(
                AppSetting.app_path[AppSetting.PathType.install] + "/bin")

            AppSetting.app_path[AppSetting.PathType.resource] = os.path.normpath(
                AppSetting.app_path[AppSetting.PathType.bin] + "/resource")

            AppSetting.app_path[AppSetting.PathType.schema] = os.path.normpath(
                AppSetting.app_path[AppSetting.PathType.bin] + "/schema")

            AppSetting.app_path[AppSetting.PathType.device_db] = os.path.normpath(
                AppSetting.app_path[AppSetting.PathType.install] + "/db")

            AppSetting.app_path[AppSetting.PathType.main_html_help] = os.path.normpath(
                AppSetting.app_path[AppSetting.PathType.install] + "/doc/topics/hlp-id-intro.html")

            AppSetting.app_path[AppSetting.PathType.gpio_html_help] = os.path.normpath(
                AppSetting.app_path[AppSetting.PathType.install] + "/doc/topics/hlp-gpio-intro.html")

            AppSetting.app_path[AppSetting.PathType.pll_html_help] = os.path.normpath(
                AppSetting.app_path[AppSetting.PathType.install] + "/doc/topics/hlp-id-pll-intro.html")

            AppSetting.app_path[AppSetting.PathType.osc_html_help] = os.path.normpath(
                AppSetting.app_path[AppSetting.PathType.install] + "/doc/topics/hlp-id-osc.html")

            AppSetting.app_path[AppSetting.PathType.iobank_html_help] = os.path.normpath(
                AppSetting.app_path[AppSetting.PathType.install] + "/doc/topics/hlp-id-iobank.html")

            AppSetting.app_path[AppSetting.PathType.gpio_bus_html_help] = os.path.normpath(
                AppSetting.app_path[AppSetting.PathType.install] + "/doc/topics/hlp-id-gpio-bus.html")

            AppSetting.app_path[AppSetting.PathType.lvds_html_help] = os.path.normpath(
                AppSetting.app_path[AppSetting.PathType.install] + "/doc/topics/hlp-id-lvds.html")

            AppSetting.app_path[AppSetting.PathType.ctrl_html_help] = os.path.normpath(
                AppSetting.app_path[AppSetting.PathType.install] + "/doc/topics/hlp-id-config.html")

			# All PLL point to the same help now
            AppSetting.app_path[AppSetting.PathType.pll_adv_html_help] = os.path.normpath(
                AppSetting.app_path[AppSetting.PathType.install] + "/doc/topics/hlp-id-pll-intro.html")

            AppSetting.app_path[AppSetting.PathType.mipi_html_help] = os.path.normpath(
                AppSetting.app_path[AppSetting.PathType.install] + "/doc/topics/hlp-id-mipi-csi2.html")

            AppSetting.app_path[AppSetting.PathType.jtag_html_help] = os.path.normpath(
                AppSetting.app_path[AppSetting.PathType.install] + "/doc/topics/hlp-id-jtag.html")

            AppSetting.app_path[AppSetting.PathType.h264_html_help] = os.path.normpath(
                AppSetting.app_path[AppSetting.PathType.install] + "/doc/topics/concept_interface_h264.html")

            AppSetting.app_path[AppSetting.PathType.ddr_html_help] = os.path.normpath(
                AppSetting.app_path[AppSetting.PathType.install] + "/doc/topics/hlp-id-ddr.html")

            AppSetting.app_path[AppSetting.PathType.res_asg_html_help] = os.path.normpath(
                AppSetting.app_path[AppSetting.PathType.install] + "/doc/topics/hlp-id-resource-assigner.html")

            AppSetting.app_path[AppSetting.PathType.pll_clk_calculator_html_help] = os.path.normpath(
                AppSetting.app_path[AppSetting.PathType.install] + "/doc/topics/hlp-id-pll-calculator.html")

			# Titanium Interface help links
            AppSetting.app_path[AppSetting.PathType.mipi_dphy_html_help] = os.path.normpath(
                AppSetting.app_path[AppSetting.PathType.install] + "/doc/topics/titanium-hsio-as-mipi.html")

            AppSetting.app_path[AppSetting.PathType.pll_comp_html_help] = os.path.normpath(
                AppSetting.app_path[AppSetting.PathType.install] + "/doc/topics/hlp-id-pll-intro.html")

            AppSetting.app_path[AppSetting.PathType.lvds_adv_html_help] = os.path.normpath(
                AppSetting.app_path[AppSetting.PathType.install] + "/doc/topics/hlp-id-lvds.html")

            AppSetting.app_path[AppSetting.PathType.gpio_comp_html_help] = os.path.normpath(
                AppSetting.app_path[AppSetting.PathType.install] + "/doc/topics/hlp-gpio-intro.html")

            AppSetting.app_path[AppSetting.PathType.clockmux_html_help] = os.path.normpath(
                AppSetting.app_path[AppSetting.PathType.install] + "/doc/topics/hlp-id-clk-network.html")

            AppSetting.app_path[AppSetting.PathType.osc_adv_html_help] = os.path.normpath(
                AppSetting.app_path[AppSetting.PathType.install] + "/doc/topics/hlp-id-osc.html")

            AppSetting.app_path[AppSetting.PathType.spi_flash_html_help] = os.path.normpath(
                AppSetting.app_path[AppSetting.PathType.install] + "/doc/topics/hlp-id-spi-flash-intro.html")

            AppSetting.app_path[AppSetting.PathType.hyper_ram_html_help] = os.path.normpath(
                AppSetting.app_path[AppSetting.PathType.install] + "/doc/topics/hlp-id-hyperram-intro.html")

            AppSetting.app_path[AppSetting.PathType.iobank_adv_html_help] = os.path.normpath(
                AppSetting.app_path[AppSetting.PathType.install] + "/doc/topics/hlp-id-iobank.html")

            AppSetting.app_path[AppSetting.PathType.mipi_adv_html_help] = os.path.normpath(
                AppSetting.app_path[AppSetting.PathType.install] + "/doc/topics/titanium-mipi.html")

            AppSetting.app_path[AppSetting.PathType.ddr_adv_html_help] = os.path.normpath(
                AppSetting.app_path[AppSetting.PathType.install] + "/doc/topics/hlp-id-ddr.html")

            AppSetting.app_path[AppSetting.PathType.pll_ssc_html_help] = os.path.normpath(
                AppSetting.app_path[AppSetting.PathType.install] + "/doc/topics/hlp-id-pll-ssc.html")

            AppSetting.app_path[AppSetting.PathType.quad_pcie_html_help] = os.path.normpath(
                AppSetting.app_path[AppSetting.PathType.install] + "/doc/topics/hlp-id-pcie.html")

            AppSetting.app_path[AppSetting.PathType.eth_10g_html_help] = os.path.normpath(
                AppSetting.app_path[AppSetting.PathType.install] + "/doc/topics/hlp-id-10gbase-kr.html")

            AppSetting.app_path[AppSetting.PathType.sgmii_html_help] = os.path.normpath(
                AppSetting.app_path[AppSetting.PathType.install] + "/doc/topics/hlp-id-sgmii.html")

            AppSetting.app_path[AppSetting.PathType.raw_serdes_html_help] = os.path.normpath(
                AppSetting.app_path[AppSetting.PathType.install] + "/doc/topics/hlp-id-pma-direct.html")

            AppSetting.app_path[AppSetting.PathType.pkg_view_html_help] = os.path.normpath(
                AppSetting.app_path[AppSetting.PathType.install] + "/doc/topics/efn-pkg-planner.html")

            AppSetting.app_path[AppSetting.PathType.pkg_view_config_html_help] = os.path.normpath(
                AppSetting.app_path[AppSetting.PathType.install] + "/doc/topics/efn-pkg-planner-select.html")

            AppSetting.app_path[AppSetting.PathType.pkg_view_pin_browser_html_help] = os.path.normpath(
                AppSetting.app_path[AppSetting.PathType.install] + "/doc/topics/efn-pkg-planner-browse.html")

            AppSetting.app_path[AppSetting.PathType.qcrv_html_help] = os.path.normpath(
                AppSetting.app_path[AppSetting.PathType.install] + "/doc/topics/hlp-id-riscv.html")

    def check(self):
        """
        Check and make sure all path setting not empty

        :returns A tuple of (pass state, path). If fail, the path is the path that is missing
        """
        missing_path = ""
        is_pass = True
        for path in AppSetting.app_path.values():
            if path == "":
                is_pass = False
                missing_path = path
                break

        return is_pass, missing_path

    def _get_config_setting(self, category_name:str , setting_name: str, is_optional=False):
        """
        Get setting in the specified category_name

        :param setting_name: Setting name inside the category_name
        :return: Setting value
        """        
        config_file = f"{AppSetting.app_path[AppSetting.PathType.install]}/config.ini"
        config = ConfigParser()
        config.read(config_file)

        if category_name not in config and not is_optional:
            raise PTInteralAssertException(f'Unable to find {category_name} setting')
        
        setting_value = ""
        if category_name in config:
            cat_options = config[category_name]
            setting_value = cat_options.get(setting_name, "")

        return setting_value
          
    def get_general_setting(self, setting_name):
        return self._get_config_setting("general", setting_name, True)

    def get_exim_setting(self, setting_name):
        """
        Get setting for export/import feature

        :param setting_name: Setting name inside the exim category
        :return: Setting value
        """
        return self._get_config_setting("exim", setting_name)

    def get_style_setting_file(self):
        return f"{AppSetting.app_path[self.PathType.install]}/style.ini"

    def get_project_setting_list(self, design_path: str, setting_names: List[str]):
        setting_map = {}

        if design_path != "":
            setting_file = os.path.normpath(
                '{}/efx_interface_settings.ini'.format(design_path))
            
            if os.path.exists(setting_file):
                config = ConfigParser()
                config.read_string('[DEFAULT]\n' + open(setting_file).read())

                for setting in setting_names:
                    setting_val = config['DEFAULT'].get(setting, "")
                    if setting_val != "":
                        setting_map[setting] = setting_val

        return setting_map
    
@dataclass
class SchemaSetting:
    app_ns_element_tag: str
    app_ns_attr: str  #: Application namespace atttribute
    app_ns_attr_tag: str
    xml_ns_attr: str  #: xml namespace attribute
    xml_ns_attr_tag: str
    app_schema: str  #: Application schema file

class SchemaService(SchemaSetting):
    """
    Application schema setting
    """
    def __init__(self):
        self.setting = SchemaService.build_setting()

    @staticmethod
    def build_setting():
        setting = SchemaSetting(
            app_ns_element_tag="efxpt",
            app_ns_attr="http://www.efinixinc.com/peri_design_db",
            app_ns_attr_tag="xmlns:efxpt",
            xml_ns_attr="http://www.w3.org/2001/XMLSchema-instance",
            xml_ns_attr_tag="xmlns:xsi",
            app_schema=""
        )

        return setting

    def write_design_namespace_attr(self, writer):
        writer.writeAttribute(self.setting.app_ns_attr_tag, self.setting.app_ns_attr)
        writer.writeAttribute(self.setting.xml_ns_attr_tag, self.setting.xml_ns_attr)

