"""
Copyright (C) 2017-2020 Efinix Inc. All rights reserved.

No portion of this code may be reused, modified or
distributed in any way without the expressed written
consent of Efinix Inc.

Created on Nov 1, 2020

@author: maryam
"""

from __future__ import annotations
import abc
from typing import Dict, TYPE_CHECKING, Any

import util.gen_util

import common_device.rule as base_rule
import common_device.pll.pll_rule as an08_pllrule

import an20_device.pll.pll_rule_adv as an20_pllrule
from tx60_device.gpio.gpio_design_comp import GPIOInputComplex
from tx60_device.lvds.lvds_design_adv import LVDSRxAdvance

if TYPE_CHECKING:
    from tx60_device.pll.pll_design_comp import PLLComplex, PLLOutputClockComplex


@util.gen_util.freeze_it
class PLLCheckerComp(an20_pllrule.PLLCheckerAdv):

    def __init__(self, design):
        """
        Constructor
        """
        self.mipitx_fastclk_names = {}
        self.mipitx_slowclk_names = {}

        super().__init__(design)

    def _build_rules(self):
        """
        Build pll rule database
        """
        self._add_rule(an08_pllrule.PLLRuleInstanceName())
        self._add_rule(an08_pllrule.PLLRuleInputFreq())
        self._add_rule(PLLRuleOutputDivider())
        self._add_rule(an08_pllrule.PLLRuleOutputClock())
        self._add_rule(an08_pllrule.PLLRuleOutputNumber())
        self._add_rule(an08_pllrule.PLLRuleResource())
        self._add_rule(an20_pllrule.PLLRuleVCOFreq())
        self._add_rule(PLLPostVCOFrequency())
        self._add_rule(an08_pllrule.PLLRuleOutputClockFreq())
        self._add_rule(an08_pllrule.PLLRuleOutputName())
        self._add_rule(PLLRuleRefClk())
        self._add_rule(PLLRuleFeedback())
        self._add_rule(an20_pllrule.PLLRuleFeedbackResource())
        self._add_rule(an08_pllrule.PLLRuleInputFreqLimit())
        self._add_rule(an20_pllrule.PLLRuleNonInternalFeedbackFreq())
        self._add_rule(PLLRuleIOFeedbackStd())
        self._add_rule(PLLRuleInternalFeedback())
        self._add_rule(PLLRuleDynamicShiftPinEmptyName())
        self._add_rule(PLLRuleDynamicShiftPinInvalidName())
        self._add_rule(PLLRuleDynamicShiftFeedback())
        self._add_rule(PLLRuleConfig())
        self._add_rule(PLLRuleMultiplier())
        self._add_rule(PLLRulePreDivider())
        self._add_rule(PLLRulePostDivider())
        self._add_rule(PLLRuleMIPIDPHYTxClock())
        self._add_rule(an20_pllrule.PLLRuleClockSelectorPinName())
        self._add_rule(an08_pllrule.PLLRuleDividerRelationWhenCascade())
        self._add_rule(PLLRuleClockDividerPhaseShift())
        self._add_rule(PLLRuleRegionalFeedbackClock())

        # Not required for Titanium PLL
        # self._add_rule(an20_pllrule.PLLRulePhaseShiftDiv())

    def check_depended_resources(self, gpio_reg, lvds_reg, dev_res_list,
                                 expected_conn_type, lvds_expected_conn_type):
        '''
        Function use to check if the depended resources
        are configured correctly.

        :param gpio_reg: GPIO Design Registry
        :param lvds_reg: LVDS Design Registry
        :param dev_res_list: List of device instance
                names that is going to be investigated
        :param expected_conn_type: The expected connection type to check
        :return: a tuple of:
            1) count: The number of resource that was configured
                    based on the list
            2) non_global: Indication that the resource was not set
                    to alternate connection type
            3) invalid_gpio: a string of gpio resource names that
                    are invalid
        '''
        count = 0
        non_global = False
        gpio_name = ""

        for gpio_res in dev_res_list:
            tmp_name = ""

            gpio = gpio_reg.get_gpio_by_device_name(gpio_res)
            if gpio is None:

                # Check if it is configured as LVDS
                if lvds_reg is not None:
                    lvds = lvds_reg.get_inst_by_device_name(gpio_res)
                    if lvds is not None:
                        # Check that it was configured as alternate type
                        # TODO: consolidate with PLLRUleAdv where the difference is
                        # only that PLLComplex allow LVDS Bidir
                        if (lvds.ops_type == lvds.OpsType.op_rx or \
                                lvds.ops_type == lvds.OpsType.op_bidir) and\
                                lvds.rx_info is not None:
                            lvds_rx = lvds.rx_info
                            if lvds_rx.conn_type == lvds_expected_conn_type:
                                count += 1

                            else:
                                non_global = True
                                tmp_name = gpio_res

            # Check the validity of the CLKIN Resource
            else:
                # If there is dependent instances, then we
                # check its usage. This could be an lvds gpio
                # as well.
                if gpio.mode == gpio.PadModeType.input:
                    gpio_input = gpio.input
                    if gpio_input is not None:
                        if gpio_input.conn_type == expected_conn_type:
                            count += 1

                        else:
                            non_global = True
                            tmp_name = gpio_res

            if tmp_name != "":
                if gpio_name != "":
                    gpio_name += "," + tmp_name
                else:
                    gpio_name = tmp_name

        return count, non_global, gpio_name

    def get_all_mipi_tx_clock_names(self):
        fastclk_names = []
        slowclk_names = []

        if self.design is not None and self.design.mipi_dphy_reg is not None:
            mipid_reg = self.design.mipi_dphy_reg

            # Get only the mipi tx instances
            tx_inst_list = mipid_reg.get_all_tx_inst()

            if tx_inst_list:
                for tx_inst in tx_inst_list:
                    if tx_inst is not None:
                        if tx_inst.tx_info is not None and tx_inst.tx_info.gen_pin is not None:
                            tx_gpin = tx_inst.tx_info.gen_pin

                            # Get the fast and slow clock from the gen pin regardless of the mode
                            # since the clocks are applicable in both modes
                            fastclk_pin = tx_gpin.get_pin_by_type_name("FASTCLK")
                            if fastclk_pin is not None and fastclk_pin.name != "":
                                if fastclk_pin.name not in fastclk_names:
                                    fastclk_names.append(fastclk_pin.name)

                            slowclk_pin = tx_gpin.get_pin_by_type_name("SLOWCLK")
                            if slowclk_pin is not None and slowclk_pin.name != "":
                                if slowclk_pin.name not in slowclk_names:
                                    slowclk_names.append(slowclk_pin.name)

        # Override the data member
        self.mipitx_fastclk_names = fastclk_names
        self.mipitx_slowclk_names = slowclk_names

    def run_all_check(self, exclude_rule=None):
        """
        Run all checks on all lvds in registry

        :param exclude_rule: A list of excluded rule name
        """
        self.get_all_mipi_tx_clock_names()
        return super().run_all_check(exclude_rule)

class PLLRuleRefClk(an20_pllrule.PLLRuleRefClk):
    """
    Checks the validity of the Reference clock
    """

    def check_external(self, checker, design_block: PLLComplex):
        '''
        Pre-requisite: It is assumed that the pll already has a resource
        assigned (pll_def != "")
        :param checker:
        :param design_block:
        :return:
        '''
        pll = design_block

        valid = True

        gpio_reg = checker.design.gpio_reg
        if gpio_reg is None:
            checker.logger.warning(
                "{}: gpio reg is empty. Skipped.".format(self.name))
            self.severity = self.SeverityType.unknown
            return valid

        lvds_reg = checker.design.lvds_reg

        _, ext_count = checker.plldev_service.get_ref_clk_type_count(
            pll.pll_def)

        # Design has it such that based on the number of external clock
        # that the resource supports (mapping of pin index):
        # 3 ->  1: ext_clock_0, 2: ext_clock_1, 3: ext_clock_2
        # 2 ->  1: ext_clock_0, 2: ext_clock_1
        is_found_pin1 = False
        if pll.get_refclk1_pin_name(checker.design) != "":
            is_found_pin1 = True

        is_found_pin2 = False
        if pll.get_refclk2_pin_name(checker.design) != "":
            is_found_pin2 = True

        is_found_pin3 = False
        if ext_count == 3:
            if pll.get_refclk3_pin_name(checker.design) != "":
                is_found_pin3 = True

        # Don't need to check the name matches or not because
        # it is auto populated based on resource config for external
        index_list = []

        if pll.ref_clock_mode == pll.RefClockModeType.external:

            # Either refclk1, refclk2 or refclk3 is set
            if pll.ext_ref_clock_id is None or \
                (pll.ext_ref_clock_id != pll.RefClockIDType.ext_clock_0 and
                 pll.ext_ref_clock_id != pll.RefClockIDType.ext_clock_1 and
                 (pll.ext_ref_clock_id != pll.RefClockIDType.ext_clock_2 and ext_count == 3)):

                self.error('External refclk pin has to be set in external mode')
                return False

            tmp_valid, index_list = self.check_single_external_assignment(checker, pll, ext_count, lvds_reg, gpio_reg,
                                                                          is_found_pin1, is_found_pin2, is_found_pin3)

            if not tmp_valid:
                return tmp_valid

        elif pll.ref_clock_mode == pll.RefClockModeType.dynamic:
            tmp_valid, index_list = self.check_dynamic_assignment(checker, pll, ext_count, lvds_reg, gpio_reg,
                                                                  is_found_pin1, is_found_pin2, is_found_pin3)

            # Exit if it is not valid
            if not tmp_valid:
                return tmp_valid

        # For some it would be an LVDS instance
        if not self.check_external_connections(checker, pll, index_list, gpio_reg, lvds_reg):
            valid = False

        return valid

    def check_single_external_assignment(self, checker, pll, ext_count, lvds_reg, gpio_reg,
                                         is_found_pin1, is_found_pin2, is_found_pin3):
        index_list = []

        device_db = checker.design.device_db
        pll_index = -1

        # index: 0 - ext_clock_0 - pin0
        # index: 1 - ext_clock_1 - pin1
        # If 3 external:
        # index: 2 - ext_clock_2 - pin2
        if pll.ext_ref_clock_id == pll.RefClockIDType.ext_clock_0 and not is_found_pin1:

            ext_clk0_info_list = pll.find_external_clock_info(pll.RefClockIDType.ext_clock_0,
                                                              checker.plldev_service, lvds_reg, gpio_reg, device_db)
            if ext_clk0_info_list:
                # PT-800: Provide the External resource information
                res_names = self.get_resource_names_on_ext_clock(
                    ext_clk0_info_list)
                # self.msg = 'Selected external clock 0 pin name has to be specified'
                self.msg = 'Reference clock at {} connected to external clock pin 0 has not been defined'.format(
                    res_names)

            else:
                # PT-744: Use a different message if it was an instance not
                # bonded out
                self.msg = 'Invalid external clock 0 resource selected: Resource Unbonded'

            self.error()
            return False, index_list

        elif pll.ext_ref_clock_id == pll.RefClockIDType.ext_clock_1 and not is_found_pin2:

            ext_clk1_info_list = pll.find_external_clock_info(pll.RefClockIDType.ext_clock_1,
                                                              checker.plldev_service, lvds_reg, gpio_reg, device_db)

            if ext_clk1_info_list:
                # PT-800: Provide the External resource information
                res_names = self.get_resource_names_on_ext_clock(
                    ext_clk1_info_list)
                # self.msg = 'Selected external clock 1 pin name has to be specified'
                self.msg = 'Reference clock at {} connected to external clock 1 pin has not been defined'.format(
                    res_names)

            else:
                # PT-744: Use a different message if it was an instance not
                # bonded out
                self.msg = 'Invalid external clock 1 resource selected: Resource Unbonded'

            self.error()
            return False, index_list

        elif pll.ext_ref_clock_id == pll.RefClockIDType.ext_clock_2 and\
                (not is_found_pin3 and ext_count == 3):

            ext_clk1_info_list = pll.find_external_clock_info(pll.RefClockIDType.ext_clock_2,
                                                              checker.plldev_service, lvds_reg, gpio_reg, device_db)

            if ext_clk1_info_list:
                # PT-800: Provide the External resource information
                res_names = self.get_resource_names_on_ext_clock(
                    ext_clk1_info_list)
                # self.msg = 'Selected external clock 1 pin name has to be specified'
                self.msg = 'Reference clock at {} connected to external clock pin 2 has not been defined'.format(
                    res_names)

            else:
                # PT-744: Use a different message if it was an instance not
                # bonded out
                self.msg = 'Invalid external clock 2 resource selected: Resource Unbonded'

            self.error()
            return False, index_list

        # Get the pin name that corresponds to the chosen resource
        ext_index = pll.get_external_clkin_index(
            checker.plldev_service, pll.ext_ref_clock_id)

        index_list.append(ext_index)

        return True, index_list

    def check_dynamic_assignment(self, checker, pll, ext_count, lvds_reg, gpio_reg,
                                 is_found_pin1, is_found_pin2, is_found_pin3):
        index_list = []

        # Both bonded out pin name needs to be specified in dynamic
        ext_pin1_bonded = True
        ext_pin2_bonded = True
        ext_pin3_bonded = False

        # Change the flag if the resource has 3 ext_count
        if ext_count == 3:
            ext_pin3_bonded = True

        # index: 0 - ext_clock_0 - pin1
        # index: 1 - ext_clock_1 - pin2
        # Additionally, if 3 external:
        # index: 2 - ext_clock_2 - pin3

        if not is_found_pin1 or not is_found_pin2 or \
                (not is_found_pin3 and ext_count == 3):
            device_db = checker.design.device_db

            # PT-882: Flag only if the missing pin is not the unbonded ones
            is_missing = False

            if not is_found_pin1:
                ext_pin1_bonded, tmp_missing = self.find_individual_external_clock_info(
                    pll, pll.RefClockIDType.ext_clock_0,
                    checker.plldev_service,
                    lvds_reg, gpio_reg, device_db)

                if tmp_missing:
                    is_missing = True

            if not is_found_pin2:
                ext_pin2_bonded, tmp_missing = self.find_individual_external_clock_info(
                    pll, pll.RefClockIDType.ext_clock_1,
                    checker.plldev_service,
                    lvds_reg, gpio_reg, device_db)

                if tmp_missing:
                    is_missing = True

            if ext_count > 2:
                if not is_found_pin3:
                    ext_pin3_bonded, tmp_missing = self.find_individual_external_clock_info(
                        pll, pll.RefClockIDType.ext_clock_2,
                        checker.plldev_service,
                        lvds_reg, gpio_reg, device_db)

                    if tmp_missing:
                        is_missing = True

            if is_missing:
                # Both needs to be specified in dynamic
                self.error('Bonded external reference clock pin has to be specified in dynamic mode')
                return False, index_list

        # We add the CLKIN index corresponding to each external usage
        if ext_pin1_bonded:
            index_list.append(0)
        if ext_pin2_bonded:
            index_list.append(1)
        if ext_pin3_bonded:
            index_list.append(2)

        return True, index_list

    def find_individual_external_clock_info(self, pll, ref_id, pll_dev_svc,
                                            lvds_reg, gpio_reg, device_db):

        ext_clk_info_list = pll.find_external_clock_info(ref_id, pll_dev_svc, lvds_reg, gpio_reg,
                                                         device_db)
        ext_pin_bonded = True
        is_missing = False

        if ext_clk_info_list:
            is_missing = True
        else:
            ext_pin_bonded = False

        return ext_pin_bonded, is_missing

    def check_external_connections(self, checker, pll, index_list, gpio_reg, lvds_reg):
        valid = True

        # For some it would be an LVDS instance
        for index in index_list:
            gpio_res_list = checker.plldev_service.get_all_resource_on_ref_clk_pin(
                pll.pll_def, index)

            # If it is a list of multiple items, it means that
            # either one is required. This should be using the GPIO Complex and
            # LVDSAdvance
            count, non_global, invalid_gpio = checker.check_depended_resources(
                gpio_reg, lvds_reg, gpio_res_list, GPIOInputComplex.ConnType.pll_clkin_conn,
                LVDSRxAdvance.ConnType.pll_clkin_conn)

            # By right there should only be one driver
            if count > 1:
                self.error("There can only be one configured resource for CLKIN[{}]".format(
                    index))
                valid = False
                break

            elif count == 0:
                if non_global:
                    # If there's a GPIO resource but it wasn't set as alternate
                    self.error('External reference clock resource {} is not ' \
                        'configured as pll_clkin connection'
                        .format(invalid_gpio))
                    valid = False
                    break

                else:
                    self.error("The resource for CLKIN[{}] is not configured".format(index))
                    valid = False
                    break

        return valid

    def check_core(self, pll, checker):

        if pll.ref_clock_mode == pll.RefClockModeType.core:
            # When in core mode, it is expected that it is hardcoded to refclk0
            if pll.ref_clock_name == "":
                self.error('Core refclk pin has to be specified in core mode')
                return False

        # There can be a resource where only 1 core is available
        else:
            # Check whether it's a resource with 1 or 2 core pins
            core_count, _ = checker.plldev_service.get_ref_clk_type_count(
                pll.pll_def)
            if core_count > 1:
                if pll.ref_clock_name == "" or pll.ref_clock1_name == "":
                    # Both should be specified
                    self.error('Both core refclk pins have to be specified in dynamic mode')
                    return False
            else:
                if pll.ref_clock_name == "":
                    # The only core pin should be specified
                    self.error('Core refclk pin has to be specified in dynamic mode')
                    return False

        return True

    def check(self, checker, design_block: PLLComplex):
        pll = design_block
        valid = True

        # Slightly different than PLLAdvance. Don't check if resource hasn't been
        # assigned because the number of core and external connection is associated
        # to the resource
        if pll.pll_def == "":
            return

        if pll.ref_clock_mode == pll.RefClockModeType.core or\
                pll.ref_clock_mode == pll.RefClockModeType.dynamic:
            valid = self.check_core(pll, checker)

        # Check next one only if core is valid
        if pll.ref_clock_mode == pll.RefClockModeType.external or \
                pll.ref_clock_mode == pll.RefClockModeType.dynamic:
            if valid:
                self.check_external(checker, pll)


class PLLRuleDynamicShiftPin(base_rule.Rule, metaclass=abc.ABCMeta):
    """
    Helper class for checking empty/invalid pin names
    Implmented class needs to inherit either RuleEmptyName or RuleInvalidName
    """
    @abc.abstractmethod
    def validate(self, pins: Dict[str, base_rule.PinData]):
        pass

    def check(self, checker, design_block: PLLComplex):
        util.gen_util.mark_unused(checker)

        # Get the mode
        pll = design_block

        # Iterate through all the output clock
        is_dyn_enabled = False
        out_clock_list = pll.get_output_clock_list()

        for clk in out_clock_list:
            if clk.is_dyn_phase:
                is_dyn_enabled = True
                break

        if is_dyn_enabled:
            if pll.gen_pin is not None:
                # Check that all the pins required are non-empty
                pll_gpin = pll.gen_pin

                pins: Dict[str, base_rule.PinData] = {}

                phase_pin = pll_gpin.get_pin_by_type_name("SHIFT_ENA")
                if phase_pin is not None:
                    pins["Shift Enable"] = base_rule.PinData(phase_pin.name)

                phase_pin = pll_gpin.get_pin_by_type_name("SHIFT_SEL")
                if phase_pin is not None:
                    pins["Shift Select"] = base_rule.PinData(phase_pin.name)

                phase_pin = pll_gpin.get_pin_by_type_name("SHIFT")
                if phase_pin is not None:
                    pins["Shift"] = base_rule.PinData(phase_pin.name)

                self.validate(pins)

class PLLRuleDynamicShiftPinEmptyName(base_rule.RuleEmptyName, PLLRuleDynamicShiftPin):
    """
    Check that the Phase Shift pin names are filled when dynamic phase
    shift is enabled. Phase Shift set to 0 when in dynamic phase shift
    """
    def __init__(self):
        super().__init__()
        self.name = "pll_rule_dynamic_shift_empty_pin"

    def check(self, checker, design_block: PLLComplex):
        super().check(checker, design_block)

        # Override default error message
        if not self.pass_status:
            self.msg = self.msg.replace("Empty pin names found: ", "Dynamic phase shift is enabled but missing pin names: ")

class PLLRuleDynamicShiftPinInvalidName(base_rule.RuleInvalidName, PLLRuleDynamicShiftPin):
    """
    Check that the Phase Shift pin names are valid when dynamic phase
    shift is enabled.
    """
    def __init__(self):
        super().__init__()
        self.name = "pll_rule_dynamic_shift_invalid_pin"


class PLLRuleInternalFeedback(base_rule.Rule):
    """
    Check that Internal feedback mode is not used since it
    is not supported
    """

    def __init__(self):
        super().__init__()
        self.name = "pll_rule_feedback_mode"

    def check(self, checker, design_block: PLLComplex):
        util.gen_util.mark_unused(checker)

        # Get the mode
        pll = design_block

        if pll.fb_mode == pll.FeedbackModeType.internal:
            self.error("Internal feedback mode is not supported")
            return


class PLLRuleDynamicShiftFeedback(base_rule.Rule):
    """
    Check that dynamic phase shift cannot be enabled on output
    used as feedback
    """

    def __init__(self):
        super().__init__()
        self.name = "pll_rule_dynamic_shift_feedback"

    def check(self, checker, design_block: PLLComplex):

        pll = design_block

        # Not checking the feedback mode because we don't support
        # internal. Hence, all feedback is based on the clk output
        feedback_clock_name = pll.fb_clock_name

        is_fb_dyn_pshift = False
        # Check if any of the output clock has it's dynamic
        # shift enabled
        fb_clk: Any = pll.get_output_clock(feedback_clock_name) # type: ignore
        if fb_clk is not None:
            # If dynamic phase shift is enabled
            if fb_clk.is_dyn_phase:
                is_fb_dyn_pshift = True

        if is_fb_dyn_pshift:
            self.error()
            if feedback_clock_name != "":
                self.msg = "Output clock {} used as feedback cannot be set with dynamic phase shift".format(
                    feedback_clock_name)
            else:
                self.msg = "Output clock used as feedback cannot be set with dynamic phase shift"
            return


class PLLRuleIOFeedbackStd(base_rule.Rule):
    """
    When IO feedback mode is enabled, refclk and fbclk
    must use the same IO type and standard
    """

    def __init__(self):
        super().__init__()
        self.name = "pll_rule_extfb_io"

    def get_instance_io_standard(self, gpio_reg, lvds_reg, res_name):
        ins_obj = None
        is_gpio = False
        io_standard = ""

        if gpio_reg is not None:
            # Find if there's any gpio configured with the resource name
            ins_obj = gpio_reg.get_gpio_by_device_name(res_name)
            if ins_obj is not None:
                is_gpio = True
                io_standard = ins_obj.io_standard

        if ins_obj is None and lvds_reg is not None:
            ins_obj = lvds_reg.get_inst_by_device_name(res_name)

        return ins_obj, is_gpio, io_standard

    def check(self, checker: PLLCheckerComp, design_block: PLLComplex): # type: ignore

        pll = design_block
        assert checker.plldev_service is not None

        if pll.fb_mode == pll.FeedbackModeType.external:

            # Find the reference clock and io
            if pll.ref_clock_mode == pll.RefClockModeType.dynamic or\
                    pll.ref_clock_mode == pll.RefClockModeType.external:

                # Getting the registry for gpio and lvds as the
                # sources of the reference clock and io feedback
                # can either  be from a gpio or lvds instances
                gpio_reg = checker.design.gpio_reg
                if gpio_reg is None:
                    checker.logger.warning(
                        "{}: gpio reg is empty. Skipped.".format(self.name))
                    self.severity = self.SeverityType.unknown
                    return

                lvds_reg = checker.design.lvds_reg

                # Get the IO FBK resource
                iofb_res_list = checker.plldev_service.get_resource_on_io_fb_pin(
                    pll.pll_def)
                io_fb_info_list = []
                if iofb_res_list:
                    # Find the instance associated to this resource
                    for iob_res in iofb_res_list:
                        # There should only be one IO Feedback
                        ins_obj, is_gpio, io_standard = \
                            self.get_instance_io_standard(
                                gpio_reg, lvds_reg, iob_res)

                        # Save only if the instance is found
                        if ins_obj is not None:
                            io_fb_info_list.append(
                                (ins_obj, is_gpio, io_standard))

                # Get the Refclk resource. We're not checking the validity of
                # the external resource as that is handled in a different check
                ref_clk_id_to_check = []
                if pll.ref_clock_mode == pll.RefClockModeType.external:
                    assert pll.ext_ref_clock_id is not None
                    ext_index = pll.get_external_clkin_index(
                        checker.plldev_service, pll.ext_ref_clock_id)
                    ref_clk_id_to_check.append(ext_index)

                else:
                    # Dynamic: The number of external clk resource varies based
                    # on the PLL resource
                    _, ext_pin_cnt = checker.plldev_service.get_ref_clk_type_count(
                        pll.pll_def)
                    if ext_pin_cnt == 3:
                        ref_clk_id_to_check = [0, 1, 2]
                    else:
                        ref_clk_id_to_check = [0, 1]

                ref_clk_info_list = []
                for clk_id in ref_clk_id_to_check:
                    # Find the design instance associated to the resource
                    ext_clk_res_list = checker.plldev_service.get_all_resource_on_ref_clk_pin(
                        pll.pll_def, clk_id)

                    if ext_clk_res_list:
                        for ref_res in ext_clk_res_list:
                            ins_obj, is_gpio, io_standard = \
                                self.get_instance_io_standard(
                                    gpio_reg, lvds_reg, ref_res)

                            if ins_obj is not None:
                                ref_clk_info_list.append(
                                    (ins_obj, is_gpio, io_standard))

                is_valid = True
                if io_fb_info_list and ref_clk_info_list:
                    # Now we compare the information between the refclk and fb
                    for io_fb in io_fb_info_list:
                        fb_ins_obj, fb_is_gpio, fb_io_std = io_fb

                        for ref_clk in ref_clk_info_list:
                            ref_ins_obj, ref_is_gpio, ref_io_std = ref_clk

                            if fb_is_gpio != ref_is_gpio or \
                                    fb_io_std != ref_io_std:
                                # Stop the first time we see differences
                                is_valid = False
                                break

                if not is_valid:
                    self.error("External feedback and reference clock have to be of the same instance type and IO standard")
                    return


class PLLRuleConfig(base_rule.Rule):
    """
    Check that all the params are valid
    """

    def __init__(self):
        super().__init__()
        self.name = "pll_rule_param"

    def check(self, checker, design_block: PLLComplex):
        util.gen_util.mark_unused(checker)

        # Get the mode
        pll = design_block

        invalid_param_names = []

        def get_param_name(param_obj):
            return param_obj.name

        # Check the PLL param
        param_info = pll.get_param_info()
        if pll.param_group is not None and param_info is not None:
            all_param = pll.param_group.get_all_param()

            for param_obj in sorted(all_param, key=get_param_name):
                if param_obj is not None:
                    is_valid, msg = pll.param_group.check_param_value(
                        param_obj.name, param_info)
                    if not is_valid:
                        checker.logger.debug("Invalid {} value: {}".format(
                            param_obj.name, msg))
                        invalid_param_names.append(param_obj.name)

        # Check the PLL output clock param
        out_clock_list = pll.get_output_clock_list()
        if out_clock_list:
            for clk in out_clock_list:
                if clk is not None:
                    outclk_param_info = clk.get_param_info()

                    if clk.param_group is not None and outclk_param_info is not None:
                        all_param = clk.param_group.get_all_param()

                        # This might check parameters that are not relevant. However, those
                        # not relevant should still have its value set to valid
                        # value (i.e. default)
                        for param_obj in sorted(all_param, key=get_param_name):
                            if param_obj is not None:
                                is_valid, msg = clk.param_group.check_param_value(
                                    param_obj.name, param_info)
                                if not is_valid:
                                    checker.logger.debug("Invalid {} value: {}".format(
                                        param_obj.name, msg))
                                    invalid_param_names.append("{}:{}".format(
                                        clk.name, param_obj.name))

        if invalid_param_names:
            self.error("Invalid parameters configuration: {}".format(
                ",".join(invalid_param_names)))
            return


class PLLRuleFeedback(an20_pllrule.PLLRuleFeedback):
    """
    Check the validity of the feedback clock in core and external modes
    """

    def check(self, checker, design_block: PLLComplex):
        util.gen_util.mark_unused(checker)

        pll = design_block

        if pll.fb_mode == pll.FeedbackModeType.core or\
                pll.fb_mode == pll.FeedbackModeType.external or \
                pll.fb_mode == pll.FeedbackModeType.local:

            if pll.fb_clock_name == "":
                self.error('Feedback clock name is required with non-internal feedback')
                return

            fb_clk_data = checker.reg.find_output_clock(pll.fb_clock_name)

            if fb_clk_data is not None:
                fb_pll_ins, fb_out_no = fb_clk_data

                # It should be of the same instance
                # TODO: remove this if we support clkout from other PLL
                if pll != fb_pll_ins:
                    self.error('Feedback clock name {} is not from the same PLL'.format(
                        pll.fb_clock_name))
                    return

                if pll.fb_mode == pll.FeedbackModeType.local:
                    if fb_out_no != 0:
                        self.error('Feedback clock in local mode has to connect to output clock 0')
                        return

                out_clk = fb_pll_ins.get_output_clock_by_number(fb_out_no)
                if out_clk.phase_setting != 0:
                    self.info('Feedback clock phase shift is not 0-degree, check that the feedback clock is in-phase with the reference clock.')
                    return

            else:
                self.error('Feedback clock {} is not connected to pll clkout'.format(
                    pll.fb_clock_name))
                return

class PLLRulePreDivider(an08_pllrule.PLLRulePreDivider):
    """
    Check if pre-divider is valid
    """

    def check(self, checker, design_block: PLLComplex):

        pll = design_block

        if checker.plldev_service is None:
            checker.logger.warning(
                "{}: device db is empty. Skipped.".format(self.name))
            self.severity = self.SeverityType.unknown
            return

        # Different then Trion which is a range
        valid_opt_str_list = checker.plldev_service.get_pre_divider_mode_names()

        # Post divider is an integer. The valid list of options are string
        if isinstance(pll.pre_divider, int) and valid_opt_str_list:
            # Convert to string
            pre_div_str = str(pll.pre_divider)
            if pre_div_str not in valid_opt_str_list:
                self.error("Pre-divider is invalid. Valid values are {}." \
                    .format(",".join(valid_opt_str_list)))

        else:
            checker.logger.warning(
                "Unable to determine the pre-divider type: {}. Skipped.".format(pll.pre_divider))
            self.severity = self.SeverityType.unknown


class PLLRuleMultiplier(an08_pllrule.PLLRuleMultiplier):
    """
    Check if multiplier is valid
    """

    def check(self, checker, design_block: PLLComplex):

        pll = design_block

        if checker.plldev_service is None:
            checker.logger.warning(
                "{}: device db is empty. Skipped.".format(self.name))
            self.severity = self.SeverityType.unknown
            return

        # Different then Trion which is a range
        valid_opt_str_list = checker.plldev_service.get_multiplier_mode_names()

        # Post divider is an integer. The valid list of options are string
        if isinstance(pll.multiplier, int) and valid_opt_str_list:
            # Convert to string
            mult_str = str(pll.multiplier)
            if mult_str not in valid_opt_str_list:
                self.error("Multiplier is invalid. Valid values are {}." \
                    .format(",".join(valid_opt_str_list)))

        else:
            checker.logger.warning(
                "Unable to determine the multiplier type: {}. Skipped.".format(pll.multiplier))
            self.severity = self.SeverityType.unknown


class PLLRulePostDivider(an08_pllrule.PLLRulePostDivider):
    """
    Check the validity of the feedback clock in core and external modes
    """

    def check(self, checker, design_block: PLLComplex):

        pll = design_block

        if checker.plldev_service is None:
            checker.logger.warning(
                "{}: device db is empty. Skipped.".format(self.name))
            self.severity = self.SeverityType.unknown
            return

        # Get the valid range
        valid_opt_str_list = checker.plldev_service.get_post_divider_mode_names()

        # Post divider is an integer. The valid list of options are string
        if isinstance(pll.post_divider, int) and valid_opt_str_list:
            # Convert to string
            post_div_str = str(pll.post_divider)
            if post_div_str not in valid_opt_str_list:
                self.error("Post-divider is invalid. Valid values are {}." \
                    .format(",".join(valid_opt_str_list)))

        else:
            checker.logger.warning(
                "Unable to determine the post-divider type: {}. Skipped.".format(pll.post_divider))
            self.severity = self.SeverityType.unknown

class PLLRuleOutputDivider(an08_pllrule.PLLRuleOutputDivider):
    """
    Check if output divider is valid
    """

    def check(self, checker, design_block: PLLComplex):

        util.gen_util.mark_unused(checker)

        pll = design_block

        out_clock_list = pll.get_output_clock_list()
        count = len(out_clock_list)
        if out_clock_list is None or count == 0:
            # Empty output clock is trapped in another rule
            return

        for clk in out_clock_list:
            # PT-1347 Remove x1 for TX PLL output divider
            if clk.out_divider < 2 or clk.out_divider > 128:
                self.error("Output divider for {} is invalid. Valid values are between 2-128." \
                    .format(clk.name))

class PLLRuleMIPIDPHYTxClock(base_rule.Rule):
    """
    PT-1010: Check that no same output clock is driving both a MIPI Tx fast and slow clock.
    This rule is only applicable to sample device
    """

    def __init__(self):
        super().__init__()
        self.name = "pll_rule_mipi_tx_clock"

    def check(self, checker: PLLCheckerComp, design_block: PLLComplex): # type: ignore
        util.gen_util.mark_unused(checker)

        # Get the mode
        pll = design_block

        # Execute check if there is any MIPI DPHY Tx clock names and on sample device ONLY
        if checker.design.device_db is not None and \
            checker.design.device_db.is_sample_device() and \
                (checker.mipitx_fastclk_names or checker.mipitx_slowclk_names):

                outclk_list = pll.get_output_clock_list()

                for clk_obj in outclk_list:
                    if clk_obj is not None:
                        clk_name = clk_obj.name

                        if clk_name in checker.mipitx_slowclk_names and\
                            clk_name in checker.mipitx_fastclk_names:
                            # We find that the same clock is assigned to both the MIPI Tx
                            # fast and slow clock
                            self.error("PLL output clock {} is not allowed to connect to "\
                                "MIPI TX Lane Serial and Parallel clocks at the same time" \
                                .format(clk_name))
                            return

class PLLPostVCOFrequency(base_rule.Rule):
    """
    Check the Post VCO Frequency aka PLL Frequency is valid
    """
    def __init__(self):
        super().__init__()
        self.name = 'pll_rule_pll_freq'

    def check(self, checker: PLLCheckerComp, design_block: PLLComplex): # type: ignore
        pll = design_block

        if checker.plldev_service is None:
            checker.logger.warning(
                "{}: device db is empty. Skipped.".format(self.name))
            self.severity = self.SeverityType.unknown
            return

        # Follow PLLRuleVCOFreq
        is_pass_calc = checker.reg.calc_single_instance_frequency(pll)
        if is_pass_calc:
            min_freq, max_freq = checker.plldev_service.get_min_max_pll_freq(checker.design, pll)

            if pll.pll_freq < min_freq or pll.pll_freq > max_freq:
                self.severity = self.SeverityType.error
                self.pass_status = False
                self.msg = f"PLL Frequency is out of range, Freq={pll.get_pll_freq_str()} Min={min_freq}MHz Max={max_freq}MHz"
        else:
            checker.logger.warning(
                "{}: Fail to calculate PLL frequency in order to check for valid range.".format(self.name))
            self.severity = self.SeverityType.unknown

class PLLRuleClockDividerPhaseShift(base_rule.Rule):
    """
    PT-1817: Give warning if the output clock has non-zero phase shift with
    the output clock divider being odd.
    """

    def __init__(self):
        super().__init__()
        self.name = "pll_rule_outclk_div_pshift"

    def check(self, checker, design_block: PLLComplex):
        util.gen_util.mark_unused(checker)

        # Get the mode
        pll = design_block

        out_clock_list = pll.get_output_clock_list()
        count = len(out_clock_list)
        if out_clock_list is None or count == 0:
            # Empty output clock is trapped in another rule
            return

        out_clk_issues = []
        for clk in out_clock_list:
            # out_divider is an integer
            if clk.out_divider > 1 and clk.out_divider % 2 == 1 and\
                    ((not clk.is_dyn_phase and clk.phase_setting > 0) or\
                     (clk.is_dyn_phase)):
                out_clk_issues.append(clk.name)

        if out_clk_issues:
            self.warning('Enable phase shift with odd output clock divider will result '\
                         'in duty cycle distortion on the output clock.  It is not advisable '\
                         'to use the clock for double data operation: {}'.format(",".join(out_clk_issues)))

class PLLRuleRegionalFeedbackClock(base_rule.Rule):
    """
    PT-2017: There is specific restriction for using external clock
    with output clock that goes to regional clock.
    """

    def __init__(self):
        super().__init__()
        self.name = "pll_rule_feedback_clk_regional"

    def check(self, checker, design_block: PLLComplex):
        # Get the mode
        pll = design_block

        # This rule is very specific to Ti60 only
        if self.is_ti60_device(checker) and pll.pll_def != "":

            # Do not need to consider dynamic since we will fix
            # fbk pin to use FBK[1] which is alwys the same region
            # as the outclk4
            if pll.fb_mode == pll.FeedbackModeType.core and \
                pll.ref_clock_mode == pll.RefClockModeType.external and\
                pll.fb_clock_name != "":
                # We don't check the validity of the feedback clock
                # since that is handled in PLLRuleFeedback
                fb_clk_data = checker.reg.find_output_clock(pll.fb_clock_name)

                if fb_clk_data is not None:
                    _, fb_out_no = fb_clk_data

                    if fb_out_no == 4:
                        # Ti60 output clock 4 are all fixed
                        # connection to regional clock network
                        # Get the external resource and check if
                        # they are from GPIOL/GPIOR resource
                        ext_index = pll.get_external_clkin_index(
                            checker.plldev_service, pll.ext_ref_clock_id)

                        # Find the design instance associated to the resource
                        ext_clk_res_list = checker.plldev_service.get_all_resource_on_ref_clk_pin(
                            pll.pll_def, ext_index)

                        if ext_clk_res_list:
                            gpio_reg = checker.design.gpio_reg
                            lvds_reg = checker.design.lvds_reg

                            for ref_res in ext_clk_res_list:                                
                                if ref_res.startswith("GPIOL_") or ref_res.startswith("GPIOR_"):
                                    # Check if it was configured
                                    if gpio_reg is not None:
                                        # Find if there's any gpio configured with the resource name
                                        ins_obj = gpio_reg.get_gpio_by_device_name(ref_res)

                                    if ins_obj is None and lvds_reg is not None:
                                        ins_obj = lvds_reg.get_inst_by_device_name(ref_res)

                                    if ins_obj is not None:
                                        self.error('Unroutable regional clock output {} to the core feedback'\
                                               ' interface with external reference clock resource'\
                                               ' set to {}. Select a different reference clock resource or'\
                                               ' assign a different output clock as feedback clock'.format(
                                                   fb_out_no, ref_res))
                                        return

    def is_ti60_device(self, checker):
        assert checker.design is not None and checker.design.device_db is not None
        device_db = checker.design.device_db
        
        _, die_name = device_db.get_device_family_die_name()
        if die_name == "Ti60":
            return True

        return False        