'''
Copyright (C) 2017-2020 Efinix Inc. All rights reserved.

No portion of this code may be reused, modified or
distributed in any way without the expressed written
consent of Efinix Inc.

Created on Nov 1, 2020

@author: maryam
'''
from __future__ import annotations
import abc
from decimal import Decimal, localcontext
import re
from typing import Dict, TYPE_CHECKING, Any

from common_device.pll.pll_design_param_info import PLLDesignParamInfo
import util.gen_util
import util.excp as app_excp

import device.db_interface as devdb_int
from device.block_instance import SecondaryConn

import common_device.rule as base_rule
import common_device.lvds.lvds_rule as lvdsrule
from util.signal_util import calc_phase_shifted_time_diff_ps, format_signal_time, float_to_decimal, calc_period_in_ps

if TYPE_CHECKING:
    from tx60_device.lvds.lvds_design_adv import LVDSAdvance, LVDSRegistryAdvance


@util.gen_util.freeze_it
class LVDSCheckerAdv(lvdsrule.LVDSChecker):
    def __init__(self, design):
        """
        Constructor
        """
        super().__init__(design)
        self.reg: LVDSRegistryAdvance

        # We overwrite with the HSIO LVDS Device services
        if self.design.device_db is not None:
            dbi = devdb_int.DeviceDBService(self.design.device_db)
            self.lvdstx_service = dbi.get_block_service(
                devdb_int.DeviceDBService.BlockType.HSIO_LVDS_TX)
            self.lvdsrx_service = dbi.get_block_service(
                devdb_int.DeviceDBService.BlockType.HSIO_LVDS_RX)

    def set_used_bank_inst(self):
        """
        :return a tuple of:
            1) map of bank name to instance names list
            2) map of instance name to bank name list
        """

        # Clear the list so it is overwritten
        self.tx_group2ins = {}
        self.tx_ins2group = {}
        self.rx_group2ins = {}
        self.rx_ins2group = {}

        lvds_tx = []
        lvds_rx = []

        # Get the list of configured instance objects.
        if self.lvdstx_service is not None:
            lvds_tx = self.lvdstx_service.get_all_instances(
                self.design)

        if self.lvdsrx_service is not None:
            lvds_rx = self.lvdsrx_service.get_all_instances(
                self.design)

        # Get the list of instaces configured as hsip gpio
        # and add it to both rx and tx instance list
        dbi = devdb_int.DeviceDBService(self.design.device_db)
        hsio_service = dbi.get_block_service(
            devdb_int.DeviceDBService.BlockType.HSIO)

        hsio_gpio = hsio_service.get_design_block_type_instances(
            self.design.BlockType.comp_gpio, self.design)

        # Proceed if we see combination of lvds and gpio usage
        if lvds_tx:
            combined_tx = lvds_tx + hsio_gpio

            # Get a map of bank to the list of lvds device instances
            _, tx_ins2banks = hsio_service.get_bank_to_instances(
                combined_tx)

            self._determine_bank_group(True, tx_ins2banks)

        if lvds_rx:
            combined_rx = lvds_rx + hsio_gpio

            # Get a map of bank to the list of lvds device instances
            _, rx_ins2banks = hsio_service.get_bank_to_instances(
                combined_rx)

            self._determine_bank_group(False, rx_ins2banks)

    def _determine_bank_group(self, is_tx, ins2banks_map):

        if is_tx:
            group2ins_map = self.tx_group2ins
            ins2group_map = self.tx_ins2group
        else:
            group2ins_map = self.rx_group2ins
            ins2group_map = self.rx_ins2group

        for ins_name in ins2banks_map.keys():
            header_name = ins_name
            stop_pos = header_name.find("_")

            if stop_pos != -1:
                group_name = header_name[:stop_pos]

                ins_list = []
                if group_name not in group2ins_map:
                    ins_list.append(ins_name)
                else:
                    ins_list = group2ins_map[group_name]
                    ins_list.append(ins_name)

                group2ins_map[group_name] = ins_list
                ins2group_map[ins_name] = group_name

    def check_usage_distance(self, res_svc, lvds, group_name, sorted_ins_names):
        is_valid = True
        pass_status = True
        severity = None
        msg = ""

        # Based on this ins names list, check if there's any
        # LVTTL that is of distance <= abs(2) with the current
        # lvds.lvds_def
        self.logger.debug("Check lvds {}:{} Map of bank:ins -> {}: {}".format(
            lvds.name, lvds.lvds_def, group_name, sorted_ins_names))

        # Get this lvds instance index (extract integers)
        cur_index_list = [int(s)
                          for s in re.findall(r'\d+', lvds.lvds_def)]

        violated_lvttl_names = []
        if cur_index_list:
            if len(cur_index_list) == 1:
                index = cur_index_list[0]
                #self.logger.debug("Read LVDS index {}".format(index))

                right_limit = index + 1
                left_limit = index - 1

                for insn in sorted_ins_names:
                    #self.logger.debug("Ins: {}-{}, r-l:{},{}".format(
                    #    insn, index, right_limit, left_limit))

                    search_index_list = [
                        int(s) for s in re.findall(r'\d+', insn)]
                    #self.logger.debug("Index list: {}".format(search_index_list))

                    # Always get the first number because the second is
                    # the mode name (i.e. lvttl1, lvttl2)
                    if search_index_list:
                        nexti = search_index_list[0]

                        #self.logger.debug("Index: {}".format(nexti))
                        if nexti <= right_limit and nexti >= left_limit and \
                                nexti != index:
                            if not res_svc.is_non_gpio_resource(insn):
                                self.logger.debug(
                                    "Found violation on {}".format(insn))
                                violated_lvttl_names.append(insn)

            else:
                is_valid = False
                severity = base_rule.Rule.SeverityType.warning
                pass_status = True
                msg = "Unable to tell the resource index of the LVDS instance {}".format(
                    lvds.name)
                return is_valid, severity, pass_status, msg

        if violated_lvttl_names:
            is_valid = False
            severity = base_rule.Rule.SeverityType.warning
            pass_status = True
            msg = 'These HSIO GPIO must be placed at least 1 pair away from LVDS {} '\
                    'in order to avoid noise coupling from GPIO to LVDS: {}'.format(
                lvds.name, ",".join(sorted(violated_lvttl_names)))

        return is_valid, severity, pass_status, msg

    def _build_rules(self):
        self._add_rule(LVDSRuleResource())
        self._add_rule(LVDSRuleUsage())
        self._add_rule(LVDSRuleTxOutputMode())
        self._add_rule(LVDSRuleTxClockOutMode())
        self._add_rule(LVDSRuleTxPLL())
        self._add_rule(LVDSRuleRx())
        self._add_rule(LVDSRuleRxPLL())
        self._add_rule(LVDSRuleRxPLLRefClk())
        self._add_rule(LVDSTxRuleUsageDistance())
        self._add_rule(LVDSRxRuleUsageDistance())
        self._add_rule(LVDSRxAlternateConn())
        self._add_rule(LVDSRxAlternateConfig())
        self._add_rule(LVDSRuleRxConfig())
        self._add_rule(LVDSRuleTxConfig())
        self._add_rule(LVDSRuleRxDeserialWidth())
        self._add_rule(LVDSRuleTxSerialWidth())
        # self._add_rule(LVDSRuleBidirTxMode()) # PT-1341 Output Enable pin name is checked by LVDSRuleTxPinEmpty/InvalidName
        self._add_rule(LVDSRuleTxVref())
        self._add_rule(LVDSRuleTxSerialRate())
        self._add_rule(LVDSRuleRxDeserialRate())
        self._add_rule(LVDSRuleTxSerialWidth2())
        self._add_rule(LVDSRuleRxDeserialWidth2())
        self._add_rule(LVDSRuleRxFIFO())
        self._add_rule(LVDSRuleTxPinEmptyName())
        self._add_rule(LVDSRuleTxPinInvalidName())
        self._add_rule(LVDSRuleRxPinEmptyName())
        self._add_rule(LVDSRuleRxPinInvalidName())
        self._add_rule(LVDSRuleRxDPA())
        self._add_rule(LVDSRuleRxDPAESDevice())
        self._add_rule(LVDSRuleTxPLLClkRegion())
        self._add_rule(LVDSRuleRxPLLClkRegion())
        self._add_rule(LVDSRuleRxDPAWidth())
        self._add_rule(LVDSRuleTxClockOutModeBypass())
        self._add_rule(LVDSRuleRxPLLFeedback())

    def run_all_check(self, exclude_rule=None):
        """
        Run all checks on all lvds in registry

        :param exclude_rule: A list of excluded rule name
        """

        # prepare the bank instance mapping
        self.set_used_bank_inst()
        self.block_tag_gen = self.gen_block_tag
        return super().run_all_check(exclude_rule)

    def check_lvds_fast_slow_clocks_conn_type(self, fast_clk_name: str, slow_clk_name: str):
        error_msg = ""

        fast_pll_data = self.pll_reg.find_output_clock(
            fast_clk_name)

        slow_pll_data = self.pll_reg.find_output_clock(
            slow_clk_name)

        if fast_pll_data is not None and slow_pll_data is not None:
            pll_fast_inst, fast_clk_no = fast_pll_data
            pll_slow_inst, slow_clk_no = slow_pll_data

            # Check if both of the clock are in the same region
            fast_conn_type = self.plldev_service.get_pll_output_clock_conn_type(pll_fast_inst, fast_clk_no)
            slow_conn_type = self.plldev_service.get_pll_output_clock_conn_type(pll_slow_inst, slow_clk_no)

            if fast_conn_type != slow_conn_type:
                def get_region_name(conn_type):
                    if conn_type == "rclk":
                        return "regional clock"

                    elif conn_type == "gclk":
                        return "global clock"

                    raise ValueError("check_lvds_fast_slow_clocks_conn_type: Unexpected connection type {}".format(
                        conn_type))

                # Error if the two clocks doesn't route to the same region
                error_msg = 'Serial and Parallel clocks generated by PLL have ' \
                           'to be driven to the same clock network.  {}: {} {}: {}'.format(
                    slow_clk_name, get_region_name(slow_conn_type),
                    fast_clk_name, get_region_name(fast_conn_type))

        return error_msg

class LVDSRuleResource(lvdsrule.LVDSRuleResource):
    """
    Check if the resource is valid
    """

    def check(self, checker: LVDSCheckerAdv, design_block: LVDSAdvance): # type: ignore
        lvds = design_block

        if self.is_name_empty(lvds.lvds_def):
            self.error("Resource name is empty.")
            return

        #res_svc = checker.reg.res_svc

        # Check if resource name is a valid device instance name
        # if res_svc is not None:
        if checker.lvdstx_service is not None:
            # Just check one since both rx and tx have the same source list
            # (shared HSIO)
            lvds_dev_list = checker.lvdstx_service.get_usable_instance_names()
            # Using the HIO Service from design doesn't seem to work
            if lvds.lvds_def not in lvds_dev_list:
                self.error("Resource {} is not a valid LVDS device instance.".format(
                    lvds.lvds_def))
                return

        else:
            checker.logger.warning(
                "{}: device db is empty, skip valid instance check.".format(self.name))


class LVDSRuleUsage(base_rule.Rule):
    """
    Check that the LVDS resource doe snot conflict with
    LVTTL instance in gpio
    """

    def __init__(self):
        super().__init__()
        self.name = "lvds_rule_usage"

    def check(self, checker: LVDSCheckerAdv, design_block: LVDSAdvance): # type: ignore
        lvds = design_block

        if self.is_name_empty(lvds.lvds_def):
            return

        # Check if there is any usage of the same resource in other registry
        if checker.reg is not None and checker.reg.res_svc is not None:
            res_svc = checker.reg.res_svc
            used_des_ins_list = res_svc.find_resource_user(lvds.lvds_def)

            if used_des_ins_list and len(used_des_ins_list) > 1:
                self.error("Resource {} was assigned multiple times.".format(
                    lvds.lvds_def))
                return


class LVDSRuleTxOutputMode(lvdsrule.LVDSRuleTxOutputMode):
    """
    Check if the output mode config is valid
    """

    def check(self, checker, design_block: LVDSAdvance):

        lvds = design_block

        # Skip if not an LVDS Tx
        if lvds.ops_type != lvds.OpsType.op_tx and lvds.ops_type != lvds.OpsType.op_bidir:
            return

        lvds_tx = lvds.tx_info

        if lvds_tx is None:
            self.error("Instance of LVDS Tx has invalid configuration")
            return

        if lvds_tx.mode == lvds_tx.ModeType.out:
            # Check that the necessary fields are filled
            if self.check_name_empty_or_invalid(lvds_tx.output_bus_name,
                "Output name must be configured in data output mode"
            ):
                return

            # Check that if it was not bypass, then the clock is not empty
            if lvds_tx.is_serial:
                if lvds_tx.is_serial_width_gt2():
                # x1 and x2 does not require fast clock
                    if self.check_name_empty_or_invalid(lvds_tx.fast_clock_name,
                        "Serial clock name must be configured in data output mode"
                    ):
                        return

                if self.check_name_empty_or_invalid(lvds_tx.slow_clock_name,
                    "Parallel clock name must be configured in data output mode"
                ):
                    return


class LVDSRuleTxClockOutMode(lvdsrule.LVDSRuleTxClockOutMode):
    """
    Check if the Clkout mode config is valid
    """

    def check(self, checker, design_block: LVDSAdvance):
        lvds = design_block

        # Skip if not an LVDS Tx
        if lvds.ops_type != lvds.OpsType.op_tx and lvds.ops_type != lvds.OpsType.op_bidir:
            return

        lvds_tx = lvds.tx_info

        if lvds_tx is None:
            self.error("Instance of LVDS Tx has invalid configuration")
            return

        if lvds_tx.mode == lvds_tx.ModeType.clkout:
            if lvds_tx.is_serial:
                # PT-1135: clkout is not supported in x1 width
                if int(lvds_tx.serial_width) == 1:
                    self.error("Clock output mode is not supported with serialization width 1")
                    return

                # Check that the necessary fields are filled.
                if lvds_tx.is_serial_width_gt2():
                    # x2 does not require fast clock
                    if self.check_name_empty_or_invalid(lvds_tx.fast_clock_name,
                        "Serial clock name must be configured in data output mode"):
                        return

                if self.check_name_empty_or_invalid(lvds_tx.slow_clock_name,
                    "Parallel clock name must be configured in data output mode"):
                    return

class LVDSRuleTxClockOutModeBypass(lvdsrule.LVDSRuleTxClockOutModeBypass):
    """
    PT-1661: Check that the clock output pin name is specified
    with Tx clkout mode + serialization disabled
    """

    def check(self, checker, design_block: LVDSAdvance):
        lvds = design_block

        # Skip if not an LVDS Tx
        if lvds.ops_type != lvds.OpsType.op_tx and lvds.ops_type != lvds.OpsType.op_bidir:
            return

        lvds_tx = lvds.tx_info

        if lvds_tx is None:
            self.error("Instance of LVDS Tx has invalid configuration")
            return

        if lvds_tx.mode == lvds_tx.ModeType.clkout:
            if not lvds_tx.is_serial and lvds_tx.slow_clock_name == "":
                self.error('Output clock name must be configured in '\
                        'clock output mode with serialization disabled')

class LVDSRuleTxPLL(lvdsrule.LVDSRuleTxPLL):
    """
    Check that the clock source is from valid PLL and that
    the frequencies are valid.
    """

    def check(self, checker: LVDSCheckerAdv, design_block: LVDSAdvance): # type: ignore

        lvds = design_block

        # Skip LVDS Tx is not used
        if lvds.ops_type != lvds.OpsType.op_tx and lvds.ops_type != lvds.OpsType.op_bidir:
            return

        lvds_tx = lvds.tx_info

        # This should skip checking for x1,x2
        if lvds_tx is not None and lvds_tx.is_serial and lvds_tx.is_serial_width_gt2():

            # Check that the fast and slow is not the same pin
            if lvds_tx.fast_clock_name != "" and \
                    lvds_tx.slow_clock_name != "":

                if lvds_tx.is_half_rate:
                    expected_denom = 2
                else:
                    expected_denom = 1

                valid, pll_inst, fast_clk_no, slow_clk_no, msg = \
                    checker.check_clock_relationship(
                        lvds_tx.fast_clock_name, lvds_tx.slow_clock_name,
                        int(lvds_tx.serial_width), expected_denom)

                if not valid:
                    self.error(msg)
                    return

                assert pll_inst is not None

                fast_out_clk = pll_inst.get_output_clock_by_number(
                    fast_clk_no)
                if fast_out_clk is None:
                    self.error("Unable to retrieve pll output clock connecting to serial clock")
                    return

                slow_out_clk = pll_inst.get_output_clock_by_number(
                    slow_clk_no)
                if slow_out_clk is None:
                    self.error("Unable to retrieve pll output clock connecting to parallel clock")
                    return

                # Check out_clk phase shift
                # Fast clk
                fast_phase_shift_deg = pll_inst.phase_setting2degree(
                    fast_out_clk.phase_setting, fast_out_clk.out_clock_freq, pll_inst.vco_freq,
                    pll_inst.post_divider)
                # Slow clk
                slow_phase_shift_deg = pll_inst.phase_setting2degree(
                    slow_out_clk.phase_setting, slow_out_clk.out_clock_freq, pll_inst.vco_freq,
                    pll_inst.post_divider)

                # Check the absolute phase shifted time difference between fast and slow clk
                with localcontext() as ctx:
                    PRECISION = 4
                    ctx.prec = PRECISION
                    t_diff, t_fastclk_shifted, t_slowclk_shifted = calc_phase_shifted_time_diff_ps(fast_out_clk.out_clock_freq,
                                                                                                   fast_phase_shift_deg,
                                                                                                   slow_out_clk.out_clock_freq,
                                                                                                   slow_phase_shift_deg)
                    
                    # PT-2691: Updated limit
                    t_min_diff = 300
                    # Nax depends on the fast clk period
                    fastclk_period_ps, err_msg = self.get_fast_clk_period(pll_inst, fast_out_clk)
                                        
                    if fastclk_period_ps is None:
                        if err_msg != "":
                            self.error(err_msg)
                        return
                        
                    # we already set the context precision
                    t_max_diff = (fastclk_period_ps * Decimal('0.5')) - Decimal('125')
                    
                    #print(f'fast ps:{t_fastclk_shifted}, slow ps: {t_slowclk_shifted}, diff: {t_diff}, max_limit: {t_max_diff}')

                    if t_diff < t_min_diff or t_diff > t_max_diff:
                        msg = f"Invalid phase shift difference: {format_signal_time(t_diff, PRECISION)}" \
                                f" = Serial: {format_signal_time(t_fastclk_shifted, PRECISION)}" \
                                f" - Parallel: {format_signal_time(t_slowclk_shifted, PRECISION)}" \
                                f" (max={format_signal_time(t_max_diff, PRECISION)}, " \
                                f"min={format_signal_time(t_min_diff, PRECISION)})"
                        self.error(msg)
                        return

    def get_fast_clk_period(self, pll_inst, fast_out_clk):
        fastclk_period_ps = None
        msg = ""

        with localcontext() as ctx:
            PRECISION = 4
            ctx.prec = PRECISION

            try:
                fastclk_period_ps = calc_period_in_ps(fast_out_clk.out_clock_freq)
            except app_excp.PTException:
                # Calculate the frequency
                pll_inst.calc_all_frequencies()
                if fast_out_clk.out_clock_freq is not None and \
                    fast_out_clk.out_clock_freq > 0.0:
                    fastclk_period_ps = calc_period_in_ps(fast_out_clk.out_clock_freq)
                else:
                    msg = f"Unexpected error: Clock frequency for {fast_out_clk.name} is not available"
                    self.error(msg)
                    fastclk_period_ps = None

        return fastclk_period_ps, msg
    
class LVDSRuleRx(lvdsrule.LVDSRuleRx):
    """
    Check if the LVDS Rx configuration is valid
    """

    def check(self, checker, design_block: LVDSAdvance):
        lvds = design_block

        # Skip if not an LVDS Rx
        if lvds.ops_type != lvds.OpsType.op_rx and lvds.ops_type != lvds.OpsType.op_bidir:
            return

        lvds_rx = lvds.rx_info

        if lvds_rx is None:
            self.error("Instance of LVDS Rx has invalid configuration type")
            return

        # Check that the necessary fields are filled
        if self.check_name_empty_or_invalid(lvds_rx.input_bus_name,
            "Input name must be configured"
        ):
            return

        # Check that if it was bypass, then the clock is empty
        if lvds_rx.is_serial:
            if lvds_rx.is_serial_width_gt2():
            # x1,x2 does not require fast clock
                if self.check_name_empty_or_invalid(lvds_rx.fast_clock_name,
                    "Serial clock name must be configured"
                ):
                    return

            if self.check_name_empty_or_invalid(lvds_rx.slow_clock_name,
                "Parallel clock name must be configured"
            ):
                return


class LVDSRuleRxPLL(lvdsrule.LVDSRuleRxPLL):
    """
    Check that the frequencies are valid.
    """

    def check(self, checker: LVDSCheckerAdv, design_block: LVDSAdvance): # type: ignore
        lvds = design_block

        # Skip if LVDS Rx is not used
        if lvds.ops_type != lvds.OpsType.op_rx and lvds.ops_type != lvds.OpsType.op_bidir:
            return

        lvds_rx = lvds.rx_info

        # Skip for bypass mode and x1,x2
        if lvds_rx is not None and lvds_rx.is_serial and lvds_rx.is_serial_width_gt2():

            if lvds_rx.fast_clock_name != "" and \
                    lvds_rx.slow_clock_name != "":

                # Expected factor depends on whether full/half rate
                if lvds_rx.is_half_rate:
                    expected_denom = 2
                else:
                    expected_denom = 1

                valid, pll_inst, fast_clk_no, slow_clk_no, msg = \
                    checker.check_clock_relationship(
                        lvds_rx.fast_clock_name, lvds_rx.slow_clock_name,
                        int(lvds_rx.serial_width), expected_denom)

                if not valid:
                    self.error(msg)
                    return

                assert pll_inst is not None
                # Check that the fastclk has 90 degrees phase shift
                fast_out_clk = pll_inst.get_output_clock_by_number(
                    fast_clk_no)
                if fast_out_clk is None:
                    self.error("Unable to retrieve pll output clock connecting to serial clock")
                    return

                slow_out_clk = pll_inst.get_output_clock_by_number(
                    slow_clk_no)
                if slow_out_clk is None:
                    self.error("Unable to retrieve pll output clock connecting to parallel clock")
                    return

                # Check out_clk phase shift
                # Fast clk
                fast_phase_shift_deg = pll_inst.phase_setting2degree(
                    fast_out_clk.phase_setting, fast_out_clk.out_clock_freq, pll_inst.vco_freq,
                    pll_inst.post_divider)
                # Slow clk
                slow_phase_shift_deg = pll_inst.phase_setting2degree(
                    slow_out_clk.phase_setting, slow_out_clk.out_clock_freq, pll_inst.vco_freq,
                    pll_inst.post_divider)

                # Check the phase shifted time difference between fast and slow clk
                with localcontext() as ctx:
                    PRECISION = 4
                    ctx.prec = PRECISION
                    t_diff, t_fastclk_shifted, t_slowclk_shifted = calc_phase_shifted_time_diff_ps(fast_out_clk.out_clock_freq,
                                                                                                   fast_phase_shift_deg,
                                                                                                   slow_out_clk.out_clock_freq,
                                                                                                   slow_phase_shift_deg)
                    # PT-2691: Check the halfrate status since the limit varies based on that
                    # Min is fixed to 250ps
                    t_min_diff = 250
                    # Nax depends on the fast clk period
                    fastclk_period_ps, err_msg = self.get_fast_clk_period(pll_inst, fast_out_clk)
                                        
                    if fastclk_period_ps is None:
                        if err_msg != "":
                            self.error(err_msg)
                        return
                    
                    # We already set the context precision
                    if lvds_rx.is_half_rate:
                        t_max_diff = (fastclk_period_ps * Decimal('0.5')) - Decimal('200')
                    else:
                        t_max_diff = (fastclk_period_ps * Decimal('0.625'))

                    #print(f'fast ps:{t_fastclk_shifted}, slow ps: {t_slowclk_shifted}, diff: {t_diff}, max_limit: {t_max_diff}')

                    if t_diff < t_min_diff or t_diff > t_max_diff:
                        msg = f"Invalid phase shift difference: {format_signal_time(t_diff, PRECISION)}" \
                                f" = Serial: {format_signal_time(t_fastclk_shifted, PRECISION)}" \
                                f" - Parallel: {format_signal_time(t_slowclk_shifted, PRECISION)}" \
                                f" (max={format_signal_time(t_max_diff, PRECISION)}, " \
                                f"min={format_signal_time(t_min_diff, PRECISION)})"
                        self.error(msg)
                        return

    def get_fast_clk_period(self, pll_inst, fast_out_clk):
        fastclk_period_ps = None
        msg = ""

        with localcontext() as ctx:
            PRECISION = 4
            ctx.prec = PRECISION

            try:
                fastclk_period_ps = calc_period_in_ps(fast_out_clk.out_clock_freq)
            except app_excp.PTException:
                # Calculate the frequency
                pll_inst.calc_all_frequencies()
                if fast_out_clk.out_clock_freq is not None and \
                    fast_out_clk.out_clock_freq > 0.0:
                    fastclk_period_ps = calc_period_in_ps(fast_out_clk.out_clock_freq)
                else:
                    msg = f"Unexpected error: Clock frequency for {fast_out_clk.name} is not available"
                    self.error(msg)
                    fastclk_period_ps = None

        return fastclk_period_ps, msg
              
class LVDSRuleRxSlowClkRefClkFreq(lvdsrule.LVDSRuleRxSlowClkRefClkFreq):
    """
    PT-1359: If the PLL ref clock is in external mode and from
    and LVDS (pll_clkin), check that the ratio between the SLow clock
    frequency and the reference input clock frequency is a whole number (integer).

    Due to PT-1381: The requirement is now different. Rather than deleting this class,
    it shall be updated with the new requirement and rule name.
    """

    def check(self, checker, design_block: LVDSAdvance):
        lvds = design_block

        # Skip if LVDS Rx is not used
        if lvds.ops_type != lvds.OpsType.op_rx and lvds.ops_type != lvds.OpsType.op_bidir:
            return

        lvds_rx = lvds.rx_info

        # Skip for bypass mode and x1,x2
        if lvds_rx is not None and lvds_rx.is_serial and lvds_rx.is_serial_width_gt2():
            self.check_ref_clk_slow_clk_rship(lvds_rx, checker)

    def check_pll_lvds_ext_refclk(self, checker, pll_inst):
        '''
        Slightly different than LVDS because we now have bidirectional type

        :param checker:
        :param pll_inst:
        :return:
        '''
        is_found_ext_lvds_rx_refclk = False

        # Check if PLL clock source is in external mode
        if pll_inst.ref_clock_mode == pll_inst.RefClockModeType.external:

            # Check if the external resource is an LVDS Rx
            ref_clock_res_name = pll_inst.get_ext_refclk_res_name(
                checker.design, pll_inst.ext_ref_clock_id)

            # Check if the resource is LVDS Rx by searching for it in the
            # lvds registry and that the ops type is Rx/Bidir with instance
            # in pll_clkin conn type
            if ref_clock_res_name != "" and checker.reg is not None:
                lvds_refclk_ins = checker.reg.get_inst_by_device_name(ref_clock_res_name)

                # Also check bidir type
                if lvds_refclk_ins is not None and \
                        (lvds_refclk_ins.ops_type == lvds_refclk_ins.OpsType.op_rx or
                                lvds_refclk_ins.ops_type == lvds_refclk_ins.OpsType.op_bidir):

                    lvds_rx_info = lvds_refclk_ins.rx_info
                    if lvds_rx_info is not None and \
                            lvds_rx_info.conn_type == lvds_rx_info.ConnType.pll_clkin_conn:
                        is_found_ext_lvds_rx_refclk = True

        return is_found_ext_lvds_rx_refclk

class LVDSRuleRxPLLRefClk(lvdsrule.LVDSRuleRxPLLRefClk):
    """
    Check that the clock source is from valid PLL and that
    the frequencies are valid.
    """

    def is_pll_refclk_valid(self, checker, pll_inst):
        '''
        Check that the pll instance has the right settings:
        1) ref clock mode is set to external
        2) ref_clock2_mode is chosen and connected to an LVDS Rx
            configured with LVDS mode
        '''

        # It has to be in external ref clock mode
        if pll_inst.ref_clock_mode != pll_inst.RefClockModeType.external:
            return False

        # It has to select either ext 0 or ext 1 or ext 2 if there are 3 externals
        # associated to the PLL
        if pll_inst.ext_ref_clock_id != pll_inst.RefClockIDType.ext_clock_0 and\
           pll_inst.ext_ref_clock_id != pll_inst.RefClockIDType.ext_clock_1 and\
           pll_inst.ext_ref_clock_id != pll_inst.RefClockIDType.ext_clock_2:
            return False

        # Find the LVDS resource for that clock

        # Get the resource name for the selected external clock
        ref_clock_res_name = pll_inst.get_ext_refclk_res_name(
            checker.design, pll_inst.ext_ref_clock_id)

        # No resource set to that external id
        if ref_clock_res_name == "":
            return False

        ins_obj = checker.device_db.find_instance(ref_clock_res_name)
        if ins_obj is None:
            return False

        ref_name = ins_obj.get_ref_name()

        _, ext_count = checker.plldev_service.get_ref_clk_type_count(
            pll_inst.pll_def)

        # If it returns 0 ext_count, it means that the PLL doesn't have any
        # resource set yet
        if ext_count == 0:
            return False

        valid = False

        if ref_name in [devdb_int.DeviceDBService.block_type2str_map[devdb_int.DeviceDBService.BlockType.HSIO],
                        devdb_int.DeviceDBService.block_type2str_map[devdb_int.DeviceDBService.BlockType.HSIO_MAX]]:

            # Check that the external clock was set to an LVDS instance. Don't need to
            # check the ext_ref_clock_id since we already checked earlier that it has to
            # be set to ext_clock0/1/2
            ref_clock_name = pll_inst.get_clock_pin_name_on_ext_clock_id(
                checker.design, pll_inst.ext_ref_clock_id)
            if ref_clock_name == "":
                return False

            # Check that the resource is in the LVDS mode
            if not checker.reg.is_lvds_mode_used(ref_clock_res_name):
                return False

            # Check that the lvds is in alternate connection: pll_clkin
            lvds_ins = checker.reg.get_inst_by_device_name(ref_clock_res_name)
            if lvds_ins is not None:
                if (lvds_ins.ops_type == lvds_ins.OpsType.op_rx or \
                        lvds_ins.ops_type == lvds_ins.OpsType.op_bidir) and\
                        lvds_ins.rx_info is not None:

                    lvds_rx = lvds_ins.rx_info

                    if lvds_rx.conn_type == lvds_rx.ConnType.pll_clkin_conn and\
                            not lvds_rx.is_serial:
                        valid = True

        return valid

    def check(self, checker, design_block: LVDSAdvance):
        lvds = design_block

        # Skip if not an LVDS Rx or bidir
        if lvds.ops_type == lvds.OpsType.op_tx:
            return

        lvds_rx = lvds.rx_info

        # Skip for bypass mode and non-normal type
        if lvds_rx is not None and lvds_rx.is_serial:

            fast_pll_data = None
            slow_pll_data = None

            if lvds_rx.fast_clock_name != "":
                fast_pll_data = checker.pll_reg.find_output_clock(
                    lvds_rx.fast_clock_name)
                if fast_pll_data is None:
                    self.error("Serial clock name is not a PLL output clock")
                    return

            if lvds_rx.slow_clock_name != "":
                slow_pll_data = checker.pll_reg.find_output_clock(
                    lvds_rx.slow_clock_name)
                if slow_pll_data is None:
                    self.error("Parallel clock name is not a PLL output clock")
                    return

            pll_fast_inst = None
            pll_slow_inst = None
            fast_ins_dep_names = []
            slow_ins_dep_names = []

            # TODO: Check whether there's pll dependencies. Update device
            # database if there is
            if fast_pll_data is not None:
                pll_fast_inst, _ = fast_pll_data

                # Check that the clock are from the right depended PLL instance
                # Warning will be overridden by error if there's one
                fast_ins_dep_names = checker.lvdsrx_service.get_instance_dependencies_name(
                    lvds.lvds_def)
                if pll_fast_inst.pll_def not in fast_ins_dep_names:
                    self.warning("Serial clock is expected to be from the following PLL instance: {}".format(
                        ",".join(sorted(fast_ins_dep_names))))

                else:
                    # Check that the pll has its refclk from an LVDS Rx in LVDS
                    # mode
                    if not self.is_pll_refclk_valid(checker, pll_fast_inst):
                        self.warning("PLL driving the serial clock should have its reference clock from an LVDS in pll_clkin connection type")

            # TODO: Check whether there's pll dependencies. Update device
            # database if there is
            if slow_pll_data is not None:
                pll_slow_inst, _ = slow_pll_data

                # Check that the clock are from the right depended PLL instance
                slow_ins_dep_names = checker.lvdsrx_service.get_instance_dependencies_name(
                    lvds.lvds_def)
                if pll_slow_inst.pll_def not in slow_ins_dep_names:
                    self.warning("Parallel clock is expected to be from the following PLL instance: {}".format(
                        ",".join(sorted(slow_ins_dep_names))))

                else:
                    # Check that the pll has its refclk from an LVDS Rx in LVDS
                    # mode
                    if not self.is_pll_refclk_valid(checker, pll_slow_inst):
                        self.warning("PLL driving the parallel clock should have its reference clock from an LVDS in pll_clkin connection type")


class LVDSTxRuleUsageDistance(lvdsrule.LVDSTxRuleUsageDistance):
    """
    Check that the LVTTL pins must be placed at least 2 columns (1 pair)
    away from LVDS within the same bank.
    """

    def check(self, checker: LVDSCheckerAdv, design_block: LVDSAdvance): # type: ignore

        lvds = design_block

        # Skip if not an LVDS Tx
        if lvds.ops_type != lvds.OpsType.op_tx and lvds.ops_type != lvds.OpsType.op_bidir:
            return

        #bank2ins, ins2banks = checker.get_used_bank2ins_tx(checker)

        # Check if within the bank group, we have combination of
        # LVDS and LVTTL usage
        if lvds.lvds_def in checker.tx_ins2group:

            # checker.logger.debug("LVDS tx: {} in {}".format(
            #    lvds.lvds_def, checker.tx_ins2banks))

            # Get the banks associated to this instance
            group_name = checker.tx_ins2group[lvds.lvds_def]

            res_svc = checker.reg.res_svc

            if group_name in checker.tx_group2ins and res_svc is not None:

                ins_names = checker.tx_group2ins[group_name]

                # Sort the name
                sorted_ins_names = sorted(ins_names)

                is_valid, severity, pass_status, msg = checker.check_usage_distance(
                    res_svc, lvds, group_name, sorted_ins_names)

                if not is_valid:
                    self.severity = severity
                    self.pass_status = pass_status
                    self.msg = msg


class LVDSRxRuleUsageDistance(lvdsrule.LVDSRxRuleUsageDistance):
    """
    Check that the LVTTL pins must be placed at least 2 columns (1 pair)
    away from LVDS within the same bank.
    """

    def check(self, checker: LVDSCheckerAdv, design_block: LVDSAdvance): # type: ignore

        lvds = design_block

        # Skip if not an LVDS Rx
        if lvds.ops_type != lvds.OpsType.op_rx and lvds.ops_type != lvds.OpsType.op_bidir:
            return

        #bank2ins, ins2banks = checker.get_used_bank2ins_tx(checker)

        # Check if within the bank group, we have combination of
        # LVDS and LVTTL usage
        if lvds.lvds_def in checker.rx_ins2group:

            # checker.logger.debug("LVDS tx: {} in {}".format(
            #    lvds.lvds_def, checker.tx_ins2banks))

            # Get the banks associated to this instance
            group_name = checker.rx_ins2group[lvds.lvds_def]

            res_svc = checker.reg.res_svc

            if group_name in checker.rx_group2ins and res_svc is not None:

                ins_names = checker.rx_group2ins[group_name]

                # Sort the name
                sorted_ins_names = sorted(ins_names)

                is_valid, severity, pass_status, msg = checker.check_usage_distance(
                    res_svc, lvds, group_name, sorted_ins_names)

                if not is_valid:
                    self.severity = severity
                    self.pass_status = pass_status
                    self.msg = msg


class LVDSRxAlternateConfig(lvdsrule.LVDSRxAlternateConfig):
    """
    Check if the LVDS Rx with the alternate connection
    type is being used correctly.

    PT-890: Downgrade usage error to warning
    """

    def check_pll_advance_dependencies(self, checker: LVDSCheckerAdv, lvds, secfunc_names, pll_dep_ins_obj):

        # If it is advance PLL, we need to check if the LVDS is
        # used as its reference clock or external feedback
        no_issues = True
        assert checker.plldev_service is not None

        if SecondaryConn.SEC_TYPE_PLL_CLKIN in secfunc_names or \
                SecondaryConn.SEC_TYPE_LVDS_REFCLK in secfunc_names:

            if pll_dep_ins_obj.ref_clock_mode == pll_dep_ins_obj.RefClockModeType.external:
                # figure out which one of the external connection is being assigned to
                # pll check will determine the correct setting on the pll reference
                # clock.

                _, ext_count = checker.plldev_service.get_ref_clk_type_count(
                    pll_dep_ins_obj.pll_def)

                if pll_dep_ins_obj.ext_ref_clock_id == pll_dep_ins_obj.RefClockIDType.ext_clock_0 or \
                        pll_dep_ins_obj.ext_ref_clock_id == pll_dep_ins_obj.RefClockIDType.ext_clock_1 or \
                        (pll_dep_ins_obj.ext_ref_clock_id == pll_dep_ins_obj.RefClockIDType.ext_clock_2 and ext_count == 3):

                    ext_index = pll_dep_ins_obj.get_external_clkin_index(
                        checker.plldev_service, pll_dep_ins_obj.ext_ref_clock_id)

                    # Check that the resource name matches
                    res_name_list = checker.plldev_service.get_all_resource_on_ref_clk_pin(
                        pll_dep_ins_obj.pll_def, ext_index)

                    checker.logger.debug("Checking lvds {} pll {} ref id: {} res_names: {}".format(
                        lvds.lvds_def, pll_dep_ins_obj.name,
                        pll_dep_ins_obj.ext_ref_clock_id, ",".join(res_name_list)))

                    found_res = False
                    for res_name in res_name_list:
                        # Check if it matches with the current GPIO being
                        # checked
                        if res_name == lvds.lvds_def:
                            found_res = True
                            break

                    if not found_res:
                        self.warning('pll_clkin connection to PLL clock source not being used in {}'.format(
                            pll_dep_ins_obj.name))
                        no_issues = False

                else:
                    self.warning(
                        'pll_clkin connection to PLL clock source ' \
                        'but none of the external clock source in PLL {} ' \
                        'is selected'.format(pll_dep_ins_obj.name)
                    )
                    no_issues = False

            elif pll_dep_ins_obj.ref_clock_mode != pll_dep_ins_obj.RefClockModeType.dynamic and \
                    (pll_dep_ins_obj.is_param_supported(PLLDesignParamInfo.Id.dyn_cfg_en) and
                     pll_dep_ins_obj.dyn_cfg_enable is False):
                # in dynamic all external connections are used and can be ignored
                # for the check

                self.warning(
                    'pll_clkin connection to PLL clock source ' \
                    'but PLL Clock source on {} is set to core'
                    .format(pll_dep_ins_obj.name)
                )
                no_issues = False

        if no_issues and (SecondaryConn.SEC_TYPE_PLL_EXTFB in secfunc_names or
                          SecondaryConn.SEC_TYPE_LVDS_EXTFB in secfunc_names):

            if pll_dep_ins_obj.fb_mode != pll_dep_ins_obj.FeedbackModeType.external:
                self.warning(
                    'pll_extfb connection to PLL external feedback pin' \
                    ' but PLL feedback on {} is not set to external'.format(
                    pll_dep_ins_obj.name)
                )

    def check(self, checker: LVDSCheckerAdv, design_block: LVDSAdvance): # type: ignore
        lvds = design_block

        # Skip if not an LVDS Rx
        if lvds.ops_type != lvds.OpsType.op_rx and lvds.ops_type != lvds.OpsType.op_bidir:
            return

        if self.is_name_empty(lvds.lvds_def):
            return

        lvds_rx = lvds.rx_info

        if lvds_rx is not None and \
                lvds_rx.is_alternate_connection() and\
                checker.lvdsrx_service is not None:

            # If it is set to alternate, then we check that the LVDS
            # does support secondary connection
            secfunc_names = checker.lvdsrx_service.get_secondary_connection(
                lvds.lvds_def)

            if secfunc_names:
                # Convert the conn_type to upper case and check if it is in
                # the sec function list
                conn_type_str = lvds_rx.conn2str_map[lvds_rx.conn_type].upper(
                )
                if conn_type_str in secfunc_names:
                    # If it is set to alternate, then we check for the PLL
                    self.check_pll_dependencies(checker, lvds, secfunc_names)


class LVDSRuleRxDeserialWidth(base_rule.Rule):
    """
    Check that the Configuration parameters are valid
    """

    def __init__(self):
        super().__init__()
        self.name = "lvds_rule_rx_serial_width"

    def check(self, checker, design_block: LVDSAdvance):
        lvds = design_block

        if (lvds.ops_type == lvds.OpsType.op_rx or lvds.ops_type == lvds.OpsType.op_bidir) and \
                lvds.rx_info is not None:

            if lvds.rx_info.is_serial:
                if int(lvds.rx_info.serial_width) == 9:
                    self.error("Unsupported deserialization width: 9")
                    return


class LVDSRuleTxSerialWidth(base_rule.Rule):
    """
    Check that the Configuration parameters are valid
    """

    def __init__(self):
        super().__init__()
        self.name = "lvds_rule_tx_serial_width"

    def check(self, checker, design_block: LVDSAdvance):
        lvds = design_block

        if (lvds.ops_type == lvds.OpsType.op_tx or lvds.ops_type == lvds.OpsType.op_bidir) and \
                lvds.tx_info is not None:

            if lvds.tx_info.is_serial:
                if int(lvds.tx_info.serial_width) == 9:
                    self.error("Unsupported serialization width: 9")
                    return


class LVDSRuleRxDeserialRate(base_rule.Rule):
    """
    Check that the Configuration parameters are valid
    """

    def __init__(self):
        super().__init__()
        self.name = "lvds_rule_deserial_rate"

    def check(self, checker, design_block: LVDSAdvance):
        lvds = design_block

        if (lvds.ops_type == lvds.OpsType.op_rx or lvds.ops_type == lvds.OpsType.op_bidir) and \
                lvds.rx_info is not None:

            if lvds.rx_info.is_serial and lvds.rx_info.is_half_rate:
                if lvds.rx_info.is_serial_width_gt2():
                    # Half rate serialzation only allowed with even width (> 2)
                    if int(lvds.rx_info.serial_width) % 2 != 0:
                        self.error("Half rate deserialization only allowed with even deserialization width")
                        return
                # PT-1112: Ignore half rate with x1/x2
                # For x1 and x2, we force user to set to half rate (LVDSRuleRxDeserialWidth2)
                # else:
                #    self.error("Half rate deserialization not allowed with deserialization width x{}".format(
                #        int(lvds.rx_info.serial_width)))
                #    return


class LVDSRuleTxSerialRate(base_rule.Rule):
    """
    Check that the Configuration parameters are valid
    """

    def __init__(self):
        super().__init__()
        self.name = "lvds_rule_serial_rate"

    def check(self, checker, design_block: LVDSAdvance):
        lvds = design_block

        if (lvds.ops_type == lvds.OpsType.op_tx or lvds.ops_type == lvds.OpsType.op_bidir) and \
                lvds.tx_info is not None:

            if lvds.tx_info.is_serial and lvds.tx_info.is_half_rate:
                if lvds.tx_info.is_serial_width_gt2():
                    if int(lvds.tx_info.serial_width) % 2 != 0:
                        self.error("Half rate serialization only allowed with even serialization width")
                        return

                # PT-1112: Ignore half rate with x1/x2
                # For x1 and x2, we force user to set to half rate (LVDSRuleTxDeserialWidth2)
                # else:
                #    self.error("Half rate serialization not allowed with serialization width x{}".format(
                #        int(lvds.tx_info.serial_width)))
                #    return


# class LVDSRuleBidirTxMode(base_rule.Rule):
#     """
#     Check that the Configuration parameters are valid
#     """

#     def __init__(self):
#         super().__init__()
#         self.name = "lvds_rule_bidir_tx"

#     def check(self, checker, design_block: LVDSAdvance):
#         lvds = design_block

#         if lvds.ops_type == lvds.OpsType.op_bidir and \
#                 lvds.tx_info is not None:

#             # OE is compulsory in bidir
#             if self.check_name_empty_or_invalid(lvds.tx_info.oe_name,
#                 "Output enable pin name must be configured in Bidirectional LVDS Tx"
#             ):
#                 return


class LVDSRuleTxVref(base_rule.Rule):
    """
    Check that the Configuration parameters are valid
    """

    def __init__(self):
        super().__init__()
        self.name = "lvds_rule_tx_vref"

    def check(self, checker: LVDSCheckerAdv, design_block: LVDSAdvance): # type: ignore
        lvds = design_block

        if (lvds.ops_type == lvds.OpsType.op_tx or lvds.ops_type == lvds.OpsType.op_bidir) and \
                lvds.tx_info is not None:

            # Check the common mode if external
            if lvds.tx_info.diff_type == lvds.tx_info.DiffType.custom:

                # Check that a VREF is configured (copied from hsio gpio rule)
                _, ins2banks = checker.lvdstx_service.get_bank_to_instances(
                    [lvds], is_raw=True)

                # TODO: Consolidate
                bank_list = []
                for ins_name in ins2banks:
                    if ins_name == lvds.lvds_def:
                        bank_list = ins2banks[ins_name]

                checker.logger.debug("Instance {} mapped to bank {}".format(
                    lvds.lvds_def, ",".join(bank_list)))

                # Find the vref on the same bank
                if bank_list:

                    # Get the device interface
                    dbi = devdb_int.DeviceDBService(checker.design.device_db)
                    if dbi is not None:

                        # Get the list of instances that supports vref
                        vref_ins_names_list = dbi.get_instance_names_has_secondary_function(
                            "VREF")

                        if vref_ins_names_list:
                            # This is a map of instance to bank name string
                            # if there is multiple bank, it is separated by ','
                            ins2banks_map: Dict[str, str] = dbi.get_instance_to_raw_io_bank_names()

                            # Find the resources with VREF and on the same bank
                            is_found = False

                            # The instance name should only be associated to
                            # the P resource
                            vref_dev_ins_name = ""
                            iobank_name = ""
                            for ins_name in vref_ins_names_list:
                                # We only want instance names that are associated
                                # to GPIO resource (p or n)
                                checker.logger.debug(
                                    "Reading vref ins {}".format(ins_name))

                                if checker.reg.res_svc is not None and\
                                        checker.reg.res_svc.is_hsio_gpio_resource(ins_name):
                                    # Find the instance that's in the same bank as the
                                    # current gpio instance bank
                                    #_, vref_ins2banks = checker.hsiogpiodev_service.get_bank_to_instances(
                                    #   [gpio], is_raw=True)
                                    vref_bank_names = ""
                                    bank_name_list = []

                                    if ins_name in ins2banks_map:
                                        vref_bank_names = ins2banks_map[ins_name]

                                        bank_name_list = vref_bank_names.split(
                                            ",")

                                    checker.logger.debug("Vref Instance {} mapped to bank {}".format(
                                        ins_name, vref_bank_names))

                                    for bname in bank_name_list:

                                        if bname in bank_list:
                                            # We stop assuming that the gpio only
                                            # has 1 bank associated although it's a list.
                                            # same goes with the vref resource.
                                            vref_dev_ins_name = ins_name
                                            is_found = True
                                            iobank_name = bname
                                            break

                                    if is_found:
                                        break

                            if is_found and checker.design.gpio_reg is not None:
                                # Check if the resource is configured as input
                                # vref
                                # vref_des_ins_list = checker.reg.res_svc.find_resource_user(
                                #    vref_dev_ins_name)
                                # Use the gpio registry to check if any GPIO
                                # VREF was configured
                                vref_ins_obj: Any = checker.design.gpio_reg.get_gpio_by_device_name(
                                    vref_dev_ins_name)

                                # Take the first one (should only be one)
                                if vref_ins_obj is not None:
                                    #vref_ins_obj = vref_des_ins_list[0]

                                    # If it is exactly the same object and not a vref, then need to specify
                                    # that this is reserved
                                    if vref_ins_obj == lvds:
                                        self.error("This resource is reserved as vref for bank ({}). Use a different resource to configure LVDS Tx custom output differential type".format(
                                            iobank_name))

                                    else:
                                        if vref_ins_obj.mode != vref_ins_obj.PadModeType.input:
                                            self.error("GPIO {} has to be configured as vref input mode to support LVDS Tx custom output differential type".format(
                                                vref_dev_ins_name))

                                        else:
                                            vref_input = vref_ins_obj.input

                                            if vref_input is not None and vref_input.conn_type != vref_input.ConnType.vref_conn:
                                                self.error("GPIO {} has to be configured as vref input connection type to support LVDS Tx custom output differential type".format(
                                                    vref_dev_ins_name))

                                else:
                                    # The resource is not configured as GPIO at
                                    # all
                                    self.error("GPIO {} has to be configured as vref input to support LVDS Tx custom output differential type".format(
                                        vref_dev_ins_name))

                            else:
                                # No vref resource bonded out
                                self.error("LVDS Tx custom output differential type cannot be used due to unbonded vref resource on the same bank {}".format(
                                    ",".join(bank_list)))

                        else:
                            # No vref resource bonded out
                            self.error("LVDS Tx custom output differential type cannot be used due to vref resource not bonded out")

                    else:
                        checker.logger.warning(
                            "{}: device db is empty, skip valid instance check.".format(self.name))


class LVDSRuleRxConfig(base_rule.Rule):
    """
    Check that the Configuration parameters are valid
    """

    def __init__(self):
        super().__init__()
        self.name = "lvds_rule_rx_param"

    def check(self, checker, design_block: LVDSAdvance):
        lvds = design_block

        if (lvds.ops_type == lvds.OpsType.op_rx or lvds.ops_type == lvds.OpsType.op_bidir) and \
                lvds.rx_info is not None:
            rx_config = lvds.rx_info

            def get_param_name(param_obj):
                return param_obj.name

            invalid_param_names = []
            param_info = rx_config.get_param_info()
            if rx_config.param_group is not None and param_info is not None:
                all_param = rx_config.param_group.get_all_param()

                for param_obj in sorted(all_param, key=get_param_name):
                    if param_obj is not None:
                        is_valid, _ = rx_config.param_group.check_param_value(
                            param_obj.name, param_info)
                        if not is_valid:
                            invalid_param_names.append(param_obj.name)

            if invalid_param_names:
                self.error("Invalid parameters configuration: {}".format(
                    ",".join(invalid_param_names)))
                return


class LVDSRuleTxConfig(base_rule.Rule):
    """
    Check that the Configuration parameters are valid
    """

    def __init__(self):
        super().__init__()
        self.name = "lvds_rule_tx_param"

    def check(self, checker, design_block: LVDSAdvance):
        lvds = design_block

        if (lvds.ops_type == lvds.OpsType.op_tx or lvds.ops_type == lvds.OpsType.op_bidir) and \
                lvds.tx_info is not None:
            tx_config = lvds.tx_info

            def get_param_name(param_obj):
                return param_obj.name

            invalid_param_names = []
            param_info = tx_config.get_param_info()
            if tx_config.param_group is not None and param_info is not None:
                all_param = tx_config.param_group.get_all_param()

                for param_obj in sorted(all_param, key=get_param_name):
                    if param_obj is not None:
                        is_valid, msg = tx_config.param_group.check_param_value(
                            param_obj.name, param_info)

                        if not is_valid:
                            checker.logger.debug("Invalid {} value: {}".format(
                                param_obj.name, msg))
                            invalid_param_names.append(param_obj.name)

            if invalid_param_names:
                self.error("Invalid parameters configuration: {}".format(
                    ",".join(invalid_param_names)))
                return


class LVDSRuleTxSerialWidth2(base_rule.Rule):
    """
    Check that the Configuration parameters are valid for Tx x2
    """

    def __init__(self):
        super().__init__()
        self.name = "lvds_rule_tx_width_1or2"

    def check(self, checker, design_block: LVDSAdvance):
        lvds = design_block

        if (lvds.ops_type == lvds.OpsType.op_tx or lvds.ops_type == lvds.OpsType.op_bidir) and \
                lvds.tx_info is not None:

            # Need to filter only on mode that supports serialization
            if lvds.tx_info.is_serial and lvds.tx_info.mode == lvds.tx_info.ModeType.out:
                if not lvds.tx_info.is_serial_width_gt2():
                    # Check that the name is set to slowclk (make sense for
                    # user).
                    width_int = int(lvds.tx_info.serial_width)

                    if self.check_name_empty_or_invalid(lvds.tx_info.slow_clock_name,
                        "Parallel clock name is required with serialization width {}".format(width_int)
                    ):
                        return

                    if not self.is_name_empty(lvds.tx_info.fast_clock_name):
                        self.error("Serialization width {} only requires the parallel clock name to be specified".format(width_int))
                        return

                    # PT-1112: Removed this as it shouldn't matter what user selects when in x1/x2
                    # Check that half-rate is enabled
                    #if not lvds.tx_info.is_half_rate:
                    #    self.error("Serialization width {} can only be used with half-rate enabled".format(
                    #        width_int))
                    #    return


class LVDSRuleRxDeserialWidth2(base_rule.Rule):
    """
    Check that the Configuration parameters are valid for Rx x2
    """

    def __init__(self):
        super().__init__()
        self.name = "lvds_rule_rx_width_1or2"

    def check(self, checker, design_block: LVDSAdvance):
        lvds = design_block

        if (lvds.ops_type == lvds.OpsType.op_rx or lvds.ops_type == lvds.OpsType.op_bidir) and \
                lvds.rx_info is not None:

            # Need to filter only on mode that supports serialization
            if lvds.rx_info.is_serial and lvds.rx_info.conn_type == lvds.rx_info.ConnType.normal_conn:
                if not lvds.rx_info.is_serial_width_gt2():

                    width_int = int(lvds.rx_info.serial_width)

                    # Check that the name is set to slowclk (make sense for
                    # user).
                    if self.check_name_empty_or_invalid(lvds.rx_info.slow_clock_name,
                        "Parallel clock name is required with deserialization width {}".format(width_int)
                    ):
                        return

                    if not self.is_name_empty(lvds.rx_info.fast_clock_name):
                        self.error("Deserialization width {} only requires the parallel clock name to be specified".format(width_int))
                        return

                    # PT-1112: Removed this as it shouldn't matter what user selects when in x1/x2
                    # Check that half-rate is enabled
                    #if not lvds.rx_info.is_half_rate:
                    #    self.error("Deserialization width {} can only be used with half-rate enabled".format(
                    #        width_int))
                    #    return


class LVDSRxAlternateConn(lvdsrule.LVDSRxAlternateConn):
    """
    Because of PT-890, the check for resource validity with
    alternate type assignment is separated out. This is aligned
    with the GPIO approach where the usage and resource validity
    are separated into different rule.
    """

    def check(self, checker, design_block: LVDSAdvance):
        lvds = design_block

        # Skip if not Rx and bidir
        if lvds.ops_type != lvds.OpsType.op_rx and lvds.ops_type != lvds.OpsType.op_bidir:
            return

        if self.is_name_empty(lvds.lvds_def):
            return

        self.check_conn_type(checker, lvds)


class LVDSRuleRxFIFO(base_rule.Rule):
    """
    Check that FIFO is not enabled for x1 and x2. Also FIFO is only valid
    with serialization enabled.
    """

    def __init__(self):
        super().__init__()
        self.name = "lvds_rule_rx_fifo"

    def check(self, checker, design_block: LVDSAdvance):
        lvds = design_block

        # Skip if not Rx and bidir
        if (lvds.ops_type == lvds.OpsType.op_rx or lvds.ops_type == lvds.OpsType.op_bidir) and \
                lvds.rx_info is not None:
            rx_config = lvds.rx_info

            if rx_config.is_fifo and rx_config.conn_type == rx_config.ConnType.normal_conn:
                # Check that serialization width is not x1 and x2
                if not rx_config.is_serial_width_gt2():
                    self.error("Clock Crossing FIFO is not supported with deserialization width {}".format(
                        int(rx_config.serial_width)))
                    return

                # The following is auto-controlled in UI but not API
                if not rx_config.is_serial:
                    self.error("Clock Crossing FIFO is only supported with deserialization enabled")
                    return

class LVDSRuleRxPin(base_rule.Rule, metaclass=abc.ABCMeta):
    """
    Helper class for checking empty/invalid pin names in LVDS Rx
    Implmented class needs to inherit either RuleEmptyName or RuleInvalidName
    """
    @abc.abstractmethod
    def validate(self, pins : Dict[str, base_rule.PinData]):
        pass

    def check(self, checker, design_block: LVDSAdvance):
        lvds = design_block

        # Return if it is Tx
        if lvds.ops_type == lvds.OpsType.op_tx:
            return

        rx_cfg = lvds.rx_info

        if rx_cfg is not None:

            pins: Dict[str, base_rule.PinData] = {}

            def get_desc_name(gpin, default_name):
                """
                Get desc name from pin, defaults to type name
                """
                desc_name = rx_cfg.get_pin_property_name(gpin.type_name)
                if desc_name == "":
                    desc_name = default_name
                return desc_name

            # Pins that are only associated to Normal mode
            if rx_cfg.conn_type == rx_cfg.ConnType.normal_conn:
                if rx_cfg.is_fifo and rx_cfg.is_serial and rx_cfg.is_serial_width_gt2():
                    # Ignore if x1 and x2
                    if rx_cfg.gen_pin is not None:
                        fifo_pins = rx_cfg.get_fifo_pin()
                        for fpin in fifo_pins:
                            gpin = rx_cfg.gen_pin.get_pin_by_type_name(fpin)
                            if gpin is not None:
                                pins[get_desc_name(gpin, fpin)] = base_rule.PinData(gpin.name)

                if rx_cfg.is_serial:
                    if rx_cfg.is_serial_width_gt2():
                        pins["Serial Clock Name"] = base_rule.PinData(rx_cfg.fast_clock_name)
                        pins["Parallel Clock Name"] = base_rule.PinData(rx_cfg.slow_clock_name)
                    else:
                        # For x1 and x2, only the assignment at is compulsory (refer rule LVDSRuleRx)
                        pins["Serial Clock Name"] = base_rule.PinData(rx_cfg.fast_clock_name, allow_empty=True)
                        pins["Parallel Clock Name"] = base_rule.PinData(rx_cfg.slow_clock_name)

                    # PT-1341: Adding check for Reset Pin Name
                    # For now ignore pin being empty (old behaviour) but raise error if invalid name
                    rst_pin = rx_cfg.gen_pin.get_pin_by_type_name("RST")
                    if rst_pin is not None:
                        pins["Reset Pin Name"] = base_rule.PinData(rst_pin.name, allow_empty=True)

            if rx_cfg.is_delay_mode_non_static():
                if rx_cfg.delay_mode == rx_cfg.DelayModeType.dynamic:
                    optional_pins = ["DLY_RST"]
                    dly_pins = rx_cfg.get_delay_pin()

                else:
                    # DPA
                    optional_pins = ["DLY_RST", "DBG", "LOCK"]
                    dly_pins = rx_cfg.get_delay_dpa_pin()

                if rx_cfg.gen_pin is not None:
                    for dpin in dly_pins:
                        gpin = rx_cfg.gen_pin.get_pin_by_type_name(dpin)

                        if gpin is not None:
                            allow_empty = False

                            # Skip checking if optional pins are filled
                            if dpin in optional_pins:
                                allow_empty = True

                            pins[get_desc_name(gpin, dpin)] = base_rule.PinData(gpin.name, allow_empty)

                # With PT-1192: the slow_clock_name now is compulsory with dynamic delay mode
                pins["Dynamic Delay Clock Pin Name"] = base_rule.PinData(rx_cfg.slow_clock_name)

            if rx_cfg.termination == rx_cfg.TerminationType.dynamic and \
                    rx_cfg.gen_pin is not None:
                term_pins = rx_cfg.get_term_pin()
                for tpin in term_pins:
                    gpin = rx_cfg.gen_pin.get_pin_by_type_name(tpin)
                    if gpin is not None:
                        pins[get_desc_name(gpin, tpin)] = base_rule.PinData(gpin.name)

            pins["Input Pin/Bus Name"] = base_rule.PinData(rx_cfg.input_bus_name)

            # PT-1341: Adding check for Dynamic Enable Pin Name
            # For now ignore pin being empty (old behaviour) but raise error if invalid name
            ena_pin = rx_cfg.gen_pin.get_pin_by_type_name("ENA")
            if ena_pin is not None:
                pins["Dynamic Enable Pin Name"] = base_rule.PinData(ena_pin.name, allow_empty=True)


            self.validate(pins)


class LVDSRuleRxPinEmptyName(base_rule.RuleEmptyName, LVDSRuleRxPin):
    """
    PT-1111: Check that mandatory pins in LVDS Rx are non-empty
    """
    def __init__(self):
        super().__init__()
        self.name = "lvds_rule_rx_empty_pins"

class LVDSRuleRxPinInvalidName(base_rule.RuleInvalidName, LVDSRuleRxPin):
    """
    Check that mandatory pins in LVDS Rx are valid
    """
    def __init__(self):
        super().__init__()
        self.name = "lvds_rule_rx_invalid_pins"


class LVDSRuleTxPin(base_rule.Rule, metaclass=abc.ABCMeta):
    """
    Helper class for checking empty/invalid pin names in LVDS Tx
    Implmented class needs to inherit either RuleEmptyName or RuleInvalidName
    """
    @abc.abstractmethod
    def validate(self, pins : Dict[str, base_rule.PinData]):
        pass

    def check(self, checker, design_block: LVDSAdvance):
        lvds = design_block

        # Return if it is Rx
        if lvds.ops_type == lvds.OpsType.op_rx:
            return

        tx_cfg = lvds.tx_info

        if tx_cfg is not None and tx_cfg.mode is not None:

            pins : Dict[str, base_rule.PinData] = {}

            # Pin is enabled in output mode and is required
            if tx_cfg.mode == tx_cfg.ModeType.out:
                pins["Output Pin/Bus Name"] = base_rule.PinData(tx_cfg.output_bus_name)

            # Pin is always enabled but only compulsory in lvds_bidir
            pins["Output Enable Pin Name"] = base_rule.PinData(tx_cfg.oe_name,
                                                        allow_empty = not (lvds.ops_type == lvds.OpsType.op_bidir)
                                                        )
            # Serial
            if tx_cfg.is_serial:
                pins["Parallel Clock Name"] = base_rule.PinData(tx_cfg.slow_clock_name)
                # x1,x2 does not need to have this field non-empty:
                # rules LVDSRuleTxOutputMode and LVDSRuleTxClockOutMode
                pins["Serial Clock Name"] = base_rule.PinData(tx_cfg.fast_clock_name,
                                                            allow_empty = not (tx_cfg.is_serial_width_gt2())
                                                            )
                # Optional pin
                pins["Reset Pin Name"] = base_rule.PinData(tx_cfg.reset_name, allow_empty=True)

            self.validate(pins)


class LVDSRuleTxPinEmptyName(base_rule.RuleEmptyName, LVDSRuleTxPin):
    """
    PT-1111: Check that mandatory pins in LVDS Tx are non-empty
    """
    def __init__(self):
        super().__init__()
        self.name = "lvds_rule_tx_empty_pins"

class LVDSRuleTxPinInvalidName(base_rule.RuleInvalidName, LVDSRuleTxPin):
    """
    Check that mandatory pins in LVDS Tx are valid
    """
    def __init__(self):
        super().__init__()
        self.name = "lvds_rule_tx_invalid_pins"


class LVDSRuleRxDPA(base_rule.Rule):
    """
    PT-978: Check that half-rate deserialization isn't use with delay mode DPA
    """

    def __init__(self):
        super().__init__()
        self.name = "lvds_rule_rx_dpa"

    def check(self, checker, design_block: LVDSAdvance):
        lvds = design_block

        # Return if it is Tx
        if lvds.ops_type == lvds.OpsType.op_tx:
            return

        rx_cfg = lvds.rx_info

        if rx_cfg is not None:
            if rx_cfg.delay_mode == rx_cfg.DelayModeType.dpa:
                if rx_cfg.is_serial and rx_cfg.is_half_rate:
                    self.error("Half-rate deserialization is not supported with DPA delay mode")

class LVDSRuleRxDPAESDevice(base_rule.Rule):
    """
    PT-978: Check that DPA delay mode is not used in ES device
    """

    def __init__(self):
        super().__init__()
        self.name = "lvds_rule_rx_dpa_es_device"

    def check(self, checker: LVDSCheckerAdv, design_block: LVDSAdvance): # type: ignore
        lvds = design_block

        # Return if it is Tx
        if lvds.ops_type == lvds.OpsType.op_tx:
            return

        rx_cfg = lvds.rx_info

        if rx_cfg is not None:
            assert checker.design.device_db is not None
            if rx_cfg.delay_mode == rx_cfg.DelayModeType.dpa and\
                checker.design.device_db.is_sample_device():
                self.error("DPA delay mode is not supported in ES device")

class LVDSRuleRxPLLClkRegion(base_rule.Rule):
    """
    PT-1311 #1 - clocks going to slowclk and fastclk of LVDS rx cannot be a mixture
    of rclk and gclk. So, clkout4 is not acceptable since it is rclk while clkout0-3 of
    pll are all gclk.
    """

    def __init__(self):
        super().__init__()
        self.name = "lvds_rule_rx_clock_region"

    def check(self, checker: LVDSCheckerAdv, design_block: LVDSAdvance): # type: ignore
        lvds = design_block

        # Skip if LVDS Rx is not used
        if lvds.ops_type != lvds.OpsType.op_rx and lvds.ops_type != lvds.OpsType.op_bidir:
            return

        lvds_rx = lvds.rx_info

        # Skip for bypass mode and x1,x2
        if lvds_rx is not None and lvds_rx.is_serial and lvds_rx.is_serial_width_gt2():

            if lvds_rx.fast_clock_name != "" and \
                    lvds_rx.slow_clock_name != "":
                assert checker.pll_reg is not None
                assert checker.plldev_service is not None

                # There's already a separate rule to check for the validity of the PLL.
                # Rather than extend that check, we specify this as a new standalone check
                # just to flag out if it's using mix of regional and global clock
                err_msg = checker.check_lvds_fast_slow_clocks_conn_type(lvds_rx.fast_clock_name, lvds_rx.slow_clock_name)

                if err_msg != "":
                    self.error(err_msg)

class LVDSRuleTxPLLClkRegion(base_rule.Rule):
    """
    PT-1311 #1 - clocks going to slowclk and fastclk of LVDS rx cannot be a mixture
    of rclk and gclk. So, clkout4 is not acceptable since it is rclk while clkout0-3 of
    pll are all gclk.
    """

    def __init__(self):
        super().__init__()
        self.name = "lvds_rule_tx_clock_region"

    def check(self, checker: LVDSCheckerAdv, design_block: LVDSAdvance): # type: ignore

        lvds = design_block

        # Skip LVDS Tx is not used
        if lvds.ops_type != lvds.OpsType.op_tx and lvds.ops_type != lvds.OpsType.op_bidir:
            return

        lvds_tx = lvds.tx_info

        # This should skip checking for x1,x2
        if lvds_tx is not None and lvds_tx.is_serial and lvds_tx.is_serial_width_gt2():

            # Check that the fast and slow is not the same pin
            if lvds_tx.fast_clock_name != "" and \
                    lvds_tx.slow_clock_name != "":

                assert checker.pll_reg is not None
                assert checker.plldev_service is not None

                err_msg = checker.check_lvds_fast_slow_clocks_conn_type(lvds_tx.fast_clock_name, lvds_tx.slow_clock_name)

                if err_msg != "":
                    self.error(err_msg)

class LVDSRuleRxDPAWidth(base_rule.Rule):
    """
    PT-1309: DPA isn't supported on x1/x2 and bypass
    """

    def __init__(self):
        super().__init__()
        self.name = "lvds_rule_rx_dpa_serial"

    def check(self, checker, design_block: LVDSAdvance):
        lvds = design_block

        # Skip LVDS Rx is not used
        if lvds.ops_type != lvds.OpsType.op_rx and lvds.ops_type != lvds.OpsType.op_bidir:
            return

        rx_cfg = lvds.rx_info
        if rx_cfg is not None:
            if rx_cfg.delay_mode == rx_cfg.DelayModeType.dpa:
                if not rx_cfg.is_serial:
                    self.error("DPA delay mode is not supported with deserialization disabled")
                elif not rx_cfg.is_serial_width_gt2():
                    self.error("DPA delay mode is not supported with deserialization width less than 3")

class LVDSRuleRxPLLFeedback(lvdsrule.LVDSRuleRxPLLFeedback):
    """
    PT-1677: Warn if PLL is not using core feedback mode when the clocks are driving the LVDS Rx.
    """

    def check(self, checker, design_block: LVDSAdvance):
        lvds = design_block

        # Skip if LVDS Rx is not used
        if lvds.ops_type != lvds.OpsType.op_rx and lvds.ops_type != lvds.OpsType.op_bidir:
            return

        lvds_rx = lvds.rx_info

        # Skip for bypass mode and x1,x2
        if checker.pll_reg is not None and lvds_rx is not None and \
                lvds_rx.is_serial and lvds_rx.is_serial_width_gt2():

            if lvds_rx.fast_clock_name != "" and \
                    lvds_rx.slow_clock_name != "" and \
                    lvds_rx.delay_mode == lvds_rx.DelayModeType.static:

                # Only check further if it is static delay type
                self.check_pll_feedback_mode_core(checker, lvds_rx)

