'''
Copyright (C) 2017-2020 Efinix Inc. All rights reserved.

No portion of this code may be reused, modified or
distributed in any way without the expressed written
consent of Efinix Inc

Created on Nov 2, 2020

@author: maryam
'''
import os
import sys
from typing import List, Tuple, Dict, TYPE_CHECKING
from enum import auto, Enum, unique

import xml.etree.ElementTree as et

import util.gen_util as pt_util

import device.db_interface as device_dbi
from device.block_definition import TimingArc

import design.db as des_db

import common_device.gpio.writer.timing as gtiming

from tx60_device.gpio.gpio_design_comp import GPIOComplex, GPIOOutputComplex

if TYPE_CHECKING:
    from writer.arc_delay_table import ArcDelayTable

sys.path.append(os.path.join(os.path.dirname(__file__), '..'))

@unique
class GPIORoutingArcLabel(Enum):
    gpio_routing_tco = "GPIO_ROUTING_TCO"
    gpio_routing_tsetup = "GPIO_ROUTING_TSETUP"
    gpio_routing_thold = "GPIO_ROUTING_THOLD"
    gpio_routing_to_core = "GPIO_ROUTING_TO_CORE"
    gpio_routing_from_core = "GPIO_ROUTING_FROM_CORE"
    gpio_routing_clkout = "GPIO_ROUTING_CLKOUT"

class GPIOTimingComplex(gtiming.GPIOTiming):
    """
    Constructor. Inheriting GPIOTiming from Trion
    """

    class VariableType(Enum):
        '''
        The list of supported timing variables.
        '''
        io_standard = auto()
        schmitt_trigger = auto()
        static_delay = auto()
        bank = auto()

    var2str_map = \
        {
            VariableType.io_standard: "io_standard",
            VariableType.schmitt_trigger: "schmitt_trigger",
            VariableType.static_delay: "static_delay",
            VariableType.bank: "gpio_bank"
        }

    str2var_map = \
        {
            "io_standard": VariableType.io_standard,
            "schmitt_trigger": VariableType.schmitt_trigger,
            "static_delay": VariableType.static_delay,
            "gpio_bank": VariableType.bank
        }

    mode2path_map = \
        {
            GPIOComplex.PadModeType.input: gtiming.GPIOTiming.GPIOPathType.input,
            GPIOComplex.PadModeType.output: gtiming.GPIOTiming.GPIOPathType.output,
            GPIOComplex.PadModeType.inout: gtiming.GPIOTiming.GPIOPathType.oe,
            GPIOComplex.PadModeType.clkout: gtiming.GPIOTiming.GPIOPathType.clkout
        }

    def __init__(self, bname, device, design, report, sdc,
                 max_model, min_model, ins2iostd_map):
        '''
        Constructor
        '''

        super().__init__(bname, device, design, report, sdc,
                         max_model, min_model, ins2iostd_map)

        # Due to PT-1842, we need to check for gpio that are part of a bank with
        # count exceeding max toggle rate. if there is, then the gpio in out/oe path
        # needs additional 400ps (pre-scale). This list stores the device instance (resource)
        # name.
        self.dev_ins_names_with_extra_delay = self._determine_inst_with_extra_delay()

        assert self._max_model is not None
        assert self._min_model is not None

        # Populate HVIO Routing delay if the device supports it
        self.routing_label2bank_tbl_max_map, self.routing_label2bank_tbl_min_map = \
            self.populate_routing_delay_per_bank()
        
        # Keep the list of instance (in resource map) map
        # to bank name for use later with GPIO
        self.ins2bankname_map: Dict[str, str] = self.get_ins_bank_names()
       
    def get_ins_bank_names(self):

        device_db = device_dbi.DeviceDBService(self._device)
        ins2bank = device_db.get_instance_to_raw_io_bank_names()

        return ins2bank
           
    def _determine_inst_with_extra_delay(self) -> List[str]:
        '''
        Find the list of GPIO instance that requires we add the additional delay to it.

        :return: a list of device instance/resource names that resides in bank with
                number of instance that exceeds the max toggle limit
        '''
        dbi = device_dbi.DeviceDBService(self._device)
        gpiodev_service = dbi.get_block_service(
            device_dbi.DeviceDBService.BlockType.GPIO)

        # Stores the device instance names that needs additional delay
        ins_with_additional_delay = []
        assert gpiodev_service is not None
        gpio_hvio = gpiodev_service.get_all_instances(self._design)
        for gpio in gpio_hvio:
            if gpio.is_output_path_used():
                ins_with_additional_delay.append(gpio.gpio_def)
        return ins_with_additional_delay

    def _get_gpio_data(self, ins_to_blk_map):
        '''
        '''
        # This will return the GPIOComplexRegistry
        gpio_reg = self._design.get_block_reg(
            des_db.PeriDesign.BlockType.gpio)

        # Map of device instance name to  gpio_obj stored
        # according to their mode type
        input_gpio = {}
        output_gpio = {}
        inout_gpio = {}
        clkout_gpio = {}

        # Map of clk name to gpio obj names associated with the clock
        clk2ins_map = {}
        # Map of device instance name to the instance object
        devinsname2obj_map = {}

        db_global = self._device.get_global_clocks()
        ins2func_map = db_global.get_instance_to_function_map()

        # Sort the gpio_obj name to the instance name
        for ins_name in sorted(ins_to_blk_map.keys(),
                               key=pt_util.natural_sort_key_for_list):

            # Skip if not a GPIO
            blk_name = ins_to_blk_map[ins_name]
            if blk_name not in device_dbi.DeviceDBService.block_str2type_map or\
                device_dbi.DeviceDBService.block_str2type_map[blk_name] != \
                    device_dbi.DeviceDBService.BlockType.GPIO:
                continue

            # Go through all instances and find out those
            # that are registered and get their clocks
            gpio_obj = gpio_reg.get_gpio_by_device_name(ins_name)

            # Skip instances that are not configured in design or
            # not a gpio instance
            if gpio_obj is None:
                continue

            if self._device is not None:
                ins_obj = self._device.find_instance(ins_name)
                devinsname2obj_map[ins_name] = ins_obj
                # self.logger.debug("Saving instance {} to devinsname2obj".format(ins_obj.get_name()))

            if gpio_obj.mode == GPIOComplex.PadModeType.input and \
                    gpio_obj.input is not None:
                input_gpio[gpio_obj.name] = gpio_obj

                # Get the clock name
                if gpio_obj.input.clock_name != "" and\
                        gpio_obj.input.is_register:
                    self._add_clk_name(
                        clk2ins_map, gpio_obj.input.clock_name, gpio_obj)
                elif gpio_obj.input.is_alternate_connection():
                    if gpio_obj.gpio_def in ins2func_map:

                        func_set = ins2func_map[gpio_obj.gpio_def]
                        conn_str = gpio_obj.input.conn2str_map[gpio_obj.input.conn_type]

                        for func_name in func_set:
                            if func_name == "GCLK" or func_name == "RCLK":
                                if func_name.lower() == conn_str:
                                    self._add_clk_name(
                                        clk2ins_map, gpio_obj.input.name, gpio_obj)

            elif gpio_obj.mode == GPIOComplex.PadModeType.output and \
                    gpio_obj.output is not None:
                output_gpio[gpio_obj.name] = gpio_obj

                if gpio_obj.output.clock_name != "" and \
                    (gpio_obj.output.register_option == GPIOOutputComplex.RegOptType.register or
                     gpio_obj.output.register_option == GPIOOutputComplex.RegOptType.inv_register):
                    self._add_clk_name(
                        clk2ins_map, gpio_obj.output.clock_name, gpio_obj)

            elif gpio_obj.mode == GPIOComplex.PadModeType.clkout and \
                    gpio_obj.output is not None:
                clkout_gpio[gpio_obj.name] = gpio_obj

                if gpio_obj.output.clock_name != "":
                    self._add_clk_name(
                        clk2ins_map, gpio_obj.output.clock_name, gpio_obj)

            elif gpio_obj.mode == GPIOComplex.PadModeType.inout:
                inout_gpio[gpio_obj.name] = gpio_obj

                if gpio_obj.input is not None and\
                        gpio_obj.input.clock_name != "" and\
                        gpio_obj.input.is_register:
                    self._add_clk_name(
                        clk2ins_map, gpio_obj.input.clock_name, gpio_obj)

                if gpio_obj.output is not None and\
                        gpio_obj.output.clock_name != "" and\
                        (gpio_obj.output.register_option == GPIOOutputComplex.RegOptType.register or
                         gpio_obj.output.register_option == GPIOOutputComplex.RegOptType.inv_register):
                    self._add_clk_name(
                        clk2ins_map, gpio_obj.output.clock_name, gpio_obj)

                if gpio_obj.output_enable is not None and\
                        gpio_obj.output_enable.clock_name != "" and\
                        gpio_obj.output_enable.is_register:
                    self._add_clk_name(
                        clk2ins_map, gpio_obj.output_enable.clock_name, gpio_obj)

        return clk2ins_map, clkout_gpio, input_gpio, \
            output_gpio, inout_gpio, devinsname2obj_map

    def _write_clock_network_delay(self, rptfile, clk2ins_map,
                                   clk2delay_map, clkout_map, input_map,
                                   output_map, inout_map):
        # Nothing to write out for Titanium
        pass

    # def _create_clkout_delay_report(self, gpio_obj, devinsname2obj_map,
    #                                 max_clk, min_clk, parameter_name):

    #     row_list = super()._create_clkout_delay_report(gpio_obj, devinsname2obj_map,
    #                                                    max_clk, min_clk, parameter_name)

    #     # Format:
    #     # <clkout/block instance name> -clock <clkout pin name in PT> -reference_pin [get_ports {<clkout pad name in core>}]

    #     # Example string:
    #     # rx_fclk_output -clock rx_fclk -reference_pin [get_ports {rx_fclk~CLKOUT~1~15}]
    #     _, clkout_str = self.get_outclk_clkout_info(
    #         gpio_obj, devinsname2obj_map)

    #     if clkout_str != "":
    #         self.clkout_str_names.add(clkout_str)

    #     return row_list

    # def _write_gpio_input_register(self, devinsname2obj_map, sdcfile, global_reg_table,
    #                                gpio_obj, max_clk, min_clk,
    #                                max_delay_add, min_delay_add, ref_pin_name):

    #     # Save the clkout pins info
    #     int_ref_pin_name, clkout_str = self.get_inclk_clkout_info(
    #         gpio_obj, devinsname2obj_map)

    #     if clkout_str != "":
    #         self.clkout_str_names.add(clkout_str)

    #     super()._write_gpio_input_register(devinsname2obj_map, sdcfile, global_reg_table,
    #                                        gpio_obj, max_clk, min_clk,
    #                                        max_delay_add, min_delay_add, int_ref_pin_name)

    # def _write_gpio_output_register(self, devinsname2obj_map, sdcfile, global_reg_table,
    #                                 gpio_obj, max_clk, min_clk,
    #                                 max_delay_add, min_delay_add, max_delay,
    #                                 min_delay, out_tied, ref_pin_name):

    #     # Save the clkout pins info
    #     int_ref_pin_name, clkout_str = self.get_outclk_clkout_info(
    #         gpio_obj, devinsname2obj_map)

    #     if clkout_str != "":
    #         self.clkout_str_names.add(clkout_str)

    #     super()._write_gpio_output_register(devinsname2obj_map, sdcfile, global_reg_table,
    #                                         gpio_obj, max_clk, min_clk,
    #                                         max_delay_add, min_delay_add, max_delay,
    #                                         min_delay, out_tied, int_ref_pin_name)

    # def _write_gpio_inout_input_register(self, devinsname2obj_map, sdcfile, gpio_obj, max_clk,
    #                                      min_clk, max_delay_add, min_delay_add,
    #                                      in_max_map, in_min_map, ref_pin_name):

    #     # Save the clkout pins info
    #     int_ref_pin_name, clkout_str = self.get_inclk_clkout_info(
    #         gpio_obj, devinsname2obj_map)

    #     if clkout_str != "":
    #         self.clkout_str_names.add(clkout_str)

    #     in_clk_name, in_tsu, in_thold, in_min_tsu, in_max_thold = super()._write_gpio_inout_input_register(
    #         devinsname2obj_map, sdcfile, gpio_obj, max_clk,
    #         min_clk, max_delay_add, min_delay_add,
    #         in_max_map, in_min_map, int_ref_pin_name)

    #     return in_clk_name, in_tsu, in_thold, in_min_tsu, in_max_thold

    # def _write_gpio_inout_output_register(self, devinsname2obj_map, sdcfile, gpio_obj, max_clk,
    #                                       min_clk, max_delay_add, min_delay_add,
    #                                       out_max_map, out_min_map, out_tied, ref_pin_name):

    #     # Save the clkout pins info
    #     int_ref_pin_name, clkout_str = self.get_outclk_clkout_info(
    #         gpio_obj, devinsname2obj_map)

    #     if clkout_str != "":
    #         self.clkout_str_names.add(clkout_str)

    #     out_clk_name, out_tco_max, out_tco_min = super()._write_gpio_inout_output_register(
    #         devinsname2obj_map, sdcfile, gpio_obj, max_clk,
    #         min_clk, max_delay_add, min_delay_add,
    #         out_max_map, out_min_map, out_tied, int_ref_pin_name)

    #     return out_clk_name, out_tco_max, out_tco_min

    # def _write_gpio_inout_oe_register(self, devinsname2obj_map, sdcfile, gpio_obj, max_clk,
    #                                   min_clk, max_delay_add, min_delay_add,
    #                                   oe_max_map, oe_min_map, ref_pin_name):

    #     # Save the clkout pins info. OE and OUT clock name is the same
    #     int_ref_pin_name, clkout_str = self.get_outclk_oe_clkout_info(
    #         gpio_obj, devinsname2obj_map)

    #     if clkout_str != "":
    #         self.clkout_str_names.add(clkout_str)

    #     out_clk_name, oe_tco_max, oe_tco_min = super()._write_gpio_inout_oe_register(
    #         devinsname2obj_map, sdcfile, gpio_obj, max_clk,
    #         min_clk, max_delay_add, min_delay_add,
    #         oe_max_map, oe_min_map, int_ref_pin_name)

    #     return out_clk_name, oe_tco_max, oe_tco_min

    def _get_delay_value(self, gpio_obj, delay_var_map, arc_table, is_max=True):
        '''
        :param gpio_obj: The user gpio design object
        :param delay_var_map: The table of variables mapped to acceptable
                        values
        :param arc_table: The arc_parameter table with the name
                        representing the variables concatenated (as
                        in the xml file). An ArcDelayTable object

        :return the delay value of that parameter based on the required
                gpio configuration
        '''

        # self.logger.debug("GPIOComplexTiming _get_delay_value for {} mode {}".format(
        #    gpio_obj.name, gpio_obj.mode))

        # Only if the arc parameter has variable associated, then
        # we figure out, else take that one value
        if arc_table.get_pcount() == 0:
            delay_val = arc_table.get_variable_delay("")
            return delay_val

        iostd_name = ""
        schmitt_trigger = "false"
        static_delay = 0
        delay_val = 0

        # Get the io std
        if gpio_obj.gpio_def in self._ins2iostd_map:
            io_std = self._ins2iostd_map[gpio_obj.gpio_def]

            # Get the value associated to the io_standard
            if self.VariableType.io_standard in delay_var_map:
                io_std_list = delay_var_map.get(
                    self.VariableType.io_standard, "")

                # self.logger.debug("Finding delay for gpio {} with io std {} and bank io std {} in {}".format(
                #    gpio_obj.name, gpio_obj.io_standard, io_std, ",".join(io_std_list)))

                # str_list = io_std.split(" ")
                iostd_name = io_std
                if iostd_name not in io_std_list:
                    # Then check if it could be based on the actual gpio instance
                    # io standard and not just pure voltage
                    gpio_io_std = gpio_obj.io_standard

                    if gpio_io_std in io_std_list:
                        iostd_name = gpio_io_std

                    else:
                        raise ValueError(
                            "I/O Standard {} not defined in timing variable list".format(iostd_name))

            else:
                raise ValueError(
                    "I/O Standard not defined in timing variable list")

        # Depending on the gpio mode, we check the setting mentioned
        # in the delay_var_map and the arc_table to find the
        # suitable delay to pick.
        if gpio_obj.mode == gpio_obj.PadModeType.input or \
            gpio_obj.mode == gpio_obj.PadModeType.inout:
            # Get the gpio setting that is relevant:
            # schmitt_trigger
            if gpio_obj.input is not None:
                if gpio_obj.input.is_schmitt_trigger_enable:
                    schmitt_trigger = "true"

        # Get the arc delay parameter name
        delay_param = arc_table.get_parameter_name()

        # Static delay value to attached depends on the delay parameter name
        # because both output and input has their own static delay. Static Delay
        # is an addition to arc connecting to the PAD either sequential or comb arc.
        if delay_param.find("_OUT_") != -1 or delay_param.find("_OE_") != -1 or\
                delay_param.find("_OUTCLK_") != -1:
            if gpio_obj.output is not None and gpio_obj.mode != gpio_obj.PadModeType.clkout:
                # clkout mode does not support static delay. checking it although design
                # would have reset it to 0
                static_delay = gpio_obj.output.delay
        elif delay_param.find("_IN_") != -1 or delay_param.find("_INCLK_"):
            if gpio_obj.input is not None and not gpio_obj.input.is_dyn_delay:
                static_delay = gpio_obj.input.delay

        # Now that we have all the required setting, we
        # get the delay based on it. The order of the
        # variables list will have to match the variable parameter
        # name. For example, the GPIO block has variable listed in following order:
        # B, D, A
        # In the delay parameter, if it uses multiple variable, then the order
        # has to follow the order above:
        # <efxpt:parameter name="B:bval,A:aval" delay=""/>
        # if it was not order, we won't be able to find a matching value
        var_list = arc_table.get_variables()
        concat_name = ""
        read_vcount = 0

        # List that stores the variable name that was used in string
        var_name_used_in_str_list = []

        for index in range(len(var_list)):
            vname = var_list[index]
            vfound = True

            if vname in self.str2var_map:
                vtype = self.str2var_map[vname]

                if vtype == self.VariableType.io_standard:
                    vsetting = vname + ":" + iostd_name
                elif vtype == self.VariableType.schmitt_trigger:
                    vsetting = vname + ":" + schmitt_trigger
                elif vtype == self.VariableType.static_delay:
                    vsetting = "{}:{}".format(vname, static_delay)
                else:
                    vfound = False

                # To count how many variables was read
                if vfound:
                    read_vcount += 1
                    var_name_used_in_str_list.append(vname)

                if concat_name == "":
                    concat_name = vsetting
                else:
                    concat_name = concat_name + "," + vsetting

            else:
                raise ValueError(
                    "Unexpected timing variable name {}".format(vname))

        # Check for any variable option that will be additional delay adders
        additional_delay = 0
        var_option_unique_name = set()
        if arc_table.is_variable_option_exists():

            var_opt_map = arc_table.get_variable_option_map()

            for var_name in var_opt_map:

                if var_name in self.str2var_map:
                    var_type = self.str2var_map[var_name]

                    if var_name in var_name_used_in_str_list:
                        # This is if two same variable was used in the variable string
                        # AND also the variable option (redundancy not allowed). Only
                        # overlap_var can exists in both
                        raise ValueError(
                            "Timing variable {} was used multiple times for parameter {}".format(
                                var_name, delay_param))

                    var_opt_list = var_opt_map[var_name]
                    for var_opt in var_opt_list:

                        # TODO: Update this if more variable uses option approach
                        if var_type == self.VariableType.static_delay:
                            # Depending on the parameter name, we need to get the delay
                            # based on the value coming from the right GPIO type (in/out).
                            # oe will use the static delay setting from out.
                            var_value = 0

                            if var_opt.get_overlap_variable() != "":
                                # Check if this match. If don't, we skip
                                if not self._is_variable_option_overlap_var_match(var_opt, concat_name):
                                    continue

                            if delay_param.find("_OUT_") != -1 or delay_param.find("_OE_") != -1:
                                if gpio_obj.output is not None:
                                    var_value = gpio_obj.output.delay
                            elif delay_param.find("_IN_") != -1:
                                if gpio_obj.input is not None and not gpio_obj.input.is_dyn_delay:
                                    var_value = gpio_obj.input.delay

                            # Add the value to the additional delay
                            additional_delay += var_opt.get_var_value_delay(
                                str(var_value))

                        var_option_unique_name.add(var_name)

                else:
                    raise ValueError(
                        "Unexpected timing variable option name {}".format(var_name))

        # Add up the variable count with the var option unique variable names
        read_vcount += len(var_option_unique_name)

        # Verify that the variable count matches
        if read_vcount != arc_table.get_pcount():
            raise ValueError(
                "Mismatch variable count, {} and the pcount parameter, {}".format(
                    read_vcount, arc_table.get_pcount()))

        # Get the delay from table based on the concat variable name
        tmp_delay = arc_table.get_variable_delay(concat_name)
        if tmp_delay is not None:
            delay_val = tmp_delay
        else:
            raise ValueError(
                "Unable to get the delay for variable {}".format(concat_name))

        # Add the additional delay to the delay value
        delay_val += additional_delay

        return delay_val

    def _is_variable_option_overlap_var_match(self, var_opt, concat_var_name):
        if var_opt is not None:
            overlap_var_str = var_opt.get_overlap_variable()

            # Cannot find a subset of the concatenate variables with the overlap
            # string. For example, if parameter name is "io_standard:ABC,schmitt_trigger=true"
            # and overlap string is "schmitt_trigger=true", then it is considered a match
            if concat_var_name.find(overlap_var_str) != -1:
                return True

        return False

    def build_delay_param_arc_table(self, gpio_block_max, gpio_block_min):
        label2arctbl_max_map = {}
        label2arctbl_min_map = {}

        for arc_label in GPIORoutingArcLabel:
            # Get the arc table for that label
            label_name = arc_label.value

            arc_table_max = self._load_arc_parameter(
                gpio_block_max, label_name, is_max=True, is_optional=True)
            
            if arc_table_max is not None:
                label2arctbl_max_map[label_name] = arc_table_max

            arc_table_min = self._load_arc_parameter(
                gpio_block_min, label_name, is_max=False, is_optional=True)

            if arc_table_min is not None:
                label2arctbl_min_map[label_name] = arc_table_min

        return label2arctbl_max_map, label2arctbl_min_map

    def get_instance_bank_name_for_routing(self, dev_ins_name,
                                           lbl_name,
                                           label2bank_tbl_map: Dict[str, Dict[str, float]]):
        '''
        Find the bank name associated to the device instance.
        Then, figure out the correct name that would exists in the
        timing table since it may not be exactly the same bank name.
        For example, in Ti375 where this is supported, the
        instance bank name might be BL0 but the timing delay is generic
        saved to bank "BL".
        '''
        bank_name = ""

        if self.ins2bankname_map and label2bank_tbl_map and\
            dev_ins_name in self.ins2bankname_map:
            
            bank_name_raw = self.ins2bankname_map[dev_ins_name]

            if lbl_name in label2bank_tbl_map:
                bank_map = label2bank_tbl_map[lbl_name]

                if bank_name_raw not in bank_map:
                    for bname in bank_map:                        
                        if bank_name_raw.find(bname) != -1:
                            bank_name = bname
                            break
                        
                else:
                    bank_name = bank_name_raw

            #print(f'get_ins_bank_name_routing: {dev_ins_name} - {bank_name_raw} - {lbl_name} - {bank_name}')
            
        return bank_name

    def get_routing_delay_value(self, lbl_name: str, bank_name: str,
                                label2bank_tbl_map: Dict[str, Dict[str, float]]):
        delay_val = 0

        if label2bank_tbl_map and lbl_name in label2bank_tbl_map:
            bank_map = label2bank_tbl_map[lbl_name]
            if bank_name in bank_map:
                delay_val = bank_map[bank_name]

        return delay_val
    
    
    def identify_routing_delay_on_ins(self, dev_ins: str, arc_type: TimingArc.TimingArcType,
                                      is_to_core: bool, tscale: float,
                                      label2bank_tbl_map: Dict[str, Dict[str, float]],
                                      is_clkout: bool = False):
        '''
        Get the additional routing delay associated to an arc based on the
        instance device resource name.

        :param dev_ins: The device instance name
        :param arc_type: The TimingArc type that we're analyzing where we want to
                determine the additional routing delay associated to it
        :param label2bank_tbl_map: The routing delay table for the GPIO where it contains
                the map of the arc label mapped to a dictionary of param to delay value
                where param is the bank name for Ti375
        :param is_to_core: To indicate the direction of the connection. True it's
                an output port and input to core. Otherwise, False.                
        :return the routing delay associated to the arc
        '''

        routing_delay = 0

        if label2bank_tbl_map:
            label_name = ""
           
            match arc_type:
                case TimingArc.TimingArcType.setup:
                    label_name = GPIORoutingArcLabel.gpio_routing_tsetup.value
                case TimingArc.TimingArcType.hold:
                    label_name = GPIORoutingArcLabel.gpio_routing_thold.value
                case TimingArc.TimingArcType.clk_to_q:
                    label_name = GPIORoutingArcLabel.gpio_routing_tco.value
                case TimingArc.TimingArcType.delay:
                    # This depends on the port direction:
                    # input port / clkout.output interface - from_core
                    # output port / input interface - to_core
                    if is_to_core:
                        label_name = GPIORoutingArcLabel.gpio_routing_to_core.value
                    elif is_clkout:
                        label_name = GPIORoutingArcLabel.gpio_routing_clkout.value
                    else:
                        label_name = GPIORoutingArcLabel.gpio_routing_from_core.value
                case _:
                    raise NotImplementedError(f'Unexpected timing arc type: {arc_type}')

            if label_name != "":
                # Determine the label to use based on the arc type
                dly_bank_name = self.get_instance_bank_name_for_routing(dev_ins, label_name, label2bank_tbl_map)

                if dly_bank_name != "":
                    routing_delay = self.get_routing_delay_value(
                        label_name, dly_bank_name, label2bank_tbl_map)
                    
                    routing_delay = routing_delay * tscale / 1000

                    #print(f"identify_routing_delay_on_ins {dev_ins}: {is_to_core}: {label_name}: {dly_bank_name} - {routing_delay}")

        return routing_delay


    def populate_routing_delay_per_bank(self):
                    #label2arctbl_max_map: Dict[GPIORoutingArcLabel, ArcDelayTable],
                    #label2arctbl_min_map: Dict[GPIORoutingArcLabel, ArcDelayTable]):

        gpio_block_max = self.get_xml_block(self._name, True)
        assert gpio_block_max is not None
        gpio_block_min = self.get_xml_block(self._name, False)
        assert gpio_block_min is not None

        blk_label2arc_max_map, blk_label2arc_min_map = \
            self.build_delay_param_arc_table(gpio_block_max, gpio_block_min)

        # Get the delay and populate per bank
        label2bank_tbl_max_map = {}
        label2bank_tbl_min_map = {}
                            
        if blk_label2arc_max_map and blk_label2arc_min_map:

            label_names = blk_label2arc_max_map.keys()
            assert len(label_names) == len(blk_label2arc_min_map)

            for lbl_name in label_names:
                max_lbl_delay_map = {}
                min_lbl_delay_map = {}
            
                max_arc_tbl = blk_label2arc_max_map[lbl_name]
                min_arc_tbl = blk_label2arc_min_map[lbl_name]

                assert max_arc_tbl.get_pcount() == 1 and\
                        min_arc_tbl.get_pcount() == 1
                
                #var_str_max = max_arc_tbl.get_variable_names()
                #var_str_min = max_arc_tbl.get_variable_names()
                #assert sorted(var_str_max) == sorted(var_str_min)
                var_name2val_max: Dict[str, List[str]] = max_arc_tbl.get_variable_values()
                var_name2val_min: Dict[str, List[str]] = min_arc_tbl.get_variable_values()

                assert len(var_name2val_max.keys()) == 1 and \
                    len(var_name2val_min.keys()) == 1
                
                assert sorted(var_name2val_max.values()) == \
                        sorted(var_name2val_min.values())
                
                if "gpio_bank" in var_name2val_max:
                    assert "gpio_bank" in var_name2val_min

                    bank_values = var_name2val_max["gpio_bank"]
                    for bank_name in bank_values:
                        var_lbl = f"gpio_bank:{bank_name}"
                        
                        max_delay = max_arc_tbl.get_variable_delay(var_lbl)
                        min_delay = min_arc_tbl.get_variable_delay(var_lbl)

                        max_lbl_delay_map[bank_name] = max_delay
                        min_lbl_delay_map[bank_name] = min_delay

                    label2bank_tbl_max_map[lbl_name] = max_lbl_delay_map
                    label2bank_tbl_min_map[lbl_name] = min_lbl_delay_map
            
        return label2bank_tbl_max_map, label2bank_tbl_min_map

            
    def get_xml_block(self, blk_name: str, is_max: bool):
        timing_block = None
        
        if not is_max:
            tree = et.ElementTree(file=self._min_model.get_filename())
        else:
            tree = et.ElementTree(file=self._max_model.get_filename())

        block_tag = ".//" + self.xml_ns + "block"
        for elem in tree.iterfind(block_tag):

            block_attrib = elem.attrib
            if block_attrib["type"] == blk_name:
                timing_block = elem
                break        

        return timing_block
    
    def get_instance_out_oe_specific_delay(self, gpio_obj, ins_obj):
        """
        Unlike Trion, Titanium don't really have additional routing delay saved.
        However, this is for the use of PT-1882 where only the OUTPUT and OE arcs
        that interface to the board need to have extra delay added when using non 1.8V
        The delay is indenpent of time scale

        :param ins_obj:
        :param tscale:
        :return:
        """
        delay = 0

        if ins_obj is not None and ins_obj.get_name() in self.dev_ins_names_with_extra_delay:
            # Max and Min value is the same and not tmodel dependent, so read just max
            gpio_block = self.get_xml_block(self._name, True)
            assert gpio_block is not None
            arc_table = self._load_arc_parameter(
                gpio_block, "GPIO_OUT_SWITCHING_DELAY_BASELINE", is_max=True)
            value = arc_table.get_variable_delay("") / 1000
            delay += value

            if gpio_obj.output is not None and \
                    gpio_obj.output.register_option == gpio_obj.output.RegOptType.none:
                arc_table = self._load_arc_parameter(
                    gpio_block, "GPIO_OUT_SWITCHING_DELAY_ADDITIONAL", is_max=True)
                value = arc_table.get_variable_delay("") / 1000
                delay += value

        return delay

    def get_output_additional_delay(self, gpio_obj, devinsname2obj_map) -> Tuple[float, float]:
        max_delay_add = 0
        min_delay_add = 0

        # Check if this instance has additional delay
        if gpio_obj.gpio_def in devinsname2obj_map:
            ins_obj = devinsname2obj_map[gpio_obj.gpio_def]

            max_delay_add = self.get_instance_out_oe_specific_delay(
                gpio_obj, ins_obj)
            min_delay_add = 0

        return max_delay_add, min_delay_add

    def get_clkout_additional_delay(self, gpio_obj, devinsname2obj_map) -> Tuple[float, float]:
        return self.get_output_additional_delay(gpio_obj, devinsname2obj_map)
