"""
Copyright (C) 2017-2020 Efinix Inc. All rights reserved.

No portion of this code may be reused, modified or
distributed in any way without the expressed written
consent of Efinix Inc.

Created on Nov 2, 2020

@author: maryam
"""

from typing import List
import util.gen_util
import device.db_interface as devdb_int

import common_device.rule as base_rule
import common_device.clock_mux.clkmux_rule as clkmux_rule
from tx60_device.clock_mux.clkmux_design_adv import DynamicMuxInput, ClockMuxAdvance, ClockMuxRegistryAdvance


@util.gen_util.freeze_it
class ClockMuxCheckerAdv(clkmux_rule.ClockMuxChecker):

    def __init__(self, design):
        """
        Constructor
        """
        # We save the info in preparation for the check later
        self.pll_clk2count_map = {}
        self.clkmuxdev_service = None

        super().__init__(design)

        # Add the service\
        if self.design.device_db is not None:
            dbi = devdb_int.DeviceDBService(self.design.device_db)
            self.clkmuxdev_service = dbi.get_block_service(
                self.design.device_db.get_clkmux_type())

        # Build the info now
        self.setup_pll_clock_names_for_later_check()

        # Refresh rbuf info
        self.refresh_clkmux_rbuf_info()

    def _get_pll_clkout_names_based_on_ins_pins(self, pll_reg, pll_pins_list):
        pll_out_clk = []
        visited_pll_ins_name = []

        for pll_pin in pll_pins_list:
            if pll_pin is not None:
                ins_name = pll_pin.get_connecting_instance_name(self.design.device_db)

                if ins_name != "" and ins_name not in visited_pll_ins_name:
                    # Find the instance in the pll registry
                    pll_design = pll_reg.get_inst_by_device_name(ins_name)
                    visited_pll_ins_name.append(ins_name)

                    if pll_design is not None:
                        # Get all the clkout names assigned to clkout0-clkout3
                        for idx in range(4):
                            out_clk = pll_design.get_output_clock_by_number(idx)
                            if out_clk is not None:
                                pll_out_clk.append(out_clk.name)

        return pll_out_clk

    def _get_output_clocks_assigned_to_pll(self, clkmux_dev_ins):
        '''
        Get the pll output clock names that are configured in the design
        and connects to the current clockmux inputs
        :param clkmux_dev_ins: The device clkmux instance
        :return: a list of PLL clkout names associated to this clockmux
        '''
        pll_out_clk_names = []

        if clkmux_dev_ins is not None:
            # Get the PLL resource assigned to PLL0 and PLL1 pins
            pll_reg = self.design.pll_reg

            pll0_pins_list = clkmux_dev_ins.get_bus_pins("PLL0")
            pll0_out_clk = self._get_pll_clkout_names_based_on_ins_pins(pll_reg, pll0_pins_list)

            pll1_pins_list = clkmux_dev_ins.get_bus_pins("PLL1")
            pll1_out_clk = self._get_pll_clkout_names_based_on_ins_pins(pll_reg, pll1_pins_list)

            # COmbined the pll0 and pll1 output clocks
            pll_out_clk_names = pll0_out_clk + pll1_out_clk

        return pll_out_clk_names

    def _get_clkmux_output_pll_assignment(self, clkmux_obj, pll_dev_ins, pll_out_clk_names):
        '''
        Return a map of the pll output clock names to the number of times it's being
        assigned to the current clock mux output.

        :param clkmux_obj: ClockMuxAdvance
        :param pll_dev_ins: The device clkmux instance
        :param pll_out_clk_names: a list of the current clockmux pll output clock names
        :return: a map of the pll output clock name and how many time it's assigned to the
                current mux output
        '''
        pll_clk2count_map = {}

        # Get the output map of the current clkmux
        output_map = clkmux_obj.get_all_io_setting()

        # Iterate through the clkmux output to see the pin that has been assigned
        for out_idx in output_map:
            io_setting = output_map[out_idx]

            if io_setting is not None:
                assert self.clkmuxdev_service is not None
                user_in_name, _, _ = self.clkmuxdev_service.get_conn_input_user_name(
                    self.design, pll_dev_ins, io_setting.output_name,
                    io_setting.output_no, io_setting, clkmux_obj.gen_pin)

                if user_in_name != "":
                    if user_in_name in pll_out_clk_names:
                        if user_in_name not in pll_clk2count_map:
                            pll_clk2count_map[user_in_name] = 1
                        else:
                            cur_count = pll_clk2count_map[user_in_name]
                            pll_clk2count_map[user_in_name] = cur_count + 1

        return pll_clk2count_map

    def refresh_clkmux_rbuf_info(self):
        # We refresh the rbuf info to consider cases where
        # the global flag can go out of sync due to change of
        # instance info
        if self.reg is not None and isinstance(self.reg, ClockMuxRegistryAdvance):
            self.reg.update_regional_buffer_info(self.design)

    def setup_pll_clock_names_for_later_check(self):
        # Reset the data each time it's constructed
        self.pll_clk2count_map = {}
        assert self.reg is not None

        all_clkmux = self.reg.get_all_clock_mux()

        for clkmux in all_clkmux:
            # Find the clockmux device instance
            device_db = self.design.device_db
            pll_dev_ins = None
            chk_pll_clk2count_map = {}
            if device_db is not None and clkmux.block_def != "":
                pll_dev_ins = device_db.find_instance(clkmux.block_def)

                chk_pll_out_clk_names = self._get_output_clocks_assigned_to_pll(pll_dev_ins)
                chk_pll_clk2count_map = self._get_clkmux_output_pll_assignment(clkmux, pll_dev_ins, chk_pll_out_clk_names)

            self.pll_clk2count_map[clkmux.block_def] = chk_pll_clk2count_map

    def check_duplicate_pll_clkout_assignment(self, clkmux_obj):
        '''
        We do the check at the top since it involves iterating through all clockmuxes
        :param cur_clkmux: The current CLKMUX design instance being analyzed
        :return: a set of the current clkmux pll clockout name that is duplicated
        '''
        duplicated_clkout_assigned = set()

        if clkmux_obj is not None and clkmux_obj.block_def != "" and self.reg is not None:

            if clkmux_obj.block_def in self.pll_clk2count_map:
                cur_pll_clk2count_map = self.pll_clk2count_map[clkmux_obj.block_def]

                # Now, look at other clkmux to see if it also has the same pll output
                # clock assigned to the clkmux output
                all_clkmux = self.reg.get_all_clock_mux()

                for clkmux in all_clkmux:
                    # Skip the current clockmux
                    if clkmux is not None and clkmux != clkmux_obj and clkmux.block_def != "":
                        chk_pll_clk2count_map = {}
                        if clkmux.block_def in self.pll_clk2count_map:
                            chk_pll_clk2count_map = self.pll_clk2count_map[clkmux.block_def]

                        for name, count in cur_pll_clk2count_map.items():
                            if name in chk_pll_clk2count_map:
                                chk_count = chk_pll_clk2count_map[name]

                                if count > 0 and chk_count > 0:
                                    duplicated_clkout_assigned.add(name)

        return duplicated_clkout_assigned

    def _build_rules(self):
        """
        Build control bank rule database
        """
        self._add_rule(clkmux_rule.ClkMuxRuleClocksRouted())
        self._add_rule(ClkMuxRuleCoreClockPin())
        self._add_rule(ClkMuxRuleDynamicClockPin())
        self._add_rule(ClkMuxRuleDynamicSelectClockPin())
        self._add_rule(ClkMuxRuleGlobalRegionalPin())
        self._add_rule(ClkMuxRuleCoreClockStaticMux())
        self._add_rule(ClkMuxRuleGlobalRegionalResource())
        self._add_rule(ClkMuxRuleRegionalConnType())
        self._add_rule(ClkMuxRulePLLMultConnection())
        self._add_rule(ClkMuxRuleTypeConfig())
        self._add_rule(ClkMuxRuleDynamicMuxPLLInput())
        self._add_rule(ClkMuxRuleDiffPLLClockSameMux())

class ClkMuxRuleCoreClockPin(base_rule.Rule):
    """
    Check that all core clock pin is used only when dynamic mux is enabled
    """

    def __init__(self):
        super().__init__()
        self.name = "clkmux_rule_core_clock_pin"

    def check(self, checker, design_block: ClockMuxAdvance):
        util.gen_util.mark_unused(checker)

        clkmux_obj = design_block

        if clkmux_obj is not None:
            # Check if any of the route pin name is non-empty
            route0_pin = clkmux_obj.get_route_pin_name(0)
            route1_pin = clkmux_obj.get_route_pin_name(1)
            route2_pin = clkmux_obj.get_route_pin_name(2)
            route3_pin = clkmux_obj.get_route_pin_name(3)

            if route0_pin != "" or route1_pin != "" or\
                route2_pin != "" or route3_pin != "":
                if not clkmux_obj.is_mux_bot0_dyn and not clkmux_obj.is_mux_bot7_dyn:
                    self.error("Core clock pin can only be used with dynamic mux enabled")

class ClkMuxRuleDynamicClockPin(base_rule.Rule):
    """
    Check that dynamic clock pin name is provided when dynamic is enabled
    """

    def __init__(self):
        super().__init__()
        self.name = "clkmux_rule_dynamic_clock_pin"

    def check(self, checker, design_block: ClockMuxAdvance):
        util.gen_util.mark_unused(checker)

        clkmux_obj = design_block

        if clkmux_obj is not None and clkmux_obj.gen_pin is not None:

            is_mux0_invalid = False
            is_mux7_invalid = False
            gen_pin = clkmux_obj.gen_pin

            if clkmux_obj.is_mux_bot0_dyn:
                mux0_pin = gen_pin.get_pin_by_type_name(
                    "{}0".format(clkmux_obj.dyn_mux_out_pin_name))

                if mux0_pin is not None and self.is_pin_name_pattern_invalid(mux0_pin.name):
                    is_mux0_invalid = True

            if clkmux_obj.is_mux_bot7_dyn:
                mux7_pin = gen_pin.get_pin_by_type_name(
                    "{}7".format(clkmux_obj.dyn_mux_out_pin_name))

                if mux7_pin is not None and self.is_pin_name_pattern_invalid(mux7_pin.name):
                    is_mux7_invalid = True

            if is_mux0_invalid or is_mux7_invalid:
                if is_mux0_invalid and is_mux7_invalid:
                    self.msg = "Dynamic clock pin names for both dynamic muxes are empty or invalid"
                elif is_mux0_invalid:
                    self.msg = "Dynamic clock pin names for dynamic mux 0 is empty or invalid"
                else:
                    self.msg = "Dynamic clock pin names for dynamic mux 7 is empty or invalid"

                self.error()


class ClkMuxRuleDynamicSelectClockPin(base_rule.Rule):
    """
    Check that dynamic clock select pin is provided when dynamic is enabled
    """

    def __init__(self):
        super().__init__()
        self.name = "clkmux_rule_dynamic_clock_select_pin"

    def check(self, checker, design_block: ClockMuxAdvance):
        util.gen_util.mark_unused(checker)

        clkmux_obj = design_block

        if clkmux_obj is not None and clkmux_obj.gen_pin is not None:

            is_mux0_invalid = False
            is_mux7_invalid = False
            gen_pin = clkmux_obj.gen_pin

            if clkmux_obj.is_mux_bot0_dyn:
                mux0_pin = gen_pin.get_pin_by_type_name(
                    "{}0".format(clkmux_obj.dyn_mux_sel_pin_name))

                if mux0_pin is not None and self.is_pin_name_pattern_invalid(mux0_pin.name):
                    is_mux0_invalid = True

            if clkmux_obj.is_mux_bot7_dyn:
                mux7_pin = gen_pin.get_pin_by_type_name(
                    "{}7".format(clkmux_obj.dyn_mux_sel_pin_name))

                if mux7_pin is not None and self.is_pin_name_pattern_invalid(mux7_pin.name):
                    is_mux7_invalid = True

            if is_mux0_invalid or is_mux7_invalid:
                if is_mux0_invalid and is_mux7_invalid:
                    self.msg = "Dynamic clock select pin names for both dynamic muxes are empty or invalid"
                elif is_mux0_invalid:
                    self.msg = "Dynamic clock select pin names for dynamic mux 0 is empty or invalid"
                else:
                    self.msg = "Dynamic clock select pin names for dynamic mux 7 is empty or invalid"

                self.error()

class ClkMuxRuleGlobalRegionalPin(base_rule.Rule):
    """
    Check that global and dynamic clock pin name are unique if both enabled.
    """

    def __init__(self):
        super().__init__()
        self.name = "clkmux_rule_global_regional_pin"

    def check(self, checker, design_block: ClockMuxAdvance):
        #util.gen_util.mark_unused(checker)

        clkmux_obj = design_block

        if clkmux_obj is not None:
            # Iterate through the regional buffer list to find if there
            # were any instances that has regional and global but didn't provide
            # any extra pin name
            regional_map = clkmux_obj.get_regional_buffer_map()

            for idx in sorted(regional_map):
                rbuf_obj = regional_map[idx]

                if rbuf_obj is not None:
                    # If the global connection was only assigned to dynamic mux,
                    # It means that the extra pin name is not mandatory.
                    if rbuf_obj.is_global:
                        # Find the corresponding instance
                        global_type, global_pin_name = rbuf_obj.get_type_used_as_global(clkmux_obj, checker.design)

                        # If it only connects to dynamic global (with regional), then global pin name
                        # is not required (dont warn any message as well)
                        if global_type == DynamicMuxInput.UseConnType.static_only or \
                                global_type == DynamicMuxInput.UseConnType.dyn_and_static:

                            regional_resource, regional_pin_name = rbuf_obj.get_user_resource_pin_name(
                                clkmux_obj, checker.design, True, checker.design.device_db)

                            if global_pin_name == "":
                                self.error("Missing global pin name for a regional connection that also connects to global buffer")

                            # Check that the global and regional pin name are different
                            elif global_pin_name == regional_pin_name:
                                self.error("Global and regional buffer connections require unique pin name")

class ClkMuxRuleCoreClockStaticMux(base_rule.Rule):
    """
    Check that core clock mux not routed to static muxes
    """

    def __init__(self):
        super().__init__()
        self.name = "clkmux_rule_core_clock_static_mux"

    def check(self, checker: ClockMuxCheckerAdv, design_block: ClockMuxAdvance): # type: ignore
        #util.gen_util.mark_unused(checker)

        clkmux_obj = design_block

        if clkmux_obj is not None and clkmux_obj.block_def != "":
            # Check the result of routing and check if any of the output
            # On the static mux is set to route pin
            assert checker.design.device_db is not None
            ins_obj = checker.design.device_db.find_instance(clkmux_obj.block_def)

            # Check if any route pins were specified
            route_pins_map = clkmux_obj.get_all_route_pin_names()

            is_static_conn_found = False
            invalid_route_pin_names = []
            output_map = clkmux_obj.get_all_io_setting()

            for idx in route_pins_map.keys():
                pin_name = route_pins_map[idx]

                if pin_name != "":
                    # Check if the same pin name is found in any of the static output (1-6)
                    for out_idx in output_map:
                        if out_idx > 0 and out_idx < 7:
                            io_setting = clkmux_obj.get_io_setting(out_idx)
                            if io_setting is not None:
                                assert checker.clkmuxdev_service is not None
                                cur_user_in_name, _, _ = checker.clkmuxdev_service.get_conn_input_user_name(
                                    checker.design, ins_obj, io_setting.output_name,
                                    io_setting.output_no, io_setting, clkmux_obj.gen_pin)

                                # This could be a signal name that is identical to
                                # the core clock name but it might be sourced from
                                # a clock defined in the periphery
                                if cur_user_in_name == pin_name:
                                    # Check where this clock name is coming from
                                    # If it is not from the route core clock, then
                                    # it is okay.
                                    if io_setting.input_name.startswith("ROUTE"):
                                        is_static_conn_found = True
                                        if pin_name not in invalid_route_pin_names:
                                            invalid_route_pin_names.append(pin_name)

            if is_static_conn_found:
                self.error("Core clock pin {} not allowed to route through static mux output".format(
                    ",".join(invalid_route_pin_names)))


class ClkMuxRuleGlobalRegionalResource(base_rule.Rule):
    """
    Check that regional connection with global has valid resource used
    """

    def __init__(self):
        super().__init__()
        self.name = "clkmux_rule_global_regional_resource"

    def _is_pll_allowed_to_connect_to_global_and_regional(self, checker, rbuf_obj):
        util.gen_util.mark_unused(checker)
        util.gen_util.mark_unused(rbuf_obj)

        return False

    def _is_mipi_hard_dphy_allowed_to_connect_to_global_and_regional(self, checker, rbuf_obj, mipi_pins_list):
        # Advance does not need MIPI Hard DPHY
        util.gen_util.mark_unused(checker)
        util.gen_util.mark_unused(rbuf_obj)
        util.gen_util.mark_unused(mipi_pins_list)

        return False

    def _check_invalid_rbuf_global_resource(self, checker, regional_map, gpio_pins_list,
                                            mipi_pins_list):
        invalid_global_buf_resource = []

        for idx in sorted(regional_map):
            rbuf_obj = regional_map[idx]

            if rbuf_obj is not None:
                # If the regional connection is assigned to global, then check
                # that it is not a resource that's invalid
                if rbuf_obj.is_global:
                    # Check that the resource is valid
                    ins_obj, ins_type = rbuf_obj.get_input_design_instance(checker.design)

                    resource_name = rbuf_obj.get_resource()

                    if ins_obj is not None:
                        if ins_type == ClockMuxAdvance.ResourceInstanceType.pll:
                            if not self._is_pll_allowed_to_connect_to_global_and_regional(checker, rbuf_obj):
                                invalid_global_buf_resource.append(resource_name)

                        elif ins_type == ClockMuxAdvance.ResourceInstanceType.no_config:

                            invalid_global_buf_resource.append(resource_name)

                        # For other type, then we check that if they are programmed
                        # as the type, there's a corresponding connection to the global pins
                        # (i.e GPIO and MIPI_CLK pins)

                        # Find resource on the GPIO and MIPI CLK Pins depending on the type
                        elif ins_type == ClockMuxAdvance.ResourceInstanceType.mipi_dphy:

                            found_valid = False
                            for mpin in mipi_pins_list:
                                pin_resource = mpin.get_connecting_instance_name(checker.design.device_db)

                                if pin_resource != "" and pin_resource == resource_name:
                                    found_valid = True
                                    break

                            if not found_valid:
                                invalid_global_buf_resource.append(resource_name)

                        elif ins_type == ClockMuxAdvance.ResourceInstanceType.mipi_hard_dphy:
                            if not self._is_mipi_hard_dphy_allowed_to_connect_to_global_and_regional(checker, rbuf_obj, mipi_pins_list):
                                invalid_global_buf_resource.append(resource_name)

                        else:
                            # This would be for LVDS and GPIO configuration
                            # The resource name of the rbuf is not really the design
                            # block_def since for GPIO the gpio_def is a subset of the HSIO
                            # (i.e. GPIOX_P_YY) instead of the resource name (GPIOX_PN_YY).
                            found_valid = False
                            for gpin in gpio_pins_list:
                                pin_resource = gpin.get_connecting_instance_name(checker.design.device_db)
                                if pin_resource != "" and pin_resource == resource_name:
                                    found_valid = True
                                    break

                            if not found_valid:
                                invalid_global_buf_resource.append(resource_name)

        return invalid_global_buf_resource

    def check(self, checker: ClockMuxCheckerAdv, design_block: ClockMuxAdvance): # type: ignore

        clkmux_obj = design_block

        if clkmux_obj is not None and clkmux_obj.block_def != "":
            # Iterate through the regional buffer list to find if there
            # were any instances that has regional and global but didn't provide
            # any extra pin name
            assert checker.design.device_db is not None
            regional_map = clkmux_obj.get_regional_buffer_map()

            clkmux_ins = checker.design.device_db.find_instance(clkmux_obj.block_def)
            if clkmux_ins is None:
                return

            # Get the GPIO pins:
            gpio_pins_list = clkmux_ins.get_bus_pins("GPIO")

            # Get the MIPI DPHY pins
            mipi_pins_list = self.get_mipi_clk_pins_list(clkmux_obj, clkmux_ins, checker)

            # List of regional buffer resource name that is also assigned to global
            # but is not a valid resource to be assigned to it
            invalid_global_buf_resource = self._check_invalid_rbuf_global_resource(
                checker, regional_map, gpio_pins_list, mipi_pins_list)

            if invalid_global_buf_resource:
                self.error("Regional buffer resource {} does not support global connection".format(
                    ",".join(invalid_global_buf_resource)))

    def get_mipi_clk_pins_list(self, clkmux_des_ins, clkmux_ins, checker: ClockMuxCheckerAdv):
        '''

        :param clkmux_des_ins: The Design Clock Mux object
        :param clkmux_ins: The Device instance Clock Mux object
        :return:
        '''
        mipi_pins_list = []
        for idx in range(clkmux_des_ins.get_mipi_clk_count()):
            pin_name = "MIPI_CLK{}".format(idx)

            # Not all index is connected
            pin_obj = clkmux_ins.get_connection_pin(pin_name)
            if pin_obj is not None:
                mipi_pins_list.append(pin_obj)


        if checker.design.mipi_hard_dphy_reg is not None and self.is_ti375_device(checker.design.device_db):
            for idx in range(clkmux_des_ins.get_reserved_count()):
                pin_name = f"RESERVED{idx}"
                pin_obj = clkmux_ins.get_connection_pin(pin_name)
                if pin_obj is not None:
                    mipi_pins_list.append(pin_obj)

        return mipi_pins_list

    def is_ti375_device(self, device_db):
        device_engineer_name = "opx_466x964_b56_d28" # Ti375 device

        engine_name: str = device_db.get_device_mapped_name()

        if engine_name.startswith(device_engineer_name):
            return True

        return False


class ClkMuxRuleRegionalConnType(base_rule.Rule):
    """
    Check that regional connection has the right connection type
    """

    def __init__(self):
        super().__init__()
        self.name = "clkmux_rule_regional_conn_type"

    def check(self, checker, design_block: ClockMuxAdvance):
        #util.gen_util.mark_unused(checker)

        clkmux_obj = design_block

        if clkmux_obj is not None:
            # Iterate through the regional buffer list to find if there
            # were any instances that has regional and global but didn't provide
            # any extra pin name
            regional_map = clkmux_obj.get_regional_buffer_map()

            # The list of instance name with invalid connection type
            invalid_rbuf_type_user_ins: List[str] = []
            for idx in sorted(regional_map):
                rbuf_obj = regional_map[idx]

                self._check_rbuf_connection_type(checker, rbuf_obj, invalid_rbuf_type_user_ins)

            if invalid_rbuf_type_user_ins:
                self.error("Regional buffer instance {} requires connection type to be set to rclk".format(
                    ",".join(invalid_rbuf_type_user_ins)))

    def _check_rbuf_connection_type(self, checker, rbuf_obj, invalid_rbuf_type_user_ins):

        if rbuf_obj is not None:
            ins_obj, ins_type = rbuf_obj.get_input_design_instance(checker.design, True)

            # Skip checking PLLComplex since it has no connection type
            if ins_obj is not None and ins_type != ClockMuxAdvance.ResourceInstanceType.pll:
                # The instance has been configured. Now check it's connection type
                if not rbuf_obj.is_regional_conn_type(ins_obj, ins_type):
                    # But if it is global and the instance has a clkmux_buf_name, then it's invalid
                    if rbuf_obj.is_global:
                        invalid_rbuf_type_user_ins.append(ins_obj.name)


class ClkMuxRulePLLMultConnection(base_rule.Rule):
    """
    Check that a specific PLL clkout is only routed to one clock mux
    """

    def __init__(self):
        super().__init__()
        self.name = "clkmux_rule_pll_output_clock"

    def check(self, checker: ClockMuxCheckerAdv, design_block: ClockMuxAdvance): # type: ignore

        clkmux_obj = design_block

        if clkmux_obj is not None and checker.reg is not None and\
                checker.design.pll_reg is not None:

            # Check the routing result to make sure that no same pin
            # name is routed in multiple clock mux using the current
            # clock mux as a reference
            duplicated_clk_names_set = checker.check_duplicate_pll_clkout_assignment(clkmux_obj)

            if duplicated_clk_names_set:
                self.error("Found the following PLL output clock routed multiple times: {}".format(
                    ",".join(duplicated_clk_names_set)))

class ClkMuxRuleTypeConfig(base_rule.Rule):
    """
    Check that the resource connected to dynamic mux is configured correctly.
    For example, if it was a GPIO connection, the resource can only be configured
    as LVDS/GPIO.
    """

    def __init__(self):
        super().__init__()
        self.name = "clkmux_rule_type_config"

    def check(self, checker, design_block: ClockMuxAdvance):

        clkmux_obj = design_block

        if clkmux_obj is not None:
            # Check if the mux has any of the dynamic mux enabled
            if clkmux_obj.is_mux_bot0_dyn or clkmux_obj.is_mux_bot7_dyn:

                # Get the list of inputs assigned for each dynamic mux
                if clkmux_obj.is_mux_bot0_dyn:
                    if not self.is_dyn_mux_input_type_instance_valid(checker, clkmux_obj, 0):
                        return

                if clkmux_obj.is_mux_bot7_dyn:
                    if not self.is_dyn_mux_input_type_instance_valid(checker, clkmux_obj, 7):
                        return

            # Check that the regional buffer is being configured correctly
            # based on the pin associated to the resource and comparing that
            # against the design instance that's being configured
            if not self.is_regional_type_instance_valid(checker, clkmux_obj):
                return

    def is_dyn_mux_input_type_instance_valid(self, checker, clkmux_obj, mux_idx):
        '''
        Check if the input to dynamic mux is of valid instance type
        :param checker: The checker object
        :param clkmux_obj: The clockmux object
        :param mux_idx: The dynamic mux index (int)
        :return: False if the input is configured incorrectly
        '''
        is_valid = True

        # Iterate through the user dynamic mux input assignment
        user_map = clkmux_obj.get_user_dyn_mux_inputs(mux_idx)

        for idx in user_map:
            mux_in_obj = user_map[idx]

            # Get the instance
            if mux_in_obj is not None:
                ins_obj, ins_type = mux_in_obj.get_input_design_instance(checker.design)
                resource_name = mux_in_obj.get_resource()

                if ins_obj is not None:
                    # Check the type
                    if ins_type == ClockMuxAdvance.ResourceInstanceType.mipi_dphy:
                        if mux_in_obj.get_connecting_pin() != "CLKOUT":                            
                            is_valid = False
                            self.error("Resource {} configured as MIPI RX Lane not allowed to connect to dynamic mux {}".format(
                                resource_name, mux_idx))
                            break

                    elif ins_type == ClockMuxAdvance.ResourceInstanceType.lvds or \
                            ins_type == ClockMuxAdvance.ResourceInstanceType.gpio:

                        # Check that the pin name is DOUT_EVENP (not CLKOUT)
                        if mux_in_obj.get_connecting_pin() != "DOUT_EVENP" and \
                                mux_in_obj.get_connecting_pin() != "IN[0]":
                            is_valid = False
                            self.error("Resource {} configured as GPIO/LVDS not allowed to connect to dynamic mux {}".format(
                                resource_name, mux_idx))
                            break

                    elif ins_type == ClockMuxAdvance.ResourceInstanceType.pll:
                        out_clk = mux_in_obj.get_pll_clkout_index_from_pin_name(ins_obj, mux_in_obj.get_connecting_pin())
                        if out_clk is None or self.is_pin_name_pattern_invalid(out_clk.name):
                            is_valid = False
                            self.error("Invalid output clock configuration of {} connected to index {} of dynamic mux {}".format(
                                resource_name, idx, mux_idx))
                            break

                elif resource_name != "":
                    # If it hasn't configured but it was assigned to the input, then
                    # we should error it
                    # If user didn't assign the index to any input, it should be set to
                    # default (i.e empty string), in which there should not be an entry in the
                    # the map

                    # If it is route pin but the route pin name is empty or invalid
                    if resource_name.startswith("ROUTE"):
                        gen_pin = clkmux_obj.gen_pin
                        route_pin = gen_pin.get_pin_by_type_name(resource_name)
                        if route_pin is not None and self.is_pin_name_pattern_invalid(route_pin.name):
                            is_valid = False
                            self.error("Assigned core clock pin name at index {} of dynamic mux {} is empty or invalid".format(
                                idx, mux_idx))

                    else:
                        res_name, pin_name = mux_in_obj.get_user_resource_pin_name(
                            clkmux_obj, checker.design, False, checker.design.device_db)

                        # It could be an unbonded resource
                        is_valid = False
                        self.error()
                        if res_name != "Unbonded":
                            if pin_name != "":
                                self.msg = "Resource {} assigned to input {} of dynamic mux {} but no valid configured instance found".format(
                                    resource_name, idx, mux_idx)
                            else:
                                self.msg = "Resource {} assigned to input {} of dynamic mux {} but no valid configured instance pin found".format(
                                    resource_name, idx, mux_idx)

                        else:
                            self.msg = "Unbonded resource assigned to input {} of dynamic mux {}".format(
                                idx, mux_idx)

                        break


                else:
                    # Not configured and no resource set to it
                    checker.logger.debug("INTERNAL ERROR: Input to dynamic mux {} of {} has no resource associated".format(
                        mux_idx, clkmux_obj.name))

                    is_valid = False
                    self.error("Invalid resource on index {} to connect to dynamic mux {}".format(
                        idx, mux_idx))
                    break

        return is_valid

    def is_regional_type_instance_valid(self, checker, clkmux_obj):
        '''
        :param checker:
        :param clkmux_obj:
        :return:
        '''
        is_valid = True

        # Iterate through all regional buffer and check those that has its resource
        # configured are done correctly according to the resource and pin name associated
        # to the regional buffer input index.

        reg_buf_map = clkmux_obj.get_regional_buffer_map()

        # Keep a map of resource to pins
        res_to_pins = {}
        for idx in reg_buf_map:
            reg_buf = reg_buf_map[idx]

            if reg_buf is not None:
                res_name = reg_buf.get_resource()
                pin_name = reg_buf.get_connecting_pin()

                if pin_name != "" and res_name in res_to_pins:
                    cur_pins = res_to_pins[res_name]
                    cur_pins.append(pin_name)
                else:
                    cur_pins = [pin_name]

                res_to_pins[res_name] = cur_pins

        for idx in reg_buf_map:
            reg_buf = reg_buf_map[idx]

            if reg_buf is not None:
                ins_obj, ins_type = reg_buf.get_input_design_instance(checker.design, True)
                resource_name = reg_buf.get_resource()
                pin_name = reg_buf.get_connecting_pin()

                # Regional buffer should not have any ROUTE. ALL should be from an
                # instance. If there's no instance configured on the resource, then we
                # can skip as it means that the regional buffer is not assigned. There can
                # be the same resource connected to diff RBUF # of different type:
                # Example: GPIOB_PN_09 on CLKMUX that goes to:
                # RBUF4: through DOUT_EVENP: configured as GPIO/LVDS Rx
                # RBUF6: through CLKOUT: configured as MIPI Rx Clock lane
                # So, in this case, we associate the ins_type to the pin and only check
                # if they are associated. If configured as MIPI Rx, then only check on the
                # RBUF that has CLKOUT pin used
                if ins_obj is not None:
                    # Check the type
                    if ins_type == ClockMuxAdvance.ResourceInstanceType.mipi_dphy:
                        # Check that the pin name is correctly associated to the instance type
                        if pin_name != "CLKOUT":
                            res_pins_list = res_to_pins.get(resource_name, [])
                            if pin_name not in res_pins_list:
                                # Check that there is other RBUF pin that has the same resource but
                                # with the CLKOUT pin:

                                is_valid = False
                                self.error("Resource {} with instance {} assigned to RBUF{} has invalid configuration".format(
                                    resource_name, ins_obj.name, idx))
                                break

                    elif ins_type == ClockMuxAdvance.ResourceInstanceType.gpio or \
                            ins_type == ClockMuxAdvance.ResourceInstanceType.lvds:
                        # Check that the pin name is correctly associated to the instance type
                        if pin_name != "DOUT_EVENP":
                            res_pins_list = res_to_pins.get(resource_name, [])
                            if pin_name not in res_pins_list:
                                is_valid = False
                                self.error("Resource {} with instance {} assigned to RBUF{} has invalid configuration".format(
                                    resource_name, ins_obj.name, idx))
                                break

                    elif ins_type == ClockMuxAdvance.ResourceInstanceType.pll:
                        # Check that it's only CLKOUT4
                        conn_valid, expected_idx_str = self._check_pll_regbuf_conn_pin(pin_name)
                        if not conn_valid:
                            is_valid = False
                            self.error("Resource {} with instance {} can only connect to regional buffer input {} through output clock {}".format(
                                resource_name, ins_obj.name, idx, expected_idx_str))
                            break


        return is_valid

    def _check_pll_regbuf_conn_pin(self, pin_name):
        is_valid = True
        outclk_idx_str = ""

        if pin_name != "CLKOUT4":
            is_valid = False
            outclk_idx_str = "4"

        return is_valid, outclk_idx_str

class ClkMuxRuleDynamicMuxPLLInput(base_rule.Rule):
    """
    PT-1345: Additional rule to check the usage of inverted PLL output clocks
    that are driving the Dynamic Mux input:
    Warning if there is a dynamic mux with input mixture of PLL that is inverted with
        other type of resources or non-inverted PLL
    """

    def __init__(self):
        super().__init__()
        self.name = "clkmux_rule_pll_clock"

    def check(self, checker, design_block: ClockMuxAdvance):

        clkmux_obj = design_block

        if clkmux_obj is not None:

            mux_idx_with_warning = []
            dyn_mux_output_with_warning = []

            if clkmux_obj.is_mux_bot0_dyn:
                # Check the input of the Mux0 dynamic mux
                if self.check_dyn_mux_input_pll_outclk(checker, clkmux_obj, 0):
                    mux_idx_with_warning.append(0)

                    out_pin_name = self.get_dyn_mux_output_pin_name(clkmux_obj, 0)
                    if out_pin_name != "":
                        dyn_mux_output_with_warning.append(out_pin_name)

            if clkmux_obj.is_mux_bot7_dyn:
                # Check the input of the Mux7 dynamic mux
                if self.check_dyn_mux_input_pll_outclk(checker, clkmux_obj, 7):
                    mux_idx_with_warning.append(7)

                    out_pin_name = self.get_dyn_mux_output_pin_name(clkmux_obj, 7)
                    if out_pin_name != "":
                        dyn_mux_output_with_warning.append(out_pin_name)

            if mux_idx_with_warning:
                self.severity = self.SeverityType.warning
                self.pass_status = True

                if dyn_mux_output_with_warning:
                    self.msg = 'Dynamic clock mux {} connected to both inverting and non-inverting' \
                                ' clock sources: Clock inversion will not be applied to {}'.format(
                        ','.join([str(elem) for elem in mux_idx_with_warning]),
                        ','.join(dyn_mux_output_with_warning))

                else:
                    self.msg = 'Dynamic clock mux {} connected to both inverting and non-inverting' \
                                ' clock sources: Clock inversion will not be applied to the mux output'.format(
                        ','.join([str(elem) for elem in mux_idx_with_warning]))

    def get_dyn_mux_output_pin_name(self, clkmux_obj, mux_idx):
        pin_name = ""

        if clkmux_obj is not None and clkmux_obj.gen_pin is not None:
            gen_pin = clkmux_obj.gen_pin

            mux_pin = gen_pin.get_pin_by_type_name(
                "{}{}".format(clkmux_obj.dyn_mux_out_pin_name, mux_idx))

            if mux_pin is not None and mux_pin.name != "":
                pin_name = mux_pin.name

        return pin_name

    def check_dyn_mux_input_pll_outclk(self, checker, clkmux,  mux_idx):
        is_warning = False

        # Check the validity of the pre-requisite
        if checker is not None:
            pll_reg = checker.design.pll_reg
            if pll_reg is None:
                return is_warning
        else:
            return is_warning

        non_pll_in_index, pll_with_inv_clock, pll_with_non_inv_clock = \
            clkmux.get_dyn_mux_input_types(mux_idx, pll_reg)

        # Now check, if it was a mix or not
        if pll_with_non_inv_clock and pll_with_inv_clock:
            # Mixture of PLL output clocks that are inverted and not inverted
            is_warning = True

        elif non_pll_in_index:
            # If it was a mixture of PLL and non-PLL as inputs to the mux,
            # Determine if the PLL requires inversion (MP: non-invert, ES: invert).
            # If the PLL requires inversion signal, then we give warning.
            # Get the Max limit of lvds gpio in Output/Inout mode
            dbi = devdb_int.DeviceDBService(checker.design.device_db)
            pll_dev_svc = dbi.get_block_service(
                checker.design.device_db.get_pll_type())

            if pll_dev_svc is not None:
                pll_default_outclk_inv = pll_dev_svc.is_pll_default_outclk_inv()

                if pll_default_outclk_inv:
                    # This is in the case where by default the PLL output clock
                    # need inversion when user doesn't indicate it is inverted (MP device)
                    if pll_with_non_inv_clock:
                        # In this case, some of the inputs are non-inverted which required
                        # an inverted interface that doesn't match with other non-PLL dynamic
                        # mux input
                        is_warning = True

                else:
                    # This is the case where by default the PLL output clock does not
                    # need inversion. However, if user stated that some of the output clock
                    # that's driving this Mux input is inverted, then it mismatch with
                    # the non-PLL inputs to the mux
                    if pll_with_inv_clock:
                        is_warning = True

            # If there was no special indication (ie, test device), we assume
            # that by default it is not inverted
            else:
                if pll_with_inv_clock:
                    is_warning = True

        return is_warning

class ClkMuxRuleDiffPLLClockSameMux(base_rule.Rule):
    """
    PT-2569: If the clock drives LVDS Rx and MIPI Lane fast/slow clocks. They should
    both be routed to the same clockmux eventhough in the routing graph, it can drive
    multiple clockmuxes (of different sides).
    """

    def __init__(self):
        super().__init__()
        self.name = "clkmux_rule_pll_serial_parallel_clocks"

    def check(self, checker, design_block: ClockMuxAdvance):

        clkmux_obj = design_block
        
        if clkmux_obj is not None and checker.design is not None and\
            checker.clkmuxdev_service is not None:
            if checker.reg.pll_clk_side_restrictions:
                #print(f"\npll_clk_side_restrictions: {checker.reg.pll_clk_side_restrictions}")
                # Check that the instances clock pin are routed to the same side
                output_map = clkmux_obj.get_all_io_setting()

                #checker.logger.debug(f"Clock mux: {clkmux_obj.name}")
                
                ins_obj = checker.design.device_db.find_instance(clkmux_obj.name)
                assert ins_obj is not None

                for pll_clk in sorted(checker.reg.pll_clk_side_restrictions):
                    shared_clk_list = checker.reg.pll_clk_side_restrictions[pll_clk]
                    pll_clk_found = False
                    
                    checker.logger.debug(f"Reading {pll_clk} mapped to {shared_clk_list}")

                    for out_idx in sorted(output_map.keys()):
                        io_setting = output_map[out_idx]

                        if io_setting is None:
                            return

                        #checker.logger.debug("out_idx: {} with {} and {}".format(out_idx, user_in_name, io_setting))

                        if io_setting.input_pin == pll_clk:
                            pll_clk_found = True
                            break

                    if pll_clk_found:
                        clk_not_same_mux = []

                        # All the shared clock should be found in this clock mux
                        for other_clk in shared_clk_list:
                            found_clk = False

                            # Check that other clock also exists in the output
                            for out_idx in sorted(output_map.keys()):
                                io_setting_other = output_map[out_idx]

                                if io_setting_other is None:
                                    return

                                if io_setting_other.input_pin == other_clk:
                                    found_clk = True
                                    break

                            if not found_clk:
                                clk_not_same_mux.append(other_clk)

                        if len(clk_not_same_mux) > 0:
                            msg = "Clock {} is not found to be routed on the same side with other PLL clocks: {}".format(
                                pll_clk, ",".join(clk_not_same_mux))
                            self.error(msg)

                        
if __name__ == "__main__":
    pass
