"""
Copyright (C) 2017-2020 Efinix Inc. All rights reserved.

No portion of this code may be reused, modified or
distributed in any way without the expressed written
consent of Efinix Inc.

Created on October 26, 2020

@author: maryam
"""

from __future__ import annotations
import sys
import os
import re
import gc
from enum import Enum, auto
from typing import Optional, TYPE_CHECKING, Dict, List, Tuple
from tx180_device.mipi_dphy.design import MIPIHardDPHYRx, MIPIHardDPHYTx
from tx180_device.pll_ssc.design import PLLSSC
from tx180_device.pll_ssc.design_param_info import PLLSSCDesignParamInfo

import util.gen_util
from util.singleton_logger import Logger
import util.excp as app_excp

from common_device.property import PropertyMetaData
from common_device.mipi_dphy.mipi_dphy_design import MIPIDPhy
from common_device.mipi.mipi_design import MIPI

from design.db_item import GenericParam, GenericParamGroup, \
    PeriDesignGenPinItem, PeriDesignRegistry, GenericParamService

import design.excp as des_excp

import device.excp as dev_excp
from device.block_connectivity import ConnVertex, MuxConnGraph

from tx60_device.hio_res_service import HIOResService
from tx60_device.clock_mux.clkmux_routing_adv import TopLevelClockMuxGraph
from tx60_device.lvds.lvds_design_adv import LVDSAdvance, LVDSRxAdvance
from tx60_device.gpio.gpio_design_comp import GPIOComplex

from tx180_device.pll.pll_design import PLLDesignV3Complex
from tx180_device.mipi_dphy.design_param_info import MIPIHardDPHYRxDesignParamInfo as RxParamInfo
from tx180_device.mipi_dphy.design_param_info import MIPIHardDPHYTxDesignParamInfo as TxParamInfo

if TYPE_CHECKING:
    from design.db import PeriDesign
    from device.db import PeripheryDevice

sys.path.append(os.path.join(os.path.dirname(__file__), '..'))

@util.gen_util.freeze_it
class ClockMuxIOSettingAdv:
    """
    Clock Mux I/O Map Info.

    This is a class of its own, in case, it needs more info.
    """

    # def __init__(self, output_name, input_pin=None, input_name=None):
    def __init__(self, conn_input_pin):
        super().__init__()

        self.output_name = ""
        self.output_no = None  #: An output index number
        # The pin name connecting to the top-mux
        self.input_pin = conn_input_pin  #: An input pin name
        # The internal pin name of the clockmux that connects to
        # top net name
        self.input_name = ""

        # A map of the mux that it has to route through
        # key - internal mux name
        # value - mux index
        self.mux_assignment = {}

        self.logger = Logger

    def __str__(self, *args, **kwargs):
        info = 'output no:{} name:{} conn_input:{} blk_pin:{}\n'.format(
            self.output_no, self.output_name, self.input_pin, self.input_name)

        for mux_name, in_index in self.mux_assignment.items():
            mux_asgn = "mux {}:{}".format(mux_name, in_index)
            info += "\n" + mux_asgn

        return info

    def set_output_no(self):
        if self.output_name != "":
            out_no = ""

            # Get the index out of the output name
            if not self.output_name.startswith("ROUTE"):
                match_obj = re.match(r'.*?([0-9]*)\]$', self.output_name)
                if match_obj:
                    out_no = match_obj.group(1)

                #out_no = re.sub('.*?([0-9]*)\]$', r'\1', self.output_name)
            else:
                match_obj = re.match(r'.*?([0-9]*)$', self.output_name)
                if match_obj:
                    out_no = match_obj.group(1)

                #out_no = re.sub('.*?([0-9]*)$', r'\1', self.output_name)

            if out_no != "":
                self.logger.debug("Output {} with index {}".format(
                    self.output_name, out_no))
                self.output_no = int(out_no)

class InputPinInfo:
    """
    Base class that stores information associated to any input
    pin that is from other resources.
    """

    def __init__(self, index, resource=""):

        self._index = index   #: The index (int)
        #: The device resource that is assigned to the buf. If it was
        # connected to a ROUTE pin, then the resource name would be the
        # ROUTE# name
        self._resource = resource

        # reference pin name (of the RESOURCE) associated to this input
        # For ROUTE resource, it will be empty
        self._connecting_pin = ""

        self.logger = Logger

    def get_index(self):
        return self._index

    def get_resource(self):
        return self._resource

    def get_display_resource_name(self, found_res_name, user_pin_name):
        resource_name = ""

        if found_res_name != "Unbonded" and\
                found_res_name != "Unused":
            if self.get_resource().startswith("ROUTE"):
                # The string should already be "Core Clock.."
                resource_name = found_res_name
            else:
                if user_pin_name != "":
                    if found_res_name.find("PLL") != -1 and found_res_name.find(".") == -1:
                        resource_name = "{}.{}".format(found_res_name, self._connecting_pin)
                    else:
                        # If it was a MIPI resource where the connecting pin is "CLKOUT",
                        # we append the CLKOUT string
                        if self._connecting_pin == "CLKOUT":
                            # There could be CLKOUT from MIPI and also OSC
                            if self.get_resource().find("OSC") == -1:
                                resource_name = "{}.BYTECLK".format(found_res_name)
                            else:
                                resource_name = found_res_name
                        else:
                            resource_name = found_res_name
                else:
                    resource_name = self.get_possible_resource_name(found_res_name)
        else:
            resource_name = found_res_name

        return resource_name

    def get_possible_resource_name(self, found_res_name):
        '''
        If it was an HSIO resource, depending on the pin type, we
        can tell whether it is MIPI DPHY or LVDS/GPIO. If it was
        LVDS/GPIO: then we put both PN and P. If it was a PLL
        resource we put the pin name as well
        :param found_res_name: The name that is found based on resource availability.
                This could be different than self._resource for cases where an HSIO
                that does not have both P and N pins bonded out.
        :return:
        '''
        res_name = ""
        if self._connecting_pin != "":
            if self._resource.find("PLL") != -1:
                res_name = "{}.{}".format(self._resource, self._connecting_pin)
            elif self._resource.find("OSC") != -1:
                res_name = self._resource
            else:
                if self._connecting_pin == "DOUT_EVENP":
                    if found_res_name != "" and found_res_name == self._resource:
                        res_name = "{},{}".format(self._resource, self._resource.replace("_PN_", "_P_"))
                    else:
                        res_name = found_res_name
                elif self._connecting_pin == "IN[0]":
                    # Dedicated GPIO
                    res_name = self._resource
                elif self._connecting_pin == "CLKOUT":
                    # If the connection pin is CLKOUT, we add the ".CLKOUT" to the resource name
                    # in order to differentiate that it's a MIPI DPHY resource
                    res_name = "{}.BYTECLK".format(self._resource)
                else:
                    # This could be for MIPI DPHY
                    res_name = "{}.{}".format(self._resource, self._connecting_pin)

        return res_name

    def get_connecting_pin(self):
        return self._connecting_pin

    def set_resource(self, resource):
        self._resource = resource

    def set_connecting_pin(self, pin_name):
        self._connecting_pin = pin_name

    def get_pll_clkout_index_from_pin_name(self, pll_inst, pin_name):
        pll_outclk = None

        if pin_name.find("CLKOUT") != -1:
            out_clk_no = re.sub(
                '.*?([0-9]*)$', r'\1', pin_name)

            if out_clk_no != "":
                clkout_idx = int(out_clk_no)
                pll_outclk = pll_inst.get_output_clock_by_number(clkout_idx)

        return pll_outclk

    def get_input_design_instance(self, design: PeriDesign, is_regional: bool = False):
        """
        Find the instances in design that are associated to the
        resource. Note that here we don't check the validity of the
        resource context since we dont have that information.

        Resource context definition: the instance type associated to the usage
        For example: GPIOL_PN_07 is a resource that is associated to GPIO_0 of
        the left clkmux. And GPIO_0 is connected to the dynamic mux index 3. Also,
        the resource has to be configured as either GPIO/LVDS instance and not a
        MIPI CLK.

        All this function is doing is trying to find the design instance that has been
        configured on that resource with possibilites that it can be either of the
        following:
        1) ROUTE pin
        2) PLL instance (without checking the output clock configuration)
        3) HSIO GPIO input special conn type instance
        4) LVDS Rx special conn type instance
        5) MIPI DPHY Rx clock instance
        6) OSCILLATOR clock output
        7) HARD MIPI DPHY clock (Ti180)

        :param hio_svc: The HIO design service
        :param design: The design database
        :return:
            ins_obj: The design object found
            ins_type: The indication of what type of instance it is

            If it was a ROUTE or unconfigured instance, ins_obj is None
            and the ins_type is set to no_config
        """

        ins_obj = None
        ins_type = ClockMuxAdvance.ResourceInstanceType.no_config

        # We only return it if the instance was correctly configured with connection
        # type gclk/gctrl/rclk/rctrl. For rclk/rctrl, it is only applicable if
        # it has buffer_name non-empty
        if self._resource != "" and design is not None:
            hio_svc = HIOResService()
            hio_svc.build(design)

            # If the resource name is "ROUTE#", then don't do anything
            if not self._resource.startswith("ROUTE"):
                # IT could be a PLL or HSIO resource
                if self._resource.startswith("PLL"):
                    if design is not None:
                        pll_reg = design.pll_reg
                        if pll_reg is not None:
                            pll_inst = pll_reg.get_inst_by_device_name(self._resource)
                            if pll_inst is not None:
                                # Check if the pin name is configured
                                if self._connecting_pin != "" and self._connecting_pin.find("CLKOUT") != -1:
                                    # Get the index and check if the pin is configured
                                    pll_out = self.get_pll_clkout_index_from_pin_name(pll_inst, self._connecting_pin)
                                    if pll_out is not None:
                                        if self._is_valid_pll_mux_input(pll_inst, pll_out, is_regional):
                                            ins_obj = pll_inst
                                            ins_type = ClockMuxAdvance.ResourceInstanceType.pll

                elif self._resource.find("OSC") != -1:
                    if design is not None:
                        osc_reg = design.osc_reg
                        if osc_reg is not None:
                            osc_inst = osc_reg.get_inst_by_device_name(self._resource)
                            if osc_inst is not None:
                                if self._connecting_pin != "" and self._connecting_pin == "CLKOUT":
                                    ins_obj = osc_inst
                                    ins_type = ClockMuxAdvance.ResourceInstanceType.osc

                elif self._resource.startswith("MIPI_"):
                    if design is not None:
                        hard_mipi_reg = design.get_block_reg(design.BlockType.mipi_hard_dphy)

                        if hard_mipi_reg is not None:
                            hmipi_inst = hard_mipi_reg.get_inst_by_device_name(self._resource)
                            if hmipi_inst is not None:
                                is_ti375 = hard_mipi_reg.is_ti375_device()
                                if self._connecting_pin != "" and self._is_valid_mipi_hard_dphy_input(hmipi_inst, is_regional, is_ti375):
                                    ins_obj = hmipi_inst
                                    ins_type = ClockMuxAdvance.ResourceInstanceType.mipi_hard_dphy

                        pll_ssc_reg = design.get_block_reg(design.BlockType.pll_ssc)
                        if ins_obj is None and pll_ssc_reg is not None:
                            pll_ssc_inst = pll_ssc_reg.get_inst_by_device_name(self._resource)
                            if pll_ssc_inst is not None:
                                if self._connecting_pin != "" and self._is_valid_pll_ssc_input(pll_ssc_inst, is_regional):
                                    ins_obj = pll_ssc_inst
                                    ins_type = ClockMuxAdvance.ResourceInstanceType.mipi_hard_dphy

                else:
                    user_ins_list = hio_svc.find_resource_user(self._resource)
                    if user_ins_list:
                        for inst in user_ins_list:
                            if isinstance(inst, GPIOComplex):
                            
                                # If it is configured as GPIO, we only want the P pin as
                                # N pin does not have alt conn support (assumption trap by rule)
                                if inst.gpio_def.find("_P_") != -1 and \
                                        self._is_valid_gpio_mux_input(inst, is_regional):
                                    ins_obj = inst
                                    ins_type = ClockMuxAdvance.ResourceInstanceType.gpio
                                    break

                                # Dedicated HVIO can also be used
                                elif inst.gpio_def.find("_N_") == -1 and \
                                        self._is_valid_gpio_mux_input(inst, is_regional):
                                    ins_obj = inst
                                    ins_type = ClockMuxAdvance.ResourceInstanceType.gpio
                                    break

                            elif isinstance(inst, LVDSAdvance) and self._is_valid_lvds_mux_input(inst, is_regional):
                                ins_obj = inst
                                ins_type = ClockMuxAdvance.ResourceInstanceType.lvds
                                break

                            elif isinstance(inst, MIPIDPhy) and self._is_valid_mdphy_mux_input(inst, is_regional):
                                ins_obj = inst
                                ins_type = ClockMuxAdvance.ResourceInstanceType.mipi_dphy
                                break

        return ins_obj, ins_type

    def _is_valid_mipi_hard_dphy_input(self, mdphy_inst, is_regional=False, is_ti375: bool= False):
        is_valid_mux_input = False

        param_service = GenericParamService(mdphy_inst.get_param_group(), mdphy_inst.get_param_info())

        if self._connecting_pin == "WORD_CLKOUT_HS":
            if mdphy_inst.ops_type == MIPI.OpsType.op_rx:
                conn_type = param_service.get_param_value(RxParamInfo.Id.byte_clkout_conn_type)
                gbuf_pin_name = param_service.get_param_value(RxParamInfo.Id.buf_byte_clkout_pin_name)
            else:
                conn_type = param_service.get_param_value(TxParamInfo.Id.byte_clkout_conn_type)
                gbuf_pin_name = param_service.get_param_value(TxParamInfo.Id.buf_byte_clkout_pin_name)

            if not is_regional and conn_type == "gclk":
                is_valid_mux_input = True
            elif conn_type == "rclk":
                if not is_regional and gbuf_pin_name != "":
                    is_valid_mux_input = True
                elif is_regional:
                    is_valid_mux_input = True

        elif self._connecting_pin == "RX_CLK_ESC" or self._connecting_pin == "RX_CLK_ESC_LAN0":
            if mdphy_inst.ops_type == MIPI.OpsType.op_rx:
                conn_type = param_service.get_param_value(RxParamInfo.Id.esc_clkout_conn_type)
                gbuf_pin_name = param_service.get_param_value(RxParamInfo.Id.buf_esc_clkout_pin_name)
            else:
                conn_type = param_service.get_param_value(TxParamInfo.Id.esc_clkout_conn_type)
                gbuf_pin_name = param_service.get_param_value(TxParamInfo.Id.buf_esc_clkout_pin_name)

            if is_ti375:
                if not is_regional and conn_type == "gclk":
                    is_valid_mux_input = True
                elif conn_type == "rclk":
                    if not is_regional and gbuf_pin_name != "":
                        is_valid_mux_input = True
                    elif is_regional:
                        is_valid_mux_input = True

            else:
                # This doesn't support GCLK
                if is_regional and conn_type == "rclk":
                    is_valid_mux_input = True

        return is_valid_mux_input

    def _is_valid_pll_ssc_input(self, pll_ssc_inst, is_regional: bool=False) -> bool:
        is_valid_mux_input = False
        param_service = GenericParamService(pll_ssc_inst.get_param_group(), pll_ssc_inst.get_param_info())
        if self._connecting_pin == "WORD_CLKOUT_HS":
            conn_type = param_service.get_param_value(PLLSSCDesignParamInfo.Id.outclk_conn_type)
            gbuf_pin_name = param_service.get_param_value(PLLSSCDesignParamInfo.Id.buf_outclk_pin_name)

            if not is_regional and conn_type == "gclk":
                is_valid_mux_input = True
            elif conn_type == "rclk":
                if not is_regional and gbuf_pin_name != "":
                    is_valid_mux_input = True
                elif is_regional:
                    is_valid_mux_input = True

        return is_valid_mux_input


    def _is_valid_pll_mux_input(self, pll_inst, pll_out, is_regional=False):
        is_valid_mux_input = False

        # It is regional only if it's PLLComplex or PLLDesignV3Complex but with conn type rclk
        if pll_inst is not None and pll_out is not None:
            if isinstance(pll_inst, PLLDesignV3Complex):
                if is_regional and pll_out.conn_type == "rclk":
                    is_valid_mux_input = True

                elif not is_regional:
                    if pll_out.conn_type == "gclk":
                        is_valid_mux_input = True
                    elif pll_out.conn_type == "rclk" and pll_out.clkmux_buf_name != "":
                        is_valid_mux_input = True
            else:
                is_valid_mux_input = True

        return is_valid_mux_input

    def _is_valid_gpio_mux_input(self, gpio, is_regional=False):
        '''
        Get the instance name of the GPIO instance only if connection type
        is correctly configured.

        :param ins_obj: A GPIOComplex object
        :return:
        '''
        is_valid_mux_input = False
        if gpio is not None and gpio.input is not None and\
            (gpio.mode == gpio.PadModeType.input or gpio.mode == gpio.PadModeType.inout):
            in_cfg = gpio.input
            if not is_regional and (in_cfg.conn_type == in_cfg.ConnType.gclk_conn or \
                in_cfg.conn_type == in_cfg.ConnType.gctrl_conn):
                is_valid_mux_input = True

            elif in_cfg.conn_type == in_cfg.ConnType.rclk_conn or \
                    in_cfg.conn_type == in_cfg.ConnType.rctrl_conn:
                # If the connection type is regional, then we check if user
                # wanted it to connect to global as well
                if in_cfg.clkmux_buf_name != "" and not is_regional:
                    is_valid_mux_input = True
                elif is_regional:
                    is_valid_mux_input = True

        return is_valid_mux_input

    def _is_valid_lvds_mux_input(self, lvds, is_regional=False):
        '''
        Get the instance name of the GPIO instance only if connection type
        is correctly configured.

        :param ins_obj: An LVDSAdvance object
        :return:
        '''
        is_valid_mux_input = False
        if lvds is not None and lvds.rx_info is not None \
                and (lvds.ops_type == lvds.OpsType.op_rx or lvds.ops_type == lvds.OpsType.op_bidir) :
            rx_cfg = lvds.rx_info

            if rx_cfg.conn_type == rx_cfg.ConnType.gclk_conn and not is_regional:
                is_valid_mux_input = True

            elif rx_cfg.conn_type == rx_cfg.ConnType.rclk_conn:
                if not is_regional and rx_cfg.clkmux_buf_name != "":
                    # If the connection type is regional, then we check if user
                    # wanted it to connect to global as well
                    is_valid_mux_input = True
                elif is_regional:
                    is_valid_mux_input = True

        return is_valid_mux_input

    def _is_valid_mdphy_mux_input(self, mdphy, is_regional=False):
        '''
        Get the instance name of the GPIO instance only if connection type
        is correctly configured.

        :param ins_obj: A MIPIDPhy object
        :return:
        '''
        is_valid_mux_input = False

        if mdphy is not None and mdphy.ops_type == mdphy.OpsType.op_rx and mdphy.rx_info is not None:
            rx_cfg = mdphy.rx_info

            if rx_cfg.mode == rx_cfg.ModeType.clock:
                # It can only be either global or rclk connection type. So, we don't
                # need to check.
                if rx_cfg.conn_type == rx_cfg.ConnType.gclk_conn and not is_regional:
                    is_valid_mux_input = True

                elif rx_cfg.conn_type == rx_cfg.ConnType.rclk_conn:
                    if not is_regional and rx_cfg.clkmux_buf_name != "":
                        is_valid_mux_input = True
                    elif is_regional:
                        is_valid_mux_input = True

        return is_valid_mux_input

    def _is_route_resource(self):
        """
        Check if the current input is a valid route resource. This is
        when the resource name is set to "ROUTE#"
        :return:
        """
        is_route = False

        if self._resource != "":
            if self._resource.startswith("ROUTE"):
                # Get the string after ROUTE
                route_index = re.sub(
                    '.*?([0-9]*)$', r'\1', self._resource)

                if route_index != "":
                    index_int = int(route_index)
                    if index_int >= 0 and index_int < 4:
                        is_route = True

        return is_route

    def is_regional_conn_type(self, ins_obj, ins_type):
        is_regional = False

        if ins_obj is not None:
            if ins_type == ClockMuxAdvance.ResourceInstanceType.gpio:
                if ins_obj.input is not None:
                    in_cfg = ins_obj.input
                    if in_cfg.conn_type == in_cfg.ConnType.rclk_conn or \
                        in_cfg.conn_type == in_cfg.ConnType.rctrl_conn:
                        is_regional = True

            elif ins_type == ClockMuxAdvance.ResourceInstanceType.lvds:
                if ins_obj.rx_info is not None:
                    rx_cfg = ins_obj.rx_info
                    if rx_cfg.conn_type == rx_cfg.ConnType.rclk_conn:
                        is_regional = True

            elif ins_type == ClockMuxAdvance.ResourceInstanceType.mipi_dphy:
                if ins_obj.rx_info is not None:
                    rx_cfg = ins_obj.rx_info
                    if rx_cfg.conn_type == rx_cfg.ConnType.rclk_conn:
                        is_regional = True

            # If it was a Ti60, where the pll doesn't have connection type, we
            # skip checking for it.
            elif ins_type == ClockMuxAdvance.ResourceInstanceType.pll:
                # Get the design output clock at the specific clkout index
                if isinstance(ins_obj, PLLDesignV3Complex) and self._connecting_pin != "":
                    outclk = self.get_pll_clkout_index_from_pin_name(ins_obj, self._connecting_pin)

                    if outclk is not None and outclk.conn_type == "rclk":
                        is_regional = True

            elif ins_type == ClockMuxAdvance.ResourceInstanceType.mipi_hard_dphy:
                param_service = GenericParamService(ins_obj.get_param_group(), ins_obj.get_param_info())

                md_conn_type = ""
                if isinstance(ins_obj, (MIPIHardDPHYTx, MIPIHardDPHYRx)):
                    # There are 2 possible RCLK: BYTE_CLKOUT or ESC_CLKOUT. It depends
                    # on which pin we're reading
                    if self._connecting_pin != "":
                        if self._connecting_pin == "WORD_CLKOUT_HS":
                            if ins_obj.ops_type == MIPI.OpsType.op_rx:
                                md_conn_type = param_service.get_param_value(RxParamInfo.Id.byte_clkout_conn_type)

                            else:
                                md_conn_type = param_service.get_param_value(TxParamInfo.Id.byte_clkout_conn_type)

                        elif self._connecting_pin == "RX_CLK_ESC" and \
                                ins_obj.ops_type == MIPI.OpsType.op_tx:
                            md_conn_type = param_service.get_param_value(TxParamInfo.Id.esc_clkout_conn_type)

                        elif self._connecting_pin == "RX_CLK_ESC_LAN0" and \
                                ins_obj.ops_type == MIPI.OpsType.op_rx:
                            md_conn_type = param_service.get_param_value(RxParamInfo.Id.esc_clkout_conn_type)

                elif isinstance(ins_obj, PLLSSC):
                    if self._connecting_pin != "":
                        if self._connecting_pin == "WORD_CLKOUT_HS":
                            md_conn_type = param_service.get_param_value(PLLSSCDesignParamInfo.Id.outclk_conn_type)

                if md_conn_type == "rclk":
                    is_regional = True

        return is_regional

    def is_global_conn_type(self, ins_obj, ins_type, device_db):
        is_global = False

        if ins_obj is not None:
            if ins_type == ClockMuxAdvance.ResourceInstanceType.gpio:
                if ins_obj.input is not None:
                    in_cfg = ins_obj.input
                    if in_cfg.conn_type == in_cfg.ConnType.gclk_conn or \
                        in_cfg.conn_type == in_cfg.ConnType.gctrl_conn:
                        is_global = True

            elif ins_type == ClockMuxAdvance.ResourceInstanceType.lvds:
                if ins_obj.rx_info is not None:
                    rx_cfg = ins_obj.rx_info
                    if rx_cfg.conn_type == rx_cfg.ConnType.gclk_conn:
                        is_global = True

            elif ins_type == ClockMuxAdvance.ResourceInstanceType.mipi_dphy:
                if ins_obj.rx_info is not None:
                    rx_cfg = ins_obj.rx_info
                    if rx_cfg.conn_type == rx_cfg.ConnType.gclk_conn:
                        is_global = True

            elif ins_type == ClockMuxAdvance.ResourceInstanceType.osc:
                if ins_obj.get_clock_name() != "":
                    is_global = True

            elif ins_type == ClockMuxAdvance.ResourceInstanceType.mipi_hard_dphy:
                pin = self._connecting_pin
                param_service = GenericParamService(ins_obj.get_param_group(), ins_obj.get_param_info())
                conn_type = ""

                if self.is_ti375_device(device_db) and pin.startswith("RX_CLK_ESC"):
                    info = None
                    if pin == "RX_CLK_ESC":
                        info = TxParamInfo
                    elif pin == "RX_CLK_ESC_LAN0":
                        info = RxParamInfo

                    if info is not None:
                        conn_type = param_service.get_param_value(info.Id.esc_clkout_conn_type)

                elif pin == "WORD_CLKOUT_HS":
                    if ins_obj.ops_type == MIPI.OpsType.op_rx:
                        conn_type = param_service.get_param_value(RxParamInfo.Id.byte_clkout_conn_type)

                    else:
                        conn_type = param_service.get_param_value(TxParamInfo.Id.byte_clkout_conn_type)

                if conn_type == "gclk":
                    is_global = True

        return is_global

    def get_user_resource_pin_name_based_on_found_instance(self, ins_obj, ins_type,
                                                           clkmux, device_db, is_regional):
        res_name = ""
        pin_name = ""

        if ins_obj is not None:
            self.logger.debug("found_instance: {} and type {}".format(
                ins_obj.name, ins_type))


            # Depending on the type, we search for the relevant pin name
            if ins_type == ClockMuxAdvance.ResourceInstanceType.gpio:
                res_name = ins_obj.gpio_def
                in_cfg = ins_obj.input

                if is_regional:
                    if self.is_regional_conn_type(ins_obj, ins_type):
                        pin_name = ins_obj.input.name

                else:
                    if self.is_regional_conn_type(ins_obj, ins_type):
                        pin_name = ins_obj.input.clkmux_buf_name
                    elif self.is_global_conn_type(ins_obj, ins_type, device_db):
                        pin_name = ins_obj.input.name

            elif ins_type == ClockMuxAdvance.ResourceInstanceType.lvds:
                res_name = ins_obj.lvds_def

                if is_regional:
                    if self.is_regional_conn_type(ins_obj, ins_type):
                        pin_name = ins_obj.rx_info.input_bus_name

                else:
                    if self.is_regional_conn_type(ins_obj, ins_type):
                        pin_name = ins_obj.rx_info.clkmux_buf_name
                    elif self.is_global_conn_type(ins_obj, ins_type, device_db):
                        pin_name = ins_obj.rx_info.input_bus_name

            elif ins_type == ClockMuxAdvance.ResourceInstanceType.mipi_dphy:
                res_name = ins_obj.block_def
                # The name at the clockout pin
                if is_regional:
                    if self.is_regional_conn_type(ins_obj, ins_type):
                        pin_name = ins_obj.rx_info.get_alt_clockout_pin_name()

                else:
                    if self.is_regional_conn_type(ins_obj, ins_type):
                        pin_name = ins_obj.rx_info.clkmux_buf_name
                    elif self.is_global_conn_type(ins_obj, ins_type, device_db):
                        pin_name = ins_obj.rx_info.get_alt_clockout_pin_name()

            elif ins_type == ClockMuxAdvance.ResourceInstanceType.pll:
                res_name = ins_obj.pll_def

                # Find the clkout pin name that matches with the pin name
                if self._connecting_pin != "":
                    # If the pin doesn't support multiple connection, then we don't
                    # need to check is_regional flag
                    # Get the PLL device interface
                    from device.db_interface import DeviceDBService
                    dbi = DeviceDBService(device_db)
                    plldev_service = dbi.get_block_service(device_db.get_pll_type())

                    out_clk = self.get_pll_clkout_index_from_pin_name(ins_obj, self._connecting_pin)
                    if out_clk is not None:
                        if plldev_service is not None and plldev_service.is_pll_clkout_pin_support_mult_clkout_conn(res_name, self._connecting_pin):
                            if is_regional:
                                if self.is_regional_conn_type(ins_obj, ins_type):
                                    pin_name = out_clk.name

                            else:
                                if self.is_regional_conn_type(ins_obj, ins_type):
                                    pin_name = out_clk.clkmux_buf_name
                                else:
                                    pin_name = out_clk.name

                        else:
                            pin_name = out_clk.name

            elif ins_type == ClockMuxAdvance.ResourceInstanceType.osc:
                res_name = ins_obj.osc_def
                clk_name = ins_obj.get_clock_name()
                if clk_name != "":
                    pin_name = clk_name

            elif ins_type == ClockMuxAdvance.ResourceInstanceType.mipi_hard_dphy:
                res_name = ins_obj.block_def

                if self._connecting_pin != "":
                    if is_regional:
                        if self.is_regional_conn_type(ins_obj, ins_type):
                            # Get the gen pin of the connecting pin name
                            cgpin = ins_obj.gen_pin.get_pin_by_type_name(self._connecting_pin)
                            if cgpin is not None:
                                pin_name = cgpin.name

                    elif self._connecting_pin == "WORD_CLKOUT_HS" or\
                        (self.is_ti375_device(device_db) and self._connecting_pin in ["RX_CLK_ESC", "RX_CLK_ESC_LAN0"]):
                        # We want to get the global pin name
                        # If the connection type is global, we use the name directly.
                        # However, if the conn type is regional, we get the global
                        # pin name from the other data member
                        if self.is_regional_conn_type(ins_obj, ins_type):
                            if self._connecting_pin == "WORD_CLKOUT_HS":
                                pin_name = self.get_mipi_hard_dphy_buf_name(ins_obj)
                            else:
                                pin_name = self.get_mipi_hard_dphy_esc_name(ins_obj)

                        else:
                            gpin = ins_obj.gen_pin.get_pin_by_type_name(self._connecting_pin)
                            if gpin is not None:
                                pin_name = gpin.name

        elif self._resource != "":
            # This means it's a route pin
            route_pin_name = self._resource
            if route_pin_name != "" and \
                    route_pin_name.startswith("ROUTE"):
                # Get the info from the clock mux
                res_name = DynamicMuxInputOptions.route_pin_title
                # Replace the '#' with the index extracted from the pin name
                route_index = re.sub(
                    '.*?([0-9]*)$', r'\1', route_pin_name)

                if route_index != "":
                    res_name = res_name.replace('#', "{}".format(route_index))
                    pin_name = clkmux.get_route_pin_name(int(route_index))

            elif device_db is not None:
                # In this case the instance is not configured (could be Unassigned/Unbonded)
                if ins_type == ClockMuxAdvance.ResourceInstanceType.no_config:
                    # Find the resource in the device. If it exists, then we assume
                    # That it either has been configured as an instance with irrelevant
                    # connection type or not configured at all (don't need to distinguish)
                    if device_db.is_instance_in_resource_map(self._resource):
                        res_name = self._resource

                    else:
                        # In this case, if it was an HSIO resource but it only has one of the 2 pins
                        # bonded out, then the resource name should be just that GPIO resource name
                        dev_res = device_db.get_resources()
                        if dev_res is not None:
                            if dev_res.is_partially_bonded(self._resource):
                                # We need to find out what are the partial resource that is
                                # bonded out and if they are _P_ only then it's valid. This is
                                # to ignore only if _N_ as they don't support global and regional
                                partial_res_name = dev_res.get_children_of_partially_bonded_instance(self._resource)
                                if partial_res_name.find("_P_") != -1:
                                    res_name = partial_res_name

                        if res_name == "":
                            # No pin name set since resource is unbonded
                            res_name = "Unbonded"

        elif self._resource == "":
            res_name = "Unused"

        return res_name, pin_name

    def get_user_resource_pin_name(self, clkmux, design, is_regional, device_db):
        """
        :param is_regional: If True, indicates that it wants to get the regional
                    clock pin name. This is only applicable if the connection
                    type is set to rclk/rctrl. If False, then it indicates
                    that it wants to read the global pin name.

                    This parameter is only applicable to non-ROUTE pins where
                    a real resource of configured instance is required

                    If regional True but conn type isn't regional, then pin name is empty
                    If regional True and conn type is regional, then it reads the normal
                        pin name
                    If regional False but conn type is regional, it means we want to
                        read the global pin name (stored in clkmux_buf_name)
                    If regional False and conn type is global, it means we want to read
                        the global pin name stored in input name
        :return:
        """
        # If the res has not been configured, there is possibilities:
        # 1) Unassigned: Resource is bonded out but not configured OR
        #                Resource is bonded out and configured with non-related conn type
        # 2) Unbonded: Resource is not bonded out in the device at all. It could be
        #               that the resource is HSIO, but either P/N is bonded. In this case
        #               if P is not bonded, it should show "Unbonded". But if P is
        #               bonded, the resource name should be used.

        # Find the object first
        ins_obj, ins_type = self.get_input_design_instance(design, is_regional)
        #self.logger.debug("get_user_resource_pin_name on {} resource {} pin {}".format(
        #    self._index, self._resource, self._connecting_pin))

        res_name, pin_name = self.get_user_resource_pin_name_based_on_found_instance(ins_obj, ins_type,
                                                           clkmux, device_db, is_regional)

        return res_name, pin_name

    def get_mipi_hard_dphy_buf_name(self, md_ins):
        clkmux_buf_name = ""

        # This is to get the global pin name when the connection was set as RCLK
        if md_ins is not None:
            if self._connecting_pin == "WORD_CLKOUT_HS":
                param_service = GenericParamService(md_ins.get_param_group(), md_ins.get_param_info())

                if isinstance(md_ins, (MIPIHardDPHYTx, MIPIHardDPHYRx)):
                    if md_ins.ops_type == MIPI.OpsType.op_rx:
                        clkmux_buf_name = param_service.get_param_value(RxParamInfo.Id.buf_byte_clkout_pin_name)

                    else:
                        clkmux_buf_name = param_service.get_param_value(TxParamInfo.Id.buf_byte_clkout_pin_name)
                elif isinstance(md_ins, PLLSSC):
                    clkmux_buf_name = param_service.get_param_value(PLLSSCDesignParamInfo.Id.buf_outclk_pin_name)

        return clkmux_buf_name

    def get_mipi_hard_dphy_esc_name(self, md_ins):
        esc_clk_name = ""

        # This is to get the global pin name when the connection was set as RCLK
        if md_ins is not None:
            if self._connecting_pin.startswith("RX_CLK_ESC"):
                param_service = GenericParamService(md_ins.get_param_group(), md_ins.get_param_info())
                info = None

                if self._connecting_pin == "RX_CLK_ESC":
                    info = TxParamInfo
                elif self._connecting_pin == "RX_CLK_ESC_LAN0":
                    info = RxParamInfo

                if info is not None:
                    esc_clk_name = param_service.get_param_value(info.Id.buf_esc_clkout_pin_name)

        return esc_clk_name

    def is_ti375_device(self, device_db: PeripheryDevice):
        device_engineer_name = "opx_466x964_b56_d28" # Ti375 device

        engine_name: str = device_db.get_device_mapped_name()

        if engine_name.startswith(device_engineer_name):
            return True

        return False


@util.gen_util.freeze_it
class RegionalBufferInput(InputPinInfo):
    """
    Class that stores the information on the regional buffer associated
    to the clock mux.
    """


    def __init__(self, index):

        super().__init__(index, "")
        self.is_global = False   #: Indicates if it is also connected to global buf

    def __str__(self, *args, **kwargs):
        info = 'index:{} resource:{} is_global:{}'.format(
            self._index,
            self._resource,
            self.is_global)

        return info

    def set_default_setting(self, device_db=None):
        """
        Apply default setting as per specification
        """
        util.gen_util.mark_unused(device_db)

        self.is_global = False
        self._resource = ""

    def update_global_info(self, clkmux, design):
        """
        Update the global flag due to the design state.
        is_global flag is set based on when user adds a global
        clock pin name and not a direct attribute/property assignment.

        After an ISF import or API run, the rbuf info might have some
        changes which needs to refresh the global flag so that other
        functions/caller who depends on the flag is getting the updated
        data.
        :return The pin name that connects to the global connection
        """
        global_pin_name = ""

        ins_obj, ins_type = self.get_input_design_instance(design)

        if ins_obj is not None and self.is_regional_conn_type(ins_obj, ins_type):
            _, global_pin_name = self.get_regional_with_global_type_and_pin_name(
                clkmux, design, ins_obj, ins_type)

            # Enable it only when the resource supports it
            if global_pin_name != "" and \
                    clkmux.is_resource_pin_has_global_and_regional(
                        design, self.get_resource(), self.get_connecting_pin()):

                self.is_global = True
            else:
                self.is_global = False

        else:
            self.is_global = False

        return global_pin_name

    def get_type_used_as_global(self, clkmux, design):
        '''
        Check if the regional buffer instance is also configured
        as a global connected to dynamic mux or dynamic + static mux

        If they are connected to regional + global, then return the
        global type and the pin name connected to the global buffer

        :param clkmux: ClockMuxAdvance instance
        :param design: PeriDesign db
        :return:
            global_type = DynamicMuxInput.UseConnType
            global_pin_name = The name of the pin that connects to the global
                    clockmux
        '''

        global_type = DynamicMuxInput.UseConnType.no_global
        global_pin_name = ""

        if self.is_global:
            ins_obj, ins_type = self.get_input_design_instance(design)

            # Only go through resource that can possibly be both regional and global
            if ins_obj is not None and self.is_regional_conn_type(ins_obj, ins_type):
                global_type, global_pin_name = self.get_regional_with_global_type_and_pin_name(
                    clkmux, design, ins_obj, ins_type)

        return global_type, global_pin_name

    def get_regional_with_global_type_and_pin_name(self, clkmux: ClockMuxAdvance, design, ins_obj, ins_type):

        global_type = DynamicMuxInput.UseConnType.no_global
        global_pin_name = ""

        # Regional PLL is never part of global
        if ins_type == ClockMuxAdvance.ResourceInstanceType.gpio:
            # Get the instance input buffer name
            if ins_obj.input is not None and ins_obj.input.clkmux_buf_name != "":
                # Check if the pin is also part of dynamic mux (as this will
                # differentiate whether it's an error (mandatory) which is
                # also routed to static mux

                global_type = clkmux.check_rbuf_global_connected_to_dynamic_and_static(
                    design, self, ins_obj)
                global_pin_name = ins_obj.input.clkmux_buf_name

        elif ins_type == ClockMuxAdvance.ResourceInstanceType.lvds:
            if ins_obj.rx_info is not None and ins_obj.rx_info.clkmux_buf_name != "":
                global_type = clkmux.check_rbuf_global_connected_to_dynamic_and_static(
                    design, self, ins_obj)
                global_pin_name = ins_obj.rx_info.clkmux_buf_name

        elif ins_type == ClockMuxAdvance.ResourceInstanceType.mipi_dphy:
            if ins_obj.rx_info is not None and ins_obj.rx_info.clkmux_buf_name != "":
                global_type = clkmux.check_rbuf_global_connected_to_dynamic_and_static(
                    design, self, ins_obj)
                global_pin_name = ins_obj.rx_info.clkmux_buf_name

        elif ins_type == ClockMuxAdvance.ResourceInstanceType.osc:
            global_type = clkmux.check_rbuf_global_connected_to_dynamic_and_static(
                    design, self, ins_obj)
            global_pin_name = ins_obj.get_clock_name()

        elif ins_type == ClockMuxAdvance.ResourceInstanceType.pll:
            # Only certain PLL can have both regional and global connection
            # Which is already checked by the call to is_regional_conn_type
            global_type = clkmux.check_rbuf_global_connected_to_dynamic_and_static(
                design, self, ins_obj)
            global_pin_name = self.get_pll_clkmux_buf_name(ins_obj)

        elif ins_type == ClockMuxAdvance.ResourceInstanceType.mipi_hard_dphy:
            global_type = clkmux.check_rbuf_global_connected_to_dynamic_and_static(
                design, self, ins_obj)

            if self._connecting_pin == "WORD_CLKOUT_HS":
                global_pin_name = self.get_mipi_hard_dphy_buf_name(ins_obj)
            else:
                global_pin_name = self.get_mipi_hard_dphy_esc_name(ins_obj)

        return global_type, global_pin_name

    def get_pll_clkmux_buf_name(self, pll_ins):
        clkmux_buf_name = ""

        if pll_ins is not None:
            # Find the output clock based on the connecting pin name
            out_clk = self.get_pll_clkout_index_from_pin_name(pll_ins, self._connecting_pin)

            if out_clk is not None:
                clkmux_buf_name = out_clk.clkmux_buf_name

        return clkmux_buf_name

    def get_pll_outclk(self, pll_ins):
        pll_outclk = None
        if pll_ins is not None:
            # Find the output clock based on the connecting pin name
            pll_outclk = self.get_pll_clkout_index_from_pin_name(pll_ins, self._connecting_pin)

        return pll_outclk

    def get_user_resource_pin_name(self, clkmux, design, is_regional, device_db):
        '''
        We are overriding this because for RBUF, it is important that we only check
        for instances that are matching with the connecting pin name.
        1) If CLKOUT, then it is expecting that the instance is of MIPI Rx lane type
        2) If DOUT_EVENP, then it is expecting either LVDS Rx or GPIO type

        Because the same resource can be configured as either of the 3 options and
        user can only choose 1. So, even if within the same clock mux there are more than
        one rbuf possibilities (diff rbuf pins but same resource), only one of the type
        should be set. This is different than the case where MIPI Rx Clock connects to >
        1 RBUF pins of the same resource.

        :param clkmux:
        :param design:
        :param is_regional:
        :param device_db:
        :return:
        '''
        res_name, pin_name, = super().get_user_resource_pin_name(clkmux, design, is_regional, device_db)
        _, ins_type = self.get_input_design_instance(design, True)

        final_pin_name = ""

        if ins_type == ClockMuxAdvance.ResourceInstanceType.mipi_dphy:
            final_res_name = res_name
            if self._connecting_pin == "CLKOUT":
                final_pin_name = pin_name

        elif ins_type == ClockMuxAdvance.ResourceInstanceType.gpio:
            final_res_name = res_name
            if self._connecting_pin == "DOUT_EVENP":
                final_pin_name = pin_name
            elif self._connecting_pin == "IN[0]":
                final_pin_name = pin_name

        elif ins_type == ClockMuxAdvance.ResourceInstanceType.lvds:
            final_res_name = res_name
            if self._connecting_pin == "DOUT_EVENP":
                final_pin_name = pin_name

        else:
            # This is also considering oscillator
            final_res_name = res_name
            final_pin_name = pin_name

        return final_res_name, final_pin_name

@util.gen_util.freeze_it
class DynamicMuxInput(InputPinInfo):
    """
    Class that represents the information associated to the dynamic
    mux input assigned by user
    """

    class UseConnType:
        """
        Indicates the global use type that is configured
        """
        no_global = auto()
        dynamic_only = auto()
        static_only = auto()
        dyn_and_static = auto()

    def __init__(self, index, resource=""):
        super().__init__(index, resource)
        self.is_static = False  # Indicate if also connected to the static mux

    def __str__(self, *args, **kwargs):
        info = 'index:{} resource:{} is_static:{}'.format(
            self._index,
            self._resource,
            self.is_static)

        return info

    def set_default_setting(self, device_db=None):
        """
        Apply default setting as per specification
        """
        util.gen_util.mark_unused(device_db)

        self.is_static = False
        self._resource = ""

    def is_assigned_as_input(self, design, rbuf, ins_obj):
        '''
        Check if the rbuf (with the associated instance) is used
        in the mux input assignment
        :param rbuf: RegionalBufferInput Type
        :param ins_obj: The design instance object
        :return: True if the same instance is also assigned to the
                current mux input
        '''
        in_ins_obj, in_type = self.get_input_design_instance(design)

        if in_ins_obj is not None and in_ins_obj == ins_obj:
            # The input assigned to the mux and the instance used in rbuf is the same
            # We don't check on the detailed connection type and the pin name
            # so as to assume correct.
            return True

        return False

@util.gen_util.freeze_it
class DynamicMuxInputOptions:
    """
    The class stores the list of possible resources associated to a specific
    dynamic mux input index
    """

    # The 'resource' name used for route pins in the UI listing
    # where '#' shall be replaced with the index
    route_pin_title = "Core Clock #"

    def __init__(self, mux_in_index):
        self._mux_index = mux_in_index  #: Dynamic mux input index

        # List of options associated to the input mux
        # The index indicate which input index it is to the mux input.
        # If the option is a route pin, the resource is "ROUTE#"
        self._options_map = {}

    def get_option_at_index(self, index):
        """

        :param index: The index being queried
        :return: InputPinInfo associated to the index
        """
        return self._options_map.get(index, None)

    def get_options_map(self):
        return self._options_map

    def get_option_map_size(self):
        if self._options_map:
            return len(self._options_map)

        return 0

    def set_option_at_index(self, index, resource_name):
        """
        Indicate the information associated to a specific index.
        Save it to the _options_map.
        :param index - The index of the option
        :param resource_name - The resource associated to the mux. "ROUTE#" if
                    it is from a specific route pin
        """
        is_added = False

        if index not in self._options_map:
            in_opt = InputPinInfo(index, resource_name)
            self._options_map[index] = in_opt

            is_added = True

        return is_added

    def get_resource_user_pin_name_map(self, design, clkmux):
        """
        Based on the current information in the options map, we
        provide a list of string that shows the possible options
        for the particular mux.

        The string returns is of the format:
        <RESOURCE> : <CLK/CTRL Pin name>

        If it was a ROUTE pin, then the resource should be similar to the title
        used in the UI (route_pin_title)

        If the resource is bonded out and not configured:
        <RESOURCE> : Unassigned
        If the resource is not bonded out:
        Unbonded

        :return: a map of index to string that reflects the resource and user
                configured pin.
        """
        input_options = {}

        if self._options_map:
            for index in sorted(self._options_map.keys()):
                pin_info = self._options_map[index]

                res_name, pin_name = pin_info.get_user_resource_pin_name(clkmux, design, False, design.device_db)
                disp_res_name = pin_info.get_display_resource_name(res_name, pin_name)

                if disp_res_name != "":
                    if pin_name != "":
                        combined_name = "{} : {}".format(disp_res_name, pin_name)
                    else:
                        if disp_res_name != "Unbonded" and \
                                disp_res_name != "Unused":
                            # In the case that it's unassigned
                            combined_name = "{} : Unassigned".format(disp_res_name)

                        else:
                            # In the case that it's unbonded
                            combined_name = "{}".format(disp_res_name)

                    input_options[index] = combined_name


        return input_options

@util.gen_util.freeze_it
class ClockMuxParamInfo(PropertyMetaData):

    class Id(Enum):
        mux_bot0_dyn_mode = auto()
        mux_bot7_dyn_mode = auto()


@util.gen_util.freeze_it
class ClockMuxAdvance(PeriDesignGenPinItem):
    """
    Clock Mux instance. This is slightly different that the Trion ClockMux
    """

    class ResourceInstanceType:
        """
        Indicates the instance type that is configured to the
        input resource.
        """
        no_config = auto()
        gpio = auto()
        lvds = auto()
        mipi_dphy = auto()
        pll = auto()
        osc = auto()
        mipi_hard_dphy = auto()

    # Temporary hardcode the list of PINS associated to a specific dynamic
    # mux input. Currently, the information is stored as a mux pattern in device file.
    # Dynamix mux 0 and mux 7 shares the same list.
    # The following is a map of the index to the PIN name
    dyn_mux_input_0 = {
        0: "PLL0[3]",
        1: "PLL1[3]",
        2: "GPIO[3]",
        3: "ROUTE3"
    }
    dyn_mux_input_1 = {
        0: "PLL0[2]",
        1: "PLL1[2]",
        2: "GPIO[2]",
        3: "ROUTE2"
    }
    dyn_mux_input_2 = {
        0: "PLL0[1]",
        1: "PLL1[1]",
        2: "GPIO[1]",
        3: "ROUTE1"
    }
    dyn_mux_input_3 = {
        0: "PLL0[0]",
        1: "PLL1[0]",
        2: "GPIO[0]",
        3: "ROUTE0"
    }

    # The list of edges associated to the dynamic mux that connects
    # the top output mux to the bottom dyn mux input
    dyn_mux_0_edges = {
        "TOP_0:l0": "BOT_0:0",
        "TOP_1:l1": "BOT_0:1",
        "TOP_2:l2": "BOT_0:2",
        "TOP_3:l3": "BOT_0:3"
    }

    dyn_mux_7_edges = {
        "TOP_4:l4": "BOT_7:0",
        "TOP_5:l5": "BOT_7:1",
        "TOP_6:l6": "BOT_7:2",
        "TOP_7:l7": "BOT_7:3"
    }

    # The list of the edges from the top mux connecting to dynamic
    # mux that goes to other static mux. With the key being
    # the mux input index to the dyn mux
    dyn_mux0_static_edges = {
        0: ("TOP_0:l0", "BOT_4:0"),
        1: ("TOP_1:l1", "BOT_3:0"),
        2: ("TOP_2:l2", "BOT_2:0"),
        3: ("TOP_3:l3", "BOT_1:0")
    }

    dyn_mux7_static_edges = {
        0: ("TOP_4:l4", "BOT_3:3"),
        1: ("TOP_5:l5", "BOT_4:3"),
        2: ("TOP_6:l6", "BOT_5:3"),
        3: ("TOP_7:l7", "BOT_6:3")
    }

    dyn_mux0_edges = {
        0: ("PLL0[3]", "TOP_#:0"),
        1: ("PLL1[3]", "TOP_#:1"),
        2: ("GPIO[3]", "TOP_#:2"),
        3: ("ROUTE3", "TOP_#:3")
    }

    dyn_mux1_edges = {
        0: ("PLL0[2]", "TOP_#:0"),
        1: ("PLL1[2]", "TOP_#:1"),
        2: ("GPIO[2]", "TOP_#:2"),
        3: ("ROUTE2", "TOP_#:3")
    }

    dyn_mux2_edges = {
        0: ("PLL0[1]", "TOP_#:0"),
        1: ("PLL1[1]", "TOP_#:1"),
        2: ("GPIO[1]", "TOP_#:2"),
        3: ("ROUTE1", "TOP_#:3")
    }

    dyn_mux3_edges = {
        0: ("PLL0[0]", "TOP_#:0"),
        1: ("PLL1[0]", "TOP_#:1"),
        2: ("GPIO[0]", "TOP_#:2"),
        3: ("ROUTE0", "TOP_#:3")
    }

    _param_info = None

    dyn_mux_out_pin_name = "DYN_MUX_OUT_"
    dyn_mux_sel_pin_name = "DYN_MUX_SEL_"

    def __init__(self, name, apply_default=True):
        '''
        Constructor for creating the clockmux instance
        :param name: The device instance name which is also
                saved to block_def
        :param apply_default:
        '''
        self.logger = Logger

        # For now, we always create the device instance name as its name
        # since this is not user configured instance
        self.name = name

        # Adding this to be consistent with the other configuration design block
        self.block_def = name

        # An indicator of whether all inputs were successfully routed
        # (True) or not (False).
        self._all_inputs_routed = False

        # The list of inputs that were not routable
        self._non_routed_inputs = []

        # The map of regional buffers index to the required
        # data - RegionalBufferInput (information is loaded
        # during design creation/loading. Not saved in mux peri.xml)
        self._regional_buffers_map: Dict[str, RegionalBufferInput] = {}

        # Each dynamic mux input has 4 options for user to choose from.
        # The options on both dynamic mux0 and mux7 are exactly the same for
        # each input (which should match with the mux pattern).
        self._dynamic_mux_inputs_map = {}

        # Redefining it here although in TX60 mux0 and mux7 input options
        # are identical
        self._dynamic_mux_inputs_map_mux7 = {}

        # The list of DynamicMuxInput associated to the relevant dynamic mux
        # that has been configured (selected) by user. The key is the input index.
        # If a particular index does not have assigned, then it will not exists
        # in the map
        self._user_dyn_mux0_inputs = {}
        self._user_dyn_mux7_inputs = {}

        # The list of dynamic mux input vertices that are removed due to the use
        # of dynamic mux. This will later be checked to see if the removed
        # pin is used but never routed. It does not have to be dyn mux dependent (0/7)
        # Instead, if it is removed from both, then we retain it in the container.
        # The name is based on the clkmux port name (i.e. GPIO[1], PLL1[0], etc.)
        self._removed_input = []

        # Dynamic muxes pins
        # 1) 4 route connectivity from core
        # 2) Dynamic mux select pin name x2
        # 3) Dynamic mux output pin name x2

        # A mapping of the output mux index to the chosen input
        # Added only if the input is routed
        self._io_map = {}

        if self._param_info is None:
            self._param_info = ClockMuxParamInfo()
            self.build_param_info(self._param_info)

        self.param_group = None
        self.build_param()

        super().__init__()
        self.build_generic_pin()

        if apply_default:
            self.set_default_setting()

    def __str__(self, *args, **kwargs):
        info = f"name:{self.name} block_def:{self.block_def}"
        return info

    def get_device(self):
        return self.block_def

    def set_device(self, device_name):
        self.block_def = device_name

    def create_chksum(self):
        return hash(frozenset([
            self.name
        ]))

    def update_regional_buffer_info(self, design_db):
        '''
        In cases where we do import ISF or API call, the is_global
        flag need to be manually updated as the other instances
        updated their setting (i.e. RCLK -> GCLK, remove global pin
        when still an RCLK, etc.).

        So, we refresh all the rbuf within this clockmux in one shot.
        '''
        for rbuf_obj in self._regional_buffers_map.values():
            if rbuf_obj is not None:
                rbuf_obj.update_global_info(self, design_db)

    def get_user_dyn_mux_inputs(self, index):
        if index == 0:
            return self._user_dyn_mux0_inputs
        elif index == 7:
            return self._user_dyn_mux7_inputs

        return {}

    def get_regional_buffer_map(self):
        return self._regional_buffers_map

    def get_regional_buffer_at_index(self, index):
        return self._regional_buffers_map.get(index, None)

    def get_mipi_clk_count(self):
        return 4

    def get_reserved_count(self):
        return 0

    def get_dyn_mux_input_none_index(self):
        return 4

    def get_dyn_mux_input_static(self, dyn_mux_idx: int, in_idx: int) -> Optional[bool]:
        '''
        Get the independent connect to core status flag for the input assigned
        to the dynamic mux input

        :param dyn_mux_idx: The dynamic mux index (0 or 7)
        :param in_idx: The input of the dynamic mux (0-3)
        :return: An boolean that indicates if the input is independently connected
            to core. If it is not being used/assigned (or if dyn mux is not enabled),
            it should return None.
        '''
        is_static = False

        if dyn_mux_idx == 0 and self.is_mux_bot0_dyn:
            if in_idx in self._user_dyn_mux0_inputs:
                mux_in_obj = self._user_dyn_mux0_inputs[in_idx]
                if mux_in_obj is not None:
                    is_static = mux_in_obj.is_static

        elif dyn_mux_idx == 7 and self.is_mux_bot7_dyn:
            if in_idx in self._user_dyn_mux7_inputs:
                mux_in_obj = self._user_dyn_mux7_inputs[in_idx]
                if mux_in_obj is not None:
                    is_static = mux_in_obj.is_static

        return is_static

    def get_mux_input_assignment(self, dyn_mux_idx: int, in_idx: int) -> int:
        '''
        Indicate which input selection has been assigned to this dynamic mux input.
        This would depend on the clkmux type since Ti60 has 4 options while Ti180 has
        6 optionss.

        :param dyn_mux_idx: The dynamic mux index (0 or 7)
        :param in_idx: The input of the dynamic mux (0-3)
        :return: An integer that indicates the selection on the input.  If
            it is not being used/assigned (or if dyn mux is not enabled),
            it should return the default (None)
            which is either 4 or 6 depending on the clkmux type.
        '''
        # Set it to None
        sel_input_idx = self.get_dyn_mux_input_none_index()
        user_mux_inputs = {}

        if dyn_mux_idx == 0 and self.is_mux_bot0_dyn:
            user_mux_inputs = self._user_dyn_mux0_inputs
        elif dyn_mux_idx == 7 and self.is_mux_bot7_dyn:
            user_mux_inputs = self._user_dyn_mux7_inputs

        if user_mux_inputs:

            if in_idx in user_mux_inputs:
                # This means that dyn mux 0 is enabled and the input is set
                # to user specific assignment
                mux_in_obj = user_mux_inputs[in_idx]
                if mux_in_obj is not None:
                    # We need to check which option is this assigned input on
                    mux_opt_obj = self.get_dyn_mux_input_options_at_index(in_idx, dyn_mux_idx)

                    if mux_opt_obj is not None:
                        mux_in_options = mux_opt_obj.get_options_map()

                        if mux_in_options:
                            sel_input_idx = self.get_index_of_current_selection(mux_in_obj, mux_in_options)

                            #self.logger.debug(
                            #    "Dynamic Mux {} input {} is assigned to option {}: {}".format(
                            #        dyn_mux_idx, in_idx, sel_input_idx, mux_in_obj.get_resource()))

        return sel_input_idx

    def get_index_of_current_selection(self, user_in_obj, options_map):
        ret_idx = None

        #self.logger.debug("dynmux user assignment resource.pin: {}.{}".format(
        #    user_in_obj.get_resource(), user_in_obj.get_connecting_pin()))

        if options_map and user_in_obj is not None:
            for idx, in_opt in options_map.items():
                self.logger.debug("index: {} resource.pin: {}.{}".format(
                    idx, in_opt.get_resource(), in_opt.get_connecting_pin()))
                if in_opt is not None:
                    if in_opt.get_resource() == user_in_obj.get_resource() and\
                        in_opt.get_connecting_pin() == user_in_obj.get_connecting_pin():
                        ret_idx = idx
                        break

        return ret_idx

    def get_dyn_mux_input_options_at_index(self, idx: int, dyn_mux_idx: int = 0):
        '''
        Get the list of input options for the specified dynamix mux at index idx

        :param dyn_mux_idx: Dynamic mux index (0/7)
        :param idx: THe index of the dynamix mux input. If 3, then we want
            the list of input options associated to dynamic mux Y at input 3.
        :return: DynamicMuxInputOptions associated to the specified idx
        '''
        # This will give the default map if mux 7 does not have its own
        # mux input options
        if dyn_mux_idx == 7 and self._dynamic_mux_inputs_map_mux7:
            return self._dynamic_mux_inputs_map_mux7.get(idx, None)

        return self._dynamic_mux_inputs_map.get(idx, None)


    def get_route_pin_name(self, index):
        """
        Get the user pin name set to the mentioned ROUTE pin at the given index
        :param index: The route pin index (0-4)
        :return The user configured pin name at that pin (empty string if not set)
        """
        user_pin_name = ""

        pin_name_to_find = "ROUTE{}".format(index)

        if self.gen_pin is not None:
            route_pin = self.gen_pin.get_pin_by_type_name(pin_name_to_find)

            if route_pin is not None:
                user_pin_name = route_pin.name

        return user_pin_name

    def get_all_route_pin_names(self):
        '''

        :return: The map of route pin index to the user specified pin name.
             If a specific index is not assigned, it will be set to empty string
        '''
        pin_names_map = {}
        for idx in range(4):
             pin_names_map[idx] = self.get_route_pin_name(idx)

        return pin_names_map

    def get_dyn_mux_out_pin_name(self, dynm_id):
        pin_name = ""
        out_pin = None

        if self.gen_pin is not None:
            if dynm_id == 0:
                out_pin = self.gen_pin.get_pin_by_type_name("DYN_MUX_OUT_0")
            else:
                out_pin = self.gen_pin.get_pin_by_type_name("DYN_MUX_OUT_7")

            if out_pin is not None:
                pin_name = out_pin.name

        return pin_name

    def add_non_routed_input(self, input_pin_name):
        if input_pin_name not in self._non_routed_inputs:
            self._non_routed_inputs.append(input_pin_name)

    def clear_routed_data(self):
        """
        Deletes the information associated to routed data
        so that it is in sync.
        """
        # Don't delete the connectivity graph as that is fixed
        self._all_inputs_routed = False
        self._non_routed_inputs = []
        self._io_map.clear()
        self._io_map = {}

        self._removed_input = []

    def set_default_setting(self, device_db=None):
        assert self._param_info is not None and self.param_group is not None
        for param_info in self._param_info.get_all_prop():
            self.param_group.set_param_value(param_info.name, param_info.default)

        dyn_mux_list = [0, 7]
        for dyn_mux in dyn_mux_list:
            for idx in range(4):
                self.reset_user_dyn_mux_input_assignment(dyn_mux, idx)

    def build_param(self):
        self.param_group = GenericParamGroup()
        for param_info in self._param_info.get_all_prop():
            self.param_group.add_param(
                param_info.name, param_info.default, param_info.data_type)

    def build_param_info(self, param_info):
        """
        Build information about supported properties

        :param param_info:
        :return:
        """
        if param_info is not None:
            param_info.add_prop(ClockMuxParamInfo.Id.mux_bot0_dyn_mode, "MUX_BOT0_DYN_EN", GenericParam.DataType.dbool, False,
                                disp="Enable Dynamic Mux 0")
            param_info.add_prop(ClockMuxParamInfo.Id.mux_bot7_dyn_mode, "MUX_BOT7_DYN_EN", GenericParam.DataType.dbool, False,
                                disp="Enable Dynamic Mux 7")

    def generate_pin_name(self):
        for pin in self.gen_pin.get_all_pin():
            pin.name = ""

    def get_param_info(self):
        return self._param_info

    def get_param_group(self):
        return self.param_group

    @property
    def is_mux_bot0_dyn(self):
        return self.param_group.get_param_value("MUX_BOT0_DYN_EN")

    @is_mux_bot0_dyn.setter
    def is_mux_bot0_dyn(self, value):
        self.param_group.set_param_value("MUX_BOT0_DYN_EN", value)

    @property
    def is_mux_bot7_dyn(self):
        return self.param_group.get_param_value("MUX_BOT7_DYN_EN")

    @is_mux_bot7_dyn.setter
    def is_mux_bot7_dyn(self, value):
        self.param_group.set_param_value("MUX_BOT7_DYN_EN", value)

    @staticmethod
    def build_port_info(device_db=None, is_mockup=False):
        """
        Build port info for SEU block from definition in device db.
        This function need to be call only once since it is static shared
        between class.

        :param device_db: Device db instance
        :param is_mockup: True, build a mockup data, else build from device db
        """

        if is_mockup:
            raise NotImplementedError
        else:
            if device_db is None:
                return
            else:

                # Pull info from device db
                from device.db_interface import DeviceDBService
                from device.block_definition import Port

                dbi = DeviceDBService(device_db)
                dev_service = dbi.get_block_service(
                    DeviceDBService.BlockType.CLKMUX_ADV)

                user_config_type = ["ROUTE0", "ROUTE1", "ROUTE2", "ROUTE3",
                                    "DYN_MUX_SEL_0", "DYN_MUX_SEL_7",
                                    "DYN_MUX_OUT_0", "DYN_MUX_OUT_7"]

                ClockMuxAdvance._device_port_map = {}

                # TODO: Need to check why without this a create tx unit test
                # failed
                if dev_service is not None:
                    device_port_map = dev_service.get_ports()

                    for name, mode_port in device_port_map.items():
                        if name in user_config_type and mode_port.get_type() != Port.TYPE_PAD:
                            ClockMuxAdvance._device_port_map[name] = mode_port

    def get_dyn_mux_pin(self, mux_idx):
        if mux_idx == 0:
            return ["DYN_MUX_SEL_0", "DYN_MUX_OUT_0"]
        elif mux_idx == 7:
            return ["DYN_MUX_SEL_7", "DYN_MUX_OUT_7"]

        return []

    def get_route_pin(self):
        return ["ROUTE0", "ROUTE1", "ROUTE2", "ROUTE3"]

    def get_non_routed_inputs(self):
        '''
        Return the list of inputs that were assigned and not
        routable.

        :return a list of instance.pin name
        '''
        return self._non_routed_inputs

    def is_all_inputs_routed(self):
        '''
        Return a flag that indicates if all the assigned input
        was routed.

        :return a flag True (all inputs routed) and False otherwise
        '''
        return self._all_inputs_routed

    def get_io_setting(self, output_no):
        """
        For a given output, get its io setting, ClockMuxIOSetting
        :param output_no: Output number
        :return: Clock mux io setting
        """
        return self._io_map.get(output_no, None)

    def add_output_setting(self, io_setting):
        if io_setting is not None:
            self.logger.debug("Saving output {}:{} in {}".format(
                io_setting.output_name, io_setting.output_no, self.name))
            self._io_map[io_setting.output_no] = io_setting

    def get_all_io_setting(self):
        """
        Get full map of output to its input
        """
        return self._io_map

    def route(self, instance, device_db, design_db):
        '''
        Route the clockmux input to the output

        :param instance: The device instance object
        :return True if all the input was successfully routed
                to the output. False if not with messages get
                printed for any failed routing clock.
        '''

        all_routed = False

        # This is not used for advance clock mux since the routing
        # needs to be handled at the top level due to there same PLL
        # output clocks connect to more than one clock muxes.

        return all_routed

    def _is_design_instance_used_as_gpio(self, des_ins_obj, pin_name):
        is_used = False

        if des_ins_obj.input is not None and \
                des_ins_obj.input.is_alternate_connection() and pin_name != "CLKOUT":
            # We only allow if it is gclk, gctrl other types
            # are not counted. Also, skip if we're reading
            # CLKOUT pin which is fixed to MIPI DPHY
            in_cfg = des_ins_obj.input
            if in_cfg.conn_type == in_cfg.ConnType.gctrl_conn or \
                    in_cfg.conn_type == in_cfg.ConnType.gclk_conn:

                is_used = True

            elif in_cfg.conn_type == in_cfg.ConnType.rctrl_conn or \
                    in_cfg.conn_type == in_cfg.ConnType.rclk_conn:

                # If it is regional connection, we check if it's also
                # enabled to connect to global
                if in_cfg.clkmux_buf_name != "":
                    # This indicates that it also wants to connect to global
                    # clkmux
                    is_used = True

        return is_used

    def _is_design_instance_used_as_lvds(self, des_ins_obj, pin_name):
        is_used = False
        if des_ins_obj.ops_type == des_ins_obj.OpsType.op_rx or \
                des_ins_obj.ops_type == des_ins_obj.OpsType.op_bidir:
            self.logger.debug("Found LVDS used")
            # CLKOUT is for MIPI DPHY
            if des_ins_obj.rx_info is not None and \
                    pin_name != "CLKOUT":
                if des_ins_obj.rx_info.conn_type == LVDSRxAdvance.ConnType.gclk_conn:
                    self.logger.debug("LVDS Rx set as gclk")
                    is_used = True
                elif des_ins_obj.rx_info.conn_type == LVDSRxAdvance.ConnType.rclk_conn and \
                        des_ins_obj.rx_info.clkmux_buf_name != "":
                    self.logger.debug("LVDS Rx set as rclk and gclk conn")
                    is_used = True
        return is_used

    def _is_design_instance_used_as_pll(self, des_ins_obj, pin_name):
        is_used = False

        # Check if the corresponding clkout is assigned
        if pin_name.find("CLKOUT") != -1:
            out_clk_no = re.sub(
                '.*?([0-9]*)$', r'\1', pin_name)
            clockout = des_ins_obj.get_output_clock_by_number(
                int(out_clk_no))

            # TODO: If there's a case where this pin does not go to
            # the clock mux when used, then we need to update this (refer to
            # base ClockMux function)
            if clockout is not None and clockout.name != "":
                is_used = True

        return is_used

    def _is_design_instance_used_as_mipi_dphy(self, des_ins_obj):
        is_used = False

        if des_ins_obj.ops_type == des_ins_obj.OpsType.op_rx and \
                des_ins_obj.rx_info is not None:

            # Check that it is in clkout mode
            if des_ins_obj.rx_info.mode == des_ins_obj.rx_info.ModeType.clock:
                # TODO: We filter it out if it is not meant for
                # the clkmux
                if des_ins_obj.rx_info.conn_type == des_ins_obj.rx_info.ConnType.gclk_conn:
                    is_used = True
                elif des_ins_obj.rx_info.conn_type == des_ins_obj.rx_info.ConnType.rclk_conn and \
                        des_ins_obj.rx_info.clkmux_buf_name != "":
                    is_used = True

        return is_used

    def _is_design_instance_used_as_osc(self, des_ins_obj):
        is_used = False

        if des_ins_obj.get_clock_name() != "":
            is_used = True

        return is_used

    def _is_design_instance_used(self, design_db, ins_name, pin_name):
        is_used = False

        # Find the instance in the design registry of the
        # expected block type
        des_ins_obj, des_blk_type = design_db.get_instance_and_type(
            ins_name)
        if des_ins_obj is not None:
            self.logger.debug("Found design instance of {}: {} and type {}".format(
                ins_name, des_ins_obj.name, des_blk_type))

            # It means the object is configured
            # Determine if the pin is connected
            # This could be gclk, gctrl
            if des_blk_type == design_db.gpio_type:
                if self._is_design_instance_used_as_gpio(des_ins_obj, pin_name):
                    is_used = True

            elif des_blk_type == design_db.lvds_type:
                if self._is_design_instance_used_as_lvds(des_ins_obj, pin_name):
                    is_used = True

            elif des_blk_type == design_db.pll_type:
                if self._is_design_instance_used_as_pll(des_ins_obj, pin_name):
                    is_used = True

            elif des_blk_type == design_db.BlockType.mipi_dphy and pin_name == "CLKOUT":
                if self._is_design_instance_used_as_mipi_dphy(des_ins_obj):
                    is_used = True

            elif des_blk_type == design_db.osc_type:
                if self._is_design_instance_used_as_osc(des_ins_obj):
                    is_used = True

        return is_used

    def _is_global_used_input(self, device_db, design_db,
                       dev_ins_name, pin_name):
        '''
        Check if the instance pin name is used/configured
        by users.
        :param device_db: Device database
        :param design_db: Design database
        :param ins_name: The instance name queried
        :param pin_name: The pin name of the instance to be queried

        :return True if it was configured and False otherwise
        '''
        is_used = False

        # If it was an HSIO resource but it was configured as a GPIO
        ins_name_list = [dev_ins_name]

        if dev_ins_name.find("_PN_") != -1:
            # We also add the P pin of the GPIO resource in
            gpio_p_dev_ins = dev_ins_name
            gpio_p_dev_ins = gpio_p_dev_ins.replace("_PN_", "_P_")

            ins_name_list.append(gpio_p_dev_ins)

        # Find the specified ins_name from device
        for ins_name in ins_name_list:
            ref_ins_obj = device_db.find_instance(ins_name)

            if ref_ins_obj is not None:
                if self._is_design_instance_used(design_db, ins_name, pin_name):
                    is_used = True

            else:
                msg = "Cannot find clkmux connecting instance {}".format(
                    ins_name)
                raise dev_excp.InstanceDoesNotExistException(
                    msg, app_excp.MsgLevel.error)

            # Stop once we find its usage
            if is_used:
                break

        return is_used

    def _get_dyn_input_clock_names(self, design, user_mux_inputs_map, skip_independent=False):
        '''

        :param design: Design DB
        :param user_mux_inputs_map: a map of the dynamic mux input to the
                InputPinInfo for what user has configured
        :param skip_independent: Skip looking at dynamic input with independent
                conn to core if True
        :return:
            idx2clk_map: A map of dynamic input index to the clock name used in design
                    if not independently connected to core
            indep_clock_names: a set of clock names that were identified as
                    input to the dynamic mux and also independently connected to core,
                    Skip clock source that is associated to ROUTE pin from being added
                    to the list
        '''
        idx2clk_map = {}
        indep_clock_names = set()

        for idx in sorted(user_mux_inputs_map):
            in_obj = user_mux_inputs_map[idx]
            if in_obj is not None:

                # Also skip if it is from a route pin since
                # it's not a clock source from periphery
                res_name = in_obj.get_resource()
                if res_name != "" and res_name.startswith("ROUTE"):
                    continue

                _, in_pin_name = in_obj.get_user_resource_pin_name(
                    self, design, False, design.device_db)

                # Skip input that are set as independent conn to core
                # depending on the skip_independent flag
                if not skip_independent or \
                        (skip_independent and not in_obj.is_static):

                    if in_pin_name != "":
                        idx2clk_map[idx] = in_pin_name

                if in_obj.is_static and in_pin_name != "":
                    indep_clock_names.add(in_pin_name)

        return idx2clk_map, indep_clock_names

    def get_dynamic_mux_input_clock_names(self, design, mux_idx:int, skip_independent=False):
        '''
        Get the map of dynamic mux input index to the clock name set in design
        that does not include ROUTE clock name.

        :param mux_idx: The dynamic mux index (0 or 7)
        :param skip_independent: If true, skip getting the name for clock that has
            the independent set to core set. If false, we take in all input
            clock names assigned
        :return: a map of the dynamic input index assigned to the clock signal name
            found in design
        '''
        idx2clk_map = {}
        indep_clk_names = set()

        if (mux_idx == 0 and self.is_mux_bot0_dyn) or \
                (mux_idx == 7 and self.is_mux_bot7_dyn):

            if mux_idx == 0:
                user_dyn_mux_in_map = self._user_dyn_mux0_inputs
            else:
                user_dyn_mux_in_map = self._user_dyn_mux7_inputs

            idx2clk_map, indep_clk_names = self._get_dyn_input_clock_names(
                design, user_dyn_mux_in_map, skip_independent)

        return idx2clk_map, indep_clk_names

    def get_all_dyn_mux_non_independent_input_names(self, design):
        '''
        PT-1460: Figure out the names of the clock that are non-route clock driving
        the dynamic input mux, skipping it if it is defined as an independent
        connection to core.

        :param design: Design DB
        :return:
            dyn_mux_clock_not_indep:
                A list of dynamic mux input (non-ROUTE) that does not
                have any independent connection to core set AT ALL (if
                they are driving both dyn mux but one is indep and the other isn't,
                the name will be excluded from the list)

            clock_mux_indep_to_core_names: A set that contains the clock names
                that are identified as dynamic input AND independently connected
                to core (exclude ROUTE clock)

        '''
        dyn_mux_clock_not_indep = []

        # List of clock names that are found driving dyn mux
        # at at the same time set to independently connect to core
        clock_mux_indep_to_core_names = set()

        # Different mux can have the same input but on one mux it is independent
        # but on the other it is not.  In this case, as long as it is independent
        # once, we skip checking on it. This is with respect to the current
        # clock mux. Outside, we check between different clockmux

        # Identify the index to the input dynamic mux to exclude
        # in case they were set on both dynamic mux 0 and 7 but with different
        # assignment on independent connect to core setting
        shared_in_idx_to_exclude = []

        if self.is_mux_bot7_dyn and self.is_mux_bot0_dyn:
            # use the mux0 as reference and then compare.
            for idx in self._user_dyn_mux0_inputs:
                if idx in self._user_dyn_mux7_inputs:
                    # Check if they are pointing to the same selected resource
                    mux0_sel_in = self._user_dyn_mux0_inputs[idx]
                    mux7_sel_in = self._user_dyn_mux7_inputs[idx]

                    if mux0_sel_in is not None and mux7_sel_in is not None:
                        if mux0_sel_in.get_resource() == mux7_sel_in.get_resource() and\
                            mux0_sel_in.get_connecting_pin() == mux7_sel_in.get_connecting_pin():

                            if mux0_sel_in.is_static or mux7_sel_in.is_static:
                                # If they are selecting the same input resource and either one
                                # is static, then we can exclude the index
                                shared_in_idx_to_exclude.append(idx)

        # If skip_independent enabled, get the name of the signal associated
        # to the input ONLY if it was NOT independently connected to core
        if self.is_mux_bot0_dyn:
            idx2clk_map, indep_clk_names = self.get_dynamic_mux_input_clock_names(
                design, 0, skip_independent=True)

            if idx2clk_map:
                if not shared_in_idx_to_exclude:
                    dyn_mux_clock_not_indep += list(idx2clk_map.values())
                else:
                    for idx in idx2clk_map:
                        if idx not in shared_in_idx_to_exclude:
                            dyn_mux_clock_not_indep.append(idx2clk_map[idx])

            if indep_clk_names:
                clock_mux_indep_to_core_names.update(indep_clk_names)

        if self.is_mux_bot7_dyn:
            idx2clk_map, indep_clk_names = self.get_dynamic_mux_input_clock_names(
                design, 7, skip_independent=True)

            if idx2clk_map:
                if not shared_in_idx_to_exclude:
                    dyn_mux_clock_not_indep += list(idx2clk_map.values())
                else:
                    for idx in idx2clk_map:
                        if idx not in shared_in_idx_to_exclude:
                            dyn_mux_clock_not_indep.append(idx2clk_map[idx])

            if indep_clk_names:
                clock_mux_indep_to_core_names.update(indep_clk_names)

        return dyn_mux_clock_not_indep, clock_mux_indep_to_core_names

    def _check_dynamic_input(self, vertex_name, conn_ins_name, conn_pin_name):
        '''
        Check if the resource pin name is either:
        1) connected to dynamic input pin and not connected to static mux
        2) connected to dynamic input pin and also connected to static mux
        3) connected to the regional buffer and also connected to global:
            - not connected to static mux
            - connected to static mux
        4) connected to the regional buffer but not connected to global

        :param vertex_name: The input vertex name which should match with
                    clkmux port name (i.e. PLL1[3], GPIO[3])
        :param conn_ins_name: The device instance name connected to the pin
        :param conn_pin_name: The pin name associated to the resource pin connecting
                    to the vertex/port name
        :return: True, if this connection requires routing to the global clock mux.
                Otherwise, False.
        '''
        is_route_clkmux = True

        # Check if it is found routed to dynamic mux
        is_dyn_mux_input = False
        dyn_in_count = 0
        dyn_mux0_index = -1
        dyn_mux7_index = -1

        # There could be a chance that the same clock is connected as input
        # to both dynamic muxes and either one has the static checked. Theoretically
        # it shouldn't matter which input (left/right) is used to connect to the static mux.

        if self.is_mux_bot0_dyn:
            for idx in sorted(self._user_dyn_mux0_inputs):
                dyn_in = self._user_dyn_mux0_inputs[idx]

                if dyn_in is not None:
                    # Compare the resource name and the pin name
                    if dyn_in.get_resource() == conn_ins_name and \
                        dyn_in.get_connecting_pin() == conn_pin_name:
                        is_dyn_mux_input = True
                        dyn_in_count+=1
                        dyn_mux0_index = idx

                        # We found the same resource set to the dynamic mux input
                        if not dyn_in.is_static:
                            is_route_clkmux = False
                        break

        if self.is_mux_bot7_dyn:
            for idx in sorted(self._user_dyn_mux7_inputs):
                dyn_in = self._user_dyn_mux7_inputs[idx]

                if dyn_in is not None:
                    # Compare the resource name and the pin name
                    if dyn_in.get_resource() == conn_ins_name and \
                            dyn_in.get_connecting_pin() == conn_pin_name:
                        is_dyn_mux_input = True
                        dyn_in_count += 1
                        dyn_mux7_index = idx

                        # We found the same resource set to the dynamic mux input
                        if not dyn_in.is_static:
                            is_route_clkmux = False
                        break

        return is_dyn_mux_input, is_route_clkmux, dyn_mux0_index, dyn_mux7_index

    def get_dyn_input_pin_names(self):
        dyn_inputs_pin_names = list(self.dyn_mux_input_0.values()) + \
                                   list(self.dyn_mux_input_1.values()) + \
                                   list(self.dyn_mux_input_2.values()) + \
                                   list(self.dyn_mux_input_3.values())
        return dyn_inputs_pin_names

    def check_removed_dyn_input_but_connected(self, device_db, design_db,
                                              ins_pin_conn_map, graph_input_vertices):

        if self.is_mux_bot0_dyn or self.is_mux_bot7_dyn:
            dyn_input_used_list = []

            if self.is_mux_bot0_dyn:
                for idx in self._user_dyn_mux0_inputs:
                    dyn_input_used_list.append(self._user_dyn_mux0_inputs[idx])

            if self.is_mux_bot7_dyn:
                for idx in self._user_dyn_mux7_inputs:
                    dyn_input_used_list.append(self._user_dyn_mux7_inputs[idx])

            dyn_inputs_pin_names = self.get_dyn_input_pin_names()

            # Iterate through all the possible dynamic mux input port
            # name at the top
            for vname in dyn_inputs_pin_names:

                # Only go through the inputs that were removed from the graph
                if vname not in graph_input_vertices:

                    if vname in ins_pin_conn_map:

                        self.logger.debug("Found the removed vertex in pin {}".format(vname))

                        # Check if the pin is connected/used
                        pin_conn_obj = ins_pin_conn_map[vname]

                        if pin_conn_obj is not None:
                            # If it is not connected to instance but a direct connection
                            # to the core:
                            conn_ins_name = pin_conn_obj.get_connecting_instance_name(
                                device_db)
                            conn_pin_name = pin_conn_obj.get_connecting_ref_pin_name()

                            #self.logger.debug("Check if {}.{} is connected".format(
                            #    conn_ins_name, conn_pin_name))

                            if conn_ins_name is not None:
                                # This is if the pin is connected to another
                                # instance
                                if self._is_global_used_input(device_db, design_db,
                                                       conn_ins_name, conn_pin_name):
                                    # SKIP if it was assigned to the dynamic input mux
                                    is_conn_to_dyn_in = False

                                    # Create the pair where the source is the <ins>.<pin>
                                    # and the dest is the vertex.name
                                    src_name = "{}.{}".format(
                                        conn_ins_name, conn_pin_name)

                                    for in_obj in dyn_input_used_list:
                                        if in_obj is not None and \
                                            in_obj.get_resource() == conn_ins_name and\
                                            in_obj.get_connecting_pin() == conn_pin_name:

                                            is_conn_to_dyn_in = True
                                            #self.logger.debug(
                                            #    "Skipping input edge: {} to {}".format(src_name, vname))
                                            break

                                    if not is_conn_to_dyn_in:
                                        self.logger.debug(
                                            "Find removed input edges: {} to {}".format(src_name, vname))

                                        # Use the device connection name if couldn't find user pin name
                                        self._removed_input.append(src_name)

                            else:
                                # This is if there is a direct connection to a
                                # user pin (i.e. ROUTE)
                                if self.gen_pin is not None:
                                    # IT should not be a bus
                                    route_pin = self.gen_pin.get_pin_by_type_name(
                                        pin_conn_obj.get_expanded_name())

                                    if route_pin is not None and route_pin.name != "":
                                        is_conn_to_dyn_in = False
                                        # SKIP if it was assigned to the dynamic input mux
                                        for in_obj in dyn_input_used_list:
                                            if in_obj is not None and \
                                                    in_obj.get_resource() == route_pin.type_name:
                                                is_conn_to_dyn_in = True
                                                #self.logger.debug(
                                                #    "Skipping route pin {} since it's assigned as input".format(vname))
                                                break

                                        # Found it
                                        if not is_conn_to_dyn_in:
                                            self.logger.debug(
                                                "Find removed ROUTE direct edges: {} to {}".format(
                                                    route_pin.type_name, vname))
                                            self._removed_input.append(vname)


    def determine_input_conn_all_connected(self, ins_obj, blk_mux_graph, device_db, design_db):
        '''
        :param ins_obj: The clockmux device instance
        :param blk_mux_graph: The MuxConnGraph associated to the block
        :return: a list of tuple of
            (src input which is instance.pin name at the top, block port name bit blasted)
        '''
        edge_list = []
        graph_input_vertices = []

        # We go through the vertices of in type only and checker the instance
        # whether there is anything connected to this pin in the design

        # Get the instance pin connectivity
        ins_pin_conn_map = ins_obj.get_inf_pins()

        if blk_mux_graph is not None:
            conn_vertices = blk_mux_graph.vertices

            for vertex in conn_vertices.values():
                #self.logger.debug(
                #    "Checking if vertex {} is connected".format(vertex.name))

                if vertex.vtype == ConnVertex.VertexType.input:
                    self.logger.debug("Reading input vertex {}".format(vertex.name))
                    graph_input_vertices.append(vertex.name)

                    if vertex.name in ins_pin_conn_map:
                        self.logger.debug("Found the vertex in pin {}".format(vertex.name))

                        # Check if the pin is connected/used
                        pin_conn_obj = ins_pin_conn_map[vertex.name]

                        if pin_conn_obj is not None:
                            # If it is not connected to instance but a direct connection
                            # to the core:
                            conn_ins_name = pin_conn_obj.get_connecting_instance_name(
                                device_db)
                            conn_pin_name = pin_conn_obj.get_connecting_ref_pin_name()
                            self.logger.debug("Check if {}.{} is connected".format(
                                conn_ins_name, conn_pin_name))

                            if conn_ins_name is not None:
                                # This is if the pin is connected to another
                                # instance
                                if self._is_global_used_input(device_db, design_db,
                                                       conn_ins_name, conn_pin_name):
                                    # Create the pair where the source is the <ins>.<pin>
                                    # and the dest is the vertex.name
                                    src_name = "{}.{}".format(
                                        conn_ins_name, conn_pin_name)
                                    pair_tuple = (src_name, vertex.name)

                                    self.logger.debug(
                                        "Find connected edges: {} to {}".format(src_name, vertex.name))
                                    edge_list.append(pair_tuple)
                            else:
                                # This is if there is a direct connection to a
                                # user pin (i.e. ROUTE)
                                if self.gen_pin is not None:
                                    # IT should not be a bus
                                    route_pin = self.gen_pin.get_pin_by_type_name(
                                        pin_conn_obj.get_expanded_name())

                                    if route_pin is not None and route_pin.name != "":
                                        pair_tuple = (
                                            route_pin.type_name, vertex.name)

                                        self.logger.debug(
                                            "Find direct edges: {} to {}".format(
                                                route_pin.type_name, vertex.name))

                                        edge_list.append(pair_tuple)

            self.check_removed_dyn_input_but_connected(
                device_db, design_db, ins_pin_conn_map, graph_input_vertices)

        return edge_list

    def routing_is_successful(self):
        # In this case, we set the appropriate flag to indicate that routing
        # was successful since there was no routing required
        self._non_routed_inputs = []
        self._all_inputs_routed = True
        self.logger.debug("All inputs are routed for {}: {}".format(
            self.name, self._all_inputs_routed))

    def set_user_dyn_mux_input_assignment(self, dyn_mux_index, in_index,
                                          resource, is_static, pin_name):

        dyn_mux = {}
        is_valid = False
        if dyn_mux_index == 0:
            dyn_mux = self._user_dyn_mux0_inputs
            is_valid = True
        elif dyn_mux_index == 7:
            dyn_mux = self._user_dyn_mux7_inputs
            is_valid = True

        if is_valid:
            if in_index not in dyn_mux:
                mux_in = DynamicMuxInput(in_index, resource)
                mux_in.set_connecting_pin(pin_name)
                mux_in.is_static = is_static

                dyn_mux[in_index] = mux_in
            else:
                # If it already exist, we may want to overwrite
                mux_in = dyn_mux[in_index]
                mux_in.is_static = is_static
                mux_in.set_resource(resource)
                mux_in.set_connecting_pin(pin_name)

                dyn_mux[in_index] = mux_in

    def reset_user_dyn_mux_input_assignment(self, dyn_mux_index, in_index):
        '''
        Removes the current input assignment (due to set to None)

        :param dyn_mux_index:
        :param in_index:
        :return:
        '''
        dyn_mux = {}
        is_valid = False
        if dyn_mux_index == 0:
            dyn_mux = self._user_dyn_mux0_inputs
            is_valid = True
        elif dyn_mux_index == 7:
            dyn_mux = self._user_dyn_mux7_inputs
            is_valid = True

        if is_valid:
            if in_index in dyn_mux:
                # Delete the assignment
                dyn_mux[in_index] = None
                del dyn_mux[in_index]

    def load_device_instance_info(self, device_db, design_db):
        '''
        Load device based info associated to the clockmux:

        1) List of regional buffers: populate self._regional_buffers_map
        2) Dynamic mux input options: populate self._dynamic_mux_inputs_map

        :param device_db:
        :param design_db:
        :return:
        '''

        if device_db is not None and design_db is not None:
            # Get the clockmux device instance
            dev_clkmux_ins = device_db.find_instance(self.block_def)

            if dev_clkmux_ins is not None:
                # Delete the info before we populate
                self._regional_buffers_map = {}
                self._dynamic_mux_inputs_map = {}

                # Populate the regional buffer
                self._populate_regional_buffers(device_db, design_db, dev_clkmux_ins)

                # Populate the dynamic mux inputs map
                self._populate_dynamic_mux_inputs(device_db, design_db, dev_clkmux_ins)

    def get_rbuf_indexes(self):
        rbuf_indexes = []

        if self._regional_buffers_map:
            for idx in sorted(self._regional_buffers_map.keys()):
                rbuf_indexes.append(idx)

        return rbuf_indexes

    def _populate_regional_buffers(self, device_db, design_db, dev_ins):
        '''
        Find all the resources associated to the RBUF pins and populate
        them.
        :param device_db: Device DB
        :param design_db: Design DB
        :param dev_ins: The device clock mux instance
        :return:
        '''

        # Find the RBUF pin in the instance
        rbuf_pins = dev_ins.get_bus_pins("RBUF")

        # Iterate through the RBUF pins
        for pin in rbuf_pins:
            if pin is not None:
                # Get the resource (device instance) associated to this pin
                rbuf_ins_name = pin.get_connecting_instance_name(device_db)
                rbuf_ins_pin_name = pin.get_connecting_ref_pin_name()
                pin_index = pin.get_index()

                if pin_index not in self._regional_buffers_map:

                    if rbuf_ins_name != "" and rbuf_ins_pin_name != "":
                        rbuf_in = RegionalBufferInput(pin_index)
                        rbuf_in.set_resource(rbuf_ins_name)
                        rbuf_in.set_connecting_pin(rbuf_ins_pin_name)

                        # Now find the resource instance (configured in design)
                        # to tell whether the global flag should be setup. We want to
                        # find the global pin name
                        #res_name, pin_name = rbuf_in.get_user_resource_pin_name(self, hio_svc, False, device_db)

                        ins_obj, ins_type = rbuf_in.get_input_design_instance(design_db, True)

                        # Check if the instance has been configured with regional conn type
                        # If it was a PLL, it will never be is_global
                        if rbuf_in.is_regional_conn_type(ins_obj, ins_type):
                            res_name, pin_name = rbuf_in.get_user_resource_pin_name_based_on_found_instance(
                                ins_obj, ins_type, self, device_db, False)

                            if res_name != "":
                                # This indicates that there is a global pin name associated
                                # to this resource but we haven't check that the connection type
                                # is regional or global

                                if pin_name != "" and res_name != "Unbonded":
                                        rbuf_in.is_global = True

                        # Save the info to the map
                        self._regional_buffers_map[pin_index] = rbuf_in

                    else:
                        if rbuf_ins_name == "" and rbuf_ins_pin_name == "":
                            # This is an error since all RBUF connection has an instance associated
                            msg = "rbuf connection with missing instance and pin name at index {} in clkmux {}".format(
                                pin_index, self.name)

                        elif rbuf_ins_name == "":
                            # This is an error since all RBUF connection has an instance associated
                            msg = "rbuf connection with missing instance name at index {} in clkmux {}".format(
                                pin_index, self.name)

                        elif rbuf_ins_pin_name == "":
                            # This is an error since all RBUF connection has an instance associated
                            msg = "rbuf connection with missing instance pin name at index {} in clkmux {}".format(
                                pin_index, self.name)

                        raise des_excp.ClockMuxRegionalBufferException(
                            msg, app_excp.MsgLevel.error)

                else:
                    # This is an error since it means there's duplicated index
                    msg = "Duplicated rbuf connection at index {} in clkmux {}".format(
                        pin_index, self.name)
                    raise des_excp.ClockMuxRegionalBufferException(
                        msg, app_excp.MsgLevel.error)


    def _get_dyn_mux_input_pins(self, index: int, mux_idx: int = 0):
        '''
        Get the pin map for a specific dynamix mux (top mux)
        :param index: The dynamix mux (0-3 which is also the same as 4-7)
                since Dyn Mux 0 uses Top mux 0-3
                and Dyn Mux 7 uses Top 4-7
                where the mux pairs have the same input list:
                (0,4), (1,5), (2,6), (3,7)
                mux_idx: The dynamic mux index.  In this case, for Tx60,
                we don't care since dyn mux0 and dyn mux7 has the same
                input options list
        :return: The map of mux input index to the pin name
        '''
        pin_map = {}

        if index >= 0 and index < 4:
            if index == 0:
                pin_map = self.dyn_mux_input_0
            elif index == 1:
                pin_map = self.dyn_mux_input_1
            elif index == 2:
                pin_map = self.dyn_mux_input_2
            else:
                pin_map = self.dyn_mux_input_3

        return pin_map

    def _populate_dynamic_mux_inputs(self, device_db, design_db, dev_ins):
        '''
        Find all the resources associated to the DYNAMIC Mux input pins and populate
        them. (self._dynamic_mux_inputs_map)
        :param device_db: Device DB
        :param design_db: Design DB
        :param dev_ins: The device clock mux instance
        :return:
        '''

        for index in range(4):
            # Iterate through all 3
            pin_map = self._get_dyn_mux_input_pins(index)

            if pin_map:
                dyn_mux_opt = DynamicMuxInputOptions(index)

                for mux_idx, pin_name in pin_map.items():

                    resource_name = ""
                    ref_pin_name = ""
                    # Find the instance at the specified pin in the clkmux instance
                    ins_pin = dev_ins.get_connection_pin(pin_name)
                    if ins_pin is not None:
                        buf_ins_name = ins_pin.get_connecting_instance_name(device_db)
                        buf_ins_pin_name = ins_pin.get_connecting_ref_pin_name()

                        if buf_ins_name is not None:
                            resource_name = buf_ins_name
                            ref_pin_name = buf_ins_pin_name

                        else:
                            # This is if it's a ROUTE pin with no instance. So, we store the pin
                            # name as the resource. The ref_pin_name will be empty
                            resource_name = pin_name

                    if resource_name != "":
                        if dyn_mux_opt.set_option_at_index(mux_idx, resource_name):
                            # Set the pin
                            inp_obj = dyn_mux_opt.get_option_at_index(mux_idx)
                            if inp_obj is not None:
                                inp_obj.set_connecting_pin(ref_pin_name)

                            else:
                                # Found duplicated index
                                msg = "Unable to find dynamic mux resource index{} of mux {} in clkmux {}".format(
                                    mux_idx, index, self.name)
                                raise des_excp.ClockMuxDynamicMuxException(
                                    msg, app_excp.MsgLevel.error)

                        else:
                            # Found duplicated index
                            msg = "Duplicated index {} dynamic mux resource of mux {} in clkmux {}".format(
                                mux_idx, index, self.name)
                            raise des_excp.ClockMuxDynamicMuxException(
                                msg, app_excp.MsgLevel.error)
                    else:
                        msg = "Unable to determine dynamic mux resource at index {} of mux {} in clkmux {}".format(
                            mux_idx, index, self.name)
                        raise des_excp.ClockMuxDynamicMuxException(
                            msg, app_excp.MsgLevel.error)

                self._dynamic_mux_inputs_map[index] = dyn_mux_opt

    def check_rbuf_global_connected_to_dynamic_and_static(self, design, reg_buf, ins_obj):
        '''
        Check the current list of dynamic mux input and check if the regional
        buffer is also connected to the global buffer. If yes, which type of
        global connection it is
        :param reg_buf: RegionalBufferInput object which has is_global flag set
        :param ins_obj: The design instance associated to the reg_buf which has
                    already been checked that it is a regional connection

        :return: DynamicMuxInput.UseConnType
        '''

        global_type = DynamicMuxInput.UseConnType.no_global

        if ins_obj is not None and reg_buf is not None:
            if self.is_mux_bot0_dyn or self.is_mux_bot7_dyn:
                is_dynamic_found = False
                is_static = False

                if self.is_mux_bot7_dyn:
                    # Check if it is part of the dyn mux 7 inputs
                    is_dyn_tmp, is_static_tmp =  self.is_rbuf_used_in_dyn_mux(design, 7, reg_buf, ins_obj)
                    if is_dyn_tmp:
                        is_dynamic_found = True
                        is_static = is_static_tmp

                if not is_dynamic_found and self.is_mux_bot0_dyn:
                    # Check if it is part of the dyn mux 0 inputs
                    is_dyn_tmp, is_static_tmp = self.is_rbuf_used_in_dyn_mux(design, 0, reg_buf, ins_obj)
                    if is_dyn_tmp:
                        is_dynamic_found = True
                        is_static = is_static_tmp

                if is_dynamic_found:
                    if is_static:
                        global_type = DynamicMuxInput.UseConnType.dyn_and_static
                    else:
                        global_type = DynamicMuxInput.UseConnType.dynamic_only

            else:
                # No dynamic mux used. Hence, we assumed that it is used as static only
                global_type = DynamicMuxInput.UseConnType.static_only

        return global_type

    def is_rbuf_used_in_dyn_mux(self, design, mux_idx, rbuf, ins_obj):
        is_dyn_mux_input = False
        is_static = False

        mux_map = {}
        if mux_idx == 0:
            mux_map = self._user_dyn_mux0_inputs
        elif mux_idx == 7:
            mux_map = self._user_dyn_mux7_inputs

        if mux_map:
            for in_idx in mux_map:
                mux_in_obj = mux_map[in_idx]

                if mux_in_obj is not None:
                    if mux_in_obj.is_assigned_as_input(design, rbuf, ins_obj):
                        is_dyn_mux_input = True
                        is_static = mux_in_obj.is_static
                        break

        return is_dyn_mux_input, is_static

    def _modify_dyn_mux_graph_conn(self, device_db, mux_idx: int, dev_ins,
                                   ins_mux_graph: MuxConnGraph, pll_dynamic_only_map):
        '''
        Edit the clkmux graph by removing unnecessary edges due to the
        use of dynamic mux feature
        :param mux_idx: The dynamic mux id
        :param dev_ins: The device clockmux instance
        :param ins_mux_graph: The MuxConnGraph
        :param pll_dynamic_only_map: Is a map of clock instance name to the
                list of PLL clkout connection "<PLLins>.<CLKOUT#>" that
                are dynamic only
        '''
        user_mux_inputs_map = {}
        static_edge_map = {}
        dyn_edges_map = {}

        # Keep a map of the current dynamic mux input assignment and static flag
        # dyn_mux_assignment[input_asg_idx] = (selected_index, is_static)
        dyn_mux_assignment = {}

        if mux_idx == 0:
            user_mux_inputs_map = self._user_dyn_mux0_inputs
            static_edge_map = self.dyn_mux0_static_edges
            dyn_edges_map = self.dyn_mux_0_edges
        elif mux_idx == 7:
            user_mux_inputs_map = self._user_dyn_mux7_inputs
            static_edge_map = self.dyn_mux7_static_edges
            dyn_edges_map = self.dyn_mux_7_edges

        if mux_idx == 0 or mux_idx == 7:

            for in_idx in range(4):
                # IF there is an input that has not been
                # assigned in the dynamic mux, it means that
                # the edges on that input is still valid (routable to static mux).
                # So we don't visit (don't cut edge) index that is not assigned

                self.logger.debug("Iterating through mux input {} of dyn mux {} in {} used idx: {}".format(
                    in_idx, mux_idx, self.block_def, user_mux_inputs_map.keys()))

                if in_idx in sorted(user_mux_inputs_map):
                    # For this index, since it's assigned to specific input,
                    # the other inputs of the same top mux should be disconnected
                    # so that the same mux is not used for a different input
                    in_obj = user_mux_inputs_map[in_idx]

                    # Check if static is enabled
                    if in_obj is not None:
                        self.logger.debug("Reading {}: resource: {} pin: {} is_static: {} available edge idx: {}".format(
                            in_idx, in_obj.get_resource(), in_obj.get_connecting_pin(), in_obj.is_static,
                        static_edge_map.keys()))

                        # We need to know which of the index is this current
                        # input being set to by looking at the resource name and pin
                        in_obj_resource = in_obj.get_resource()
                        in_obj_pin_name = in_obj.get_connecting_pin()

                        if not in_obj.is_static and in_idx in static_edge_map:
                            # remove the connection to the static mux from this top mux only
                            # We should also remove the connection to the other muxes but need
                            # to consider if suddenly the same input is also used in the other
                            # dynamic mux but is_static is set to True.  So, we hold off
                            # deleting it on other input until we have visited both dynamic
                            # muxes.
                            edge_tuple = static_edge_map[in_idx]
                            src, dest = edge_tuple

                            self.logger.debug("{}: Remove static edge of input {} mux {}: {} to {}".format(
                                self.block_def, in_idx, mux_idx, src, dest))

                            ins_mux_graph.remove_edge(src, dest)

                            # Check if this is a PLL
                            if in_obj_resource.startswith("PLL"):
                                pll_pin_name = "{}.{}".format(in_obj_resource, in_obj_pin_name)

                                if self.block_def not in pll_dynamic_only_map:
                                    pll_dynamic_only_map[self.block_def] = [pll_pin_name]
                                else:
                                    pll_dyn_only_pins_list = pll_dynamic_only_map[self.block_def]
                                    if pll_pin_name not in pll_dyn_only_pins_list:
                                        pll_dyn_only_pins_list.append(pll_pin_name)
                                        pll_dynamic_only_map[self.block_def] = pll_dyn_only_pins_list

                        elif in_obj.is_static and in_obj_resource.startswith("PLL"):
                            # In case if we see that this is static and the same
                            # PLL.CLKOUT was previously thought of purely dynamic, then
                            # we remove it from the list
                            if self.block_def in pll_dynamic_only_map:
                                pll_dyn_only_pins_list = pll_dynamic_only_map[self.block_def]
                                pll_pin_name = "{}.{}".format(in_obj_resource, in_obj_pin_name)
                                if pll_pin_name in pll_dyn_only_pins_list:
                                    # Remove the pin and update the map
                                    pll_dyn_only_pins_list.remove(pll_pin_name)
                                    pll_dynamic_only_map[self.block_def] = pll_dyn_only_pins_list

                        # Find the other 3 inputs of the same mux input on this index
                        # For example, this is index1 input of dynamux mux 0 which is from
                        # TOP_1 mux. Find all the input to TOP_1 mux and disconnect
                        # all inputs of TOP_1 mux except this one that's pass through to
                        # the dynamic mux.

                        # This is a map of the top mux input index to the CLKMUX
                        # pin name that drives the input
                        # in_pin_map = self._get_dyn_mux_input_pins(in_idx)

                        # This is the map of the index to a tuple that indicates
                        # the edge from the clkmux pin to the top mux output
                        in_mux_edges_map = self.get_mux_in_edges(mux_idx, in_idx)

                        # TODO: save the top mux input index to the DynamicMuxInput as well
                        # so that we can get the info directly rather than search for it

                        for pin_idx in sorted(in_mux_edges_map):
                            edge_tup = in_mux_edges_map[pin_idx]

                            vsrc, vdest = edge_tup

                            # Check if this is assigned to the ROUTE pin in which the resource
                            # name is called "ROUTE#"
                            if in_obj_resource.startswith("ROUTE"):
                                # Don't remove the edge on the ROUTE pin
                                if not vsrc.startswith("ROUTE"):
                                    # Remove the edge from the graph
                                    self.logger.debug("{}: Remove non ROUTE input {} in mux inpt {} mux {}: {} to {}".format(
                                        self.block_def, pin_idx, in_idx, mux_idx, vsrc, vdest))
                                    ins_mux_graph.remove_edge(vsrc, vdest)

                                else:
                                    dyn_mux_assignment[in_idx] = (pin_idx, in_obj.is_static)

                            else:
                                # The pin assigned to the mux is other than route pin

                                # Check which index this current input is from
                                ins_pin = dev_ins.get_connection_pin(vsrc)
                                if ins_pin is not None:
                                    # Check if the instance matches with resource name and
                                    # also if the pin name matches
                                    if ins_pin.get_connecting_instance_name(device_db) != in_obj_resource or \
                                        ins_pin.get_connecting_ref_pin_name() != in_obj_pin_name:
                                        # This is the index that it's connected to
                                        self.logger.debug(
                                            "{}: Remove input {} in mux inpt {} mux {}: {} to {}".format(
                                                self.block_def, pin_idx, in_idx, mux_idx, vsrc, vdest))
                                        ins_mux_graph.remove_edge(vsrc, vdest)

                                    else:
                                        dyn_mux_assignment[in_idx] = (pin_idx, in_obj.is_static)

            # Remove the connection to the bottom dynamic mux (always when enabled)
            for src, dest in dyn_edges_map.items():
                # Remove the edges from src to dest in the graph
                ins_mux_graph.remove_edge(src, dest)

        return dyn_mux_assignment

    def get_top_input_dyn_mux_edges_map(self, dyn_mux_idx: int, mux_input_idx: int):
        lookup_map = {}
        if mux_input_idx == 0:
            lookup_map = self.dyn_mux0_edges
        elif mux_input_idx == 1:
            lookup_map = self.dyn_mux1_edges
        elif mux_input_idx == 2:
            lookup_map = self.dyn_mux2_edges
        elif mux_input_idx == 3:
            lookup_map = self.dyn_mux3_edges

        return lookup_map

    def get_mux_in_edges(self, dyn_mux_idx: int, mux_input_idx: int):
        '''
        Get the mapping of the edges of pins associated to the mux associated to the
        dynamix mux input index.
        :param dyn_mux_idx: The dynamic mux id which is either 0 or 7
        :param mux_input_idx: The input index to the dynamic mux
        :return: a map of the specific dyn mux input index to the edges (at the top)
        '''
        # Dyn mux 0 top mux index is TOP_0 - TOP_3
        # Dyn mux 7 top mux index is TOP_4 - TOP_7
        if dyn_mux_idx == 0:
            offset = 0
        else:
            offset = 4

        lookup_map = self.get_top_input_dyn_mux_edges_map(dyn_mux_idx, mux_input_idx)

        new_lookup = {}
        if lookup_map:
            # Replace the entry TOP_# with the relevant top mux name
            new_lookup = {}

            for idx in lookup_map:
                tuple_edge = lookup_map[idx]

                src_name, dest_name = tuple_edge

                updated_dest_name = dest_name
                new_num = "{}".format(mux_input_idx + offset)

                updated_dest_name = updated_dest_name.replace("#", new_num)

                #self.logger.debug("get_mux_in_edges: mux: {} index: {} src: {}, dest: {}".format(
                #    dyn_mux_idx,mux_input_idx, src_name, updated_dest_name))

                new_lookup[idx] = (src_name, updated_dest_name)

        return new_lookup

    def determine_graph_conn(self, device_db, dev_ins, ins_mux_graph, pll_dynamic_only_map):
        '''
        Check if the current graph that shows the mux internal connectivity
        need to be amended based on the used feature.
        :param ins_mux_graph: MuxConnGraph of the current clkmux instance
        :param pll_dynamic_only_map: Is a may of clock instance name to the
                list of PLL clkout connection "<PLLins>.<CLKOUT#>" that
                are dynamic only
        '''

        mux0_dyn_in_assignment = {}
        mux7_dyn_in_assignment = {}

        # Check if any of dynamic mux is enabled
        if self.is_mux_bot0_dyn:
            mux0_dyn_in_assignment = self._modify_dyn_mux_graph_conn(
                device_db, 0, dev_ins, ins_mux_graph, pll_dynamic_only_map)

        if self.is_mux_bot7_dyn:
            mux7_dyn_in_assignment = self._modify_dyn_mux_graph_conn(
                device_db, 7, dev_ins, ins_mux_graph, pll_dynamic_only_map)

        #self.logger.debug("before revisit_dyn_mux_input".format(self.name))
        #ins_mux_graph.print_graph()

        # Remove the connection on the other mux unless it still need to
        # connect to static mux and the mux input that it drives is not used/
        # mux input is also driving the same input
        self._revisit_dyn_mux_input_assignments(ins_mux_graph, mux0_dyn_in_assignment, mux7_dyn_in_assignment)

        # Cleanup by removing vertices that have no neighbor and it's a primary
        # input (no predecessor)
        if mux0_dyn_in_assignment or mux7_dyn_in_assignment:
            ins_mux_graph.remove_dangling_primary_input_vertices()

        #self.logger.debug("after revisit_dyn_mux_input and cleaning up".format(self.name))
        #ins_mux_graph.print_graph()

    def _revisit_dyn_mux_input_assignments(self, ins_mux_graph, mux0_dyn_in_assignment, mux7_dyn_in_assignment):
        '''
        Revisit the connection in the MuxConnGraph for input that goes to the dynamic mux.  We need
        to remove edges from the Top if the input that has been assigned to the only dyanamic mux
        without connecting to any of the static muxes.

        :param ins_mux_graph: MuxConnGraph object of the current clkmux design instance
        :param mux0_dyn_in_assignment: a map of the dynamic mux 0 input assignment and static flag
                [input_asg_idx] = (selected_index, is_static)
        :param mux7_dyn_in_assignment: a map of the dynamic mux 7 input assignment and static flag
                [input_asg_idx] = (selected_index, is_static)
        '''

        # Remove the connection on the other mux unless it still need to
        # connect to static mux and the mux input that it drives is not used/
        # mux input is also driving the same input
        # For example: dyn mux 0 input 2 is connect to index 3 with is_static True
        # If dyn_mux 7 input 2 is connected to:
        # 1) other than index 3, we shall remove the connection from the top to
        #   index 3 of dyn mux 7 input
        # 2) index 3, then we don't need to do anything
        # Another example, if same scenario as example above but is_static is False
        # 1) If dyn_mux 7 input 2 is not connected to index3, remove the connection from top to
        # index 3 of dyn mux 7 input
        # 2) If dyn_mux 7 is not set as dynamic mux, then we directly remove connection
        # from the top to index 3 of top muxes driving input 2 dyn mux 7 since the input is not
        # intended to connect to static muxes

        if mux0_dyn_in_assignment or mux7_dyn_in_assignment:
            # There was dynamic mux used in this clock mux
            self._revisit_dyn_mux_pair_assignment(
                ins_mux_graph, 0, 7, mux0_dyn_in_assignment, mux7_dyn_in_assignment)
            self._revisit_dyn_mux_pair_assignment(
                ins_mux_graph, 7, 0, mux7_dyn_in_assignment, mux0_dyn_in_assignment)

    def _revisit_dyn_mux_pair_assignment(
            self, ins_mux_graph, current_dyn_mux_id, compared_dyn_mux_id,
            current_dyn_in_assignment, compared_dyn_in_assignment):

        '''
        Check the input assignment in the current_dyn_mux_id (0,7) made in current_dyn_in_assignment
        and compare it against the other dynamic mux (compared_dyn_mux_id) with its input assignment

        :param ins_mux_graph: MuxConnGraph object
        :param current_dyn_mux_id: current dynamic mux id we're analyzing (0,7)
        :param compared_dyn_mux_id: the other dynamic mux id to compare against
        :param current_dyn_in_assignment: The input assignment for the current dynamic mux
        :param compared_dyn_in_assignment: The input assignment fo the other dynamic mux being compared
        '''

        for mux_input_idx in current_dyn_in_assignment:
            cur_sel_in_tup = current_dyn_in_assignment[mux_input_idx]
            cur_in_idx, cur_is_static = cur_sel_in_tup

            if compared_dyn_in_assignment:
                if mux_input_idx not in compared_dyn_in_assignment:
                    # the other dyn mux is enabled but no specific input to this mux input
                    if not cur_is_static:
                        # Remove the edge from the top pin to this compared mux since
                        # it is not connected to the static mux.
                        # Find the edge to the compared
                        self._remove_dyn_input_conn_to_other_mux_input(
                            ins_mux_graph, compared_dyn_mux_id, mux_input_idx, cur_in_idx, current_dyn_mux_id)

                # If both dynamic mux (enabled) and has the same selected pin
                # but one is is_static True and the other False, it cuts the connection
                # to False (is_static False will cut edge from the top mux output to the
                # static mux input). If we want to make both available, we need to add back
                # the edge to the one that was removed since the same input is selected
                # in both muxes but only one has the is_static flag set to True. If
                # we want to be flexible (higher routability chances), we enable both.
                else:
                    # The same mux input in the other dyn mux is configured. But it is
                    # not necessary that they are both selecting the same input selection
                    compared_sel_in_tup = compared_dyn_in_assignment[mux_input_idx]
                    compared_in_idx, compared_is_static = compared_sel_in_tup

                    # Check if both dyn mux is selecting the same input index
                    # at the same mux input (top mux) but with different static flag.
                    # If different, we enable back the edge from the top mux output
                    # to the other static mux input that has the is_static to false
                    if cur_in_idx == compared_in_idx:
                        if cur_is_static != compared_is_static:
                            static_edge_map = {}

                            if not cur_is_static:
                                # Enable the connection from Dynamic Mux 0 top output at
                                # mux_input_idx to the static mux that it connects to
                                static_edge_map = self._get_dyn_mux_static_edge(current_dyn_mux_id)

                            elif not compared_is_static:
                                # Enable the connection from Dynamic Mux 7 top output at
                                # mux_input_idx to the static mux that it connects to
                                static_edge_map = self._get_dyn_mux_static_edge(compared_dyn_mux_id)

                            if static_edge_map:
                                edge_tuple = static_edge_map[mux_input_idx]
                                src, dest = edge_tuple
                                ins_mux_graph.add_edge(src, dest)

                        elif not cur_is_static:
                            # Their static flag is the same and it's set to false
                            self._remove_dyn_input_conn_to_other_mux_input(
                                ins_mux_graph, compared_dyn_mux_id, mux_input_idx, compared_in_idx, current_dyn_mux_id)

                    # If the input selection is different between the different mux of the
                    # same mux input, then we remove it as long as is_static is false.
                    else:
                        # They are selecting a different input. Always remove since
                        # the top mux is already selecting a different index unless
                        # static flag is set.
                        if not cur_is_static:
                            self._remove_dyn_input_conn_to_other_mux_input(
                                ins_mux_graph, compared_dyn_mux_id, mux_input_idx, cur_in_idx, current_dyn_mux_id)

            else:
                # If the other mux is not dynamic, then disconnect the edge if
                # this conn is not static.
                if not cur_is_static:
                    self._remove_dyn_input_conn_to_other_mux_input(
                        ins_mux_graph, compared_dyn_mux_id, mux_input_idx, cur_in_idx, current_dyn_mux_id)


    def _get_dyn_mux_static_edge(self, mux_idx):
        static_edge_map = {}
        if mux_idx == 0:
            static_edge_map = self.dyn_mux0_static_edges
        elif mux_idx == 7:
            static_edge_map = self.dyn_mux7_static_edges

        return static_edge_map

    def _remove_dyn_input_conn_to_other_mux_input(self, ins_mux_graph: MuxConnGraph, to_mux_idx: int,
                                                  mux_input_idx: int, sel_in_idx: int, sel_mux_idx: int):
        '''
        Remove the edge from the top-level pin to the to_mux_idx at mux_input_idx input
        where the edge to be removed is at the index sel_in_idx of that mux.

        :param ins_mux_graph: MuxConnGraph that's to be modified
        :param to_mux_idx: The dynamic mux that it's not connected to (0/7). The other dyn mux which is the
                    one being compared to by the caller
        :param mux_input_idx: The mux input index that this is concerned with. This
                    is used to retrieve the dynamic_mux_input mapping
        :param sel_in_idx: The index of the dynamic mux input that needs to be disconnected
                    from the top level pin (i.e PLL0[3], GPIO[2]). This is
                    going to refer to the dyn_mux_edges map
        '''

        # Get the dynamic_mux_input at the specified mux_input_idx. This is
        # a generic map where it's identicalt to both dyn mux 0 and 7
        dyn_mux_input_map = self._get_dyn_mux_input_pins(mux_input_idx, to_mux_idx)

        # Get the pin name at the specified input index
        in_pin_name = dyn_mux_input_map.get(sel_in_idx, "")

        if in_pin_name != "":
            # Find the edge to remove based on the to_mux_idx.

            # This is a map of the top-level pin to the top mux input pin
            dyn_mux_edges_map = self.get_mux_in_edges(to_mux_idx, mux_input_idx)

            # Get the edge for this specific input pin
            edge_tup = dyn_mux_edges_map.get(sel_in_idx, None)
            if edge_tup is not None:
                vsrc, vdest = edge_tup

                # Remove this edge
                ins_mux_graph.remove_edge(vsrc, vdest)

                # Also, we should remove the connection to other static mux (if exists) since
                # it should only connect to the dynamic mux chosen (since is_static is false)
                if ins_mux_graph.is_vertex_exists(vsrc):
                    src_vertex = ins_mux_graph.get_vertex(vsrc)
                    # Remove all neighbors, including itself (since we force dynamic input
                    # connection later)
                    src_vertex.neighbors = []

    def get_dyn_mux_input_types(self, mux_idx, pll_reg):
        '''

        :param mux_idx: The dynamic mux index
        :param pll_reg: PLLRegistry
        :return:
            non_pll_in_index: the list of input index that is assigned to non-pll
            pll_with_inv_clock: set of pll instance object that has its output clock inverted
            pll_with_non_inv_clock: set of input from pll instance object that has its
                    output clock non inverted
        '''
        non_pll_in_index = []
        # A list of pll instance names that fit its list type
        pll_with_inv_clock_set = set()
        pll_with_non_inv_clock_set = set()

        if pll_reg is not None:
            # We check the dynamic mux flag again here
            if (mux_idx == 0 and self.is_mux_bot0_dyn) or \
                    (mux_idx == 7 and self.is_mux_bot7_dyn):

                clkmux_in_map = self.get_user_dyn_mux_inputs(mux_idx)

                for in_idx, mux_obj in clkmux_in_map.items():
                    if mux_obj is not None:
                        # Check if resource is PLL
                        res_name = mux_obj.get_resource()

                        pll_ins = pll_reg.get_inst_by_device_name(res_name)
                        if pll_ins is not None:
                            # We found the PLL design instance associated to the resource
                            if pll_ins.is_outclk_inverted:
                                pll_with_inv_clock_set.add(pll_ins)
                            else:
                                pll_with_non_inv_clock_set.add(pll_ins)

                        else:
                            non_pll_in_index.append(in_idx)

        return non_pll_in_index, pll_with_inv_clock_set, pll_with_non_inv_clock_set

    def is_resource_pin_has_global_and_regional(self, design, res_name, conn_pin_name):
        '''
        Check if in this CLKMUX instance, the specificed resource.pin connects to both
        a global (MIPI_CLK, GPIO, PLL) and regional pins (RBUF)
        :param design: Design DB
        :param res_name: Device resource name that we're checking for
        :param conn_pin_name: Device instance pin name of the res name that we're checking for
        :return: True if it connects to both. Otherwise, False.
        '''
        is_conn_to_both = False

        # Get the device instance
        if design.device_db is not None:
            import device.db_interface as devdb_int
            dbi = devdb_int.DeviceDBService(design.device_db)
            svc = dbi.get_block_service(design.device_db.get_clkmux_type())

            if svc is not None:
                is_conn_to_both = svc.is_resource_pin_connects_to_regional_and_global_pins(
                    self.block_def, res_name, conn_pin_name)

        return is_conn_to_both

    def get_removed_inputs(self):
        return self._removed_input

    def check_removed_inputs_for_dyn_mux(self):
        if len(self._removed_input) > 0:
            return True

        return False

@util.gen_util.freeze_it
class ClockMuxRegistryAdvance(PeriDesignRegistry):
    """
    Container for all Clock Mux instance
    """
    # A map of the resource name to user presentable mux name
    clk_def_to_user_name = {
        "CLKMUX_B": "Bottom",
        "CLKMUX_L": "Left",
        "CLKMUX_R": "Right",
        "CLKMUX_T": "Top"
    }

    user_name_to_clk_def = {
        "Bottom": "CLKMUX_B",
        "Left": "CLKMUX_L",
        "Right": "CLKMUX_R",
        "Top": "CLKMUX_T"
    }

    def __init__(self):

        # This is equivalent to _dev2inst_map in PeriDesignRegistry
        #self._clock_mux = {}  #: A map clk mux name and its instance, ClockMux

        # Of type TopLevelClockMuxGraph
        self._top_level_mux_graph = None

        # A flag to indicate whether reassigments was made. This
        # is later used to tell whether we need to reroute or not
        self._input_changed = False

        # The device clock mux block definition
        self._clkmux_block = None

        # An indicator of whether all inputs were successfully routed
        # (True) or not (False).This is done at the registry because
        # there are shared connection across the different muxes. So
        # we route at the top-level rather than per clkmux instances.
        # The we save and distribute the result to individual clock muxes
        self._all_inputs_routed = False

        # The list of inputs that were not routable
        self._non_routed_inputs = []

        # A map of input name to the routing path
        self._route_results = {}
        self.logger = Logger

        # Save any constraint with the PLL clocks that requires to be grouped
        # to the same clockmux side
        self.pll_clk_side_restrictions: Dict[str, List[str]] = {}

        self.skip_pll_same_side_restriction = False

        super().__init__()

    def __str__(self, *args, **kwargs):
        info = 'count:{}'.format(len(self.get_clock_mux_map()))
        return info

    def reset_result(self):
        self._all_inputs_routed = False
        self._non_routed_inputs = []
        self._route_results = {}

        clkmux_list = self.get_all_clock_mux()
        for clkmux_obj in clkmux_list:
            if clkmux_obj is not None:
                clkmux_obj.clear_routed_data()

    def get_clkmux_disp_name(self, clkmux_ins):
        if clkmux_ins is not None:
            return self.clk_def_to_user_name.get(clkmux_ins.name, clkmux_ins.name)

        return ""

    def get_name_based_on_display_name(self, disp_name: str):
        return self.user_name_to_clk_def.get(disp_name, None)

    def get_clock_mux_map(self):
        return self._dev2inst_map

    def create_chksum(self):
        return hash(frozenset([len(self.get_clock_mux_map())]))

    def set_default_setting(self, device_db=None):
        # Not implemented
        pass

    def get_all_clock_mux(self):
        return list(self._dev2inst_map.values())

    def get_all_inst(self):
        """
        Support PeriDesignRegistry API

        :return: Clk mux list

        .. notes:
           This function is not needed if the ClockMuxRegistry is migrated to use PeriDesignRegistry
        """
        return self.get_all_clock_mux()

    def get_clock_mux_by_name(self, name):
        return self.get_clock_mux_map().get(name, None)

    def get_clock_mux_count(self):
        return len(self.get_clock_mux_map())

    def apply_device_db(self, device_db):

        if device_db is None:
            return

        self.device_db = device_db

        # Build the port info for generic pin first
        ClockMuxAdvance.build_port_info(device_db, False)

    def _add_clkmux_ins_graph(self, device_db, ins_name, dev_ins, clkmux_ins,
                              blk_mux_graph, pll_dynamic_only_map):
        '''
        Add the instance clkmux graph to the top-level graph after considering
        the feature used in a clkmux instance.

        :param ins_name: The clkmux instance name which is also the device instance name
        :param dev_ins: The device instance object for the current clockmux
        :param clkmux_ins: The clkmux design instance object
        :param blk_mux_graph: MuxConnGraph object of the clkmux block
        :param pll_dynamic_only_map: Is a map of clock instance name to the
                list of PLL clkout connection "<PLLins>.<CLKOUT#>" that
                are dynamic only
        :return MuxConnGraph graph that is associated to the clkmux instance.
            It could be an exact copy of the blk_mux_graph or modified version depending
            on the feature used in the clkmux
        '''
        ins_mux_graph = None

        if blk_mux_graph is not None:
            ins_mux_graph = MuxConnGraph()
            ins_mux_graph.duplicate_graph(blk_mux_graph)

            clkmux_ins.determine_graph_conn(device_db, dev_ins, ins_mux_graph, pll_dynamic_only_map)

        return ins_mux_graph

    def route_all(self, design_db, device_db):
        '''
        Route all the clockmux available in design
        '''

        try:
            # We need to route at the top since there are signals that connect
            # to more than one clock mux
            if self._top_level_mux_graph is None:
                self._top_level_mux_graph = TopLevelClockMuxGraph()
            else:
                # Check if the graph has been routed before
                # and if any of the input has changed

                # We need to recreate because the input edges need to be cleared
                # TODO: recreate only if there has been a change
                self._top_level_mux_graph = TopLevelClockMuxGraph()
                # call garbage collector
                gc.collect()

            # Clear the routing for reroute
            self._top_level_mux_graph.reset_graph()
            self.reset_result()

            clkmux_map = self.get_clock_mux_map()
            if self._clkmux_block is None:
                # We use one of the instance in the list to get the refblock

                for ins_name in clkmux_map:
                    ins_obj = device_db.find_instance(ins_name)
                    if ins_obj is not None and ins_obj.get_block_definition() is not None:
                        self._clkmux_block = ins_obj.get_block_definition()
                        break

                # At this point the block should not be None
                if self._clkmux_block is None:
                    raise ValueError(
                        "Unable to find clock mux block definition")

            blk_mux_graph = None
            blk_conn = self._clkmux_block.get_internal_conn()
            if blk_conn is not None:
                # Get the mux_graph
                blk_mux_graph = blk_conn.get_mux_graph()

            if blk_mux_graph is not None:
                # A map of the clock mux name to the list of tuple (src, dest)
                # where src is <instance>.<pin> and destination is the port of the
                # clock mux block that it connects to
                mux_input_conn_map = {}

                # A map of clock mux design instance maps to list of  PLL input pins
                # in the format of <PLL instance>.<CLKOUT PIN> that
                # has been set as dynamic connection only with no static.
                # These PLL clkout should not be have their edges connected to other clock mux
                # since PLL can connect to more that 2 clock muxes. If we see that it is used
                # on multiple sides with some static and some not static, then we conclude
                # it is also static and remove from the map
                pll_dynamic_only = {}
                found_removed_input = False

                # Build the connectivity in the top-level graph
                for ins_name in sorted(clkmux_map.keys()):
                    # Get the instance object in device
                    ins_obj = device_db.find_instance(ins_name)

                    if ins_obj is not None:
                        clkmux_ins = clkmux_map[ins_name]

                        self.logger.debug(
                            "Adding {} graph to top".format(ins_name))

                        # We modify the graph based on the specific clkmux
                        # instance configuration since they are fixed.
                        # 1) Remove edges if it was dynamic enabled + not static : to static muxes
                        # 2) Remove edges from the top mux to the enabled dynamic mux
                        ins_mux_graph = self._add_clkmux_ins_graph(
                            device_db, ins_name, ins_obj, clkmux_ins, blk_mux_graph, pll_dynamic_only)

                        self._top_level_mux_graph.add_graph(
                            ins_name, ins_mux_graph)

                        # Based on the instance, we get the list of top-level instance.pin connected
                        # to the block port name (bitblasted). We need to also check
                        # if the regional buf wants to be routed through the global mux
                        edge_list = clkmux_ins.determine_input_conn_all_connected(
                            ins_obj, ins_mux_graph, device_db, design_db)

                        # We just save it even if it could be empty (no
                        # connection at all at the mux)
                        if edge_list:
                            mux_input_conn_map[ins_name] = edge_list

                        # CHeck if any of the mux input has been removed due to the
                        # dynamic mux input assignment. At this point, we don't care which
                        # mux has it. Once we see it, we definitely say it's a routing failure.
                        if clkmux_ins.check_removed_inputs_for_dyn_mux():
                            found_removed_input = True

                # If there is PLL that is only dynamic, then we remove it from
                # the irrelevant PLL input edge list
                if pll_dynamic_only:
                    self.resolve_pll_dynamic_connection(mux_input_conn_map, pll_dynamic_only)

                # If there's any other special resource pin connection that should be
                # removed from the clock mux input so that they don't get routed to the core
                self.update_clkmux_with_removing_input_connecting_to_core(design_db, mux_input_conn_map)

                # Identify pll clock with constraints:
                # Map of PLL input to the other PLL input that it depends on.
                # This means that they both have to be on the same side of gclkbox if they go
                # to different clkmux side.  If the PLL only drive one clkmux, then we can
                # skip saving it so that it doesn't add overhead to routing
                if not self.skip_pll_same_side_restriction:
                    pll_clk_with_dependencies = self.identify_pll_clk_with_constraints(design_db, mux_input_conn_map)
                    self._top_level_mux_graph.clk_dependencies_same_mux = pll_clk_with_dependencies
                    self.pll_clk_side_restrictions = pll_clk_with_dependencies
                else:
                    self._top_level_mux_graph.clk_dependencies_same_mux = {}
                    self.pll_clk_side_restrictions = {}

                # Now that we have the top-level graph at the device level, we
                # then do the input assignment, connecting the top-level
                # pins to the unique clock muxes input pins
                found_input_connected = False
                for ins_name in mux_input_conn_map:
                    edge_list = mux_input_conn_map[ins_name]

                    # Skip if it's empty (no connection on the mux)
                    if edge_list:
                        found_input_connected = True
                        self.logger.debug(
                            "Found input connected on mux {}".format(ins_name))

                        for edge_pair in edge_list:
                            in_port, mux_pin = edge_pair

                            self._top_level_mux_graph.add_input_connection_to_graph(
                                ins_name, in_port, mux_pin)

                # Don't need to route if input pin on all muxes are not
                # connected
                if found_input_connected:
                    # Route the inputs
                    route_results, unrouted_input = self._top_level_mux_graph.route_inputs()

                    if unrouted_input:
                        self._all_inputs_routed = False
                        self._non_routed_inputs = unrouted_input

                    # route_results is a map of input pin to a list of vertex names
                    # that makes up the path
                    self._route_results = route_results

                    # Returns a map of clockmux name to
                    # the list of ClockMuxIOSetting
                    top_to_mux_setting_list = self._top_level_mux_graph.iterate_result(
                        self._route_results, unrouted_input)

                    if not self._non_routed_inputs and not found_removed_input:
                        self.routing_on_all_clkmux_successful()

                    # Found inputs not route-able
                    else:
                        # set the clkmux non-routed inputs accordingly
                        for ins_name in clkmux_map:
                            clkmux_obj = clkmux_map[ins_name]

                            if ins_name not in mux_input_conn_map and \
                                 len(clkmux_obj.get_removed_inputs()) == 0:
                                # This clockmux does not have anythign to write or
                                # all inputs goes to dynamic without any input removed
                                clkmux_obj.routing_is_successful()
                            else:
                                # Copy the unrouted inputs to the mux
                                mux_input_tuple_list = mux_input_conn_map.get(ins_name, [])

                                # Also check if any of those removed top vertices due to
                                # dynamic mux usage has real connection (no longer part of graph)
                                # but should still be flagged as failure in routing.
                                # Example is in Ti180 where say GPIO_1 which only connects to TOP_2:2 and TOP_6:2
                                # but those TOP_2 and TOP_6 mux have been used for dynamic mux and assigned to
                                # other input.  Since that is taken, the GPIO[1] is removed from graph altogether
                                # which means that the clockmux routing is not successful
                                removed_inputs = clkmux_obj.get_removed_inputs()
                                self.logger.debug("Removed inputs: {}".format(",".join(removed_inputs)))

                                for rem_in in removed_inputs:
                                    clkmux_obj.add_non_routed_input(rem_in)

                                found_unrouted_pair = False
                                for edge_pair in mux_input_tuple_list:
                                    src_node, sink_node = edge_pair

                                    if src_node in self._non_routed_inputs:
                                        found_unrouted_pair = True
                                        # So this mux has unrouted inputs
                                        clkmux_obj.add_non_routed_input(
                                            src_node)
                                        self.logger.debug("Unrouted pair for {}: {}".format(clkmux_obj.name, src_node))

                                if not found_unrouted_pair and len(removed_inputs) == 0:
                                    # This specific clockmux is routed successfully
                                    clkmux_obj.routing_is_successful()

                    if top_to_mux_setting_list:
                        for clkmux_name in top_to_mux_setting_list:
                            if clkmux_name in clkmux_map:
                                clkmux_obj = clkmux_map[clkmux_name]

                                if clkmux_obj is not None:
                                    setting_list = top_to_mux_setting_list[clkmux_name]

                                    for io_setting_obj in setting_list:
                                        clkmux_obj.add_output_setting(
                                            io_setting_obj)

                # After done routing, we need to distribute back the result
                # to the individual muxes based on the input that was connected
                else:
                    # Set the result on all clockmux. In this case, there was
                    # no input connected, so consider it is successful
                    self.routing_on_all_clkmux_successful()

        except Exception as exp:
            self.logger.error("Error routing clock mux: {}".format(exp))

    def identify_pll_clk_with_constraints(self, design_db:PeriDesign, 
                                          mux_input_conn_map: Dict[str, List[Tuple[str, str]]]) -> Dict[str, List[str]]:
        """
        PT-2569: Additional constraint in cases where PLL clock drives more than one CLKMUX
        and there are multiple pll clock that drives LVDS or MIPI Tx Lane in which all
        those PLL shall need to drive the same side of the gclkmux.  This function is to
        identify if we have such PLL clock in pairs and save it for later use during routing.
        
        :param: design_db: PeriDesign
        :param mux_input_conn_map: Dict of the clkmux instance mapped to the edge list
        :return pll_clk_with_dependencies: Dict of the PLL clock input name mapped to the other
                pll input name list that needs to be paired to the same gclkmux side.
                For example if LVDS fast and slow clk driven by:
                    a) PLL_TR0.CLKOUT2 and PLL_TR0.CLKOUT4
                    b) PLL_TR0.CLKOUT4 and PLL_TR0.CLKOUT3
                    We'll find:
                    PLL_TR0.CLKOUT2: [PLL_TR0.CLKOUT4]
                    PLL_TR0.CLKOUT4: [PLL_TR0.CLKOUT2, PLL_TR0.CLKOUT3]
                    PLL_TR0.CLKOUT3: [PLL_TR0.CLKOUT4]
        """
        pll_clk_with_dependencies: Dict[str, List[str]] = {}

        # Find PLL clock that is driving LVDS Rx/Tx (width > 2) and MIPI Lane Tx
        pll_reg = design_db.pll_reg
        lvds_reg = design_db.lvds_reg
        mipi_lane_reg = design_db.mipi_dphy_reg

        # Nothing to do if there is no PLL or both lvds and mipi lane not supported
        if pll_reg is None or (lvds_reg is None and mipi_lane_reg is None):
            return pll_clk_with_dependencies
        
        # Check if there is PLL driving more than one side of the clkmux based
        # on looking at the clkmux conn graph
        pll_inputs_mult_clkmux = self.get_pll_conn_mult_clkmux(mux_input_conn_map)

        if pll_inputs_mult_clkmux:
            tmp_dep_map:  Dict[str, List[str]] = {}
            if mipi_lane_reg is not None:
                self.get_mipi_lane_pll_clocks(pll_reg, mipi_lane_reg, tmp_dep_map)

            if lvds_reg is not None:
                self.get_lvds_pll_clocks(pll_reg, lvds_reg, tmp_dep_map)

            # If none of the identified pll clock dependencies is in the list
            # of clk that drives multiple clkmux ins, then we can clear the list
            if tmp_dep_map:
                result = all(item not in pll_inputs_mult_clkmux for item in tmp_dep_map)
                if not result:
                    # At least one of the pll clock that has dependencies
                    # drive mult clockmux
                    pll_clk_with_dependencies = tmp_dep_map

        return pll_clk_with_dependencies
    
    def get_pll_conn_mult_clkmux(self, mux_input_conn_map):

        # List of pll clock pin name with multiple clkmux ins
        pll_inputs_mult_clkmux: List[str] = []

        # Map of pll clock pin name maped to the clkmux ins name that it drives
        pll_clk_to_clkmux_ins_map: Dict[str, List[str]] = {}

        for mux_ins_name in mux_input_conn_map:
            edge_list = mux_input_conn_map[mux_ins_name]

            for edge_pair in edge_list:
                # first element is the PLL instance name + pin
                # second element is the clkmux block port name:
                # PLL_TR0.CLKOUT0, PLL1[0]
                in_port, _ = edge_pair

                mux_list = []
                if in_port in pll_clk_to_clkmux_ins_map:
                    mux_list = pll_clk_to_clkmux_ins_map[in_port]

                    if mux_ins_name not in mux_list and in_port not in pll_inputs_mult_clkmux:
                        # This means that it's driving multiple mux
                        pll_inputs_mult_clkmux.append(in_port)

                if mux_ins_name not in mux_list:
                    mux_list.append(mux_ins_name)

                pll_clk_to_clkmux_ins_map[in_port] = mux_list

        return pll_inputs_mult_clkmux
    
    def find_pll_clock_pair_dependencies(self, pll_reg, user_clk_name_pair_list,
                                         pll_clk_with_dependencies):

        # This should only contain pair with non-empty string
        for clk_pair in user_clk_name_pair_list:
            fast_clk_pin, slow_clk_pin = clk_pair

            fast_pll_data = pll_reg.find_output_clock(fast_clk_pin)
            slow_pll_data = pll_reg.find_output_clock(slow_clk_pin)

            if fast_pll_data is not None and slow_pll_data is not None:
                pll_fast_inst, fast_clk_no = fast_pll_data
                pll_slow_inst, slow_clk_no = slow_pll_data

                if pll_fast_inst.pll_def != ""  and \
                    pll_slow_inst.pll_def != "":
                    fast_clk_port_name = "{}.CLKOUT{}".format(
                        pll_fast_inst.pll_def, fast_clk_no)
                    
                    slow_clk_port_name = "{}.CLKOUT{}".format(
                        pll_slow_inst.pll_def, slow_clk_no)
                    
                    dep_map = {
                        fast_clk_port_name: slow_clk_port_name,
                        slow_clk_port_name: fast_clk_port_name
                    }

                    for clk1, clk2 in dep_map.items():
                        dep_list = []
                        if clk1 in pll_clk_with_dependencies:
                            dep_list = pll_clk_with_dependencies[clk1]
                        
                        if clk2 not in dep_list:
                            dep_list.append(clk2)
                    
                        pll_clk_with_dependencies[clk1] = dep_list

    def get_lvds_pll_clocks(self, pll_reg, lvds_reg, pll_clk_with_dependencies):
        # fast,slow clock pair list
        user_clk_name_pair_list: List[Tuple[str, str]] = []

        for lvds in lvds_reg.get_all_inst():
            if lvds.ops_type == lvds.OpsType.op_tx or lvds.ops_type == lvds.OpsType.op_bidir:
                lvds_tx = lvds.tx_info

                if lvds_tx is not None and lvds_tx.is_serial and lvds_tx.is_serial_width_gt2():
                    # Find the slow and fast clk
                    if lvds_tx.fast_clock_name != "" and lvds_tx.slow_clock_name != "":
                        user_clk_name_pair_list.append((lvds_tx.fast_clock_name, lvds_tx.slow_clock_name))

            if lvds.ops_type == lvds.OpsType.op_rx or lvds.ops_type == lvds.OpsType.op_bidir:
                lvds_rx = lvds.rx_info

                if lvds_rx is not None and lvds_rx.is_serial and lvds_rx.is_serial_width_gt2():
                    # Find the slow and fast clk
                    if lvds_rx.fast_clock_name != "" and lvds_rx.slow_clock_name != "":
                        user_clk_name_pair_list.append((lvds_rx.fast_clock_name, lvds_rx.slow_clock_name))
            
        if pll_reg is not None and user_clk_name_pair_list:
            self.find_pll_clock_pair_dependencies(pll_reg, user_clk_name_pair_list,
                                                  pll_clk_with_dependencies)

    def get_mipi_lane_pll_clocks(self, pll_reg, mipi_lane_reg,
                                 pll_clk_with_dependencies):
        # fast,slow clock pair list
        user_clk_name_pair_list: List[Tuple[str, str]] = []
        
        for mipi_lane in mipi_lane_reg.get_all_tx_inst():
            # Only applicable to MIPI Lane Tx
            tx_info = mipi_lane.tx_info
            if tx_info is not None and tx_info.gen_pin is not None:
                slow_clk_pin = tx_info.gen_pin.get_pin_name_by_type("SLOWCLK")
                fast_clk_pin = tx_info.gen_pin.get_pin_name_by_type("FASTCLK")

                # We don't check for error and instead find the clock pin
                # considering that can be an error since routing is run during check
                if slow_clk_pin != "" and fast_clk_pin != "":
                    user_clk_name_pair_list.append((fast_clk_pin, slow_clk_pin))

        if pll_reg is not None and user_clk_name_pair_list:
            self.find_pll_clock_pair_dependencies(pll_reg, user_clk_name_pair_list,
                                                  pll_clk_with_dependencies)

    def resolve_pll_dynamic_connection(self, mux_input_conn_map, pll_dynamic_only):
        """

        :param mux_input_conn_map: A map of the clock mux name to the list of tuple (src, dest)
                # where src is <instance>.<pin> and destination is the port of the
                # clock mux block that it connects to
        :param pll_dynamic_only: Is a map of clock instance name to the
                list of PLL clkout connection "<PLLins>.<CLKOUT#>" that
                are dynamic only
        :return:
        """
        for clkmux_name in pll_dynamic_only:
            pll_dynamic_pin_list = pll_dynamic_only[clkmux_name]

            if pll_dynamic_pin_list:
                for pll_pin_name in pll_dynamic_pin_list:
                    # Iterate through muxes that are not this one mux
                    for mux_ins in mux_input_conn_map:
                        if mux_ins != clkmux_name:
                            edge_list = mux_input_conn_map[mux_ins]

                            new_edge_list = []
                            for edge_pair in edge_list:
                                in_port, mux_pin = edge_pair

                                if pll_pin_name != in_port:
                                    new_edge_list.append(edge_pair)
                                else:
                                    self.logger.debug("Removing dynamic only {}:{} from {}:{}".format(
                                        clkmux_name, pll_pin_name, mux_ins, in_port))

                            # Override with the reduced list
                            mux_input_conn_map[mux_ins] = new_edge_list

    def update_clkmux_with_removing_input_connecting_to_core(self, design, mux_input_conn_map):
        """
        Update the mux input connection map by removing the top level connection for
        clocks that should not be routed through the clkmux to core.
        This is for exceptional cases that is known.

        :param mux_input_conn_map:A map of the clock mux name to the list of tuple (src, dest)
                # where src is <instance>.<pin> and destination is the port of the
                # clock mux block that it connects to
        """
        # Nothing to do for Ti60
        return False

    def routing_on_all_clkmux_successful(self):
        clkmux_map = self.get_clock_mux_map()

        for clkmux_obj in clkmux_map.values():
            clkmux_obj.routing_is_successful()

    def load_device_clkmux_ins(self, device_db, design_db):
        '''
        Populate all the clkmux instance with the necessary information
        that needs to be loaded at the beginning (design load/build).

        PRE-REQUISITE: The clockmux instance has already been created

        The passed device db will overwrite the registry's device_db (if not None).


        :param device_db:
        :return:
        '''
        if device_db is None and self.device_db is None:
            return
        elif device_db is not None:
            self.device_db = device_db

        # We operate based on the self.device_db

        # Iterate through each clockmux
        clkmux_ins_list = self.get_all_clock_mux()

        for clkmux_ins in clkmux_ins_list:
            clkmux_ins.load_device_instance_info(self.device_db, design_db)

    def create_clock_mux(self, name, apply_default=True):

        # The name is equivalent to the device clkmux instance name
        clk_mux = ClockMuxAdvance(name)
        clk_mux.block_def = name
        self._register_new_instance(clk_mux)

        self._dev2inst_map[name] = clk_mux

        return clk_mux

    def create_instance(self, name, apply_default=True, auto_pin=False):
        """
        Create and register a lvds instance. Default is lvds tx.

        :param auto_pin:
        :param name: Instance name has to be exactly the same as the device instance name
        :param apply_default: True to apply default setting as per spec
        :return: A lvds instance
        :raise: db_item.CreateException
        """
        with self._write_lock:
            clkmux_ins = self.create_clock_mux(name, apply_default)

            return clkmux_ins

    def is_device_valid(self, device_def):
        """
        Override
        """

        # Doing import at top level cause circular import problem
        import device.db_interface as devdb_int
        dbi = devdb_int.DeviceDBService(self.device_db)
        svc = dbi.get_block_service(self.device_db.get_clkmux_type())

        dev_list = svc.get_usable_instance_names()
        if device_def not in dev_list:
            return False

        return True

    def update_regional_buffer_info(self, design_db, clkmux_name=""):
        if clkmux_name != "":
            clkmux_obj = self.get_clock_mux_by_name(clkmux_name)

            if clkmux_obj is not None:
                clkmux_obj.update_regional_buffer_info(design_db)
        else:

            clkmux_ins_list = self.get_all_clock_mux()

            for clkmux_ins in clkmux_ins_list:
                clkmux_ins.update_regional_buffer_info(design_db)

if __name__ == "__main__":
    pass
