from __future__ import annotations

from typing import TYPE_CHECKING, Dict, Set
from prettytable import PrettyTable

from common_device.writer import Summary
from design.db import PeriDesign
from design.db_item import GenericParamService
from device.clocks import GlobalClocks
from device.excp import ConfigurationInvalidException
from tx375_device.soc.design import SOC
from tx375_device.soc.design_param_info import SOCParamId
from tx375_device.soc.device_service import SOCDeviceService
from util.excp import pt_assert

from util.gen_util import natural_sort_key_for_list

if TYPE_CHECKING:
    from typing import TextIO, Dict, List
    from design.db import PeriDesign
    from device.clocks import GlobalClocks

class SOCSummaryWriter(Summary):

    def __init__(self, device_db, name, section: int):
        super().__init__(device_db, name)
        self.__section = section

    @property
    def display_block_name(self) -> str:
        return "Quad-Core Risc-V Processor"

    def generate_summary_header(self, outfile: TextIO):
        outfile.write(f"\n---------- {self.__section}. {self.display_block_name} Usage Summary (begin) ----------\n")

    def generate_summary_footer(self, outfile: TextIO):
        outfile.write(f"\n---------- {self.display_block_name} Usage Summary (end) ----------\n")

    def generate_no_configured_instance_statement(self, outfile: TextIO):
        outfile.write(f"\nNo {self.display_block_name} was configured\n")

    def generate_inst_resource_table(self, design_insts: List[SOC]) -> PrettyTable:
        table = PrettyTable([
            "Instance Name", "Resource", "Clock Region"
        ])
        for inst in sorted(design_insts, key=lambda i: natural_sort_key_for_list(i.name)):
            clk_region = self.get_device().get_instance_clock_region(inst.block_def)
            row_list = [
                inst.name,
                inst.block_def,
                clk_region
            ]

            table.add_row(row_list)

        return table

    def generate_inst_detail(self, outfile: TextIO, design: PeriDesign, blk_service: SOCDeviceService, design_inst: SOC):
        param_info = design_inst.param_info
        param_group = design_inst.param_group

        def get_disp_name(prop_id: SOCParamId) -> str:
            prop_info = param_info.get_prop_info(prop_id)
            return prop_info.disp_name

        def write_details(outfile: TextIO, details: List[str, str]):
            # helper function to write data to file
            for disp_name, value in details:
                outfile.write('{:43s}: {}\n'.format(disp_name, value))

        clk_details: List[str, str] = []
        param_svc = GenericParamService(param_group, param_info)
        write_details(outfile, [
            ("Instance Name", design_inst.name),
            ("Resource", design_inst.block_def),
        ])

        # Sys clk frequency
        sys_clk_info = design_inst.find_sys_clock_info(blk_service, design.pll_reg)
        assert sys_clk_info is not None
        pll_def, pll_inst_name, pll_clk_pin_name, _ = sys_clk_info

        clk_details.append(
            ("System Clock Resource", pll_def)
        )
        clk_details.append(
            ("System Clock Instance", pll_inst_name)
        )
        clk_details.append(
            ("System Clock", pll_clk_pin_name)
        )
        sys_clk_freq = design.pll_reg.get_output_clock_frequency(pll_inst_name, pll_clk_pin_name)
        clk_details.append(
            ("System Clock Frequency", f'{sys_clk_freq} MHz')
        )

        # Mem clk frequency
        mem_clk_info = design_inst.find_mem_clock_info(blk_service, design.pll_reg)
        assert mem_clk_info is not None
        pll_def, pll_inst_name, pll_clk_pin_name, _ = mem_clk_info

        clk_details.append(
            ("Memory Clock Resource", pll_def)
        )
        clk_details.append(
            ("Memory Clock Instance", pll_inst_name)
        )
        clk_details.append(
            ("Memory Clock", pll_clk_pin_name)
        )

        mem_clk_freq = design.pll_reg.get_output_clock_frequency(pll_inst_name, pll_clk_pin_name)
        clk_details.append(
            ("Memory Clock Frequency", f'{mem_clk_freq} MHz')
        )
        write_details(outfile, clk_details)
        outfile.write("\n")

        other_details: List[str, str] = []
        other_details.append(
            (get_disp_name(SOCParamId.PIPELINE_SOC_AXI_MEM_INTERFACE_EN), param_svc.get_param_value(SOCParamId.PIPELINE_SOC_AXI_MEM_INTERFACE_EN))
        )
        other_details.append(
            (get_disp_name(SOCParamId.AXI_MASTER_EN), param_svc.get_param_value(SOCParamId.AXI_MASTER_EN))
        )
        other_details.append(
            (get_disp_name(SOCParamId.CUSTOM_INSTRUCTION_0_EN), param_svc.get_param_value(SOCParamId.CUSTOM_INSTRUCTION_0_EN))
        )
        other_details.append(
            (get_disp_name(SOCParamId.CUSTOM_INSTRUCTION_1_EN), param_svc.get_param_value(SOCParamId.CUSTOM_INSTRUCTION_1_EN))
        )
        other_details.append(
            (get_disp_name(SOCParamId.CUSTOM_INSTRUCTION_2_EN), param_svc.get_param_value(SOCParamId.CUSTOM_INSTRUCTION_2_EN))
        )
        other_details.append(
            (get_disp_name(SOCParamId.CUSTOM_INSTRUCTION_3_EN), param_svc.get_param_value(SOCParamId.CUSTOM_INSTRUCTION_3_EN))
        )
        other_details.append(
            ("JTAG Interface", param_svc.get_param_value(SOCParamId.JTAG_TYPE))
        )
        for disp_name, value in other_details:
            outfile.write('{:50s}: {}\n'.format(disp_name, value))

    def generate_summary(self, design: PeriDesign, outfile: TextIO, ins_to_blk_map: Dict[str, str], blk_service: SOCDeviceService):
        self.generate_summary_header(outfile)

        all_design_insts = blk_service.get_all_instances(design)
        if all_design_insts:
            insts_with_assignment = list(filter(lambda inst: inst.block_def in ins_to_blk_map, all_design_insts))
            pt_assert(len(insts_with_assignment) > 0, f'Instance of {self._name} has invalid design configuration', ConfigurationInvalidException)
            pt_assert(all([inst is not None for inst in insts_with_assignment]), f'Instance of {self._name} has invalid design configuration', ConfigurationInvalidException)

            table = self.generate_inst_resource_table(insts_with_assignment)
            outfile.write(f"\n{table.get_string()}\n")

            for index, inst in enumerate(sorted(insts_with_assignment, key=lambda i: natural_sort_key_for_list(i.name))):
                if len(insts_with_assignment) > 1:
                    outfile.write(f"\n***** {self.display_block_name} {index} *****\n")
                else:
                    outfile.write("\n")

                self.generate_inst_detail(outfile, design, blk_service, inst)
        else:
            self.generate_no_configured_instance_statement(outfile)

        self.generate_summary_footer(outfile)

    def generate_global_summary(self, design: PeriDesign, db_global: GlobalClocks, ins2func_map: Dict[str, Set[str]], ins_to_block_map: Dict[str, str], table: PrettyTable):
        pass
