from __future__ import annotations
from typing import TYPE_CHECKING

from common_gui.base_viewer import BaseViewer, BaseViewerModel
from design.db import PeriDesign
from design.db_item import GenericParamService
from device.db_interface import DeviceDBService
from tx375_device.soc.design import SOC
from tx375_device.soc.design_param_info import SOCParamId


if TYPE_CHECKING:
    from tx375_device.soc.design import SOCRegistry


class SOCViewerModel(BaseViewerModel):
    """
    QT Model for the SOC Properties table used by SOCViewer
    """

    def load(self, design: PeriDesign, block_name: str):
        """
        Load design instance info from design to this Qt Model

        :param design: A Periphery Design
        :type design: PeriDesign
        :param block_name: Target design instance name to be loaded
        :type block_name: str
        """

        self.design = design
        block_reg: SOCRegistry = self.design.get_block_reg(PeriDesign.BlockType.soc)
        assert block_reg is not None

        block = block_reg.get_inst_by_name(block_name)
        assert block is not None
        assert isinstance(block, SOC)

        param_svc = GenericParamService(block.param_group, block.param_info)
        def add_param_row(param_id: SOCParamId):
            self.add_generic_param_by_type(block.param_group, block.param_info, include_type=[param_id])

        def add_pin_rows_by_category(category: str):
            pins = block.get_pin_by_class(category)
            for p in pins:
                disp_name = block.get_pin_property_name(p.type_name)
                self.add_row_item(disp_name, p.name)

        def add_sys_clk_info():
            dev_db_service = DeviceDBService(self.design.device_db)
            dev_service = dev_db_service.get_block_service(DeviceDBService.BlockType.SOC)
            info = block.find_sys_clock_info(dev_service, self.design.pll_reg)
            if info is None:
                return
            self.add_row_item("System Clock Resource", info[0])
            self.add_row_item("System Clock Instance", info[1])
            self.add_row_item("System Clock Pin Name", info[2])

        def add_mem_clk_info():
            dev_db_service = DeviceDBService(self.design.device_db)
            dev_service = dev_db_service.get_block_service(DeviceDBService.BlockType.SOC)
            info = block.find_mem_clock_info(dev_service, self.design.pll_reg)
            if info is None:
                return
            self.add_row_item("Memory Clock Resource", info[0])
            self.add_row_item("Memory Clock Instance", info[1])
            self.add_row_item("Memory Clock Pin Name", info[2])

        pin_section_color = BaseViewerModel.get_secondary_highlight_colour()

        self.add_row_item("Instance Name", block.name)
        self.add_row_item("SOC Resource", block.get_device())
        assert block.param_group is not None, f"{block.name} {id(block)}"
        assert block.param_info is not None
        add_param_row(SOCParamId.OCR_FILE_PATH)

        self.add_row_item("Clock / Control", "", highlight=True)
        add_param_row(SOCParamId.SYS_CLK_SOURCE)
        add_sys_clk_info()
        add_param_row(SOCParamId.MEM_CLK_SOURCE)
        add_mem_clk_info()
        add_param_row(SOCParamId.PIPELINE_SOC_AXI_MEM_INTERFACE_EN)
        add_pin_rows_by_category('soc:clk_control')

        self.add_row_item('User AXI Master', '', highlight=True)
        add_param_row(SOCParamId.AXI_MASTER_EN)
        value = param_svc.get_param_value(SOCParamId.AXI_MASTER_EN)
        if value:
            add_pin_rows_by_category('soc:axi_master:ctrl')
            self.add_row_item('Read Address Channel', '', highlight=True, highlight_colour=pin_section_color)
            add_pin_rows_by_category('soc:axi_master:read_addr')
            self.add_row_item('Read Address Channel', '', highlight=True, highlight_colour=pin_section_color)
            add_pin_rows_by_category('soc:axi_master:write_addr')
            self.add_row_item('Read Address Channel', '', highlight=True, highlight_colour=pin_section_color)
            add_pin_rows_by_category('soc:axi_master:write_resp')
            self.add_row_item('Read Address Channel', '', highlight=True, highlight_colour=pin_section_color)
            add_pin_rows_by_category('soc:axi_master:read_data')
            self.add_row_item('Read Address Channel', '', highlight=True, highlight_colour=pin_section_color)
            add_pin_rows_by_category('soc:axi_master:write_data')

        self.add_row_item('User AXI Slave', '', highlight=True)
        add_pin_rows_by_category('soc:axi_slave:irq')
        self.add_row_item('Read Address Channel', '', highlight=True, highlight_colour=pin_section_color)
        add_pin_rows_by_category('soc:axi_slave:read_addr')
        self.add_row_item('Read Address Channel', '', highlight=True, highlight_colour=pin_section_color)
        add_pin_rows_by_category('soc:axi_slave:write_addr')
        self.add_row_item('Read Address Channel', '', highlight=True, highlight_colour=pin_section_color)
        add_pin_rows_by_category('soc:axi_slave:write_resp')
        self.add_row_item('Read Address Channel', '', highlight=True, highlight_colour=pin_section_color)
        add_pin_rows_by_category('soc:axi_slave:read_data')
        self.add_row_item('Read Address Channel', '', highlight=True, highlight_colour=pin_section_color)
        add_pin_rows_by_category('soc:axi_slave:write_data')

        show_cfu = any([
            param_svc.get_param_value(SOCParamId.CUSTOM_INSTRUCTION_0_EN),
            param_svc.get_param_value(SOCParamId.CUSTOM_INSTRUCTION_1_EN),
            param_svc.get_param_value(SOCParamId.CUSTOM_INSTRUCTION_2_EN),
            param_svc.get_param_value(SOCParamId.CUSTOM_INSTRUCTION_3_EN)
        ])
        if show_cfu:
            self.add_row_item('Custom Instruction', '', highlight=True)
            add_pin_rows_by_category('soc:custom_instr')
            value = param_svc.get_param_value(SOCParamId.CUSTOM_INSTRUCTION_0_EN)
            if value is True:
                self.add_row_item('Custom Instruction Interface 0', '', highlight=True, highlight_colour=pin_section_color)
                add_pin_rows_by_category('soc:custom_instr:instr_0')
            value = param_svc.get_param_value(SOCParamId.CUSTOM_INSTRUCTION_1_EN)
            if value is True:
                self.add_row_item('Custom Instruction Interface 1', '', highlight=True, highlight_colour=pin_section_color)
                add_pin_rows_by_category('soc:custom_instr:instr_1')
            value = param_svc.get_param_value(SOCParamId.CUSTOM_INSTRUCTION_2_EN)
            if value is True:
                self.add_row_item('Custom Instruction Interface 2', '', highlight=True, highlight_colour=pin_section_color)
                add_pin_rows_by_category('soc:custom_instr:instr_2')
            value = param_svc.get_param_value(SOCParamId.CUSTOM_INSTRUCTION_3_EN)
            if value is True:
                self.add_row_item('Custom Instruction Interface 3', '', highlight=True, highlight_colour=pin_section_color)
                add_pin_rows_by_category('soc:custom_instr:instr_3')

        self.add_row_item('External Interrupt', '', highlight=True)
        add_pin_rows_by_category('soc:user_irq')

        self.add_row_item('Debug', '', highlight=True)
        add_param_row(SOCParamId.JTAG_TYPE)
        value = param_svc.get_param_value(SOCParamId.JTAG_TYPE)
        if value == 'FPGA':
            add_pin_rows_by_category('soc:fpga_jtag')
        elif value == 'CPU':
            add_pin_rows_by_category('soc:cpu_jtag')


class SOCViewer(BaseViewer):
    """
    Display SOC properties in table format
    """

    def build(self, block_name: str, design: PeriDesign):
        self.block_name = block_name
        self.design = design

        block_reg: SOCRegistry = self.design.get_block_reg(PeriDesign.BlockType.soc)
        assert block_reg is not None

        block = block_reg.get_inst_by_name(block_name)
        # Remarks: This function called even block is not created
        if block is None:
            return

        self.model = SOCViewerModel()
        self.model.load(design, block_name)

        self.tv_viewer.setModel(self.model)
        self.formatter.refresh_table_view_format(self.tv_viewer)
