from __future__ import annotations
from typing import TYPE_CHECKING, Dict, Optional

from PyQt5.QtCore import pyqtSlot, Qt
from PyQt5.QtWidgets import QWidget, QVBoxLayout, QSpacerItem, QSizePolicy, QFileDialog, QLabel
from common_gui.builder.base import ChoiceParamWidget, FileDialog, FilePathParamWidget
from common_gui.builder.primitive_widget import TextWgtSpec

from design.db import PeriDesign
from common_gui.base_config import BaseConfig
from design.db_item import ParamGroupMonitor
from tx375_device.soc.dep_graph import SOCDependencyGraph
from tx375_device.soc.gui.graph_observer import SOCGraphObserver
from tx375_device.soc.gui.presenter import SOCPresenter
import util.gui_util as gui_util
from common_gui.builder.util import build_param_widgets, build_pin_widgets_by_category

from tx375_device.soc.design_param_info import SOCParamId

if TYPE_CHECKING:
    from common_gui.builder.base import WidgetGroup
    from tx375_device.soc.design import SOCRegistry, SOC
    from tx375_device.soc.gui.main_window import SOCWindow


class SOCConfig(BaseConfig):

    def __init__(self, parent: SOCWindow):
        super().__init__(parent)

        self.le_name = parent.le_name
        self.cb_resource = parent.cb_resource

        self.vl_sys_clk_param = parent.vl_sys_clk_param
        self.lbl_sys_clk_res = parent.lbl_sys_clk_res
        self.lbl_sys_clk_inst = parent.lbl_sys_clk_inst
        self.lbl_sys_clk_pin = parent.lbl_sys_clk_pin

        self.vl_mem_clk_param = parent.vl_mem_clk_param
        self.lbl_mem_clk_res = parent.lbl_mem_clk_res
        self.lbl_mem_clk_inst = parent.lbl_mem_clk_inst
        self.lbl_mem_clk_pin = parent.lbl_mem_clk_pin

        self.vl_base_param = parent.vl_base_param
        self.vl_clk_param = parent.vl_clk_param
        self.vl_clk_pin = parent.vl_clk_pin
        self.vl_axi_m_param = parent.vl_axi_m_param
        self.vl_axi_m_pin = parent.vl_axi_m_pin
        self.tab_axi_m_pin = parent.tab_axi_m_pin
        self.vl_axi_s_param = parent.vl_axi_s_param
        self.vl_axi_s_pin = parent.vl_axi_s_pin
        self.tab_axi_s_pin = parent.tab_axi_s_pin
        self.vl_jtag_param = parent.vl_jtag_param
        self.vl_jtag_pin = parent.vl_jtag_pin
        self.vl_custom_instr_param = parent.vl_custom_instr_param
        self.vl_custom_instr_pin = parent.vl_custom_instr_pin
        self.tab_custom_instr_pin = parent.tab_custom_instr_pin

        self.vl_irq_pin = parent.vl_irq_pin

        self.param_widget_map: Dict[str, WidgetGroup] = {}
        self.pin_widget_map: Dict[str, WidgetGroup] = {}
        self.dep_graph: Optional[SOCDependencyGraph] = None
        self.graph_observer = SOCGraphObserver(self)

    def build(self, block_name: str, design: PeriDesign):
        self.design = design
        self.disable_mouse_wheel_scroll()

        block_reg: SOCRegistry = self.design.get_block_reg(PeriDesign.BlockType.soc)
        assert block_reg is not None

        soc = block_reg.get_inst_by_name(block_name)
        if soc is None:
            return

        if self.is_monitor is True:
            self.stop_monitor_changes()

        self.block_reg = block_reg
        self.block_inst = soc

        # Rebuilt UI
        gui_util.remove_layout_children(self.vl_base_param)
        gui_util.remove_layout_children(self.vl_clk_param)
        gui_util.remove_layout_children(self.vl_clk_pin)
        gui_util.remove_layout_children(self.vl_sys_clk_param)
        gui_util.remove_layout_children(self.vl_mem_clk_param)
        gui_util.remove_layout_children(self.vl_axi_m_param)
        gui_util.remove_layout_children(self.vl_axi_m_pin)
        gui_util.remove_layout_children(self.vl_axi_s_param)
        gui_util.remove_layout_children(self.vl_axi_s_pin)
        gui_util.remove_layout_children(self.vl_jtag_param)
        gui_util.remove_layout_children(self.vl_jtag_pin)
        gui_util.remove_layout_children(self.vl_custom_instr_param)
        gui_util.remove_layout_children(self.vl_custom_instr_pin)
        gui_util.remove_layout_children(self.vl_irq_pin)
        gui_util.remove_all_tabs(self.tab_axi_m_pin)
        gui_util.remove_all_tabs(self.tab_axi_s_pin)
        gui_util.remove_all_tabs(self.tab_custom_instr_pin)

        self.pin_widget_map.clear()
        self.param_widget_map.clear()

        # widget_builder = PrimitiveWidgetBuilder()
        pinfo = self.block_inst.param_info
        self.param_widget_map.update(
            build_param_widgets(self.vl_sys_clk_param, param_defs=[
                pinfo.get_prop_info(SOCParamId.SYS_CLK_SOURCE)
            ])
        )
        widget = self.param_widget_map[SOCParamId.SYS_CLK_SOURCE.value]
        info = pinfo.get_prop_info(SOCParamId.SYS_CLK_SOURCE)
        storage_options = [str(setting) for setting in info.valid_setting.get_options()]
        display_options = storage_options.copy()
        display_options[0] = "None"
        for i in range(widget.editor_widget.count()):
            widget.editor_widget.setItemData(i, display_options[i], Qt.DisplayRole)
            widget.editor_widget.setItemData(i, storage_options[i], Qt.UserRole)

        self.param_widget_map.update(
            build_param_widgets(self.vl_mem_clk_param, param_defs=[
                pinfo.get_prop_info(SOCParamId.MEM_CLK_SOURCE)
            ])
        )
        widget = self.param_widget_map[SOCParamId.MEM_CLK_SOURCE.value]
        info = pinfo.get_prop_info(SOCParamId.MEM_CLK_SOURCE)
        storage_options = [str(setting) for setting in info.valid_setting.get_options()]
        display_options = storage_options.copy()
        display_options[0] = "None"
        for i in range(widget.editor_widget.count()):
            widget.editor_widget.setItemData(i, display_options[i], Qt.DisplayRole)
            widget.editor_widget.setItemData(i, storage_options[i], Qt.UserRole)

        self.param_widget_map.update(
            build_param_widgets(self.vl_axi_m_param, param_defs=[
                pinfo.get_prop_info(SOCParamId.AXI_MASTER_EN)
            ])
        )

        info = pinfo.get_prop_info(SOCParamId.OCR_FILE_PATH)
        dialog = FileDialog(self.design.location)
        dialog.use_relative_path(True)
        dialog.setWindowTitle("Select On-Chip-Ram Configration File")
        dialog.setDirectory(self.design.location)
        dialog.setNameFilters([
            "Intel Hex Files (*.hex)",
            "Binary Files (*.bin)"
        ])
        dialog.setFileMode(QFileDialog.ExistingFile)
        widget = FilePathParamWidget(dialog)
        widget.set_label_text(info.disp_name)
        widget.label_widget.hide_icon()
        self.vl_base_param.addWidget(widget)
        widget.editor_widget.setObjectName(TextWgtSpec.widget_prefix + info.name)
        self.param_widget_map[SOCParamId.OCR_FILE_PATH.value] = widget

        custom_label = QLabel("AXI Interface Pipeline")
        self.vl_clk_pin.addWidget(custom_label)
        self.param_widget_map.update(
            build_param_widgets(
                self.vl_clk_pin, param_defs= pinfo.get_prop_info_by_id_list(
                    [SOCParamId.PIPELINE_SOC_AXI_MEM_INTERFACE_EN])
                )
            )

        # self.param_widget_map.update(
        #     build_param_widgets(self.vl_axi_s_param, param_defs=[
        #         pinfo.get_prop_info(SOCParamId.AXI_SLAVE_EN)
        #     ])
        # )

        self.param_widget_map.update(
            build_param_widgets(self.vl_custom_instr_param, param_defs=[
                pinfo.get_prop_info(SOCParamId.CUSTOM_INSTRUCTION_0_EN),
                pinfo.get_prop_info(SOCParamId.CUSTOM_INSTRUCTION_1_EN),
                pinfo.get_prop_info(SOCParamId.CUSTOM_INSTRUCTION_2_EN),
                pinfo.get_prop_info(SOCParamId.CUSTOM_INSTRUCTION_3_EN),
            ])
        )

        self.param_widget_map.update(
            build_param_widgets(self.vl_jtag_param, param_defs=[
                pinfo.get_prop_info(SOCParamId.JTAG_TYPE),
            ])
        )

        self.pin_widget_map.update(
            build_pin_widgets_by_category(self.block_inst,
                                          self.vl_clk_pin,
                                          category='soc:clk_control')
        )

        self.pin_widget_map.update(
            build_pin_widgets_by_category(self.block_inst,
                                          self.vl_jtag_pin,
                                          category='soc:cpu_jtag')
        )

        self.pin_widget_map.update(
            build_pin_widgets_by_category(self.block_inst,
                                          self.vl_jtag_pin,
                                          category='soc:fpga_jtag')
        )

        self.pin_widget_map.update(
            build_pin_widgets_by_category(self.block_inst,
                                          self.vl_custom_instr_pin,
                                          category='soc:custom_instr')
        )

        self.pin_widget_map.update(
            build_pin_widgets_by_category(self.block_inst,
                                          self.vl_axi_m_pin,
                                          category='soc:axi_master:ctrl')
        )

        self.pin_widget_map.update(
            build_pin_widgets_by_category(self.block_inst,
                                          self.vl_axi_s_pin,
                                          category='soc:axi_slave:irq')
        )

        axi_m_tab_svc = gui_util.TabWidgetService(self.tab_axi_m_pin)

        def build_tab_for_pins(category):
            tab = QWidget()
            vl = QVBoxLayout(tab)
            self.pin_widget_map.update(
                build_pin_widgets_by_category(self.block_inst,
                                              vl,
                                              category=category)
            )
            vl.addItem(QSpacerItem(20, 10, QSizePolicy.Maximum, QSizePolicy.Expanding))
            return tab
        axi_m_tab_svc.add_tab(
            build_tab_for_pins('soc:axi_master:read_addr'),
            'Read Address Channel')
        axi_m_tab_svc.add_tab(
            build_tab_for_pins('soc:axi_master:write_addr'),
            'Write Address Channel')
        axi_m_tab_svc.add_tab(
            build_tab_for_pins('soc:axi_master:write_resp'),
            'Write Response Channel')
        axi_m_tab_svc.add_tab(
            build_tab_for_pins('soc:axi_master:read_data'),
            'Read Data Channel')
        axi_m_tab_svc.add_tab(
            build_tab_for_pins('soc:axi_master:write_data'),
            'Write Data Channel')

        axi_s_tab_svc = gui_util.TabWidgetService(self.tab_axi_s_pin)
        axi_s_tab_svc.add_tab(
            build_tab_for_pins('soc:axi_slave:read_addr'),
            'Read Address Channel')
        axi_s_tab_svc.add_tab(
            build_tab_for_pins('soc:axi_slave:write_addr'),
            'Write Address Channel')
        axi_s_tab_svc.add_tab(
            build_tab_for_pins('soc:axi_slave:write_resp'),
            'Write Response Channel')
        axi_s_tab_svc.add_tab(
            build_tab_for_pins('soc:axi_slave:read_data'),
            'Read Data Channel')
        axi_s_tab_svc.add_tab(
            build_tab_for_pins('soc:axi_slave:write_data'),
            'Write Data Channel')

        custom_instr_tab_svc = gui_util.TabWidgetService(self.tab_custom_instr_pin)
        custom_instr_tab_svc.add_tab(
            build_tab_for_pins('soc:custom_instr:instr_0'),
            'Interface 0')

        custom_instr_tab_svc.add_tab(
            build_tab_for_pins('soc:custom_instr:instr_1'),
            'Interface 1')

        custom_instr_tab_svc.add_tab(
            build_tab_for_pins('soc:custom_instr:instr_2'),
            'Interface 2')

        custom_instr_tab_svc.add_tab(
            build_tab_for_pins('soc:custom_instr:instr_3'),
            'Interface 3')

        self.pin_widget_map.update(
            build_pin_widgets_by_category(self.block_inst,
                                          self.vl_irq_pin,
                                          category='soc:user_irq')
        )

        self.dep_graph = SOCDependencyGraph(self.block_inst.param_info, self.block_inst.gen_pin)
        if self.block_inst.netlist_config:
            for param, _ in self.block_inst.netlist_config.parameters:
                if param.endswith('_PIN'):
                    pin_type_name = param[:-4]
                    self.dep_graph.set_pin_attributes(pin_type_name, is_available=True, is_editable=False)
                else:
                    param_name = param
                    self.dep_graph.set_param_attributes(param_name, is_available=True, is_editable=False)
        self.param_monitor = ParamGroupMonitor(self.dep_graph)
        # Synchronize the dep graph with block data
        for param in self.block_inst.param_group.get_all_param():
            self.param_monitor.on_param_changed(self.block_inst.param_group, param.name)

        self.presenter = SOCPresenter(self)
        self.presenter.setup_design(self.design)
        self.presenter.load_base_property(self.block_inst, self.le_name, self.cb_resource)
        self.presenter.load_generic_param_property(self.block_inst, self.param_widget_map)
        self.presenter.load_generic_pin_property(self.block_inst, self.pin_widget_map)

        self.presenter.load_sys_clk_property(self.block_inst, self.lbl_sys_clk_res,
                                             self.lbl_sys_clk_inst, self.lbl_sys_clk_pin)

        self.presenter.load_mem_clk_property(self.block_inst, self.lbl_mem_clk_res,
                                             self.lbl_mem_clk_inst, self.lbl_mem_clk_pin)

        self.graph_observer.on_graph_updated(self.dep_graph, self.dep_graph.get_all_pin_nodes() + self.dep_graph.get_all_param_nodes())

        gui_util.set_child_combo_spinbox_filter(self)
        self.monitor_changes()

    def monitor_changes(self):
        self.le_name.editingFinished.connect(self.on_name_changed)
        self.cb_resource.currentIndexChanged.connect(self.on_device_changed)
        self.sig_resource_updated.connect(self.on_refresh_clk_info)
        self.monitor_param_widget(self.param_widget_map, True, self.on_param_changed)
        self.monitor_pin_widget(self.pin_widget_map, True, self.on_pin_changed)
        self.monitor_clk_pin_inversion_widget(self.pin_widget_map, True, self.on_clkpin_invert_changed)
        self.dep_graph.register_observer(self.graph_observer)
        self.block_inst.param_group.register_param_observer(self.param_monitor)
        self.is_monitor = True

    def stop_monitor_changes(self):
        self.le_name.editingFinished.disconnect(self.on_name_changed)
        self.cb_resource.currentIndexChanged.disconnect(self.on_device_changed)
        self.sig_resource_updated.disconnect(self.on_refresh_clk_info)
        self.monitor_param_widget(self.param_widget_map, False, self.on_param_changed)
        self.monitor_pin_widget(self.pin_widget_map, False, self.on_pin_changed)
        self.monitor_clk_pin_inversion_widget(self.pin_widget_map, False, self.on_clkpin_invert_changed)
        self.dep_graph.unregister_observer(self.graph_observer)
        self.block_inst.param_group.unregister_param_observer(self.param_monitor)
        self.is_monitor = False

    @pyqtSlot()
    def on_refresh_clk_info(self):
        """
        When resource updated, refresh the sys clk / mem clk info
        """
        self.presenter.load_sys_clk_property(self.block_inst, self.lbl_sys_clk_res,
                                             self.lbl_sys_clk_inst, self.lbl_sys_clk_pin)

        self.presenter.load_mem_clk_property(self.block_inst, self.lbl_mem_clk_res,
                                             self.lbl_mem_clk_inst, self.lbl_mem_clk_pin)

    @pyqtSlot()
    def on_param_changed(self):
        assert self.block_inst is not None

        widget = self.sender()
        # print(f'on_param_changed {widget}, name = {widget.objectName()}')
        widget_name = widget.objectName()
        if SOCParamId.SYS_CLK_SOURCE.value in widget_name:
            user_data = True
        elif SOCParamId.MEM_CLK_SOURCE.value in widget_name:
            user_data = True
        else:
            user_data = False

        if self.presenter.save_generic_param_property(self.block_inst, widget, user_data=user_data):
            self.sig_prop_updated.emit()

        if SOCParamId.SYS_CLK_SOURCE.value in widget_name:
            # print("Reload sys")
            self.presenter.load_sys_clk_property(self.block_inst, self.lbl_sys_clk_res,
                                                 self.lbl_sys_clk_inst, self.lbl_sys_clk_pin)
        elif SOCParamId.MEM_CLK_SOURCE.value in widget_name:
            # print("Reload mem")
            self.presenter.load_mem_clk_property(self.block_inst, self.lbl_mem_clk_res,
                                                 self.lbl_mem_clk_inst, self.lbl_mem_clk_pin)

    @pyqtSlot()
    def on_pin_changed(self):
        assert self.block_inst is not None

        widget = self.sender()
        if self.presenter.save_generic_pin_property(self.block_inst, widget):
            self.sig_prop_updated.emit()

    @pyqtSlot(int)
    def on_clkpin_invert_changed(self, state):
        assert self.block_inst is not None
        assert self.presenter is not None

        widget = self.sender()

        if self.presenter.save_generic_pin_clk_invert_property(self.block_inst, widget):
            self.sig_prop_updated.emit()
