from enum import Enum, unique
from typing import List
from common_device.property import PropUsageType, PropertyMetaData, OptionValidator
from design.db_item import GenericParam

# Alias for enum inside class
PropData = PropertyMetaData.PropData
DataType = GenericParam.DataType


@unique
class SOCParamId(Enum):
    NAME = "NAME"
    RESOURCE = "RESOURCE"
    AXI_MASTER_EN = "AXI_MASTER_EN"
    AXI_SLAVE_EN = "AXI_SLAVE_EN"
    CUSTOM_INSTRUCTION_0_EN = "CUSTOM_INSTRUCTION_0_EN"
    CUSTOM_INSTRUCTION_1_EN = "CUSTOM_INSTRUCTION_1_EN"
    CUSTOM_INSTRUCTION_2_EN = "CUSTOM_INSTRUCTION_2_EN"
    CUSTOM_INSTRUCTION_3_EN = "CUSTOM_INSTRUCTION_3_EN"

    SYS_CLK_SOURCE = "SYS_CLK_SOURCE"
    MEM_CLK_SOURCE = "MEM_CLK_SOURCE"

    JTAG_TYPE = "JTAG_TYPE"
    OCR_FILE_PATH = "OCR_FILE_PATH"

    PIPELINE_SOC_AXI_MEM_INTERFACE_EN = "PIPELINE_SOC_AXI_MEM_INTERFACE_EN"

class SOCParamInfo(PropertyMetaData):
    """
    Store the property information for Hard SOC block
    """

    def __init__(self):
        super().__init__()

        prop_datas: List[PropData] = [
            PropData(id=SOCParamId.NAME,
                     name=SOCParamId.NAME.value,
                     data_type=DataType.dstr,
                     default='',
                     disp_name='Name',
                     usage_type=PropUsageType.GUI),
            PropData(id=SOCParamId.RESOURCE,
                     name=SOCParamId.RESOURCE.value,
                     data_type=DataType.dstr,
                     default='',
                     disp_name='Resource',
                     usage_type=PropUsageType.PNR),
            PropData(id=SOCParamId.AXI_MASTER_EN,
                     name=SOCParamId.AXI_MASTER_EN.value,
                     data_type=DataType.dbool,
                     default=False,
                     disp_name='Enable AXI Master Interface',
                     usage_type=PropUsageType.SYN,
                     category='soc:axi_master:ctrl'),
            PropData(id=SOCParamId.AXI_SLAVE_EN,
                     name=SOCParamId.AXI_SLAVE_EN.value,
                     data_type=DataType.dbool,
                     default=False,
                     disp_name='Enable AXI Slave Interface',
                     usage_type=PropUsageType.SYN,
                     category='soc:axi_slave:irq'),
            PropData(id=SOCParamId.CUSTOM_INSTRUCTION_0_EN,
                     name=SOCParamId.CUSTOM_INSTRUCTION_0_EN.value,
                     data_type=DataType.dbool,
                     default=False,
                     disp_name='Enable Custom Instruction Interface 0',
                     usage_type=PropUsageType.SYN,
                     category='soc:custom_instr:instr_0'),
            PropData(id=SOCParamId.CUSTOM_INSTRUCTION_1_EN,
                     name=SOCParamId.CUSTOM_INSTRUCTION_1_EN.value,
                     data_type=DataType.dbool,
                     default=False,
                     disp_name='Enable Custom Instruction Interface 1',
                     usage_type=PropUsageType.SYN,
                     category='soc:custom_instr:instr_1'),
            PropData(id=SOCParamId.CUSTOM_INSTRUCTION_2_EN,
                     name=SOCParamId.CUSTOM_INSTRUCTION_2_EN.value,
                     data_type=DataType.dbool,
                     default=False,
                     disp_name='Enable Custom Instruction Interface 2',
                     usage_type=PropUsageType.SYN,
                     category='soc:custom_instr:instr_2'),
            PropData(id=SOCParamId.CUSTOM_INSTRUCTION_3_EN,
                     name=SOCParamId.CUSTOM_INSTRUCTION_3_EN.value,
                     data_type=DataType.dbool,
                     default=False,
                     disp_name='Enable Custom Instruction Interface 3',
                     usage_type=PropUsageType.SYN,
                     category='soc:custom_instr:instr_3'),

            PropData(id=SOCParamId.SYS_CLK_SOURCE,
                     name=SOCParamId.SYS_CLK_SOURCE.value,
                     data_type=DataType.dstr,
                     default="Unassign",
                     valid_setting=OptionValidator(["Unassign", "Clock 0", "Clock 1", "Clock 2"]),
                     disp_name='System Clock Source',
                     usage_type=PropUsageType.SYN,
                     category='soc:clk_control'),
            PropData(id=SOCParamId.MEM_CLK_SOURCE,
                     name=SOCParamId.MEM_CLK_SOURCE.value,
                     data_type=DataType.dstr,
                     default="Unassign",
                     valid_setting=OptionValidator(["Unassign", "Clock 0", "Clock 1", "Clock 2"]),
                     disp_name='Memory Clock Source',
                     usage_type=PropUsageType.SYN,
                     category='soc:clk_control'),
            PropData(id=SOCParamId.PIPELINE_SOC_AXI_MEM_INTERFACE_EN,
                     name=SOCParamId.PIPELINE_SOC_AXI_MEM_INTERFACE_EN.value,
                     data_type=DataType.dbool,
                     default=False,
                     disp_name='Enable the pipeline for SoC AXI memory interface',
                     usage_type=PropUsageType.SYN,
                     category='soc:clk_control'),

            PropData(id=SOCParamId.JTAG_TYPE,
                     name=SOCParamId.JTAG_TYPE.value,
                     data_type=DataType.dstr,
                     default="DISABLE",
                     valid_setting=OptionValidator(['FPGA', 'CPU', 'DISABLE']),
                     disp_name='JTAG Interface Type',
                     usage_type=PropUsageType.SYN,
                     category='soc:debug'),
            PropData(id=SOCParamId.OCR_FILE_PATH,
                     name=SOCParamId.OCR_FILE_PATH.value,
                     data_type=DataType.dstr,
                     default="",
                     disp_name='On-Chip Ram Configuration File',
                     usage_type=PropUsageType.SYN),
        ]

        for prop_data in prop_datas:
            self.add_prop_by_data(prop_data.id, prop_data)
