from __future__ import annotations
from typing import TYPE_CHECKING, List, Tuple, Dict

from tx375_device.raw_serdes.device_service import QuadRawSerdesDeviceService
from tx375_device.raw_serdes.design import RawSerdes
from tx375_device.raw_serdes.raw_serdes_pin_dep_graph import get_tie_high_ports
from tx375_device.raw_serdes.raw_serdes_prop_id import RawSerdesConfigParamInfo

from common_device.quad.res_service import QuadResService
from common_device.quad.writer.interface_config import LaneInterfaceConfig

if TYPE_CHECKING:
    from tx375_device.common_quad.design import QuadLaneCommon
    from common_device.quad.lane_design import LaneBasedItem
    from design.db import PeriDesign
    from design.db_item import GenericPin

class LaneRawSerdesInterfaceConfig(LaneInterfaceConfig):

    def __init__(self, device_db, name):
        super().__init__(device_db, name)

        # List of tuple pair of quad index  (qx,qy) that makes up
        # one bundle of x8
        self.bundle_x8_pairs: List[Tuple[int, int]] = []
        self.quad_lane_to_clk_src_insname: Dict[Tuple[int, int], str] = {}

    def determine_bundle_x8_pairs(self, design_db):
        if design_db is not None and design_db.raw_serdes_reg is not None:
            self.bundle_x8_pairs, self.quad_lane_to_clk_src_insname = \
                 design_db.raw_serdes_reg.determine_bundle_x8_quad_pairs_and_clk_src()

    def get_device_service(self):
        return QuadRawSerdesDeviceService(self._device_db, self._name)
    
    def verify_instance_type(self, ins: LaneBasedItem):
        assert isinstance(ins, RawSerdes)
    
    def get_mode_name(self, res_name):
        """
        Return the mode name associated to the res_name
        """
        _, lane_idx = QuadResService.break_res_name(res_name)

        mode_name = ""
        if lane_idx != -1:
            mode_name = f'LN{lane_idx}_RAW_SERDES'

        return mode_name
        
    def get_common_pin_class(self):
        from tx375_device.raw_serdes.gui.builder import RawSerdesTabBuilder

        return RawSerdesTabBuilder.common_pin_categories

    def get_user_blk_name(self):
        return "PMA Direct"

    def get_lane_tie_high_ports(self) -> List[str]:
        return get_tie_high_ports()
            
    def is_input_clock_inf_name(self, inf_name) -> bool:
        if inf_name in ["RAW_SERDES_TX_CLK", "RAW_SERDES_RX_CLK"]:
            return True

        return False
    
    def get_lane_available_pins_for_connection(self, inst: LaneBasedItem):
        '''
        Depending on the lane instance, there might be some pins that is considered
        needed/available even if it's not enabled
        '''
        lane_avail_pins = inst.get_available_pins()

        # Check if the instance is of bundle mode, then we need to add in
        # the input clock pin name as weel
        bundle_mode = inst.param_group.get_param_value(
            RawSerdesConfigParamInfo.Id.ss_raw_bundle_mode_lane_NID.value)

        if bundle_mode != "x1":
            mode = inst.param_group.get_param_value(
                RawSerdesConfigParamInfo.Id.ss_raw_mode_lane_NID.value)  
            
            # Don't duplicate the port if it already exists
            if mode.find("Tx") != -1 and "RAW_SERDES_TX_CLK" not in lane_avail_pins:
                lane_avail_pins.append("RAW_SERDES_TX_CLK")

            if mode.find("Rx") != -1 and "RAW_SERDES_RX_CLK" not in lane_avail_pins:
                lane_avail_pins.append("RAW_SERDES_RX_CLK")

        return lane_avail_pins                

       
    def _get_user_pin_name_bundle_x2_x4(self, design_db: PeriDesign, 
                                        inst: LaneBasedItem,
                                        gen_pin:GenericPin,
                                        bundle_mode: str):
        user_pin_name = ""

        quad_idx, lane_idx = QuadResService.break_res_name(inst.get_device())

        if bundle_mode == 'x4':
            in_res_name = f"Q{quad_idx}_LN1"
        else:
            if lane_idx in [2, 3]:
                in_res_name = f"Q{quad_idx}_LN3"
            else:
                in_res_name = f"Q{quad_idx}_LN1"

        if design_db is not None and design_db.raw_serdes_reg is not None:
            user_pin_name = design_db.raw_serdes_reg.get_user_clock_input_pin_name(
                in_res_name, gen_pin.type_name)

        return user_pin_name
    
    def get_user_pin_name(self, design: PeriDesign, inst: LaneBasedItem, gen_pin:GenericPin):
        user_pin_name = gen_pin.name

        if inst is not None and inst.param_group is not None and\
            self.is_input_clock_inf_name(gen_pin.type_name):

            bundle_mode = inst.param_group.get_param_value(
                RawSerdesConfigParamInfo.Id.ss_raw_bundle_mode_lane_NID.value)

            mode = inst.param_group.get_param_value(
                RawSerdesConfigParamInfo.Id.ss_raw_mode_lane_NID.value)

            if bundle_mode in ['x2', 'x4']:

                if gen_pin.type_name == "RAW_SERDES_TX_CLK" or \
                    (gen_pin.type_name == "RAW_SERDES_RX_CLK" and mode.find("Rx FIFO") != -1):

                    user_pin_name = design.raw_serdes_reg.get_user_clock_pin_name(
                        inst, self.bundle_x8_pairs, self.quad_lane_to_clk_src_insname,
                        gen_pin.type_name)
            
            elif bundle_mode == "x8" and mode == "Rx FIFO" and\
                gen_pin.type_name == "RAW_SERDES_RX_CLK" and design.raw_serdes_reg is not None:
                
                user_pin_name, _ = design.raw_serdes_reg.get_user_pin_name_bundle_x8(
                    inst, gen_pin.type_name, self.bundle_x8_pairs, self.quad_lane_to_clk_src_insname)
                
        return user_pin_name

    def get_bundle_mode_x1_clkout_pin_names(self, mode: str, inf_name: str) -> List[str]:
        clkout_pin_names = []
        
        # PT-2393: decide which clockout to use based
        # on the mode and burst mode
        if inf_name == "RAW_SERDES_TX_CLK":
            # TX will always have loopback regardless of
            # mode
            clkout_pin_names.append("PCS_CLK_TX")

        elif inf_name == "RAW_SERDES_RX_CLK":
            # For RX clocks
            if mode.find("Rx FIFO") != -1:
                # We only need loopback with FIFO mode.
                # Rx Register mode don't have clkout loopback. 
                clkout_pin_names.append("PCS_CLK_RX")

        return clkout_pin_names

    def get_bundle_mode_x2_clkout_pin_names(self, design_db: PeriDesign,
                                            inst: LaneBasedItem, mode: str, inf_name: str) -> List[str]:
        clkout_pin_names = []

        _, lane_idx = QuadResService.break_res_name(inst.get_device())

        clk_user_name = ""
        in_res_name = ""

        if design_db is not None and design_db.raw_serdes_reg is not None:
            in_res_name = design_db.raw_serdes_reg.find_res_of_clock_input_in_bundle(
                self.bundle_x8_pairs, self.quad_lane_to_clk_src_insname, inst)

            if in_res_name != "":
                clk_user_name = design_db.raw_serdes_reg.get_user_clock_input_pin_name(
                    in_res_name, inf_name)
            
        if inf_name == "RAW_SERDES_TX_CLK":
            # LN1 -> PCS TX on LN0,LN1
            # LN3 -> PCS TX on LN2,LN3
            # No clock input at LN0, LN2
            
            # If this is not the instance where the input is from, then
            # We add the clockc input pin name here

            if in_res_name.find(f"LN{lane_idx}") == -1:
                # clock input is not for curreent lane
                clkout_pin_names.append(f"PCS_CLK_TX:{clk_user_name}")
            else:
                clkout_pin_names.append("PCS_CLK_TX")           

        elif inf_name == "RAW_SERDES_RX_CLK" and mode.find("Rx FIFO") != -1:
            # For RX clocks. No loopback for Register mode
            # For FIFO mode:
            # LN1 -> PCS RX on LN0,LN1
            # LN3 -> PCS RX on LN2,LN3
            # No clock input at LN0, LN2
            
            # If this is not the instance where the input is from, then
            # We add the clockc input pin name here
            if in_res_name.find(f"LN{lane_idx}") == -1:
                clkout_pin_names.append(f"PCS_CLK_RX:{clk_user_name}")
            else:
                clkout_pin_names.append("PCS_CLK_RX")

        return clkout_pin_names
    
    def get_bundle_mode_x4_clkout_pin_names(self, design_db: PeriDesign,
                                            inst: LaneBasedItem, mode: str, inf_name: str) -> List[str]:
        clkout_pin_names = []

        quad_idx, lane_idx = QuadResService.break_res_name(inst.get_device())
        #in_res_name = f"Q{quad_idx}_LN1"

        clk_user_name = ""
        in_res_name = ""
        if design_db is not None and design_db.raw_serdes_reg is not None:
            in_res_name = design_db.raw_serdes_reg.find_res_of_clock_input_in_bundle(
                self.bundle_x8_pairs, self.quad_lane_to_clk_src_insname, inst)

            if in_res_name != "":
                clk_user_name = design_db.raw_serdes_reg.get_user_clock_input_pin_name(
                    in_res_name, inf_name)
 
        if inf_name == "RAW_SERDES_TX_CLK":
            # LN1 -> PCS TX on LN0,LN1,LN2,LN3
            # No clock input at LN0, LN2, LN3
            if in_res_name.find(f"LN{lane_idx}") == -1:
                clkout_pin_names.append(f"PCS_CLK_TX:{clk_user_name}")
            else:
                clkout_pin_names.append("PCS_CLK_TX")             

        elif inf_name == "RAW_SERDES_RX_CLK" and mode.find("Rx FIFO") != -1:
            # For RX clocks. No loopback for Register mode
            # For FIFO mode:
            # LN1 -> PCS RX on LN0,LN1,LN2,LN3
            # No clock input at LN0, LN2, LN3            
            if in_res_name.find(f"LN{lane_idx}") == -1:
                clkout_pin_names.append(f"PCS_CLK_RX:{clk_user_name}")
            else:
                clkout_pin_names.append("PCS_CLK_RX")

        return clkout_pin_names    

    def get_bundle_mode_x8_clkout_pin_names(self, design_db: PeriDesign,
                                            inst: LaneBasedItem, mode: str, inf_name: str) -> List[str]:
        clkout_pin_names = []

        # Tx not supported
        if inf_name == "RAW_SERDES_TX_CLK" or design_db.raw_serdes_reg is None:
            return clkout_pin_names
        
        _, lane_idx = QuadResService.break_res_name(inst.get_device())

        clk_user_name, is_src_cur_quad = design_db.raw_serdes_reg.get_user_pin_name_bundle_x8(
            inst, inf_name, self.bundle_x8_pairs, self.quad_lane_to_clk_src_insname)

        if inf_name == "RAW_SERDES_RX_CLK" and mode == "Rx FIFO":
            # Only supports Rx FIFO mode
            # LN1 -> PCS RX on LN0,LN1,LN2,LN3 or both quads
            # No clock input at LN0, LN2, LN3 (lower quad), all (upper quad)           
            if clk_user_name != "" and \
                (not is_src_cur_quad or (is_src_cur_quad and lane_idx != 1)):
                # Not the lane with the source clk input
                clkout_pin_names.append(f"PCS_CLK_RX:{clk_user_name}")
            else:
                clkout_pin_names.append("PCS_CLK_RX")

        return clkout_pin_names    

    def get_clkout_pin_names(self, inst: LaneBasedItem, inf_name:str, design_db: PeriDesign) -> List[str]:
        """
        For PMA Direct, we want to override the list in base class
        since it depends on the configuration of the instance parameters.

        :param inf_name: The clock input name that we want
                to find the corresponding clkout name to use
        :return a list of clkout inf name that is associated to
                the pass input clock name that has auto-connection
                (loopback of clk input back to clkout interface)
        """
        clkout_pin_names = []

        if inst is not None and inst.param_group is not None:
            mode = inst.param_group.get_param_value(
                RawSerdesConfigParamInfo.Id.ss_raw_mode_lane_NID.value)
            bundle_mode = inst.param_group.get_param_value(
                RawSerdesConfigParamInfo.Id.ss_raw_bundle_mode_lane_NID.value)
            
            if bundle_mode == "x8":
                clkout_pin_names = self.get_bundle_mode_x8_clkout_pin_names(design_db, inst, mode, inf_name)
            elif bundle_mode == "x4":
                clkout_pin_names = self.get_bundle_mode_x4_clkout_pin_names(design_db, inst, mode, inf_name)
            elif bundle_mode == "x2":
                clkout_pin_names = self.get_bundle_mode_x2_clkout_pin_names(design_db, inst, mode, inf_name)
            else:
                clkout_pin_names = self.get_bundle_mode_x1_clkout_pin_names(mode, inf_name)

        return clkout_pin_names