from __future__ import annotations
import os
import re
from decimal import Decimal, localcontext
from enum import Enum
from typing import Dict, TYPE_CHECKING, Optional, List

import pt_version

import util.gen_util
from util.app_setting import AppSetting
from util.signal_util import float_to_decimal

import device.db_interface as devdb_int
import common_device.rule as base_rule
from common_device.quad.rule import QuadLaneChecker, QuadRuleHW, QuadRuleOSCClock, QuadRuleInvalidHexValue, QuadRuleExternalClock
from common_device.quad.res_service import QuadResService

from design.db_item import GenericParamService, GenericParam, GenericPin

from tx375_device.common_quad.quad_prop_id import CommonQuadConfigParamInfo as CommonQuadParamInfo
from tx375_device.common_quad.design import QuadLaneCommon
from tx375_device.raw_serdes.design import RawSerdes
from tx375_device.raw_serdes.raw_serdes_prop_id import RawSerdesConfigParamInfo as RawSerdesParamInfo
from tx375_device.raw_serdes.design_param_info import RawSerdesDesignParamInfo

if TYPE_CHECKING:
    from tx375_device.raw_serdes.device_service import QuadRawSerdesDeviceService

IS_SHOW_HIDDEN = True if pt_version.PT_DEBUG_VERSION == True else False

@util.gen_util.freeze_it
class RawSerdesChecker(QuadLaneChecker):

    def __init__(self, design):
        """
        Constructor
        """
        self.quad_other_map = {
            0: [1],
            1: [0, 2],
            2: [1, 3],
            3: [2]
        }

        super().__init__(design)
        self.block_tag = "pma_direct"
        self.reg = design.raw_serdes_reg
        self.common_quad_reg = design.common_quad_lane_reg
        self.dev_service = None # type: Optional[QuadRawSerdesDeviceService]
        self.x8_quad_to_ins_map : Dict[int, List[str]]= {}
        self.is_topaz_device = False
        
        if self.design is not None:
            self.reg = self.design.raw_serdes_reg
            self.x8_quad_to_ins_map = self.identify_x8_bundle_mode_instances()

            if self.design.device_db is not None:
                dbi = devdb_int.DeviceDBService(self.design.device_db)
                self.dev_service = dbi.get_block_service(
                    devdb_int.DeviceDBService.BlockType.QUAD_LANE_10G)

                self.is_topaz_device = devdb_int.DeviceDBService.is_topaz_device(self.design.device_db)

    def identify_x8_bundle_mode_instances(self) -> Dict[int, List[str]]:
        # Get all the lane instances associated to each quad
        # thas has bundle mode set to x8
        if self.reg is not None:
            x8_quad_ins_map, _ = self.reg.identify_bundle_mode_instances(is_x8_ins_only=True)
            return x8_quad_ins_map

        return {}

    def _build_rules(self):
        """
        Build raw_serdes rule database
        """
        self._add_rule(RawSerdesRuleInstanceName())
        self._add_rule(RawSerdesRuleResource())
        # Not activated and not yet removed in case it
        # is going to be used later with multiple lane-protocol support
        #self._add_rule(RawSerdesRuleTopology())
        self._add_rule(RawSerdesRuleExternalClock())
        self._add_rule(RawSerdesRuleInvalidHexValue())
        self._add_rule(RawSerdesRuleHW())
        self._add_rule(RawSerdesRuleQuadResource())
        self._add_rule(RawSerdesRuleOSCClock())
        self._add_rule(RawSerdesRuleRxClock())
        self._add_rule(RawSerdesRuleTxClock())
        self._add_rule(RawSerdesPLLConfig())
        self._add_rule(RawSerdesDataRateLimit())
        self._add_rule(RawSerdesBundleMode())
        self._add_rule(RawSerdesRegionalClock())
        self._add_rule(RawSerdesBundleModex8Mode())
        self._add_rule(RawSerdesBundleModex8RxClkConnType())
        self._add_rule(RawSerdesBundleModex8QuadPair())
        self._add_rule(RawSerdesRuleDataRateTimingModel())

class RawSerdesRuleInstanceName(base_rule.Rule):
    """
    Check if the instance name is valid
    """

    def __init__(self):
        super().__init__()
        self.name = "pma_direct_rule_inst_name"

    def check(self, checker, design_block: RawSerdes):
        util.gen_util.mark_unused(checker)

        raw_serdes = design_block

        if self.check_name_empty_or_invalid(raw_serdes.name,
                                            "Instance name is empty.",
                                            "Valid characters are alphanumeric characters with dash and underscore only."
                                            ):
            return

class RawSerdesRuleResource(base_rule.Rule):
    """
    Check if the resource name is valid.
    """

    def __init__(self):
        super().__init__()
        self.name = "pma_direct_rule_resource"

    def check(self, checker: RawSerdesChecker, design_block: RawSerdes):

        raw_serdes = design_block

        if self.is_name_empty(raw_serdes.get_device()):
            self.error("Resource name is empty.")
            return

        # Check if resource name is a valid device instance name
        if checker.reg is not None and checker.dev_service is not None:
            raw_serdes_list = checker.dev_service.get_usable_instance_names()

            if raw_serdes.get_device() not in raw_serdes_list:
                self.error("Resource is not a valid PMA Direct device instance")
                return

        else:
            checker.logger.warning(
                "{}: device db is empty, skip valid instance check.".format(self.name))

class RawSerdesRuleTopology(base_rule.Rule):
    """
    Check if instance is part of a valid topology.
    """

    def __init__(self):
        super().__init__()
        self.name = "pma_direct_rule_topology"

    def check(self, checker: RawSerdesChecker, design_block: RawSerdes):

        raw_serdes = design_block

        # Resource is <quad>_<lane#>. THe following are valid topology
        # 1. all lanes used: LN0_LN1_LN2_LN3
        # 2. Only one lane per quad
        # 3. 2 lanes at LN0 and LN2 ONLY

        if checker.reg is not None and raw_serdes.get_device() != "":
            assert checker.reg.res_svc is not None

            quad_idx, _ = checker.reg.res_svc.break_res_name(raw_serdes.get_device())

            # Find instances using the same quad
            if quad_idx != -1:
                ins2reg_map = checker.reg.res_svc.find_instance_same_quad(f'Q{str(quad_idx)}')

                invalid_topology_msg = ""
                if len(ins2reg_map) != 4 and len(ins2reg_map) != 1:
                    if len(ins2reg_map) == 2:
                        res_used_list = ins2reg_map.keys()
                        ins_count = 0
                        for res_name in res_used_list:
                            if res_name.find("LN0") != -1 or\
                                res_name.find("LN2") != -1:
                                ins_count += 1

                        if ins_count != 2:
                            invalid_topology_msg = "Only Lanes 0 and 2 can be used together"

                    else:
                        invalid_topology_msg = "Allowed lane topology: 1, 2 or 4 lanes used only"

                if invalid_topology_msg != "":
                    self.error(invalid_topology_msg)

class RawSerdesRuleExternalClock(QuadRuleExternalClock):
    def __init__(self):
        super().__init__()
        self.name = "pma_direct_rule_external_clock"
        self.protocol_name = "PMA Direct"

class RawSerdesRuleInvalidHexValue(QuadRuleInvalidHexValue):
    def __init__(self):
        super().__init__()
        self.name = "pma_direct_rule_invalid_hex_value"


class RawSerdesRuleHW(QuadRuleHW):
    '''
    Run the DRC from ICD script
    '''
    def __init__(self):
        super().__init__()
        self.name = "pma_direct_rule_hw_drc"

    def get_param_id_enum(self):
        return RawSerdesParamInfo.Id


class RawSerdesRuleQuadResource(base_rule.Rule):
    '''
    Safety check to make sure that there is no conflict
    in resource usage
    '''
    def __init__(self):
        super().__init__()
        self.name = "pma_direct_rule_res_usage"

    def check(self, checker: RawSerdesChecker, design_block: RawSerdes):
        blk = design_block

        # only check if resource is set
        if blk.get_device() == "":
            return

        same_quad_ins_list, _ = checker.find_shared_instance_quad(blk)

        # Check against instance of the same quad
        if same_quad_ins_list:
            for inst in same_quad_ins_list:
                if inst.get_device() == blk.get_device():
                    self.error(f"Resource name conflict with instance {inst.name}")
                    return

        # Check if it conflict with other protocol
        if checker.design is not None and checker.design.quad_pcie_reg is not None:
            quad_idx, _ = QuadResService.break_res_name(blk.get_device())

            if quad_idx != -1 and \
                checker.design.quad_pcie_reg.is_device_used(f'QUAD_{quad_idx}'):
                self.error(f"Resource conflicts with PCI Express resource QUAD_{quad_idx}")
                return

class RawSerdesRuleOSCClock(QuadRuleOSCClock):
    """
    Check that the oscillator has been configured.
    """

    def __init__(self):
        super().__init__()
        self.name = "pma_direct_rule_osc_clock"

class RawSerdesRuleRxClock(base_rule.Rule):
    '''
    Check that the RAW_SERDES_RX_CLK isn't empty
    '''
    def __init__(self):
        super().__init__()
        self.name = "pma_direct_rule_rx_clk"
   
    def check(self, checker: RawSerdesChecker, design_block: RawSerdes):
        blk = design_block

        if blk.gen_pin is not None:

            clk_name = blk.gen_pin.get_pin_name_by_type("RAW_SERDES_RX_CLK")

            #quad_idx, cur_lane_idx = QuadResService.break_res_name(blk.get_device())

            if clk_name == "":
                # If bonding mode x2, x4, only the clock input at specific lane is needed
                param_service = GenericParamService(blk.get_param_group(), blk.get_param_info())
                
                bundle_mode = param_service.get_param_value(RawSerdesParamInfo.Id.ss_raw_bundle_mode_lane_NID)
                lane_mode = param_service.get_param_value(RawSerdesParamInfo.Id.ss_raw_mode_lane_NID)

                if lane_mode.find("Rx Register") != -1:
                    # always compulsory for register mode regardless of bundle mode
                    # Will be flaged for x8 Rx Register as well even though it's not supported.
                    # User will just have to change to the correct mode with x8
                    # The clk_resource_en is not applicable to the pin when Rx Register mode 
                    self.error("Interface Receive Clock Input pin name cannot be empty")

                else:
                    # Usage on Rx depends on bundle mode and user indication on > x1
                    # if that lane is enabled as clock source
                    if bundle_mode == "x1":
                        # always compulsory for x1
                        self.error("Interface Receive Clock Input pin name cannot be empty")

                    else:
                        self.check_bundle_mode_enabled_clock(blk)

    def check_bundle_mode_enabled_clock(self, blk):
        
        param_service = GenericParamService(blk.get_param_group(), blk.get_param_info())
        
        is_clk_res_enabled = param_service.get_param_value(RawSerdesDesignParamInfo.Id.clk_resource_en)

        # The check to ensure that at least one of the lane is enabled is
        # cooversd in the RawSerdesBundleMode rule
        if is_clk_res_enabled:
            self.error("Interface Receive Clock Input pin name cannot be empty on lane with clock resource enabled")

class RawSerdesRuleTxClock(base_rule.Rule):
    '''
    Check that the RAW_SERDES_TX_CLK isn't empty
    '''
    def __init__(self):
        super().__init__()
        self.name = "pma_direct_rule_tx_clk"

    def check(self, checker: RawSerdesChecker, design_block: RawSerdes):
        blk = design_block


        # Tx only supports FIFO mode. So the check is always applied
        if blk.gen_pin is not None:

            clk_name = blk.gen_pin.get_pin_name_by_type("RAW_SERDES_TX_CLK")

            #_, cur_lane_idx = QuadResService.break_res_name(blk.get_device())

            # PT-2557:Tx clock pin will not be applicable/visible in UI with Rx FIFO
            # So, we don't need to check for it any longer
            if clk_name == "":
                # If bonding mode x2, x4, only the clock input at specific lane is needed
                param_service = GenericParamService(blk.get_param_group(), blk.get_param_info())
                
                bundle_mode = param_service.get_param_value(RawSerdesParamInfo.Id.ss_raw_bundle_mode_lane_NID)

                if bundle_mode == "x1":
                    self.error("Interface Transmit Clock Input pin name cannot be empty")

                elif bundle_mode != "x8":
                    self.check_bundle_mode_enabled_clock(blk)

    def check_bundle_mode_enabled_clock(self, blk):
        
        param_service = GenericParamService(blk.get_param_group(), blk.get_param_info())
        
        is_clk_res_enabled = param_service.get_param_value(RawSerdesDesignParamInfo.Id.clk_resource_en)

        # The check to ensure that at least one of the lane is enabled is
        # cooversd in the RawSerdesBundleMode rule
        if is_clk_res_enabled:
            self.error("Interface Transmit Clock Input pin name cannot be empty on lane with clock resource enabled")

class RawSerdesPLLConfig(base_rule.Rule):
    '''
    Check that the PLL config is valid
    '''
    def __init__(self):
        super().__init__()
        self.name = "pma_direct_rule_pll_config"

    def check(self, checker: RawSerdesChecker, design_block: RawSerdes):
        blk = design_block
        cmn_ins = checker.design.raw_serdes_reg.get_cmn_inst_by_lane_name(blk.name)
        
        if cmn_ins is None:
            return
        
        # Get the value of the 3 key parameters
        
        param_service = GenericParamService(blk.get_param_group(), blk.get_param_info())
        cmn_param_service = GenericParamService(cmn_ins.get_param_group(), cmn_ins.get_param_info())

        data_rate = param_service.get_param_value(RawSerdesParamInfo.Id.ss_raw_data_rate_lane_NID)
        refclk_freq = cmn_param_service.get_param_value(CommonQuadParamInfo.Id.ss_raw_refclk_freq)
        data_width = param_service.get_param_value(RawSerdesParamInfo.Id.ss_raw_serdes_width_lane_NID)

        pll_cfg = blk.pll_cfg
        if pll_cfg is not None:
            cfg_map = pll_cfg.get_pll_config_settings(data_rate, refclk_freq, data_width, IS_SHOW_HIDDEN)

            if cfg_map is not None:
                
                invalid_param = []
                param_info = blk.get_param_info()

                # Check that the setting matches with the current value set
                for pid, pval in cfg_map.items():
                    # PT-2519 Skip for deemphasis setting
                    if isinstance(pid, RawSerdesParamInfo.Id):
                        continue

                    pval_cur = param_service.get_param_value(pid)

                    if pval_cur != pval:
                        prop_info = param_info.get_prop_info_by_name(pid.value)
                        if prop_info is not None:
                            invalid_param.append(prop_info.disp_name)

                if invalid_param:
                    self.error("PMA Direct selection has invalid PLL config settings. Please reassign the preset: {}".format(
                        ",".join(invalid_param)))
            else:
                # This is a combination that doesn't exists in the table
                self.error('Invalid PMA Direct preset with combination '\
                        'of Data Rate: {}, SerDes Width: {} and Reference '\
                            'Clock Frequency: {} MHz'.format(
                                data_rate, data_width, refclk_freq))

        else:
            self.error('Internal Unexpected Error: No PMA Direct PLL configuration map')
            
class RawSerdesDataRateLimit(base_rule.Rule):
    '''
    Check that the data rate is within limit for Topaz
    '''
    def __init__(self):
        super().__init__()
        self.name = "pma_direct_rule_data_rate"

    def check(self, checker: RawSerdesChecker, design_block: RawSerdes):
        blk = design_block

        if checker.design is not None and checker.design.device_db is not None:
            device_db = checker.design.device_db

            if checker.is_topaz_device:

                timing_name:str = device_db.get_current_timing_model_name()

                param_service = GenericParamService(blk.get_param_group(), blk.get_param_info())
                data_rate = param_service.get_param_value(RawSerdesParamInfo.Id.ss_raw_data_rate_lane_NID)

                if timing_name in ["C3", "I3"]:
                    if data_rate > 12.5:
                        self.error(f"Data rate exceeds maximum limit: 12.5 Gbps")
                
                elif timing_name in ["C2", "I2"]:
                    if data_rate > 10.3:
                        self.error(f"Data rate exceeds maximum limit: 10.3 Gbps")

class RawSerdesBundleMode(base_rule.Rule):
    '''
    PT-2393: Bonding mode  x2 and x4 has strict lane configuration
    '''
    def __init__(self):
        super().__init__()
        self.name = "pma_direct_rule_bonding_mode"

    def check_mult_bundle_mode(self, blk, checker: RawSerdesChecker,
                               same_quad_ins, bundle_mode):
        '''
        This only checks for bundle mode x2 and x4
        '''
        
        # x2 len can be 4 if there are 2 x2 modes in the same quad
        if bundle_mode == "x4" and len(same_quad_ins) != 4:
            expected_lane_count = bundle_mode[1:]
            self.error(f'Bonding mode {bundle_mode} requires {expected_lane_count} lanes configured in the same QUAD')
            return
        
        # Not checking x4 since we already called get_insts_by_quad_res
        # which means it should return all 4 lanes belonging to the same quad
        else:
            # List of instances that make up the bundle with the current instance
            bundle_mode_ins = []
            
            if bundle_mode == 'x2':
                # Only LN0,1 or LN2,3 pair is acceptable                    
                _, cur_lane_idx = QuadResService.break_res_name(blk.get_device())
                
                # It could be 2 but of different ocmbination
                for ins in same_quad_ins:
                    res_name = ins.get_device()

                    # Skip current instance
                    if res_name != "" and res_name != blk.get_device():
                        _, other_idx = QuadResService.break_res_name(res_name)

                        # It could be all lanes are used but they are using
                        # x2 (so 2 pairs).

                        # Save the instance that are paired with the current
                        if (cur_lane_idx in [0, 1] and other_idx in [0, 1]) or\
                            (cur_lane_idx in [2, 3] and other_idx in [2, 3]):
                            bundle_mode_ins.append(ins)

                if len(bundle_mode_ins) != 1:
                    self.error(f'Bonding mode {bundle_mode} is valid on lane resource pairs LN0,LN1 or LN2,LN3 of the same QUAD only')
                    return

                # Add the instance itself
                bundle_mode_ins.append(blk)

            else:
                bundle_mode_ins = same_quad_ins

            # Check for consistencies across lanes in same bundle
            self.check_lanes_in_bundle(blk, checker, bundle_mode, bundle_mode_ins)

    def check_lanes_in_bundle(self, blk: RawSerdes, checker: RawSerdesChecker,
                              bundle_mode: str, bundle_mode_ins: List[RawSerdes]):
        mismatch_bundle_mode_ins_names = []
        # Check that all the instance in the same bundle has the same setting:
        # bundle mode, control reg mode and preset
        param_service = GenericParamService(blk.get_param_group(), blk.get_param_info())       
        lane_mode = param_service.get_param_value(RawSerdesParamInfo.Id.ss_raw_mode_lane_NID)
        data_width = param_service.get_param_value(RawSerdesParamInfo.Id.ss_raw_serdes_width_lane_NID)
        data_rate = param_service.get_param_value(RawSerdesParamInfo.Id.ss_raw_data_rate_lane_NID)

        # For x8, we need to check for refclk freq set in diff quad
        refclk_freq = None

        if bundle_mode == "x8" and checker.reg is not None:
            cmn_ins = checker.reg.get_cmn_inst_by_lane_name(blk.name)
            if cmn_ins is not None:
                cmn_param_service = GenericParamService(cmn_ins.get_param_group(),
                                                    cmn_ins.get_param_info())
                refclk_freq = cmn_param_service.get_param_value(CommonQuadParamInfo.Id.ss_raw_refclk_freq)

        clock_resoruce_list = []
        # Since refclk is a common parameter across lanes in same quad, we skip checking for that
        for ins in bundle_mode_ins:
            param_service = GenericParamService(ins.get_param_group(), ins.get_param_info())
            clk_resource_en = param_service.get_param_value(RawSerdesDesignParamInfo.Id.clk_resource_en)

            if clk_resource_en:
                clock_resoruce_list.append(ins.name)

            if ins.get_device() != blk.get_device():
                prop_mismatch_list = []

                # Lane with the same bundle group has different lane setting
                same_setting_check = [
                    (RawSerdesParamInfo.Id.ss_raw_data_rate_lane_NID, data_rate, "data rate"),
                    (RawSerdesParamInfo.Id.ss_raw_serdes_width_lane_NID, data_width, "serdes width"),
                    (RawSerdesParamInfo.Id.ss_raw_mode_lane_NID, lane_mode, "mode"),
                    (RawSerdesParamInfo.Id.ss_raw_bundle_mode_lane_NID, bundle_mode, "bonding mode"),
                ]
                for param_id, exp_value, display_str in same_setting_check:
                    other_value = param_service.get_param_value(param_id)
                    if other_value != exp_value:
                        prop_mismatch_list.append(display_str)

                if bundle_mode == "x8" and checker.reg is not None:
                    others_cmn_ins = checker.reg.get_cmn_inst_by_lane_name(ins.name)
                    if others_cmn_ins is not None:
                        cmn_param_service = GenericParamService(others_cmn_ins.get_param_group(),
                                                            others_cmn_ins.get_param_info())
                        others_refclk_freq = cmn_param_service.get_param_value(CommonQuadParamInfo.Id.ss_raw_refclk_freq)

                        if others_refclk_freq != refclk_freq:
                            prop_mismatch_list.append("reference clock frequency")

                if len(prop_mismatch_list) > 0:
                    diff_str = "{}: {}".format(ins.name, ",".join(prop_mismatch_list))
                    mismatch_bundle_mode_ins_names.append(diff_str)

        if mismatch_bundle_mode_ins_names:
            self.error('Mismatch properties against lane instances in the same bonding {}: {}'.format(
                bundle_mode, ",".join(mismatch_bundle_mode_ins_names)))
            return

        # Only 1 clock resource is allowed
        if bundle_mode != 'x1':
            if len(clock_resoruce_list) == 0:
                self.error(f'Clock resource is not selected for bonding mode {bundle_mode}')

            elif len(clock_resoruce_list) > 1:
                self.error('Only 1 clock resource is allowed to be selected '\
                    f'for bonding mode {bundle_mode}, selected instances: ' + \
                    ', '.join(clock_resoruce_list))


    def check_bundle_mode_x8(self, blk: RawSerdes, checker: RawSerdesChecker, 
                             same_quad_ins: List[RawSerdes],
                             bundle_mode: str):
        '''
        This only checks for bundle mode x8
        '''

        # Check if the quad is filled
        if len(same_quad_ins) != 4:
            self.error(f'Bonding mode x8 requires all lanes in the quad to be configured')
            return
        
        # Find the other quad associated to this quad that
        # makes up the bundle
        quad_idx, _ = QuadResService.break_res_name(blk.get_device())
        
        assert quad_idx in checker.quad_other_map
        other_quad_list = checker.quad_other_map[quad_idx]

        # if there is more than one and both configured, we need to
        # choose the correct quad to use for the check
        quad_ins_map = checker.x8_quad_to_ins_map

        # List of instance names
        other_quad_lanes_ins_names: List[str] = []

        # We skip checking if invalid pair (1,3)
        if len(quad_ins_map) == 4:
            # All quads used as x8. Then, the valid pair
            # 0,1 and 2,3
            expected_pair =  {
                0: 1,
                1: 0,
                2: 3,
                3: 2
            }

            other_quad = expected_pair.get(quad_idx, None)
            if other_quad is not None:
                if other_quad in quad_ins_map:
                    other_quad_lanes_ins_names = quad_ins_map[other_quad]

                else:
                    # The other quad not configured
                    self.error(f"Expected Q{other_quad} lanes to be configured as part of bonding mode x8")
                    return
            else:
                self.error("Unable to determine the quad pair for bonding mode x8 instance")
                return

        elif len(quad_ins_map) == 2:

            for quad in other_quad_list:
                # check if the other quad was also configured as x8
                if quad in quad_ins_map:
                    # it could also be (1, 2)
                    other_quad_lanes_ins_names = quad_ins_map[quad]
                    break

        # Find the actual instances of the other lanes based
        # that we've collected based on names
        if len(other_quad_lanes_ins_names) != 4:
            self.error("Bonding mode x8 requires 8 lane instances to be configured")
        
        else:
            other_quad_ins = []

            for iname in other_quad_lanes_ins_names:
                other_ins = checker.reg.get_inst_by_name(iname)
                if other_ins is not None:
                    other_quad_ins.append(other_ins)

            combined_ins_lanes = same_quad_ins + other_quad_ins

            if len(combined_ins_lanes) != 8:
                self.error(f"Bonding mode x8 requires 8 lane instances but found {len(combined_ins_lanes)} configured instances")
            
            else:
                self.check_lanes_in_bundle(blk, checker, bundle_mode, combined_ins_lanes)

    def check(self, checker: RawSerdesChecker, design_block: RawSerdes):
        blk = design_block

        param_service = GenericParamService(blk.get_param_group(), blk.get_param_info())       
        bundle_mode = param_service.get_param_value(RawSerdesParamInfo.Id.ss_raw_bundle_mode_lane_NID)

        if bundle_mode in ["x2", "x4", "x8"] and blk.get_device() != "":
            raw_serdes_reg = checker.design.raw_serdes_reg
                    
            # Get the list of lane intances associated to the quad
            same_quad_ins: List[RawSerdes] = raw_serdes_reg.get_insts_by_quad_res(blk.get_device())

            if bundle_mode == "x8":
                self.check_bundle_mode_x8(blk, checker, same_quad_ins, bundle_mode)

            else:
                if len(same_quad_ins) > 0:
                    self.check_mult_bundle_mode(blk, checker, same_quad_ins, bundle_mode)

                else:
                    # This is impossible since at least there should be 1 returned
                    expected_lane_count = bundle_mode[1:]
                    self.error(f'Bonding mode {bundle_mode} requires {expected_lane_count} lanes configured in the same QUAD')

class RawSerdesRegionalClock(base_rule.Rule):
    '''
    PT-2393: Some rclk on the quad-lane is not valid
        due to no routable path for loopback input-clkout
        in the allowed regions in core
    '''
    def __init__(self):
        super().__init__()
        self.name = "pma_direct_rule_rclk"

    def check(self, checker: RawSerdesChecker, design_block: RawSerdes):
        blk = design_block

        param_service = GenericParamService(blk.get_param_group(), blk.get_param_info())       
        bundle_mode = param_service.get_param_value(RawSerdesParamInfo.Id.ss_raw_bundle_mode_lane_NID)
        lane_mode = param_service.get_param_value(RawSerdesParamInfo.Id.ss_raw_mode_lane_NID)

        is_clk_en = param_service.get_param_value(RawSerdesDesignParamInfo.Id.clk_resource_en)

        # We are restricting based on bundle_mode + lane_mode (where
        # the loopback path spans multiple lanes and is compulsory))
        if bundle_mode in ['x2', 'x4'] and blk.get_device() != "" and is_clk_en:

            tx_conn_type = param_service.get_param_value(RawSerdesDesignParamInfo.Id.tx_clk_conn_type)            
            rx_conn_type = param_service.get_param_value(RawSerdesDesignParamInfo.Id.rx_clk_conn_type)

            is_invalid = False

            inf_name = "RAW_SERDES_TX_CLK"
            invalid_pin_type_names = []
            if tx_conn_type == "rclk" and blk.is_disconnect_clock_with_clkout_loopback(checker.design, inf_name):
                is_invalid = True
                invalid_pin_type_names.append("Transmit Clock Input")

            inf_name = "RAW_SERDES_RX_CLK"
            if rx_conn_type == "rclk" and lane_mode.find("Rx FIFO") != -1 and \
                blk.is_disconnect_clock_with_clkout_loopback(checker.design, inf_name):
                is_invalid = True                
                invalid_pin_type_names.append("Receive Clock Input")
    
            if is_invalid:
                self.error('Use global clock due to unroutable clock loopback on regional clock with bonding mode:{} Mode: {} on pins: {}'.format(
                    bundle_mode, lane_mode, ",".join(invalid_pin_type_names)))

class RawSerdesBundleModex8Mode(base_rule.Rule):
    '''
    PT-2511: Only Mode Rx FIFO is allowed with x8
    '''
    def __init__(self):
        super().__init__()
        self.name = "pma_direct_rule_x8_mode"

    def check(self, checker: RawSerdesChecker, design_block: RawSerdes):
        blk = design_block

        param_service = GenericParamService(blk.get_param_group(), blk.get_param_info())       
        bundle_mode = param_service.get_param_value(RawSerdesParamInfo.Id.ss_raw_bundle_mode_lane_NID)
        lane_mode = param_service.get_param_value(RawSerdesParamInfo.Id.ss_raw_mode_lane_NID)
        
        if bundle_mode == "x8":

            if lane_mode != "Rx FIFO":
                self.error("Bonding mode x8 is only supported in Mode RX FIFO")

        elif lane_mode == "Rx FIFO":
            self.error("Mode RX FIFO is only supported with bonding mode x8")

class RawSerdesBundleModex8RxClkConnType(base_rule.Rule):
    '''
    PT-2511: regional clock has limitation where in most cases it won't
    be able to span for 2 quads clkout location. So, we just disallow rclk.
    '''
    def __init__(self):
        super().__init__()
        self.name = "pma_direct_rule_x8_rx_clk_conn_type"

    def check(self, checker: RawSerdesChecker, design_block: RawSerdes):
        blk = design_block

        param_service = GenericParamService(blk.get_param_group(), blk.get_param_info())       
        bundle_mode = param_service.get_param_value(RawSerdesParamInfo.Id.ss_raw_bundle_mode_lane_NID)       

        if bundle_mode == "x8":
            rx_conn_type = param_service.get_param_value(RawSerdesDesignParamInfo.Id.rx_clk_conn_type)
            user_pin_name = blk.gen_pin.get_pin_name_by_type("RAW_SERDES_RX_CLK")

            if rx_conn_type == "rclk" and user_pin_name != "":
                self.error('Bonding mode x8 does not support rclk clock input connection type')

class RawSerdesBundleModex8QuadPair(base_rule.Rule):
    '''
    PT-2511: x8 bundle mode only supports limited quad pairing

    But if quad pair found but it's used with multiple
    other quad, then we need to check that they are valid.
    ie. Q0,Q1 AND Q1,Q2 -> Invalid
        Q1 gets both Q0 and  Q2. But Q2 is the one paired with Q3 -> valid

    '''
    def __init__(self):
        super().__init__()
        self.name = "pma_direct_rule_x8_quad_pair"

    def check(self, checker: RawSerdesChecker, design_block: RawSerdes):
        blk = design_block

        param_service = GenericParamService(blk.get_param_group(), blk.get_param_info())       
        bundle_mode = param_service.get_param_value(RawSerdesParamInfo.Id.ss_raw_bundle_mode_lane_NID)

        if bundle_mode == "x8" and blk.get_device() != "":
            
            # Get the quad index of the current lane instance
            quad_idx, _ = QuadResService.break_res_name(blk.get_device())
                
            if quad_idx != -1:
                # Allowed pairing: (Q0,Q1), (Q2,Q3), (Q1,Q2)
                # checker.quad_other_map = {
                #     0: [1],
                #     1: [0, 2],
                #     2: [1, 3],
                #     3: [2]
                # }
                assert quad_idx in checker.quad_other_map
                #valid_pair_list = [
                #    (0,1), (2,3), (1,2)
                #]

                x8_quad_ins_map = checker.x8_quad_to_ins_map

                if len(x8_quad_ins_map) < 4 and len(x8_quad_ins_map) > 0:
                    # Not all quads are used for x8. Else, we assume that
                    # the permutation is correct if all quad is in that x8 map
                    paired_quad_idx_list = checker.quad_other_map[quad_idx]
                    valid_quad_cnt = 0
                    for pair_idx in paired_quad_idx_list:
                        if pair_idx in x8_quad_ins_map:
                            valid_quad_cnt += 1

                    if valid_quad_cnt > 1:
                        # It means that only 3 quad has x8 and that is invalid.
                        # we rule out 2 quad pair because we found < 4 quad used
                        # with x8 bundle
                        if len(paired_quad_idx_list) > 1:
                            quad_names = ','.join(f"Q{idx}" for idx in paired_quad_idx_list)
                            self.error(f'Bonding mode x8 on quad Q{quad_idx} expected to pair up with instances of one of the following quads: {quad_names}')

                        else:
                            self.error(f'Bonding mode x8 on quad Q{quad_idx} expected to pair up with quad Q{paired_quad_idx_list[0]}')

                    elif valid_quad_cnt < 1:
                        # No quad pair found at all
                        self.error(f'Bonding mode x8 requires 2 quads with PMA Direct configuration')

class RawSerdesRuleDataRateTimingModel(base_rule.Rule):
    '''
    PT-2658: The data rate usage is limited by speedgrade
    '''
    def __init__(self):
        super().__init__()
        self.name = "pma_direct_rule_data_rate_timing_model"

    def check(self, checker: RawSerdesChecker, design_block: RawSerdes):
        blk = design_block

        if checker.design is not None and checker.design.device_db is not None:
            device_db = checker.design.device_db
            timing_name: str = device_db.get_current_timing_model_name()

            param_service = GenericParamService(blk.get_param_group(), blk.get_param_info())       
            data_rate = param_service.get_param_value(RawSerdesParamInfo.Id.ss_raw_data_rate_lane_NID)

            if data_rate > 10.3125 and timing_name not in ("C4", "C4L", "I4", "I4L"):
                self.error(f'Data rate greater than 10.3125 Gbps is not supported in timing model {timing_name}')