import os
from csv import DictReader
from typing import Mapping, Tuple, Optional, TYPE_CHECKING, TypeAlias, Dict, Any, List
from enum import Enum

from util.signal_util import float_to_decimal
from tx375_device.common_quad.quad_prop_id import CommonQuadConfigParamInfo as CommonQuadParamInfo
from tx375_device.raw_serdes.raw_serdes_prop_id import RawSerdesConfigParamInfo as RawSerdesParamInfo
from tx375_device.raw_serdes.raw_serdes_pll_cfg_prop_id import RawSerdesPLLConfigParamInfo as RawSerdesPLLParamInfo
from tx375_device.raw_serdes.quad_param_info import get_hidden_param_values

from design.db_item import GenericParamService
from util.app_setting import AppSetting

if TYPE_CHECKING:
    from tx375_device.raw_serdes.design import RawSerdes
    from tx375_device.common_quad.design import QuadLaneCommon

PRESET_KEY_TYPE: TypeAlias = Tuple[float, float, str]

class RawSerdesPLLConfig:

    class DesignSetting(Enum):
        show_all_data_rate = "show_all_data_rate"

    def __init__(self, dataset_file: str| os.PathLike):
        self.dataset: Mapping[
            Tuple[float, float, str],
            Mapping[RawSerdesPLLParamInfo.Id| RawSerdesParamInfo.Id, str]
        ] = self.load_dataset(dataset_file)

        # Read in any settings.ini file to get settings related to raw serdes
        self.settings_map : Dict[str, str]= {}
            
    def load_settings(self, location: str):
        settings_parsed: Dict[str, str] = {}

        if location != "":
            setting = AppSetting()
            raw_serdes_setting_names = [dsetting.value for dsetting in self.DesignSetting]

            settings_parsed = setting.get_project_setting_list(location, raw_serdes_setting_names)            

        self.settings_map = settings_parsed.copy()

    def load_dataset(self, dataset_file: str| os.PathLike) -> Mapping:
        # Load dataset from file
        results = {}

        PRIMARY_KEYS = [
            RawSerdesParamInfo.Id.ss_raw_data_rate_lane_NID,
            CommonQuadParamInfo.Id.ss_raw_refclk_freq,
            RawSerdesParamInfo.Id.ss_raw_serdes_width_lane_NID
        ]

        with open(dataset_file, 'r', encoding='utf-8') as fptr:
            reader = DictReader(fptr, )
            for data in reader:
                for pk in PRIMARY_KEYS:
                    assert pk.value in data, f'{pk.value} not in PRIMARY_KEYS'

                # The value in file is string but the param data type may not be string
                # We are casting the value to match param data type. So, it may be no
                # longer the same in terms of string (100 vs 100.0)
                data_rate = float(data[RawSerdesParamInfo.Id.ss_raw_data_rate_lane_NID.value])
                ref_clk_freq = float(data[CommonQuadParamInfo.Id.ss_raw_refclk_freq.value])
                data_width = data[RawSerdesParamInfo.Id.ss_raw_serdes_width_lane_NID.value]

                # key of the map is the combination of the above 3 params:
                key = (data_rate, ref_clk_freq, data_width)
                assert key not in results
                for pk in PRIMARY_KEYS:
                    data.pop(pk.value)

                # PLL config params
                record: Dict[RawSerdesParamInfo.Id | RawSerdesPLLParamInfo.Id, Any] = {}

                for name, value in data.items():
                    if value == "":
                        continue
                    if RawSerdesParamInfo.Id.has_key(name):
                        lane_id = RawSerdesParamInfo.Id(name)
                        record[lane_id] = value
                    else:
                        param_id = RawSerdesPLLParamInfo.Id(name)
                        record[param_id] = value

                results[key] = record

        return results

    def get_pll_config_settings(self, data_rate: float, refclk_freq: float, data_width: str, 
                                is_allow_hidden: bool = False) -> Optional[Mapping[RawSerdesPLLParamInfo.Id| RawSerdesParamInfo.Id, str]]:
        data_width_strip = data_width.strip()

        if is_allow_hidden or not self.is_preset_hidden(data_rate, refclk_freq, data_width_strip):
            return self.dataset.get((data_rate, refclk_freq, data_width_strip), None)
        
        return None

    def _is_data_rate_hidden(self, data_rate: float) -> bool:
        hidden_param_val_map = get_hidden_param_values()

        if RawSerdesParamInfo.Id.ss_raw_data_rate_lane_NID.value in hidden_param_val_map:
            hidden_data_rate = hidden_param_val_map[RawSerdesParamInfo.Id.ss_raw_data_rate_lane_NID.value]
            
            if data_rate in hidden_data_rate:
                if self.DesignSetting.show_all_data_rate.value in self.settings_map:
                    if self.settings_map[self.DesignSetting.show_all_data_rate.value] != "on":
                        return True
                    
                else:
                    return True                

        return False

    def is_preset_hidden(self, data_rate: float, ref_clk_freq: float, data_width: str) -> bool:
        '''
        '''
        is_hidden = False

        param_name2val = {
            RawSerdesParamInfo.Id.ss_raw_data_rate_lane_NID.value: data_rate,
            CommonQuadParamInfo.Id.ss_raw_refclk_freq.value: ref_clk_freq,
            RawSerdesParamInfo.Id.ss_raw_serdes_width_lane_NID.value: data_width
        }

        hidden_param_val_map = get_hidden_param_values()

        for pname, pval in param_name2val.items():
            if pname in hidden_param_val_map:
                if pval in hidden_param_val_map[pname]:
                    is_hidden = True
                    break

            # There's no longer hidden data rate. So we comment this out in case
            # it's needed in future
            # if pname == RawSerdesParamInfo.Id.ss_raw_data_rate_lane_NID.value and \
            #     self._is_data_rate_hidden(pval):
            #     is_hidden = True
            #     break

        return is_hidden
    
    def is_preset_supported(self, data_rate, ref_clk_freq, data_width, allow_hidden):
        if self.get_pll_config_settings(data_rate, ref_clk_freq, data_width, allow_hidden) is not None:
            return True
        
        return False
    
    def is_preset_str_supported(self, preset_name: str, allow_hidden: bool):
        is_found = False

        if preset_name.find("-") != -1:
            preset_names_list: List[str] = self.get_all_preset_names(allow_hidden)
            preset_elem: List[str] = preset_name.split("-")
            if len(preset_elem) == 3:
                for preset_key in preset_names_list:
                    golden_list: List[str] = preset_key.split("-")
                    assert len(golden_list) == 3

                    # Preset is a string but in fact they value is
                    # param type specific (ie float). So, when we compare,
                    # we want to compare  using the param data type instead
                    # of string (ie string formatted 100 vs 100.0)
                    # 0 - data rate, 1 - freq (float)
                    # 2 - data width (string)
                    if float_to_decimal(float(golden_list[0]), 4) == float_to_decimal(float(preset_elem[0]), 4) and \
                        float_to_decimal(float(golden_list[1]), 4) == float_to_decimal(float(preset_elem[1]), 4) and \
                        golden_list[2] == preset_elem[2]:
                        is_found = True
                        break

        return is_found
    
    def get_all_preset_names(self, show_hidden: bool=False) -> List[str]:
        '''
        :param filter_hidden: If True, then we need to filter out preset that is
                made up of hidden param values. Else, show all that exists.
        Get the list of all preset names supported in dataset
        :return a sorted list of string with each string represent the combined keys
                in the format "<data_rate>-<refclk_freq>-<width>"
        '''
        key_list = []

        # sort the list of tuples and also filter out preset that has
        # param value that is hidden if flag is set (PT-2589)
        for key_tup in sorted(list(self.dataset.keys())):
            data_rate, ref_clk_freq, data_width = key_tup
            if show_hidden or not self.is_preset_hidden(data_rate, ref_clk_freq, data_width):
                key_str = "{}-{}-{}".format(data_rate, ref_clk_freq, data_width)
                key_list.append(key_str)

        return key_list
