"""

Copyright (C) 2017-2024 Efinix Inc. All rights reserved.

No portion of this code may be reused, modified or
distributed in any way without the expressed written
consent of Efinix Inc.

Created on August 05, 2024

@author: Shirley Chan
"""
from __future__ import annotations
from enum import Enum
from typing import Optional, Set, TYPE_CHECKING, Dict
import itertools

from PyQt5 import QtCore, QtGui
from PyQt5.QtWidgets import QWidget, QDialog, QListView, QLineEdit

import pt_version

from tx375_device.common_quad.design import QuadLaneCommon
from tx375_device.common_quad.quad_prop_id import CommonQuadConfigParamInfo as CommonQuadParamInfo
from tx375_device.raw_serdes.design import RawSerdes
from tx375_device.raw_serdes.raw_serdes_prop_id import RawSerdesConfigParamInfo

from tx375_device.raw_serdes.gui.Ui_tx375_device_pll_preset_selector import Ui_dialog_preset as UIFORM_CLASS

if TYPE_CHECKING:
    from tx375_device.raw_serdes.pll_config import PRESET_KEY_TYPE

IS_SHOW_HIDDEN = True if pt_version.PT_DEBUG_VERSION == True else False


class PresetEditor(QtCore.QObject):
    """
    Editor for selecting preset for PLL settings

    It displays a list of preset.
    User can filter the list to narrow down the options.
    """
    sig_set_preset = QtCore.pyqtSignal()

    def __init__(self, parent: QWidget):
        """
        Constructor
        """
        super().__init__(parent)
        self.dialog = QDialog(parent)
        self.ui_dialog = UIFORM_CLASS()
        self.ui_dialog.setupUi(self.dialog)

        self.data_model: Optional[PresetDataModel] = None
        self.filter_view = PresetViewFilter(self.ui_dialog.lv_preset, self.ui_dialog.le_filter)

        self.blk_inst: Optional[RawSerdes] = None
        self.is_built = False

        # For getting unit for preset keys
        self.param2unit: Dict[Enum, str] = {}

        # Remove the default icon
        for button in self.ui_dialog.btn_box.buttons():
            button.setIcon(QtGui.QIcon())

    def build(self,inst: RawSerdes, cmn_inst: QuadLaneCommon):
        """
        Build the editor.

        Two data models are build. One the actual data model and the other is the filtered model.
        The model shown on the UI is the filtered model. If not filter set, the filtered model
        is the same as actual model.

        :param inst: inst instance
        """

        if inst is None:
            return

        self.blk_inst = inst
        assert self.blk_inst.pll_cfg is not None
        all_preset = self.blk_inst.pll_cfg.get_all_preset_names(IS_SHOW_HIDDEN)

        if self.data_model is None:
            self.data_model = PresetDataModel()
            self.data_model.param2unit = self.param2unit

        preset_set = set(all_preset)
        self.data_model.load(preset_set, self.blk_inst.get_default_pll_config_key(cmn_inst))
        self.ui_dialog.lv_preset.setModel(self.data_model)
        self.filter_view.set_source_model(self.data_model)

        if not self.is_built:
            self.monitor_changes()
            self.is_built = True

    def monitor_changes(self):
        self.dialog.accepted.connect(self.on_ok)
        self.dialog.rejected.connect(self.on_cancel)

    # noinspection PyArgumentList
    @QtCore.pyqtSlot()
    def on_ok(self):
        self.sig_set_preset.emit()

    # noinspection PyArgumentList
    @QtCore.pyqtSlot()
    def on_cancel(self):
        # Do nothing
        pass

    def show(self):
        """
        Show the editor. Blocking.
        """

        if self.blk_inst is None:
            return

        self.ui_dialog.le_filter.clear()

        self.dialog.exec()

    def get_selected_preset(self) -> Optional[PRESET_KEY_TYPE]:
        """
        Get user selection.

        :return: A tuple of preset information
        """
        assert self.data_model is not None

        selected_list = self.ui_dialog.lv_preset.selectedIndexes()
        if selected_list is None:
            return None

        count = len(selected_list)
        if count > 0:
            selected_index = selected_list[0]
            source_index = self.filter_view.get_source_index(selected_index)
            item = self.data_model.itemFromIndex(source_index)

            preset_info = item.data(QtCore.Qt.UserRole)
            return preset_info

        return None


class PresetDataModel(QtGui.QStandardItemModel):
    """
    UI data model that drives the preset list
    """

    def __init__(self):
        """
        Constructor
        """
        super().__init__()
        self.count = 0  #: Current row count
        self.default_preset = None
        self.default_preset_str = "default"
        self.param2unit: Dict[Enum, str] = {}

    def load(self, preset_set: Set, default_preset):
        """
        Load preset into selector

        :param preset_set: set of preset
        """
        self.unload()
        self.default_preset = default_preset

        # PT-2672: Order the preset based on the data rate decreasing order
        def splitondigits(string):
            return [int("".join(chars)) if digits else "".join(chars) \
                for digits,chars in itertools.groupby(string, str.isdigit)]

        preset_list = list(preset_set)
        sorted_preset_list = sorted(preset_list[:], key=splitondigits, reverse=True)
        assert len(sorted_preset_list) == len(preset_set)

        for preset in sorted_preset_list:
            self.add_row_item(preset)

    def unload(self):
        """
        Remove all rows from the model
        """
        self.removeRows(0, self.rowCount())
        self.count = 0

    def add_row_item(self, preset_str: str):
        """
        Add a row into the info table model
        """
        presets = preset_str.split("-")
        assert len(presets) == 3, f"Error when trying to add a row for preset {preset_str}"

        data_rate, ref_clk_freq, data_width = presets
        preset_key: PRESET_KEY_TYPE = (float(data_rate), float(ref_clk_freq), data_width)
        display_str = self.convert_db2display_preset(preset_key)

        if preset_key == self.default_preset:
            display_str += f" ({self.default_preset_str})"

        item = QtGui.QStandardItem(display_str)
        item.setData(preset_key, QtCore.Qt.UserRole)
        item.setToolTip(display_str)

        self.insertRow(self.count, item)
        self.count = (self.count + 1)

    def convert_db2display_preset(self, preset_key: PRESET_KEY_TYPE):
        """
        Convert Preset Key from param value to display value in the selector
        """
        data_rate, ref_clk_freq, data_width = preset_key
        data_width = data_width.replace(" bits", "")
        preset_list = []
        param2value_list = [
            (RawSerdesConfigParamInfo.Id.ss_raw_data_rate_lane_NID, data_rate),
            (CommonQuadParamInfo.Id.ss_raw_refclk_freq, ref_clk_freq),
            (RawSerdesConfigParamInfo.Id.ss_raw_serdes_width_lane_NID, data_width),
        ]
        for param_id, value in param2value_list:
            unit = self.param2unit.get(param_id)
            assert unit is not None
            preset_list.append(f"{value} {unit}")

        return "-".join(preset_list)


class PresetViewFilter(QtCore.QObject):

    def __init__(self, lv_filter_view: QListView, le_filter_input: QLineEdit):
        """
        Constructor
        """
        super().__init__()

        self.filter_model = QtCore.QSortFilterProxyModel()
        self.setup_filter_model()

        self.lv_filter_view = lv_filter_view
        self.le_filter_input = le_filter_input

        self.le_filter_input.textChanged.connect(self.on_filter_changed)

    def setup_filter_model(self):
        # If source model change, automatically update the proxy filter model
        self.filter_model.setDynamicSortFilter(True)
        self.filter_model.setFilterCaseSensitivity(QtCore.Qt.CaseInsensitive)
        self.filter_model.setSortCaseSensitivity(QtCore.Qt.CaseInsensitive)

    def get_source_model(self):
        return self.filter_model.sourceModel()

    def get_source_index(self, filter_index):
        return self.filter_model.mapToSource(filter_index)

    def set_source_model(self, data_model):
        self.filter_model.setSourceModel(data_model)
        self.lv_filter_view.setModel(self.filter_model)

    # noinspection PyArgumentList
    @QtCore.pyqtSlot(str)
    def on_filter_changed(self, filter_text):
        regexp = QtCore.QRegExp(filter_text, QtCore.Qt.CaseInsensitive, QtCore.QRegExp.RegExp2)
        self.filter_model.setFilterRegExp(regexp)


if __name__ == "__main__":
    pass
