"""

Copyright (C) 2017-2024 Efinix Inc. All rights reserved.

No portion of this code may be reused, modified or
distributed in any way without the expressed written
consent of Efinix Inc.

Created on May 13, 2024

@author: Shirley Chan
"""

from __future__ import annotations
from typing import Callable, Dict, List, TYPE_CHECKING
from PyQt5.QtCore import pyqtSlot, Qt
import re
from enum import Enum

from PyQt5.QtWidgets import (QHBoxLayout, QSizePolicy,
                             QLabel, QMessageBox,
                             QPushButton, QSpacerItem)

import pt_version

from util.gen_util import mark_unused

from design.db import PeriDesign
from design.db_item import PeriDesignGenPinItem, GenericPin

from common_device.property import PropertyMetaData
from common_device.quad.gui.lane_config import BaseLaneConfig

from tx375_device.common_quad.design_param_info import QuadDesignParamInfo
from tx375_device.common_quad.quad_prop_id import CommonQuadConfigParamInfo as CommonQuadParamInfo

from tx375_device.raw_serdes.design import RawSerdesRegistry, RawSerdes
from tx375_device.raw_serdes.design_param_info import RawSerdesDesignParamInfo as RawSerdesParamInfo
from tx375_device.raw_serdes.raw_serdes_prop_id import RawSerdesConfigParamInfo
from tx375_device.raw_serdes.quad_param_info import get_hidden_parameters

from tx375_device.raw_serdes.gui.presenter import RawSerdesConfigUIPresenter
from tx375_device.raw_serdes.gui.graph_observer import RawSerdesGraphObserver, CommonQuadGraphObserver
from tx375_device.raw_serdes.gui.builder import RawSerdesTabBuilder
from tx375_device.raw_serdes.gui.pll_preset_editor import PresetEditor


if TYPE_CHECKING:
    from tx375_device.raw_serdes.gui.main_window import RawSerdesWindow


FILTER_PROP_FUNC = Callable[[PropertyMetaData], List[PropertyMetaData.PropData]]
FILTER_PIN_FUNC = Callable[[PeriDesignGenPinItem], List[GenericPin]]

IS_SHOW_HIDDEN = True if pt_version.PT_DEBUG_VERSION == True else False


def get_flr_filter_disp_name(disp_name: str) -> str:
    disp_name_list = disp_name.split("C")
    assert len(disp_name_list) == 2
    larger_disp = disp_name_list[0] + "C"
    smaller_disp = disp_name_list[1]
    return f'<html><head/><body><p>{larger_disp}<span style=" vertical-align:sub; font-size:18px;">{smaller_disp} </span></p></body></html>'

def get_unit_from_str(target_str: str):
    pattern = r"\(([a-zA-Z]*)\)"
    unit = ""

    for result in re.finditer(pattern, target_str):
        unit = result.group(1)
        break

    return unit


class RawSerdesConfig(BaseLaneConfig):
    """
    Controller for specific instance configuration
    """

    def __init__(self, parent: RawSerdesWindow):
        super().__init__(parent)

        self.presenter: RawSerdesConfigUIPresenter
        self.graph_observer: RawSerdesGraphObserver
        self.common_graph_observer: CommonQuadGraphObserver
        self.block_inst: RawSerdes
        self.block_reg: RawSerdesRegistry

        # Preset related
        self.pb_preset_editor = QPushButton()
        self.preset_editor = PresetEditor(parent)
        self.preset_label_map: Dict[Enum, QLabel] = {}
        self.param2unit: Dict[Enum, str] = {} # unit display

    def build_preset_readonly_label(self):
        assert self.block_inst is not None and self.common_inst is not None

        param_id_list = [
            RawSerdesConfigParamInfo.Id.ss_raw_data_rate_lane_NID,
            RawSerdesConfigParamInfo.Id.ss_raw_serdes_width_lane_NID,
            CommonQuadParamInfo.Id.ss_raw_refclk_freq,
        ]
        assert self.presenter is not None
        for param in param_id_list:
            if isinstance(param, CommonQuadParamInfo.Id):
                inst = self.common_inst
            else:
                inst = self.block_inst

            container = QHBoxLayout()
            container.setObjectName(f"hl_{param.value}")
            self.vl_base.addLayout(container)

            prop_info = inst.param_info.get_prop_info_by_name(param.value)
            assert prop_info is not None
            disp_lbl = QLabel()
            display_str = prop_info.disp_name

            unit = get_unit_from_str(display_str)
            self.param2unit[param] = unit
            display_str += ":"
            disp_lbl.setText(display_str)
            disp_lbl.setObjectName("lbl_disp_" + prop_info.name)
            container.addWidget(disp_lbl)

            val = inst.param_group.get_param_value(param)
            val = self.presenter.convert_db2display_value(param.value, val)
            val_lbl = QLabel()
            val_lbl.setTextInteractionFlags(
                Qt.LinksAccessibleByMouse|Qt.TextSelectableByKeyboard|Qt.TextSelectableByMouse)
            val_lbl.setText(str(val))
            val_lbl.setObjectName("lbl_" + prop_info.name)
            container.addWidget(val_lbl)
            spacer = QSpacerItem(20, 5, QSizePolicy.Maximum, QSizePolicy.Fixed)
            self.vl_base.addSpacerItem(spacer)

            self.preset_label_map[param] = val_lbl

        self.preset_editor.param2unit = self.param2unit

    def build_preset_editor_widget(self):
        pb_preset_editor = QPushButton()
        pb_preset_editor.setObjectName("pb_outclk_editor")
        pb_preset_editor.setText("Select Preset")

        self.vl_base.addWidget(pb_preset_editor)

        return pb_preset_editor

    def hide_pll_settings(self):
        assert self.block_inst is not None
        param_id_list = [
            RawSerdesConfigParamInfo.Id.ss_raw_data_rate_lane_NID,
            RawSerdesConfigParamInfo.Id.ss_raw_serdes_width_lane_NID
        ]

        for param in param_id_list:
            widget = self.param_widget_map.get(param.value)

            if widget is not None:
                widget.set_visible(False)

        widget = self.common_param_widget_map.get(CommonQuadParamInfo.Id.ss_raw_refclk_freq.value)
        assert widget is not None
        widget.set_visible(False)

    def get_design_block_type(self) -> PeriDesign.BlockType:
        return PeriDesign.BlockType.raw_serdes

    def get_ref_clk_cmn_params(self):
        return [
            CommonQuadParamInfo.Id.ss_raw_refclk_freq,
            CommonQuadParamInfo.Id.PIPE_CONFIG_CMN__config_reg_2__pma_cmn_refclk_sel,
            QuadDesignParamInfo.Id.ref_clk_pin_name,
        ]

    def get_hidden_parameters(self) -> List[str]:
        return get_hidden_parameters()

    def build_blk_graph_observer(self):
        return RawSerdesGraphObserver(self)

    def build_cmn_graph_observer(self):
        return CommonQuadGraphObserver(self)

    def build_tab_builder(self):
        return RawSerdesTabBuilder()

    def build_presenter(self):
        return RawSerdesConfigUIPresenter(self)

    def load_all_widgets_info(self):
        assert self.common_inst is not None and isinstance(self.block_inst, RawSerdes)
        self.preset_editor.build(inst=self.block_inst, cmn_inst=self.common_inst)
        super().load_all_widgets_info()
        self.hide_pll_settings()

    def get_conn_type2pin_type(self):
        return {
            RawSerdesParamInfo.Id.tx_clk_conn_type: "RAW_SERDES_TX_CLK",
            RawSerdesParamInfo.Id.rx_clk_conn_type: "RAW_SERDES_RX_CLK",
        }

    def build_mix_widgets(self) -> None:
        inst = self.block_inst
        assert inst is not None and self.presenter is not None and self.res_svc is not None

        usable_pin_types = []
        device = inst.get_device()

        if device != "":
            usable_pin_types = self.res_svc.get_usable_pin_types_by_res_name(device, inst.quad_type)

        category = "CLOCK and RESET"
        container = self.category2vl.get(category)
        assert container is not None

        # Clock resource
        param_id = RawSerdesParamInfo.Id.clk_resource_en
        en_param = inst.param_info.get_prop_info(param_id)
        assert en_param is not None
        assert en_param.category == category
        self.param_widget_map.update(
            self.gen_param_widgets_by_builder(inst, container,
            filter_func=lambda param_info: [en_param]))

        params = []
        pins = []
        for param_id, pin_type in self.get_conn_type2pin_type().items():
            # param
            param = inst.param_info.get_prop_info(param_id)
            assert param is not None
            if param.category != category:
                continue
            params.append(param)

            # pin
            pin = inst.gen_pin.get_pin_by_type_name(pin_type)
            pins.append(pin)

            pin_widgets_map, params_widgets_map = self.gen_mixed_widget_by_builder(
                inst,container,
                pins=pins, params=params,
                is_last_spacer=False,
                usable_pin_types=usable_pin_types
            )
            self.pin_widget_map.update(pin_widgets_map)
            self.param_widget_map.update(params_widgets_map)
            params.clear()
            pins.clear()

    def build_category2filter_func(self):
        def clk_reset_param(param_info: PropertyMetaData, category: str):
            return [param for param in param_info.get_prop_by_category(category)\
                if param.id not in self.get_clock_reset_order()]

        def clk_reset_pin(blk: PeriDesignGenPinItem, category: str):
            pins = [pin for pin in blk.get_pin_by_class(category) if pin.type_name not in set(self.get_conn_type2pin_type().values())]
            return pins

        def ctrl_reg_filter(param_info: PropertyMetaData, category: str):
            param_list = param_info.get_prop_by_category(category)
            set_font_param_list = self.get_ctrl_reg_order()
            order = {item:idx for idx, item in enumerate(set_font_param_list)}
            param_list = sorted(param_list, key=lambda prop: order.get(prop.id, len(param_list)))

            return param_list

        category = "CLOCK and RESET"
        self.category2filter_func[category] = (lambda p, c=category: clk_reset_param(p, c),
                                               lambda p, c=category: clk_reset_pin(p, c))

        category = "Per Lane Parameters - Control Register "
        self.category2filter_func[category] = (lambda p, c=category: ctrl_reg_filter(p, c), None)
        self.category2keep_filter_order[category] = (True, False)

    def build_param_widgets(self) -> None:
        super().build_param_widgets()

        # Preset
        self.build_preset_readonly_label()
        self.pb_preset_editor = self.build_preset_editor_widget()

    def get_ctrl_reg_order(self) -> List:
        return [
            RawSerdesConfigParamInfo.Id.ss_raw_data_rate_lane_NID,
            RawSerdesConfigParamInfo.Id.ss_raw_serdes_width_lane_NID,
            RawSerdesConfigParamInfo.Id.ss_raw_mode_lane_NID,
            RawSerdesConfigParamInfo.Id.ss_raw_bundle_mode_lane_NID,
            RawSerdesConfigParamInfo.Id.ss_raw_tx_eq_mode_lane_NID,
            RawSerdesConfigParamInfo.Id.ss_raw_main_c0_lane_NID,
            RawSerdesConfigParamInfo.Id.ss_raw_pre_c_lane_NID,
            RawSerdesConfigParamInfo.Id.ss_raw_post_c_lane_NID,
        ]

    def get_clock_reset_order(self) -> List:
        return [
            RawSerdesParamInfo.Id.clk_resource_en,
        ] + list(self.get_conn_type2pin_type().keys())

    def get_editable_preset_param(self):
        return [
            RawSerdesConfigParamInfo.Id.ss_raw_tx_eq_mode_lane_NID,
            RawSerdesConfigParamInfo.Id.ss_raw_deem_lane_NID,
        ] + self.get_flr_filter_param()

    @staticmethod
    def get_flr_filter_param():
        return [
            RawSerdesConfigParamInfo.Id.ss_raw_main_c0_lane_NID,
            RawSerdesConfigParamInfo.Id.ss_raw_pre_c_lane_NID,
            RawSerdesConfigParamInfo.Id.ss_raw_post_c_lane_NID,
        ]

    def update_option_display(self):
        super().update_option_display()

    def monitor_changes(self):
        self.pb_preset_editor.clicked.connect(self.on_show_preset_editor)
        self.preset_editor.sig_set_preset.connect(self.on_set_preset)
        super().monitor_changes()

    def stop_monitor_changes(self):
        self.pb_preset_editor.clicked.disconnect(self.on_show_preset_editor)
        self.preset_editor.sig_set_preset.disconnect(self.on_set_preset)
        super().stop_monitor_changes()

    @pyqtSlot(bool)
    def on_show_preset_editor(self, is_clicked):
        mark_unused(is_clicked)
        self.preset_editor.show()

    @pyqtSlot()
    def on_set_preset(self):
        self.blockSignals(True)
        self.stop_monitor_changes()

        if self.block_inst is None:
            self.blockSignals(False)
            self.monitor_changes()
            return

        preset_info = self.preset_editor.get_selected_preset()

        if preset_info is None or self.common_inst is None:
            self.blockSignals(False)
            self.monitor_changes()
            return

        device_name = self.block_inst.get_device()
        if device_name != "":
            all_inst = self.res_svc.find_resource_user(device_name)

            if len(all_inst) > 1:
                # Get common reference clock value
                _, cur_refclk_freq, _ = self.block_reg.get_pll_config_base_param_values(self.block_inst)
                _, selected_refclk_freq, _ = preset_info

                # Ask user to confirm the change if different clock is selected
                if cur_refclk_freq != selected_refclk_freq:
                    msg = "The selected preset will affect other lane instance(s) "\
                            "due to different reference clock frequency."
                    result = QMessageBox.warning(self.parent, "Preset Update", msg,
                            QMessageBox.StandardButton.Yes| QMessageBox.StandardButton.Cancel)

                    if result == QMessageBox.StandardButton.Cancel:
                        self.monitor_changes()
                        self.blockSignals(False)
                        return

        self.block_inst.set_pll_config_by_preset_key(preset_info, self.common_inst, IS_SHOW_HIDDEN)

        self.graph_observer.load_preset_info()
        self.graph_observer.on_graph_updated(
            self.block_inst.dependency_graph,
            updated_nodes=list(param_id.name for param_id in self.get_editable_preset_param()))

        self.monitor_changes()
        self.blockSignals(False)

        self.sig_prop_updated.emit()

    def get_label_display_name(self, widget_spec, prop_info):
        if prop_info.id in RawSerdesConfig.get_flr_filter_param():
            widget_spec.display_name = get_flr_filter_disp_name(prop_info.disp_name)
        return super().get_label_display_name(widget_spec, prop_info)
