import sys
import os


import pt_version

import util.gen_util

from design.db_patch import DesignVersion, DesignPatch
from design.db_item import GenericParamService

from tx375_device.raw_serdes.design import RawSerdes, RawSerdesRegistry
from tx375_device.raw_serdes.raw_serdes_prop_id import RawSerdesConfigParamInfo as RawSerdesParamInfo
from tx375_device.raw_serdes.design_param_info import RawSerdesDesignParamInfo

from tx375_device.common_quad.quad_prop_id import CommonQuadConfigParamInfo as CommonQuadParamInfo

sys.path.append(os.path.join(os.path.dirname(__file__), '..'))


IS_SHOW_HIDDEN = True if pt_version.PT_DEBUG_VERSION == True else False


@util.gen_util.freeze_it
class RawSerdesPatch(DesignPatch):
    """
    RawSerdes design backward compatibility patcher
    """

    def __init__(self, raw_serdes_reg: RawSerdesRegistry, device_db):
        assert raw_serdes_reg.common_quad_reg is not None
        self.common_reg = raw_serdes_reg.common_quad_reg

        # For temporary saving param information since param will be removed
        self.cmn_inst2param_info = {}

        super().__init__(raw_serdes_reg, device_db)
        self.block_reg: RawSerdesRegistry

        #
        # Patches sequence can imply dependency between patches. When the patch is applied,
        # it will be applied from the lowest version number to the highest.
        # This means it is safe for patch function to assume certain db states.
        #

        # PT-2453
        msg = "PMA Direct: Move some params between common instance and PMA Direct instance"
        self.add_patch(DesignVersion.PATCH_VERSION_20242002,
                       RawSerdesPatch.patch_update_pll_related_param_betwee_cmn_and_lane, msg)

        # PT-2464
        msg = "PMA Direct: Update ss_raw_mode_lane_NID value"
        self.add_patch(DesignVersion.PATCH_VERSION_20242003,
                       RawSerdesPatch.patch_update_ss_raw_mode_lane_nid_value, msg)

        # PT-2469
        msg = "PMA Direct: Reassign pll config parameters from default"
        self.add_patch(DesignVersion.PATCH_VERSION_20242004,
                       RawSerdesPatch.patch_update_pll_config_params, msg)

        # PT-2471
        msg = "PMA Direct: Clock output from core is auto-assigned to the clock input pin name"
        self.add_patch(DesignVersion.PATCH_VERSION_20242005,
                       RawSerdesPatch.patch_update_fixed_input_to_clkout_pin, msg)

        # PT-2558
        msg = "PMA Direct: Add new param CLK_RESOURCE_EN for selecting clock resource"
        self.add_patch(DesignVersion.PATCH_VERSION_20242010,
                       RawSerdesPatch.patch_clk_resource_select, msg)

    def patch_update_pll_related_param_betwee_cmn_and_lane(self, lane_inst: RawSerdes):
        """
        Move some params between common instance and raw serdes due to Raw PLL settings

        Updated params:
        - ss_raw_data_rate, ss_raw_mode: common instance -> Raw Serdes
        - REF_CLK_FREQUENCY: raw serdes -> common instance

        :param lane_inst: RawSerdes instance
        :return: True, if design is patched, else False
        """
        is_patch = False

        cmn_inst = self.block_reg.get_cmn_inst_by_lane_name(lane_inst.name)
        assert cmn_inst is not None

        lane_param_group = lane_inst.param_group
        lane_param_info = lane_inst.param_info
        cmn_param_group = cmn_inst.param_group

        tmp_param_info = self.cmn_inst2param_info.get(cmn_inst.name, {})

        cmn_param_list = [
            "ss_raw_data_rate",
            "ss_raw_mode"
        ]
        for cmn_param_name in cmn_param_list:
            lane_param_name = cmn_param_name + "_lane_NID"
            # Load previous common param value
            param_val = tmp_param_info.get(cmn_param_name, None)
            prop_info = lane_param_info.get_prop_info_by_name(lane_param_name)

            # PT-2453 ss_raw_data_rate is in raw_serdes (2024.M.155)
            if lane_param_group.get_param_by_name(cmn_param_name) is not None:
                lane_param_group.remove_param(cmn_param_name)
                is_patch = True

            if lane_param_group.get_param_by_name(lane_param_name) is None or\
                prop_info is None:
                continue

            if param_val is not None:
                lane_param_group.set_param_value(lane_param_name, param_val)
                is_patch = True
            # Skip if it is not removed and param is not in param group
            elif cmn_param_group.get_param_by_name(cmn_param_name) is None:
                continue
            else:
                # Save current common param value and remove it from param group
                param_val = cmn_param_group.get_param_value(cmn_param_name)
                settings = prop_info.valid_setting

                # PT-2453 Currently all the params are options,
                # later can update it if support others like range
                if isinstance(settings, list):
                    # Set to default if not valid
                    if param_val not in settings:
                        param_val = prop_info.default
                else:
                    continue

                tmp_param_info[cmn_param_name] = param_val
                self.cmn_inst2param_info[cmn_inst.name] = tmp_param_info

                lane_param_group.set_param_value(lane_param_name, param_val)
                cmn_param_group.remove_param(cmn_param_name)
                is_patch = True

        # REF_CLK_FREQUENCY: Lane param to common instance
        inst_list = sorted(self.block_reg.get_insts_by_quad_res(cmn_inst.get_device()),
                           key=lambda inst: inst.name)

        if len(inst_list) > 0:
            is_first_inst = (inst_list[0] == lane_inst)
            cmn_param_service = GenericParamService(cmn_inst.get_param_group(), cmn_inst.get_param_info())

            lane_param_name = "REF_CLK_FREQUENCY"
            param_id = CommonQuadParamInfo.Id.ss_raw_refclk_freq

            if lane_param_group.get_param_by_name(lane_param_name) is not None:
                # Save only for first instance
                if is_first_inst:
                    val = lane_param_group.get_param_value(lane_param_name)
                    cmn_param_service.set_param_value(param_id, val)

                lane_inst.param_group.remove_param(lane_param_name)
                is_patch = True

        if is_patch:
            func_name = util.gen_util.get_function_name()
            self.logger.warning(
                "{}: Patch {} applied to {}".format(func_name, DesignVersion.PATCH_VERSION_20242002,
                                                    lane_inst.name))

        return is_patch

    def patch_update_ss_raw_mode_lane_nid_value(self, lane_inst: RawSerdes):
        is_patch = False

        value_mapping = {
            "FIFO": "Tx FIFO, Rx FIFO",
            "Register": "Tx FIFO, Rx Register"
        }
        param_id = RawSerdesParamInfo.Id.ss_raw_mode_lane_NID
        param = lane_inst.param_group.get_param_by_name(param_id.value)

        # If param not found, or is valid, no need to patch
        if param is None or lane_inst.param_info.is_valid_setting(param_id, param.value):
            return is_patch

        updated_val = value_mapping.get(param.value, None)

        # If no mapping found, use default value
        if updated_val is None:
            prop_info = lane_inst.param_info.get_prop_info_by_name(param_id.value)
            if prop_info is None:
                return is_patch
            updated_val = prop_info.default

        # Check if the value is valid
        if lane_inst.param_info.is_valid_setting(param_id, updated_val):
            param.value = updated_val
            is_patch = True

        if is_patch:
            func_name = util.gen_util.get_function_name()
            self.logger.warning(
                "{}: Patch {} applied to {} with updated value {}".format(func_name, DesignVersion.PATCH_VERSION_20242003,
                                                        lane_inst.name, updated_val))

        return is_patch

    def patch_update_pll_config_params(self, lane_inst: RawSerdes):
        is_patch = False

        # Read the 3 params and find the combination in the pll config tae
        # if found, check if the pll config param has been updated to match
        # the combination in table. If it hasn't been updated, then resync ONLY
        # the pll config param

        cmn_ins = self.block_reg.get_cmn_inst_by_lane_name(lane_inst.name)
        assert cmn_ins is not None

        param_service = GenericParamService(lane_inst.get_param_group(), lane_inst.get_param_info())
        cmn_param_service = GenericParamService(cmn_ins.get_param_group(), cmn_ins.get_param_info())

        data_rate = param_service.get_param_value(RawSerdesParamInfo.Id.ss_raw_data_rate_lane_NID)
        refclk_freq = cmn_param_service.get_param_value(CommonQuadParamInfo.Id.ss_raw_refclk_freq)
        data_width = param_service.get_param_value(RawSerdesParamInfo.Id.ss_raw_serdes_width_lane_NID)

        if lane_inst.pll_cfg is not None:
            cfg_map = lane_inst.pll_cfg.get_pll_config_settings(data_rate, refclk_freq, data_width, IS_SHOW_HIDDEN)

            if cfg_map is not None:
                # Get parameters by type pll_config
                param_info = lane_inst.get_param_info()
                pll_params = param_info.get_prop_by_category("pll_config")

                updated_params = []
                for param in pll_params:
                    cur_param_value = param_service.get_param_value(param.id)
                    if param.id in cfg_map and cfg_map[param.id] != cur_param_value:
                        # Override the current PLL config value with the value in the table
                        param_service.set_param_value(param.id, cfg_map[param.id])
                        updated_params.append("{}: {}".format(param.name, cfg_map[param.id]))
                        is_patch = True

                if is_patch:
                    func_name = util.gen_util.get_function_name()
                    self.logger.warning(
                        "{}: Patch {} applied to {} with updated pll config value {}".format(func_name, DesignVersion.PATCH_VERSION_20242004,
                                                                lane_inst.name, ",".join(updated_params)))

        return is_patch

    def patch_update_fixed_input_to_clkout_pin(self, lane_inst: RawSerdes):
        """
        The generic pin PCS_CLK_RX and PCS_CLK_TX should be removed since
        it is a fixed connection from input to clkout clock which also
        depends on the mode and bundle mode.
        """
        is_patch = False

        deprecated_pin = ["PCS_CLK_RX", "PCS_CLK_TX"]

        if lane_inst is not None and lane_inst.gen_pin is not None:

            for pin_name in deprecated_pin:
                gpin = lane_inst.gen_pin.get_pin_by_type_name(pin_name)

                if gpin is not None:
                    lane_inst.gen_pin.delete_pin_by_type(pin_name)
                    is_patch = True
                    assert lane_inst.gen_pin.get_pin_by_type_name(pin_name) is None

            if is_patch:
                func_name = util.gen_util.get_function_name()
                self.logger.warning(
                    "{}: Patch {} applied to {}".format(
                        func_name, DesignVersion.PATCH_VERSION_20242005,
                        lane_inst.name))

        return is_patch

    def patch_clk_resource_select(self, lane_inst: RawSerdes):
        is_patch = False

        param_id = RawSerdesParamInfo.Id.ss_raw_mode_lane_NID
        mode = lane_inst.param_group.get_param_value(param_id.value)

        param_id = RawSerdesDesignParamInfo.Id.clk_resource_en
        is_enable = lane_inst.param_group.get_param_value(param_id.value)

        # Do not need patch if it is enabled
        if is_enable:
            return is_patch

        is_clock = False

        # Skip for register mode
        pin_type_list = ['RAW_SERDES_TX_CLK']
        if "Rx Register" not in mode:
            pin_type_list.append('RAW_SERDES_RX_CLK')

        # Check if pin name not empty
        for pin_type in pin_type_list:
            pin_name = lane_inst.gen_pin.get_pin_name_by_type(pin_type)
            if pin_name != "":
                is_clock = True
                break

        if is_clock:
            is_patch = True
            lane_inst.param_group.set_param_value(param_id.value, True)

        if is_patch:
            func_name = util.gen_util.get_function_name()
            self.logger.warning(
                "{}: Patch {} applied to {}".format(
                    func_name, DesignVersion.PATCH_VERSION_20242010,
                    lane_inst.name))

        return is_patch
