from __future__ import annotations
from typing import List, Dict, Optional, TYPE_CHECKING, TypeAlias, Tuple
import os

from util.app_setting import AppSetting

from device.block_definition import ModeInterface, Port
from device.db import PeripheryDevice

from common_device.property import PropertyMetaData
from common_device.quad.res_service import QuadResService, QuadType
from common_device.quad.lane_design import LaneBasedItem, LaneBaseRegistry

from tx375_device.raw_serdes.gui.dep_graph import RawSerdesDependencyGraph
from tx375_device.raw_serdes.design_param_info import RawSerdesDesignParamInfo as RawSerdesParamInfo
from tx375_device.raw_serdes.quad_param_info import build_param_info, get_supported_common_parameters
from tx375_device.raw_serdes.raw_serdes_prop_id import RawSerdesConfigParamInfo
from tx375_device.raw_serdes.raw_serdes_pin_dep_graph import get_hidden_ports
from tx375_device.raw_serdes.pll_cfg_param_info import get_pll_config_common_param, build_param_info as pll_cfg_build_param_info
from tx375_device.raw_serdes.pll_config import RawSerdesPLLConfig, PRESET_KEY_TYPE
from tx375_device.raw_serdes.raw_serdes_pll_cfg_prop_id import RawSerdesPLLConfigParamInfo as RawSerdesPLLParamInfo
from tx375_device.common_quad.quad_prop_id import CommonQuadConfigParamInfo as CommonQuadParamInfo
from tx375_device.common_quad.design import QuadLaneCommon
from tx375_device.common_quad.design_param_info import QuadDesignParamInfo as CmnDeisgnParamInfo


from design.db_item import (
    GenericParam,
    GenericParamService,
)

if TYPE_CHECKING:
    from common_device.quad.res_service import QuadResService
    from tx375_device.quad_pcie.device_service import QuadPCIEDeviceService
    from tx375_device.common_quad.design import QuadLaneCommonRegistry

ParamIdTypeAlias: TypeAlias = RawSerdesConfigParamInfo.Id| RawSerdesParamInfo.Id


class RawSerdes(LaneBasedItem):

    _device_port_map: Dict = {}
    dev_service = None # type: Optional[QuadPCIEDeviceService]

    def __init__(self, name: str, block_def: str = "",
                 apply_default: bool = True, pll_cfg: Optional[RawSerdesPLLConfig] = None ):
        super().__init__(name, block_def=block_def)
        self.quad_type = QuadType.raw_serdes
        self._dep_graph = RawSerdesDependencyGraph(self)
        self.pll_cfg = pll_cfg

        if apply_default:
            self.set_default_setting()

    @staticmethod
    def str2enumtype(prop_id: RawSerdesParamInfo.Id, value: str):
        str2enum_map = None
        return str2enum_map

    @staticmethod
    def enumtype2str(prop_id: RawSerdesParamInfo.Id, val):
        enum2str_map = None
        return enum2str_map

    @staticmethod
    def get_all_precision() -> Dict[ParamIdTypeAlias, int]:
        return {
            RawSerdesConfigParamInfo.Id.ss_raw_data_rate_lane_NID: 5,
        }

    @staticmethod
    def get_precision(param_id: ParamIdTypeAlias):
        return RawSerdes.get_all_precision().get(param_id, None)

    def set_default_setting(self, device_db=None):
        param_service = GenericParamService(self._param_group, self._param_info)
        for param_id in RawSerdesParamInfo.Id:
            default_val = self._param_info.get_default(param_id)
            param_service.set_param_value(param_id, default_val)

    def build_param_info(self):
        param_info = RawSerdesParamInfo()

        prop_data_list: List[PropertyMetaData.PropData] = [
            PropertyMetaData.PropData(id=RawSerdesParamInfo.Id.clk_resource_en,
                name=RawSerdesParamInfo.Id.clk_resource_en.value,
                data_type=GenericParam.DataType.dbool,
                default=False,
                disp_name='Used as Clock Resource',
                category='CLOCK and RESET'),
            PropertyMetaData.PropData(id=RawSerdesParamInfo.Id.tx_clk_conn_type,
                name=RawSerdesParamInfo.Id.tx_clk_conn_type.value,
                data_type=GenericParam.DataType.dstr,
                default="rclk",
                valid_setting=["gclk", "rclk"],
                disp_name='Transmit Clock Input Connection Type',
                category='CLOCK and RESET'),
            PropertyMetaData.PropData(id=RawSerdesParamInfo.Id.rx_clk_conn_type,
                name=RawSerdesParamInfo.Id.rx_clk_conn_type.value,
                data_type=GenericParam.DataType.dstr,
                default="rclk",
                valid_setting=["gclk", "rclk"],
                disp_name='Receive Clock Input Connection Type',
                category='CLOCK and RESET'),
            ]

        for prop_data in prop_data_list:
            param_info.add_prop_by_data(prop_data.id, prop_data)

        # Add the additional parameter that is from ICD
        more_param_info = build_param_info()
        # Combined the Enum and the ParamInfo
        param_info.concat_param_info(more_param_info)

        # Add the PLL config params
        pll_cfg_param_info = pll_cfg_build_param_info()
        # Combined the Enum and the ParamInfo
        param_info.concat_param_info(pll_cfg_param_info)

        return param_info

    def _is_clkout_pin(self, pin_type_name: str) -> bool:
        return pin_type_name in {
                                 'PCS_CLK_RX',
                                 'PCS_CLK_TX',
                                }

    def get_input_clock_type_names(self) -> List[str]:
        return ['RAW_SERDES_RX_CLK', 'RAW_SERDES_TX_CLK']
    
    def get_used_pcs_clk_names(self, reg) -> Dict[str, Tuple[str, LaneBasedItem]]:
        clk2user_name_ins_map = {}

        in2clkout_map = {
            'RAW_SERDES_TX_CLK': 'PCS_CLK_TX',
            'RAW_SERDES_RX_CLK': 'PCS_CLK_RX'
        }

        # Get the lane index
        _, lane_idx = QuadResService.break_res_name(self.get_device())

        bundle_x8_pairs, quad_lane_to_clk_src_insname = reg.determine_bundle_x8_quad_pairs_and_clk_src()

        # Check the bundle mode and take in the PCS_CLK of the
        # lane that is relevant only.
        for in_inf,clkout_inf in in2clkout_map.items():
            user_clk_name = reg.get_user_clock_pin_name(
                self, bundle_x8_pairs, quad_lane_to_clk_src_insname, in_inf)

            if user_clk_name != "":
                port_name = '{}{}'.format(clkout_inf, lane_idx)
                clk2user_name_ins_map[port_name] = (user_clk_name, self)

        return clk2user_name_ins_map
    
    def _is_input_clk_pin(self, pin_type_name: str) -> bool:
        return pin_type_name in self.get_input_clock_type_names()

    def is_skip_clk_port_name(self, port_name: str) -> bool:
        if port_name in ["PCS_CLK_RX", "PCS_CLK_TX"]:
            return True

        return False

    # Alias so that it can be access outside of the class
    def is_clkout_pin(self, pin_type_name: str) -> bool:
        return self._is_clkout_pin(pin_type_name)

    def get_clk_gpin_type_name(self, clkout_name):
        '''
        This is a very specific case to find the corresponding
        port that connects to the passed clockout name that is skip
        :param clkout_name: Skipped clock port name
        :return the corresponding created generic pin type name
        '''
        clkout_to_input_name_map = {
            "PCS_CLK_RX": "RAW_SERDES_RX_CLK",
            "PCS_CLK_TX": "RAW_SERDES_TX_CLK"
        }

        return clkout_to_input_name_map.get(clkout_name, "")

    @staticmethod
    def build_port_info(device_db: Optional[PeripheryDevice] = None, is_mockup: bool = False):
        """
        Build port info for Raw serdes which is mode based.
        This function need to be call only once since it is static shared
        between class.

        :param device_db: Device db instance
        :param is_mockup: True, build a mockup data, else build from device db
        """
        assert is_mockup is False
        assert isinstance(device_db, PeripheryDevice)

        from tx375_device.quad_pcie.device_service import QuadPCIEDeviceService

        dev_service = RawSerdesRegistry.get_blk_service(device_db)
        assert isinstance(dev_service, QuadPCIEDeviceService)

        RawSerdes.dev_service = dev_service

        mode_name_list = ['LN0_RAW_SERDES', 'LN1_RAW_SERDES', 'LN2_RAW_SERDES', 'LN3_RAW_SERDES']
        device_port_map: Dict[str, ModeInterface] = {}

        for mode_name in mode_name_list:
            device_port_map.update(dev_service.get_interface_by_mode(mode_name))

        RawSerdes._device_port_map = {}

        skip_port = ["PCS_CLK_RX", "PCS_CLK_TX"]
        for name, mode_port in device_port_map.items():
            if name in skip_port:
                continue

            class_str = mode_port.get_class()
            if class_str is not None and class_str.endswith(":SW_COMMON"):
                continue

            if mode_port.get_type() != Port.TYPE_PAD:

                # All the interface have description except some
                # which are in exlucded list
                assert mode_port.get_description() not in ("", None)
                RawSerdes._device_port_map[name] = mode_port

    def get_parameters_for_lpf_gen(self, common_reg: Optional[QuadLaneCommonRegistry]):
        '''
        Create a dictionary of paramter settings targetted
        for bitstream generation (input to hw script):
        user_registers = {
            "q#": {
                param_name (str): param_value (str),
            }
        }
        All parameters need to be written out eventhough not used
        in context
        '''
        ins_param_map = {}

        # Cannot determine when resource hasn't been assigned
        if self.get_device() == "":
            return ins_param_map

        def get_quad_name_lane_idx(res_name):
            quad_idx, lane_idx = QuadResService.break_res_name(res_name)
            if quad_idx != -1:
                return f'QUAD_{quad_idx}', lane_idx

            return "", lane_idx

        res2hwname_map = {
            "QUAD_0": "q0",
            "QUAD_1": "q1",
            "QUAD_2": "q2",
            "QUAD_3": "q3"
        }
        quad_ins_name, lane_idx = get_quad_name_lane_idx(self.get_device())

        hw_name = res2hwname_map.get(quad_ins_name, "")
        if hw_name != "":
            param_map = {}
            # we will need to populate all the parameter
            # that is only defined in the quad_pcie_param_info.py
            param_service = GenericParamService(self._param_group, self._param_info)
            # Combined the ID from RawSerdesConfigParamInfo and RawSerdesPLLParamInfo
            hw_param_id_list = list(RawSerdesConfigParamInfo.Id) + list(RawSerdesPLLParamInfo.Id)
            for param_id in hw_param_id_list:

                # The following parameter is for SW purpose use only
                # and not meant for pcr generation (verified with ICD)
                if param_id.name.startswith("sw_"):
                    continue

                pval = param_service.get_param_value(param_id)
                pname = self._param_info.get_prop_name(param_id)
                assert pname not in param_map

                # If it was a boolean, we need to convert it to 0,1
                prop_data = self._param_info.get_prop_info_by_name(pname)
                assert prop_data is not None
                pval = self.translate_hw_param_value(pval, prop_data.data_type)

                # convert to string
                # also replace the pname
                # with the correct lane prefix based on the resource where
                # "L_NID" -> L#, with # the resource lane id
                revised_pname = self.translate_hw_param_name(pname, lane_idx)

                param_map[revised_pname] = str(pval)

            if common_reg is not None:
                param_map.update(self.get_cmn_parameters_from_cmn_reg(common_reg))

            ins_param_map[hw_name] = param_map

        return ins_param_map

    def get_clock_source_usage(self, port_name: str):
        '''
        The instance pin name in gen pin is a translated
        name per mode. Other block (ie. clockmux) only save
        based on the block port name. So, we need to get
        the mapped interface name and its info through this function

        :return the gen pin interface name and the connection type of
                the clock assigned
        '''
        inf_name = ""
        conn_type = None

        tup_info = self.get_clock_input_pin_info(port_name)
        if tup_info is not None:
            inf_name, param_id = tup_info

            param_service = GenericParamService(
                self.get_param_group(), self.get_param_info())

            conn_type = param_service.get_param_value(param_id)

        return inf_name, conn_type

    def get_clk_port_to_inf_map(self):
        clk_port2inf_map = {
            "CH_NID_TX_FWD_CLK": ("RAW_SERDES_TX_CLK", RawSerdesParamInfo.Id.tx_clk_conn_type),
            "CH_NID_RX_FWD_CLK": ("RAW_SERDES_RX_CLK", RawSerdesParamInfo.Id.rx_clk_conn_type)
            # Not listing customized gen pin for the REFCLK since that
            # is not handled as part of gen pin
        }
        return clk_port2inf_map

    def get_clock_input_pin_info(self, port_name: str):
        clk_port2inf_map = self.get_clk_port_to_inf_map()

        port_info = None

        if self.get_device() != "":
            # We can only tell which port it is if the resource has been set
            quad_idx, lane_idx = QuadResService.break_res_name(self.get_device())

            if quad_idx != -1 and lane_idx != -1:
                for temp_name in clk_port2inf_map:
                    pname = temp_name.replace("_NID", f'{lane_idx}')

                    if port_name == pname:
                        port_info = clk_port2inf_map[temp_name]
                        break

        return port_info
    
    def is_input_unroutable_ti135(self, quad_idx, lane_idx, bundle_mode, pname):
        is_unroutable = False

        match quad_idx:
            case 0:
                if bundle_mode == "x4":
                    if (lane_idx == 2 and pname in ["CH2_RX_FWD_CLK", "CH2_TX_FWD_CLK"]) or \
                        (lane_idx == 3 and pname in ["CH3_RX_FWD_CLK", "CH3_TX_FWD_CLK"]):
                        is_unroutable = True

            case 1:
                if lane_idx == 1 and bundle_mode in ['x2', 'x4']:
                    if pname in ["CH1_RX_FWD_CLK", "CH1_TX_FWD_CLK"]:
                        is_unroutable = True

                elif bundle_mode == 'x4':
                    if pname in ["CH3_RX_FWD_CLK", "CH3_TX_FWD_CLK"]:
                        is_unroutable = True

            case _:
                is_unroutable = False

        return is_unroutable

    def is_input_unroutable_ti375(self, quad_idx, lane_idx, bundle_mode, pname):
        is_unroutable = False

        match quad_idx:
            case 0:
                if bundle_mode == "x4":
                    if (lane_idx == 2 and pname in ["CH2_RX_FWD_CLK", "CH2_TX_FWD_CLK"]) or \
                        (lane_idx == 3 and pname in ["CH3_RX_FWD_CLK", "CH3_TX_FWD_CLK"]):
                        is_unroutable = True

            case 1:
                if lane_idx == 1 and bundle_mode in ['x2', 'x4']:
                    if pname in ["CH1_RX_FWD_CLK", "CH1_TX_FWD_CLK"]:
                        is_unroutable = True

                elif bundle_mode == 'x4':
                    if pname in ["CH3_RX_FWD_CLK", "CH3_TX_FWD_CLK"]:
                        is_unroutable = True

            case 2:
                if lane_idx == 2:
                    if pname in ["CH2_RX_FWD_CLK", "CH2_TX_FWD_CLK"]:
                        is_unroutable = True

                elif lane_idx == 0 and bundle_mode == "x4":
                    if pname in ["CH0_RX_FWD_CLK", "CH0_TX_FWD_CLK"]:
                        is_unroutable = True

            case 3:
                if bundle_mode == "x4":
                    if lane_idx == 0 and pname in ["CH0_RX_FWD_CLK", "CH0_TX_FWD_CLK"] or \
                         lane_idx == 1 and pname in ["CH1_RX_FWD_CLK", "CH1_TX_FWD_CLK"]:
                        is_unroutable = True

            case _:
                is_unroutable = False

        return is_unroutable
    
    def is_input_quad_lane_mux_unroutable(self, device_db, quad_idx, lane_idx, bundle_mode,
                                           pname):
        '''
        :return True if the rclkmux with the current quad  + lane + bundle_mode
                is never routable with loopback on the chosen lane clock source.
                Else, False
        '''
        is_unroutable = False

        _, dev_die_name = device_db.get_device_family_die_name()

        if dev_die_name == "Ti135":
            is_unroutable = self.is_input_unroutable_ti135(quad_idx, lane_idx, bundle_mode, pname)

        else:
            is_unroutable = self.is_input_unroutable_ti375(quad_idx, lane_idx, bundle_mode, pname)

        return is_unroutable

    def is_disconnect_clock_with_clkout_loopback(self, design_db, inf_name: str) -> bool:
        '''
        This is assumed that the caller is checking for rclk connectivity
        through rclkmux.  This function is used to check if the instance
        can use the rclk connection based on its state since we want to not
        attempt to route some rclk knowing that it  can never loopback if it
        needs to drive back the clkout from core due to incompatible region
        '''
        is_disconnect = False
        param_service = GenericParamService(self.get_param_group(), self.get_param_info())

        bundle_mode = param_service.get_param_value(RawSerdesConfigParamInfo.Id.ss_raw_bundle_mode_lane_NID)
        lane_mode = param_service.get_param_value(RawSerdesConfigParamInfo.Id.ss_raw_mode_lane_NID)

        if bundle_mode in ['x2', 'x4'] and design_db is not None and \
            design_db.device_db is not None:

            if inf_name == "RAW_SERDES_TX_CLK" or \
                (inf_name == "RAW_SERDES_RX_CLK" and lane_mode.find("Rx FIFO") != -1):

                # Check device and quad info
                quad_idx, lane_idx = QuadResService.break_res_name(self.get_device())

                if quad_idx != -1 and lane_idx != -1:
                    clk_port2inf_map = self.get_clk_port_to_inf_map()

                    for temp_name, tup_info in clk_port2inf_map.items():
                        pname = temp_name.replace("_NID", f'{lane_idx}')
                        ref_inf_name, _ = tup_info

                        # Example: pname = CH1_TX_FWD_CLK, inf_name = RAW_SERDES_TX_CLK
                        if ref_inf_name == inf_name:
                            # This is the interface we're looking at (RX/TX)
                            # Check if the corresponding port name is one
                            # of those invalid ones
                            if self.is_input_quad_lane_mux_unroutable(
                                design_db.device_db, quad_idx, lane_idx, 
                                bundle_mode, pname):
                                is_disconnect = True

                            break

        return is_disconnect

    # Overload this since we have some hw name mapped to interface
    # name and it may be searched by other non-pcie itself
    def get_inf_pin_by_type_name(self, type_name: str):
        translated_name = type_name
        tup_info = self.get_clock_input_pin_info(type_name)

        if tup_info is not None:
            mapped_name, _ = tup_info
            translated_name = mapped_name

        return self.gen_pin.get_pin_by_type_name(translated_name)

    def get_hidden_ports(self):
        return get_hidden_ports()

    @staticmethod
    def get_supported_common_parameters():
        return get_supported_common_parameters()

    @staticmethod
    def get_pll_config_common_param():
        return get_pll_config_common_param()

    def set_pll_config_by_preset_key(self, preset_key: PRESET_KEY_TYPE, cmn_inst: QuadLaneCommon,
                                     is_allow_hidden: bool):
        is_set = False

        if self.pll_cfg is None:
            return is_set

        data_rate, ref_clk_freq, data_width = preset_key
        settings = self.pll_cfg.get_pll_config_settings(data_rate, ref_clk_freq, data_width, is_allow_hidden)

        if settings is None or len(settings) <= 0:
            return is_set

        # Update param (Common)
        cmn_id = CommonQuadParamInfo.Id.ss_raw_refclk_freq
        cmn_inst.param_group.set_param_value(
            cmn_id.value, ref_clk_freq)

        # Update param (Lane)
        param_service = GenericParamService(
            self.get_param_group(), self.get_param_info())

        preset_key_info = {
            RawSerdesConfigParamInfo.Id.ss_raw_data_rate_lane_NID: data_rate,
            RawSerdesConfigParamInfo.Id.ss_raw_serdes_width_lane_NID: data_width
        }
        for param_id, value in preset_key_info.items():
            param_service.set_param_value(param_id, value)

        assert isinstance(self._dep_graph, RawSerdesDependencyGraph)

        # PT-2506 Since these param can be changed by user before/ after preset,
        # we should reset the param only when user update preset
        lane_param_list = [
            RawSerdesConfigParamInfo.Id.ss_raw_deem_lane_NID,
            RawSerdesConfigParamInfo.Id.ss_raw_main_c0_lane_NID,
            RawSerdesConfigParamInfo.Id.ss_raw_pre_c_lane_NID,
            RawSerdesConfigParamInfo.Id.ss_raw_post_c_lane_NID,
            RawSerdesConfigParamInfo.Id.ss_raw_tx_eq_mode_lane_NID,
        ]
        for lane_param in lane_param_list:
            self._dep_graph.on_pll_config_key_changed(
                self._dep_graph, self.param_group, cmn_id.value, lane_param.value)

        is_set = True
        return is_set

    def get_default_pll_config_key(self, cmn_inst: QuadLaneCommon):
        data_rate = self.param_info.get_default(RawSerdesConfigParamInfo.Id.ss_raw_data_rate_lane_NID)
        data_width = self.param_info.get_default(RawSerdesConfigParamInfo.Id.ss_raw_serdes_width_lane_NID)
        ref_clk_freq = cmn_inst.param_info.get_default(CommonQuadParamInfo.Id.ss_raw_refclk_freq)

        return data_rate, ref_clk_freq, data_width

class RawSerdesRegistry(LaneBaseRegistry):
    def __init__(self, common_quad_reg: Optional[QuadLaneCommonRegistry]=None):
        setting = AppSetting()
        efxpt_home = setting.app_path[AppSetting.PathType.install]
        csvfile = os.path.normpath(efxpt_home + "/db/die/block_models/raw_serdes_pll_config.csv")
        self.pll_cfg = RawSerdesPLLConfig(csvfile)

        super().__init__(common_quad_reg)

    def load_settings_file(self, design_location: str):
        self.pll_cfg.load_settings(design_location)

    def apply_device_db(self, device_db: PeripheryDevice):
        assert device_db is not None
        self.device_db = device_db
        RawSerdes.build_port_info(self.device_db, False)

    def get_instance_class(self):
        return RawSerdes

    @staticmethod
    def get_blk_service(device_db: PeripheryDevice):
        from device.db_interface import DeviceDBService
        dbi = DeviceDBService(device_db)
        return dbi.get_block_service(DeviceDBService.BlockType.QUAD_PCIE)

    def create_instance(self, name, apply_default=True, auto_pin=False):
        with self._write_lock:
            inst = self.get_instance_class()(name)
            inst.pll_cfg = self.pll_cfg

            if self.common_quad_reg is not None:
                # Create common instance
                cmn_inst = self.create_new_common_quad_lane_inst("")
                assert isinstance(cmn_inst, QuadLaneCommon)
                self.common_quad_reg.connect_lane_inst_2cmn_inst(inst.quad_type, inst.name, cmn_inst)

                # Load dep graph for inst
                inst.build_dep(cmn_inst, self.common_quad_reg)

            if apply_default:
                inst.set_default_setting(self.device_db)
            if auto_pin:
                inst.generate_pin_name()

            self._register_new_instance(inst)
            return inst

    def get_pll_config_base_param_values(self, inst: RawSerdes):
        '''
        :return the following parameter values based on the order:
            data_rate, refclk_freq, serdes width
        '''
        data_rate = None
        refclk_freq = None
        serdes_width = None

        if inst is not None:
            cmn_ins = self.get_cmn_inst_by_lane_name(inst.name)
            assert cmn_ins is not None
            param_service = GenericParamService(inst.get_param_group(),
                                                inst.get_param_info())
            cmn_param_service = GenericParamService(cmn_ins.get_param_group(),
                                                    cmn_ins.get_param_info())
            data_rate = param_service.get_param_value(RawSerdesConfigParamInfo.Id.ss_raw_data_rate_lane_NID)
            refclk_freq = cmn_param_service.get_param_value(CommonQuadParamInfo.Id.ss_raw_refclk_freq)
            serdes_width = param_service.get_param_value(RawSerdesConfigParamInfo.Id.ss_raw_serdes_width_lane_NID)

        return data_rate, refclk_freq, serdes_width

    def check_ins_bundle_mode_rclk_usage(self):
        is_used = False

        all_inst = self.get_all_inst()

        for inst in all_inst:
            param_service = GenericParamService(inst.get_param_group(),
                                                inst.get_param_info())

            bundle_mode = param_service.get_param_value(RawSerdesConfigParamInfo.Id.ss_raw_bundle_mode_lane_NID)

            if bundle_mode in ['x2', 'x4']:
                tx_conn_type = param_service.get_param_value(RawSerdesParamInfo.Id.tx_clk_conn_type)
                rx_conn_type = param_service.get_param_value(RawSerdesParamInfo.Id.rx_clk_conn_type)
                clken = param_service.get_param_value(RawSerdesParamInfo.Id.clk_resource_en)

                lane_mode = param_service.get_param_value(RawSerdesConfigParamInfo.Id.ss_raw_mode_lane_NID)

                # This is a rough guess because it could be that the clock/lane is
                # not the one that requires loopback. So once we found a scenario
                # that needs it we stop. The caller will look further.
                # Only if it is clock that requires loopback with rclk that  needs this set to True
                if clken is True and (tx_conn_type == 'rclk' or \
                    (rx_conn_type == 'rclk' and lane_mode.find("Rx FIFO") != -1)):
                    is_used = True
                    break

        return is_used

    def get_ti375_device_rclk_mult_bundle_fix_assignment(self, bundle_mode: str):
        '''
        When in bundle mode x2,x4 (FIFO), there is a fixed mux input to
        use for specific quad-lane due to ensure that the rclk can
        route back to clkout interface in the same overlap region
        as the  clk input.
        '''

        bundle2fixedmux_rclk = {}

        x2_quad_pin2ins_map = {
            "QUAD_0": {
                        "CH0_RX_FWD_CLK": ["RCLKMUX_R0", "RCLKMUX_R1"],
                        "CH0_TX_FWD_CLK": ["RCLKMUX_R0", "RCLKMUX_R1"],
                        "CH1_RX_FWD_CLK": ["RCLKMUX_R0", "RCLKMUX_R1"],
                        "CH1_TX_FWD_CLK": ["RCLKMUX_R0", "RCLKMUX_R1"],
                        "CH2_RX_FWD_CLK": ["RCLKMUX_R2"],
                        "CH2_TX_FWD_CLK": ["RCLKMUX_R2"],
                        "CH3_RX_FWD_CLK": ["RCLKMUX_R2"],
                        "CH3_TX_FWD_CLK": ["RCLKMUX_R2"],
                      },
            "QUAD_1": {
                        "CH0_RX_FWD_CLK": ["RCLKMUX_R4"],
                        "CH0_TX_FWD_CLK": ["RCLKMUX_R4"],
                        "CH2_RX_FWD_CLK": ["RCLKMUX_R4"],
                        "CH2_TX_FWD_CLK": ["RCLKMUX_R4"],
                        "CH3_RX_FWD_CLK": ["RCLKMUX_R5"],
                        "CH3_TX_FWD_CLK": ["RCLKMUX_R5"],
                      },
            "QUAD_2": {
                        "CH0_RX_FWD_CLK": ["RCLKMUX_R6"],
                        "CH0_TX_FWD_CLK": ["RCLKMUX_R6"],
                        "CH1_RX_FWD_CLK": ["RCLKMUX_R7"],
                        "CH1_TX_FWD_CLK": ["RCLKMUX_R7"],
                        "CH3_RX_FWD_CLK": ["RCLKMUX_R7"],
                        "CH3_TX_FWD_CLK": ["RCLKMUX_R7"],
                      },
            "QUAD_3": {
                        "CH0_RX_FWD_CLK": ["RCLKMUX_R9"],
                        "CH0_TX_FWD_CLK": ["RCLKMUX_R9"],
                        "CH1_RX_FWD_CLK": ["RCLKMUX_R9"],
                        "CH1_TX_FWD_CLK": ["RCLKMUX_R9"],
                        "CH2_RX_FWD_CLK": ["RCLKMUX_R10", "RCLKMUX_R11"],
                        "CH2_TX_FWD_CLK": ["RCLKMUX_R10", "RCLKMUX_R11"],
                        "CH3_RX_FWD_CLK": ["RCLKMUX_R10", "RCLKMUX_R11"],
                        "CH3_TX_FWD_CLK": ["RCLKMUX_R10", "RCLKMUX_R11"],
                      },
        }

        # QUAD_1 and QUAD_3 are not included because rule
        # will trap usage of RCLK
        x4_quad_pin2ins_map = {
            "QUAD_0": {
                        "CH0_RX_FWD_CLK": ["RCLKMUX_R1"],
                        "CH0_TX_FWD_CLK": ["RCLKMUX_R1"],
                        "CH1_RX_FWD_CLK": ["RCLKMUX_R1"],
                        "CH1_TX_FWD_CLK": ["RCLKMUX_R1"],
                      },
            "QUAD_1": {
                        "CH0_RX_FWD_CLK": ["RCLKMUX_R4"],
                        "CH0_TX_FWD_CLK": ["RCLKMUX_R4"],
                        "CH2_RX_FWD_CLK": ["RCLKMUX_R4"],
                        "CH2_TX_FWD_CLK": ["RCLKMUX_R4"],
                      },
            "QUAD_2": {
                        "CH1_RX_FWD_CLK": ["RCLKMUX_R7"],
                        "CH1_TX_FWD_CLK": ["RCLKMUX_R7"],
                        "CH3_RX_FWD_CLK": ["RCLKMUX_R7"],
                        "CH3_TX_FWD_CLK": ["RCLKMUX_R7"],
                      },
            "QUAD_3": {
                        "CH2_RX_FWD_CLK": ["RCLKMUX_R10"],
                        "CH2_TX_FWD_CLK": ["RCLKMUX_R10"],
                        "CH3_RX_FWD_CLK": ["RCLKMUX_R10"],
                        "CH3_TX_FWD_CLK": ["RCLKMUX_R10"],
                      }
        }

        if bundle_mode == "x2":
            bundle2fixedmux_rclk = x2_quad_pin2ins_map
        elif bundle_mode == "x4":
            bundle2fixedmux_rclk = x4_quad_pin2ins_map

        return bundle2fixedmux_rclk

    def get_ti135_device_rclk_mult_bundle_fix_assignment(self, bundle_mode: str):
        '''
        When in bundle mode x2,x4 (FIFO), there is a fixed mux input to
        use for specific quad-lane due to ensure that the rclk can
        route back to clkout interface in the same overlap region
        as the  clk input.
        '''

        bundle2fixedmux_rclk = {}

        x2_quad_pin2ins_map = {
            "QUAD_0": {
                        "CH0_RX_FWD_CLK": ["RCLKMUX_R0", "RCLKMUX_R1"],
                        "CH0_TX_FWD_CLK": ["RCLKMUX_R0", "RCLKMUX_R1"],
                        "CH1_RX_FWD_CLK": ["RCLKMUX_R0", "RCLKMUX_R1"],
                        "CH1_TX_FWD_CLK": ["RCLKMUX_R0", "RCLKMUX_R1"],
                        "CH2_RX_FWD_CLK": ["RCLKMUX_R2"],
                        "CH2_TX_FWD_CLK": ["RCLKMUX_R2"],
                        "CH3_RX_FWD_CLK": ["RCLKMUX_R2"],
                        "CH3_TX_FWD_CLK": ["RCLKMUX_R2"],
                      },
            "QUAD_1": {
                        "CH0_RX_FWD_CLK": ["RCLKMUX_R4"],
                        "CH0_TX_FWD_CLK": ["RCLKMUX_R4"],
                        "CH2_RX_FWD_CLK": ["RCLKMUX_R4"],
                        "CH2_TX_FWD_CLK": ["RCLKMUX_R4"],
                        "CH3_RX_FWD_CLK": ["RCLKMUX_R5"],
                        "CH3_TX_FWD_CLK": ["RCLKMUX_R5"],
                      }
        }

        x4_quad_pin2ins_map = {
            "QUAD_0": {
                        "CH0_RX_FWD_CLK": ["RCLKMUX_R1"],
                        "CH0_TX_FWD_CLK": ["RCLKMUX_R1"],
                        "CH1_RX_FWD_CLK": ["RCLKMUX_R1"],
                        "CH1_TX_FWD_CLK": ["RCLKMUX_R1"],
                      },
            "QUAD_1": {
                        "CH0_RX_FWD_CLK": ["RCLKMUX_R4"],
                        "CH0_TX_FWD_CLK": ["RCLKMUX_R4"],
                        "CH2_RX_FWD_CLK": ["RCLKMUX_R4"],
                        "CH2_TX_FWD_CLK": ["RCLKMUX_R4"],
                      }
        }

        if bundle_mode == "x2":
            bundle2fixedmux_rclk = x2_quad_pin2ins_map
        elif bundle_mode == "x4":
            bundle2fixedmux_rclk = x4_quad_pin2ins_map

        return bundle2fixedmux_rclk

    def get_allowed_rclkmux_input_conn(self, device_db):
        # Do nothing if device db doesnt exists
        if device_db is None:
            return {}

        # Determine device
        _, dev_die_name = device_db.get_device_family_die_name()

        # Map of <QUAD>:<CLK PORT NAME> to the list of RCLKMUX_NAME that it should be
        # For those rclk input that has no restriction, it will not be included
        # in the map
        allowed_in_rclkmux_conn = {}

        gen_port_names = {
            "RAW_SERDES_TX_CLK": "CH_NID_TX_FWD_CLK",
            "RAW_SERDES_RX_CLK": "CH_NID_RX_FWD_CLK"
        }

        all_inst = self.get_all_inst()

        for inst in all_inst:
            param_service = GenericParamService(inst.get_param_group(),
                                                inst.get_param_info())

            bundle_mode = param_service.get_param_value(RawSerdesConfigParamInfo.Id.ss_raw_bundle_mode_lane_NID)
            clken = param_service.get_param_value(RawSerdesParamInfo.Id.clk_resource_en)

            if bundle_mode in ['x2', 'x4'] and inst.get_device() != "" and clken is True:

                # Check device and quad info
                quad_idx, lane_idx = QuadResService.break_res_name(inst.get_device())

                if quad_idx != -1 and lane_idx != -1:
                    # The map shows the valid rclkmux connectivity based
                    # on the bundle mode - quad -> clk gen pin name: clkmux
                    # We want to be able to see that the input vertex (ie QUAD_0:CH1_RX_FWD_CLK)
                    # have an edge to the specified clockmux only (RCLKMUX_R1 ok but not RCLKMUX_R0)
                    # NOTE that the RCLKMUX input vertex name is  RCLKMUX_R1_QUAD_0:CH1_RX_FWD_CLK
                    # (based on build_top_level_mux_graph: mux_in_name = <CLKMUX_INS_NAME>_<IN_NAME>)

                    if dev_die_name == "Ti135":
                        bundle2fixedmux_rclk = self.get_ti135_device_rclk_mult_bundle_fix_assignment(bundle_mode)
                    else:
                        bundle2fixedmux_rclk = self.get_ti375_device_rclk_mult_bundle_fix_assignment(bundle_mode)

                    quad_name = f'QUAD_{quad_idx}'
                    if quad_name in bundle2fixedmux_rclk:

                        pin2rclkmux_map = bundle2fixedmux_rclk[quad_name]

                        lane_mode = param_service.get_param_value(RawSerdesConfigParamInfo.Id.ss_raw_mode_lane_NID)

                        for inf_name, gen_pname in gen_port_names.items():
                            if inf_name == "RAW_SERDES_TX_CLK" or \
                                (inf_name == "RAW_SERDES_RX_CLK" and lane_mode.find("Rx FIFO") != -1):

                                pname = gen_pname.replace("_NID", f'{lane_idx}')

                                if pname in pin2rclkmux_map:
                                    valid_rclkmux_list = pin2rclkmux_map[pname]

                                    allowed_key = f'{quad_name}:{pname}'
                                    rclk_list = []

                                    if allowed_key in allowed_in_rclkmux_conn:
                                        rclk_list = allowed_in_rclkmux_conn[allowed_key]

                                    rclk_list += valid_rclkmux_list

                                    allowed_in_rclkmux_conn[allowed_key] = list(set(rclk_list))

        return allowed_in_rclkmux_conn

    def get_user_clock_input_pin_name(self, res_name:str, pin_type_name: str)->str:
        '''
        Find the clock input name at the specific resource lane instance (res_name)
        :param design_db: Design DB
        :param res_name: The instance with the res_name that we want to
                    get the pin name from
        :param pin_type_name: The clock input pin type name to get the user
                    clock input name (either RX/TX FWD CLK)
        :return user clock input pin name
        '''
        user_pin_name = ""

        ins_clk_input = self.get_inst_by_device_name(res_name)

        if ins_clk_input is not None:
            user_pin_name = ins_clk_input.gen_pin.get_pin_name_by_type(pin_type_name)

        return user_pin_name

    def identify_bundle_mode_instances(self, is_x8_ins_only, skip_if_res_none=False) -> Tuple[Dict[int, List[str]], Dict[Tuple[int, int], str]]:
        '''
        :param is_x8_ins_only: When True, we are only getting the quads that is associated to
                x8 into the quad_to_ins_map result. If False, the information in the
                quad_to_ins_map will contain all instances with bundle mode > x1
        :param skip_if_res_none: When True, it will return empty containers in the result
                when we found at least one instance with no resource set.
        :return 
            quad_to_ins_map:
                a map of quad index to the list of instance names that is associated
                to the quad with x8 configured
            quad_to_clk_source_inst:
                a map of the Tuple(quad index, lane index) to the instance name that has its clock resource enable
                set on any instance with bundle mode > x1 (only if resource not None)
        '''
        # Get all the lane instances associated to each quad
        # thas has bundle mode set to x8
        quad_to_ins_map: Dict[int, List[str]]= {}

        quad_to_clk_source_inst: Dict[Tuple[int, int], str] = {}

        all_inst = self.get_all_inst()
        for ins in all_inst:
            # Return nothing when we found at least one
            # instance without resource if flag is set
            if ins.get_device() == "" and skip_if_res_none:
                quad_to_ins_map = {}
                quad_to_clk_source_inst = {}
                break

            elif ins.get_device() != "":

                param_service = GenericParamService(ins.get_param_group(), ins.get_param_info())
                bundle_mode = param_service.get_param_value(RawSerdesConfigParamInfo.Id.ss_raw_bundle_mode_lane_NID)
                is_clk_src = param_service.get_param_value(RawSerdesParamInfo.Id.clk_resource_en)

                quad_num, quad_idx = QuadResService.break_res_name(ins.get_device())

                if bundle_mode == "x8" or (not is_x8_ins_only and bundle_mode in ["x2", "x4"]):

                    if quad_num not in quad_to_ins_map:
                        quad_to_ins_map[quad_num] = [ins.name]
                    else:
                        ins_list = quad_to_ins_map[quad_num]
                        ins_list.append(ins.name)
                        quad_to_ins_map[quad_num] = ins_list

                # Save the instance name with clk enabled on a > x1 bundle
                if is_clk_src and bundle_mode in ['x2', 'x4', 'x8']:
                    quad_to_clk_source_inst[(quad_num, quad_idx)] = ins.name      

        return quad_to_ins_map, quad_to_clk_source_inst

    def determine_bundle_x8_quad_pairs_and_clk_src(self) -> Tuple[List[Tuple[int, int]], Dict[Tuple[int, int], str]]:
        '''
        This is used for determining the quad pairs based on the
        lanes configured. This can only be meaningful if all the
        lane instance have been assigned to a resource.  If at least
        one instance doesn't have resource, then we don't check further.
        Caller should know when to call it based on design state.

        :return 
            quad_pair:
                a list of tuple that contains quad index that make up
                the quad pair for an x8 bundle
            quad_to_clk_source_inst: 
                a map of the instance name identified as the clock source 
                to the tuple pair of quad index (<quad#>,<quad#>)
                that has been identified as making up the x8 bundle mode
        '''

        quad_other_map = {
            0: [1],
            1: [0, 2],
            2: [1, 3],
            3: [2]
        }

        quad_idx_to_ins_names_map, quad_to_clk_source_inst = \
            self.identify_bundle_mode_instances(is_x8_ins_only=True, skip_if_res_none=True)

        quad_pair : List[Tuple[int, int]]= []

        if quad_idx_to_ins_names_map:
            if len(quad_idx_to_ins_names_map) == 4:

                # When all 4 quads are configured with x8 (2 quad pair)
                is_valid = True
                for idx in quad_other_map.keys():
                    if idx not in quad_idx_to_ins_names_map:
                        is_valid = False
                        break

                if is_valid:
                    # All quads used up
                    quad_pair.append((0,1))
                    quad_pair.append((2,3))

            elif len(quad_idx_to_ins_names_map) == 2:
                # Need to figure out which pair
                visited_quad_idx = set()

                for idx in sorted(quad_idx_to_ins_names_map.keys()):
                    if idx in quad_other_map and idx not in visited_quad_idx:
                        others_quad_idx = quad_other_map[idx]

                        for odx in others_quad_idx:
                            # If the other quad is also a configured
                            # quad. This doesn't really check validity
                            # but just to know which quad has x8 bundle mode
                            if odx in quad_idx_to_ins_names_map:
                                visited_quad_idx.add(idx)
                                visited_quad_idx.add(odx)

                                # maintain quad index order so don't
                                # need to sort later
                                if idx < odx:
                                    quad_pair.append((idx, odx))
                                else:
                                    quad_pair.append((odx, idx))

        return quad_pair, quad_to_clk_source_inst
    
    def is_quad_used_by_cur_reg(self, device_name: str) -> bool:
        used_inst_list = self.get_insts_by_quad_res(device_name)
        return len(used_inst_list) > 0

    def update_cmn_inst(self, cmn_inst_name: str):
        if self.common_quad_reg is None:
            return

        # Check if common instance removed
        cmn_inst = self.common_quad_reg.get_inst_by_name(cmn_inst_name)
        if not isinstance(cmn_inst, QuadLaneCommon):
            return

        # PT-2559: Reset param if common instance since param only visible in raw serdes
        if not self.is_quad_used_by_cur_reg(cmn_inst.get_device()): # type: ignore
            self.reset_cmn_inst(cmn_inst)

    def reset_cmn_inst(self, cmn_inst: QuadLaneCommon):
        param_id_list = [CmnDeisgnParamInfo.Id.phy_reset_en] + \
            cmn_inst.get_phy_reset_related_pcr_list()

        # This param only visible in raw serdes
        for param_id in param_id_list:
            cmn_inst.reset_param_by_id(param_id)

    def reset_phy_reset_related_pcr(self, cmn_inst: QuadLaneCommon):
        # This param only visible in raw serdes
        for param_id in cmn_inst.get_phy_reset_related_pcr_list():
            cmn_inst.reset_param_by_id(param_id)

    def assign_inst_device(self, inst: LaneBasedItem, new_dev: str):
        if inst.get_device() == new_dev or self.common_quad_reg is None:
            return super().assign_inst_device(inst, new_dev)

        # Get common inst name
        cur_device = inst.get_device()
        cmn_inst = self.get_cmn_inst_by_lane_name(inst.name)
        assert cmn_inst is not None
        cmn_inst_name = cmn_inst.name

        # Assign resource
        super().assign_inst_device(inst, new_dev)

        # Need to update common param as the some params need to reset for other protocols
        if cur_device != "":
            self.update_cmn_inst(cmn_inst_name)

        # Reload Reset pin related PCR value (device dependent)
        inst.dependency_graph.propgrate_param(inst.param_group,
                                        CmnDeisgnParamInfo.Id.phy_reset_en.value)

    def delete_inst(self, name: str):
        # Get common inst name
        inst = self.get_inst_by_name(name)

        if inst is None or self.common_quad_reg is None or inst.get_device() == "":
            return super().delete_inst(name)

        cmn_inst = self.get_cmn_inst_by_lane_name(name)
        assert cmn_inst is not None
        cmn_inst_name = cmn_inst.name

        # Delete instance
        super().delete_inst(name)

        # Update common instance
        self.update_cmn_inst(cmn_inst_name)

    def reset_inst_device(self, inst: LaneBasedItem):
        # Get common inst name
        if self.common_quad_reg is None:
            return super().reset_inst_device(inst)

        cmn_inst = self.get_cmn_inst_by_lane_name(inst.name)
        assert cmn_inst is not None
        cmn_inst_name = cmn_inst.name

        # Reset device
        if inst.get_device() == "":
            return super().reset_inst_device(inst)

        super().reset_inst_device(inst)

        # Update common instance
        self.update_cmn_inst(cmn_inst_name)

    def find_res_of_clock_input_in_bundle(self, bundle_x8_pairs,
                                          quad_lane_to_clk_src_insname, 
                                          ins_obj):
        '''
        Get the resource name of the instance that was set with clock resource
        found associated to the instance resource.
        :param bundle_x8_pairs: List of tuple that shows the x8 bundle pair quad name
        :param quad_lane_to_clk_src_insname
        :param ins_obj: The raw serdes instance that we're looking at and trying to
                find the clock input resource

        :return the resource name of the clock source for the bundle with the passed
                ins_obj
        '''
        res_name = ""

        bundle_mode = ins_obj.param_group.get_param_value(
                RawSerdesConfigParamInfo.Id.ss_raw_bundle_mode_lane_NID.value)

        if ins_obj is not None:
            quad_idx, lane_idx = QuadResService.break_res_name(ins_obj.get_device())

            # Only x8 we can have 2 quads while the other, it has to be the same
            # quad. We are assuming at this stage the design is valid  where only
            # one clock input is found per bundle. So, we're not checking for duplicates
            # (more than one clock input per bundle)
            other_quad = None

            if bundle_mode == 'x8':
                # Get the other quad index with this instance

                for quad_pair in bundle_x8_pairs:
                    q1, q2 = quad_pair

                    if quad_idx == q1 or quad_idx == q2:
                        # We found the pair
                        if quad_idx == q1:
                            other_quad = q2
                        else:
                            other_quad = q1
                        break

            for tup_quad_lane in quad_lane_to_clk_src_insname:
                quad_in, lane_in = tup_quad_lane

                if quad_in == quad_idx and bundle_mode != 'x8':
                    if bundle_mode == 'x4':
                        # This is it
                        res_name = f"Q{quad_in}_LN{lane_in}"
                        break

                    elif bundle_mode == 'x2':
                        # The valid lane pair is only [0,1] or [2,3]
                        if lane_idx in [0, 1]:
                            if lane_in in [0, 1]:
                                res_name = f"Q{quad_in}_LN{lane_in}"
                                break
                        elif lane_idx in [2, 3]:
                            if lane_in in [2, 3]:
                                res_name = f"Q{quad_in}_LN{lane_in}"
                                break                       

                elif bundle_mode == 'x8':
                    if quad_in == quad_idx or quad_in == other_quad:
                        res_name = f"Q{quad_in}_LN{lane_in}"
                        break

        return res_name

    def get_user_clock_pin_name(self, ins_obj: LaneBasedItem, bundle_x8_pairs: List[Tuple[int, int]],
                                quad_lane_to_clk_src_insname, input_clock_port_name: str):
        '''
        Override the base class so that we can get the user clock
        name from other lanes.
        :param ins_obj: The raw serdes design instance
        :param bundle_x8_pairs: List of x8 quad pair in tuples
        :param quad_lane_to_clk_src_insname:
        :param input_clock_port_name: The clock input port name used in generic pin
                    (ie RAW_SERDES_RX_CLK)

        :return the user clock input name that should be connected to the
                clkout pin. This should take care of crossing lane where
                bundle mode is x2,x4 where the lane doesn't have the clock input
                name and it should be taken from other lanes depending on the
                clk_en, bundle mode and the lane resource
        '''
        user_clk_name = ""

        param_service = GenericParamService(ins_obj.get_param_group(), 
                                            ins_obj.get_param_info())
        
        bundle_mode = param_service.get_param_value(RawSerdesConfigParamInfo.Id.ss_raw_bundle_mode_lane_NID)
        mode_type = param_service.get_param_value(RawSerdesConfigParamInfo.Id.ss_raw_mode_lane_NID)
                
        if bundle_mode in ['x2', 'x4']:
            
            if (input_clock_port_name == "RAW_SERDES_RX_CLK" and mode_type.find("Rx FIFO") != -1) or \
                input_clock_port_name == "RAW_SERDES_TX_CLK":

                quad_idx, lane_idx = QuadResService.break_res_name(ins_obj.get_device())

                if quad_idx != -1 and lane_idx != -1:
                   
                    in_res_name = self.find_res_of_clock_input_in_bundle(
                        bundle_x8_pairs, quad_lane_to_clk_src_insname, ins_obj)
                    
                    if in_res_name != "":
                        user_clk_name = self.get_user_clock_input_pin_name(
                            in_res_name, input_clock_port_name)

            else:
                user_clk_name = ins_obj.gen_pin.get_pin_name_by_type(input_clock_port_name)    
        
        elif bundle_mode == "x8" and input_clock_port_name == "RAW_SERDES_RX_CLK" and mode_type == "Rx FIFO":
            user_clk_name, _ = self.get_user_pin_name_bundle_x8(
                ins_obj, input_clock_port_name, bundle_x8_pairs, quad_lane_to_clk_src_insname)
        else:
            user_clk_name = ins_obj.gen_pin.get_pin_name_by_type(input_clock_port_name)

        return user_clk_name

    def get_user_pin_name_bundle_x8(self, inst: LaneBasedItem, inf_name: str,
                                     bundle_x8_pairs: List[Tuple[int, int]],
                                     quad_lane_to_clk_src_insname):
        '''
        Direct access for x8.
        :param ins_obj: The raw serdes design instance
        :param inf_name: The clock input port name used in generic pin
                    (ie RAW_SERDES_RX_CLK)
        :param bundle_x8_pairs: List of x8 quad pair in tuples
        :param quad_lane_to_clk_src_insname:

        :return the user clock input name that should be connected to the
                clkout pin. This should take care of crossign lane where
                bundle mode is x2,x4 where the lane doesn't have the clock input
                name and it should be taken from LN1/LN3 depending on the
                bundle mode and the lane resource
        '''
        user_pin_name = ""
        is_cur_quad = True

        # Find the clock source pin name. By that, we have to determine
        # which resource it is on this bundle 
        quad_idx, _ = QuadResService.break_res_name(inst.get_device())

        if bundle_x8_pairs:

            in_res_name = self.find_res_of_clock_input_in_bundle(
                        bundle_x8_pairs, quad_lane_to_clk_src_insname, inst)
            
            if in_res_name != "":
                if in_res_name.find(f"Q{quad_idx}") == -1:
                    is_cur_quad = False

                user_pin_name = self.get_user_clock_input_pin_name(
                    in_res_name, inf_name)
                
        return user_pin_name, is_cur_quad
