

from __future__ import annotations
from enum import Enum
from typing import TYPE_CHECKING, List, Optional, Callable

import util.gen_util as gutil

import common_device.writer as writer
from common_device.property import PropertyMetaData

from design.db_item import GenericParamService, GenericParamGroup, GenericParam
from tx375_device.quad_pcie.design_param_info import QuadPCIEDesignParamInfo
from tx375_device.quad_pcie.quad_pcie_prop_id import QuadPCIEConfigParamInfo

if TYPE_CHECKING:
    from tx375_device.quad_pcie.design import QuadPCIE


class QuadPCIESummary(writer.Summary):
    '''
    Class for Summary report generation.
    '''

    def __init__(self, device_db, name, section):
        '''
        constructor
        '''
        super().__init__(device_db, name)

        # Integer that indicates the section within the summary
        self.__section = section

    def generate_summary(self, design,
                         outfile,
                         ins_to_blk_map,
                         blk_service):
        '''
        Writes out related summary

        :param device_db: device DeviceDBService
        :param outfile: file handle that has already been opened
        :param ins_to_blk_map: A map of instance name to the
                    reference block name
        '''

        outfile.write(
            "\n---------- {}. PCI Express Usage Summary (begin) ----------\n".format(self.__section))

        # Get the list of user instances from design
        # The list should contain non-NOne type object
        all_pcie = blk_service.get_all_instances(design)

        if all_pcie:

            def get_name(pcie_obj):
                return pcie_obj.name

            index = 0
            for pcie_obj in sorted(all_pcie, key=get_name):
                if pcie_obj is not None and pcie_obj.get_device() in ins_to_blk_map:

                    param_service = GenericParamService(
                        pcie_obj.get_param_group(), pcie_obj.get_param_info())

                    if len(all_pcie) > 1:
                        outfile.write("\n***** PCIe {} *****".format(index))

                    outfile.write(
                        '\n{:58s}: {}\n'.format(
                            "Instance Name", pcie_obj.name))
                    outfile.write(
                        '{:58s}: {}\n'.format(
                            "Resource", pcie_obj.get_device()))

                    # Write all the param
                    self.write_instance_parameters(outfile, pcie_obj, param_service)

                    # Write only some of the generic pin that
                    # requires customization
                    assert pcie_obj.gen_pin is not None
                    outfile.write("\nPin Names\n")
                    self.write_selected_gen_pin_names(outfile, pcie_obj, param_service)

                    index += 1

        else:
            outfile.write("\nNo PCI Express was configured\n")

        outfile.write("\n---------- PCI Express Usage Summary (end) ----------\n")

    def write_selected_gen_pin_names(self, outfile, pcie_obj: QuadPCIE,
                                     param_service):
        assert pcie_obj.gen_pin is not None

        axi_clk_pin = pcie_obj.gen_pin.get_pin_by_type_name("AXI_CLK")
        if axi_clk_pin is not None:
            self.write_gen_pin_info(outfile, pcie_obj, axi_clk_pin)

        apb_en = param_service.get_param_value(
            QuadPCIEDesignParamInfo.Id.apb_en)

        if apb_en:
            apb_clk_pin = pcie_obj.gen_pin.get_pin_by_type_name("USER_APB_CLK")
            if apb_clk_pin is not None:
                self.write_gen_pin_info(outfile, pcie_obj, apb_clk_pin)

        pwr_en = param_service.get_param_value(QuadPCIEDesignParamInfo.Id.pwr_mgmt_en)

        if pwr_en:
            pm_clk_pin = pcie_obj.gen_pin.get_pin_by_type_name("PM_CLK")
            if pm_clk_pin is not None:
                self.write_gen_pin_info(outfile, pcie_obj, pm_clk_pin)

        debug_en = param_service.get_param_value(QuadPCIEDesignParamInfo.Id.debug_en)

        if debug_en:
            div2_clk_pin = pcie_obj.gen_pin.get_pin_by_type_name("FORWARDED_DIV2_CLK")
            if div2_clk_pin is not None:
                self.write_gen_pin_info(outfile, pcie_obj, div2_clk_pin)

                # Print the connection type
                param_info = pcie_obj.get_param_info()
                prop_info = param_info.get_prop_info_by_name(
                    QuadPCIEDesignParamInfo.Id.fpga_div2_clk_conn_type.value)

                if prop_info is not None:
                    disp_name = prop_info.disp_name
                    conn_type = param_service.get_param_value(
                        QuadPCIEDesignParamInfo.Id.fpga_div2_clk_conn_type)
                    outfile.write('{:58s}: {}\n'.format(disp_name, conn_type))


    def write_gen_pin_info(self, outfile, pcie_obj, pin_obj):
        if pin_obj is not None:
            outfile.write('{:58s}: {}\n'.format(
                    pcie_obj.get_pin_property_name(pin_obj.type_name), pin_obj.name))

    def write_params_based_on_category(self, outfile, category: str,
                                       group_display_name: str,
                                       param_info: PropertyMetaData,
                                       param_group: GenericParamGroup,
                                       pcie_inst: QuadPCIE,
                                       custom_order: Optional[List[Enum]] = None):

        prop_list: List[PropertyMetaData.PropData] = param_info.get_prop_by_category(
            category)
        available_prop_names = pcie_inst.get_available_params()

        prop_id_list: List[QuadPCIEDesignParamInfo.Id] = [
            x.id for x in prop_list if x.name in available_prop_names]
        if custom_order:
            order = {item: idx for idx, item in enumerate(custom_order)}
            prop_id_list.sort(key=lambda id: order.get(id, len(prop_id_list)))

        outfile.write("{}\n".format(group_display_name))
        param_list: List[GenericParam] = param_group.get_all_param()

        sort_func: Callable[[GenericParam], str] = lambda this_param: param_info.get_prop_info_by_name(
            this_param.name).disp_name  # type: ignore
        sorted_list = sorted(param_list, key=sort_func)

        for param in sorted_list:
            prop_info = param_info.get_prop_info_by_name(param.name)
            if prop_info is None or prop_info.id not in prop_id_list:
                continue

            disp_name = prop_info.disp_name
            self.write_param_value_pair(
                outfile, prop_info.id, prop_info.data_type, disp_name, param.value)

    def write_params_based_on_param_list(self, outfile,
                                       group_display_name: str,
                                       param_info: PropertyMetaData,
                                       param_group: GenericParamGroup,
                                       pcie_inst: QuadPCIE,
                                       custom_order: List[Enum]):

        available_prop_names = pcie_inst.get_available_params()

        outfile.write("{}\n".format(group_display_name))

        for param_id in custom_order:
            if param_id.value not in available_prop_names:
                continue

            prop_info = param_info.get_prop_info_by_name(param_id.value)
            param = param_group.get_param_by_name(param_id.value)
            if prop_info is None or param is None:
                continue

            disp_name = prop_info.disp_name
            self.write_param_value_pair(
                outfile, prop_info.id, prop_info.data_type, disp_name, param.value)

    def write_param_value_pair(self, outfile, pid, data_type, disp_name, value):
        if data_type == GenericParam.DataType.dstr:
            outfile.write('{:58s}: {}\n'.format(disp_name, value))
        elif data_type == GenericParam.DataType.dbool:
            outfile.write('{:58s}: {}\n'.format(
                disp_name, f"{value}"))
        elif data_type == GenericParam.DataType.dflo:
            outfile.write('{:58s}: {}\n'.format(
                disp_name, "{:.3f}".format(value)))
        elif data_type == GenericParam.DataType.dint:
            outfile.write('{:58s}: {}\n'.format(disp_name, f"{value}"))


    def write_instance_parameters(self, outfile, pcie_inst: QuadPCIE, param_service):
        param_info = pcie_inst.get_param_info()
        param_group = pcie_inst.param_group

        outfile.write("\n")
        self.write_params_based_on_category(
            outfile, 'Base', 'Base', param_info, param_group, pcie_inst)
        outfile.write('\n')
        self.write_params_based_on_param_list(
            outfile, 'Reference Clock', param_info, param_group, pcie_inst,
            custom_order=[
                QuadPCIEDesignParamInfo.Id.ref_clk_frequency,
                QuadPCIEConfigParamInfo.Id.PIPE_CONFIG_CMN__config_reg_2__pma_cmn_refclk_sel,
                QuadPCIEDesignParamInfo.Id.ref_clk_pin_name,
                QuadPCIEConfigParamInfo.Id.ss_refclk_onboard_osc,
            ])
        outfile.write('\n')

        custom_params = [
            QuadPCIEDesignParamInfo.Id.axi_master_en,
            QuadPCIEDesignParamInfo.Id.axi_slave_en,
            # QuadPCIEDesignParamInfo.Id.apb_en,
            QuadPCIEDesignParamInfo.Id.status_en,
            QuadPCIEDesignParamInfo.Id.cfg_snoop_en,
            QuadPCIEDesignParamInfo.Id.pwr_mgmt_en,
            # QuadPCIEDesignParamInfo.Id.vendor_en, #Hidden for now
            QuadPCIEDesignParamInfo.Id.debug_en]

        for pid in custom_params:
            prop_info = param_info.get_prop_info_by_name(pid.value)
            if prop_info is not None:
                disp_name = prop_info.disp_name
                value = param_service.get_param_value(pid)
                outfile.write('{:58s}: {}\n'.format(disp_name, value))
