'''
Copyright (C) 2017-2023 Efinix Inc. All rights reserved.

No portion of this code may be reused, modified or
distributed in any way without the expressed written
consent of Efinix Inc.

Created on Sep 9, 2023

@author: maryam
'''
from __future__ import annotations
import re
from typing import Tuple, TYPE_CHECKING, Optional, List, Dict

import util.gen_util
from util.signal_util import float_to_decimal

import device.db_interface as devdb_int
import common_device.rule as base_rule
from common_device.quad.rule import QuadRuleHW, QuadRuleOSCClock, QuadRuleInvalidHexValue
from common_device.quad.res_service import QuadResService

from design.db_item import GenericParamService, GenericParam
from tx375_device.quad_pcie.gui.presenter import QuadPCIEConfigUIPresenter
from tx375_device.quad_pcie.design_param_info import QuadPCIEDesignParamInfo as PCIEParamInfo
from tx375_device.quad_pcie.quad_pcie_prop_id import QuadPCIEConfigParamInfo as PCIEConfigParamInfo
from tx375_device.quad_pcie.design import QuadPCIE

if TYPE_CHECKING:
    from design.db import PeriDesign

    from tx375_device.fpll.design import EfxFpllV1, EfxFpllV1OutputClock, EfxFpllV1Registry
    from tx375_device.fpll.device_service import EfxFpllV1DeviceService
    from tx375_device.quad_pcie.device_service import QuadPCIEDeviceService


@util.gen_util.freeze_it
class QuadPCIEChecker(base_rule.Checker):

    def __init__(self, design):
        """
        Constructor
        """
        super().__init__(design)
        self.block_tag = "pcie"
        self.reg = None
        self.dev_service = None # type: Optional[QuadPCIEDeviceService]
        self.presenter = QuadPCIEConfigUIPresenter()
        self.is_gen4_supported = True
        self.pll_dev_service: Optional[EfxFpllV1DeviceService] = None

        if self.design is not None:
            self.reg = self.design.quad_pcie_reg
            self.presenter.setup_design(design)

            if self.design.device_db is not None:
                dbi = devdb_int.DeviceDBService(self.design.device_db)
                self.dev_service = dbi.get_block_service(
                    devdb_int.DeviceDBService.BlockType.QUAD_PCIE)

                # Override
                self.is_gen4_supported = self.dev_service.is_gen4_supported()
                self.pll_dev_service = dbi.get_block_service(
                    self.design.device_db.get_pll_type())

    def _build_rules(self):
        """
        Build mipi rule database
        """
        self._add_rule(QuadPCIERuleInstanceName())
        self._add_rule(QuadPCIERuleResource())
        self._add_rule(QuadPCIERuleAXIUsage())
        self._add_rule(QuadPCIERulePLLRefclk())
        self._add_rule(QuadPCIERulePERSTnPin())
        self._add_rule(QuadPCIERuleAXIClock())
        self._add_rule(QuadPCIERuleAPBClock())
        self._add_rule(QuadPCIERuleOSCClock())
        self._add_rule(QuadPCIERuleHW())
        self._add_rule(QuadPCIERuleInvalidHexValue())
        self._add_rule(QuadPCIERulePMClockName())
        self._add_rule(QuadPCIeRuleExternalClock())
        self._add_rule(QuadPCIeRuleGenerationLimit())

    def find_pll_output_clock(self, clk_name: str)-> Tuple[Optional[EfxFpllV1],
                                                           Optional[EfxFpllV1OutputClock]]:
        if self.design is not None and self.design.pll_reg is not None:
            all_pll_inst = self.design.pll_reg.get_all_inst()

            for pll_inst in all_pll_inst:
                out_clk = pll_inst.get_output_clock(clk_name)

                # Return on first one found. They should be
                # unique as trapped later in interface writer
                if out_clk is not None:
                    return pll_inst, out_clk

        return None, None


class QuadPCIERuleInstanceName(base_rule.Rule):
    """
    Check if the instance name is valid
    """

    def __init__(self):
        super().__init__()
        self.name = "pcie_rule_inst_name"

    def check(self, checker, design_block: QuadPCIE):
        util.gen_util.mark_unused(checker)

        pcie = design_block

        if self.check_name_empty_or_invalid(pcie.name,
                                            "Instance name is empty.",
                                            "Valid characters are alphanumeric characters with dash and underscore only."
                                            ):
            return


class QuadPCIERuleResource(base_rule.Rule):
    """
    Check if the resource name is valid.
    """

    def __init__(self):
        super().__init__()
        self.name = "pcie_rule_resource"

    def check(self, checker: QuadPCIEChecker, design_block: QuadPCIE):

        pcie = design_block

        if self.is_name_empty(pcie.get_device()):
            self.error("Resource name is empty.")
            return

        # Check if resource name is a valid device instance name
        if checker.reg is not None and checker.dev_service is not None:
            pcie_list = checker.dev_service.get_usable_instance_names()

            if pcie.get_device() not in pcie_list:
                self.error("Resource is not a valid PCIe device instance")
                return

        else:
            checker.logger.warning(
                "{}: device db is empty, skip valid instance check.".format(self.name))


class QuadPCIERuleAXIUsage(base_rule.Rule):
    """
    Check that AXI is used either one or both.
    """

    def __init__(self):
        super().__init__()
        self.name = "pcie_rule_axi"

    def check(self, checker, design_block: QuadPCIE):

        pcie = design_block

        param_service = GenericParamService(
            pcie.get_param_group(), pcie.get_param_info())

        master_axi_en = param_service.get_param_value(
            PCIEParamInfo.Id.axi_master_en)
        slave_axi_en = param_service.get_param_value(
            PCIEParamInfo.Id.axi_slave_en)

        if not master_axi_en and not slave_axi_en:
            self.error(
                "At least either AXI Master or AXI Slave has to be enabled")


class QuadPCIERulePLLRefclk(base_rule.Rule):
    """
    Check that PLL Refclk has been set correctly.
    """

    def __init__(self):
        super().__init__()
        self.name = "pcie_rule_refclk"

    def check(self, checker: QuadPCIEChecker, design_block: QuadPCIE):

        pcie = design_block

        if pcie.get_device() != "":

            param_service = GenericParamService(
                pcie.get_param_group(), pcie.get_param_info())

            ref_clk_source = param_service.get_param_value(
                PCIEConfigParamInfo.Id.PIPE_CONFIG_CMN__config_reg_2__pma_cmn_refclk_sel)

            # Find the resource associated to the PERSTN
            assert checker.design is not None

            # We only support external in the first release
            if ref_clk_source == "External":
                # This is from a package pin. Check that it is not
                # unbonded/exists in package pin
                self.check_external_refclk(pcie, checker, param_service)
            else:
                # self.warning("Only external reference clock is supported")

                if ref_clk_source == "Internal":
                    # If it is core check that user provided a pin name
                    refclk_pin_name = param_service.get_param_value(
                        PCIEParamInfo.Id.ref_clk_pin_name)

                    if refclk_pin_name == "":
                        self.error(
                            "Core refclk pin has to be specified in Internal mode")

                else:
                    assert ref_clk_source == "PLL"
                    # No checking since design requires information
                    # on which pll was selected
                    # self.check_pll_refclk(pcie, checker, param_service)

    def check_external_refclk(self, pcie_ins: QuadPCIE,
                              checker: QuadPCIEChecker, param_service: GenericParamService):
        ext_ref_clk = param_service.get_param_value(
            PCIEConfigParamInfo.Id.PMA_CMN__cmn_plllc_gen_preg__cmn_plllc_pfdclk1_sel_preg)
        assert checker.dev_service is not None

        pkg_pins_list = checker.dev_service.get_external_refclk_pins(
                pcie_ins.get_device(), f"REFCLK{ext_ref_clk[-1]}")

        if len(pkg_pins_list) == 0:
            self.error("Invalid {} selection due to pins not available in device".format(
                checker.presenter.get_displayed_ext_clk_name(ext_ref_clk)))
            return

        # PT-2577 Required to have PLL instance
        # if reference clock is from on-board osc/ not available
        param_id = PCIEConfigParamInfo.Id.ss_refclk_onboard_osc
        is_enable = pcie_ins.param_group.get_param_value(param_id.value)
        is_param_available = pcie_ins.is_param_available(param_id)

        # Nothing to do if not available
        if is_enable or not is_param_available:
            return

        pll_reg = checker.design.pll_reg
        if pll_reg is None or checker.pll_dev_service is None:
            return

        self.validate_pll_inst_settings(
            pll_reg, checker, pcie_ins) # type: ignore


    def check_pll_refclk(self, pcie_ins: QuadPCIE,
                         checker: QuadPCIEChecker, param_service:   GenericParamService):
        if checker.design.pll_reg is not None:
            if checker.design.device_db is not None:
                assert checker.dev_service is not None

                # Map of design option name to the inf name
                # inf2port_map, port2inf_map = checker.dev_service.get_mode_interface_port_mapping(
                #     pcie_ins.get_device(), "pcie", "REFCLK", is_port_to_inf=True)

                # Check that the user has chosen the correct option
                # pll_ref_clk_id = param_service.get_param_value(
                #    PCIEParamInfo.Id.pll_ref_clk_id)

                # Get the REFCLK
                refclk_idx = 2
                refclk_inf_name = "REFCLK[{}]".format(refclk_idx)

                # Get the resources associated to the chosen PLL clock
                refclk_res_name, refclk_pin_name = checker.dev_service.get_resource_on_mode_internal_pin(
                    pcie_ins.get_device(), refclk_inf_name, index=None, mode_name="pcie")

                assert refclk_res_name is not None and refclk_res_name != ""

                pll_obj = checker.design.pll_reg.get_inst_by_device_name(
                    refclk_res_name)

                if pll_obj is not None:
                    if refclk_pin_name.startswith("CLKOUT"):
                        # Extract the number
                        pin_clk_no = refclk_pin_name
                        pin_clk_no = re.sub(
                            '.*?([0-9]*)$', r'\1', pin_clk_no)

                        pll_out = pll_obj.get_output_clock_by_number(
                            int(pin_clk_no))
                        if pll_out is None or pll_out is not None and pll_out.name == "":
                            self.error("Reference clock PLL resource {} Output Clock {} has not been configured".format(
                                refclk_res_name, pin_clk_no))

                else:
                    self.error("Reference clock PLL resource {} has not been configured".format(
                        refclk_res_name))

        else:
            # No pll in this device (unlikely unless unit  test)
            self.error(
                "Invalid reference clock since PLL does not exists in the device")

    def is_ref_clock_src_valid(self, pll_inst_list: List[EfxFpllV1],
                pll_dev_service: EfxFpllV1DeviceService, design: PeriDesign,
                exp_refclk_src_list: List[str]):
        """
        Check if expected reference clock resource is used in PLL.

        :return is valid reference clock resource used and a list of valid PLL instance
        """
        # Check if ref clk setting valid
        valid_inst_list: List[EfxFpllV1] = []

        lvds_reg = design.lvds_reg
        gpio_reg = design.gpio_reg
        if lvds_reg is None or gpio_reg is None:
            return False, valid_inst_list

        for inst in pll_inst_list:
            is_valid_external = False
            if inst.ref_clock_mode != inst.RefClockModeType.external:
                continue

            ref_clk_info = inst.find_external_clock_info(inst.ext_ref_clock_id,
                        pll_dev_service, lvds_reg, gpio_reg, design.device_db)
            cur_src_list = [src for src, _, _ in ref_clk_info if src != ""]
            is_valid_external = any([exp_src in cur_src_list for exp_src in exp_refclk_src_list])
            if is_valid_external:
                valid_inst_list.append(inst)

        return len(valid_inst_list) > 0, valid_inst_list

    def validate_pll_inst_settings(self, reg: EfxFpllV1Registry,
                                   checker: QuadPCIEChecker, pcie_ins: QuadPCIE):
        design = checker.design
        if design.device_db is None or design.pll_reg is None\
            or checker.dev_service is None or checker.pll_dev_service is None:
            return
        quad_idx, _ = QuadResService.break_res_name(pcie_ins.get_device())

        # Check PLL settings
        if quad_idx == -1:
            return

        # At least 1 PLL instance with device assigned
        is_res_valid = False
        valid_clk_list: List[EfxFpllV1] = []
        invalid_clk_mapping: Dict[EfxFpllV1, int] = {}
        valid_pll_res_list = []

        # Check PLL resource and output clock
        pll_ref_clk_list = ["PMA_CMN_REFCLK_PLL_1", "PMA_CMN_REFCLK_PLL_2"]
        for pll_ref_clk in pll_ref_clk_list:
            pll_res_name_list, ref_pin = checker.dev_service.get_all_resource_on_ins_pin(
                pcie_ins.get_device(), pll_ref_clk, None)

            if len(pll_res_name_list) <= 0 or ref_pin in ("", None):
                continue

            pll_res = pll_res_name_list[0]
            valid_pll_res_list.append(pll_res)

            inst: Optional[EfxFpllV1] = reg.get_inst_by_device_name(pll_res)
            if inst is None:
                continue

            clk_idx = QuadPCIE._convert_str2pll_outclk_num(ref_pin)
            assert clk_idx != 1
            is_res_valid = True

            # Check output clock
            clk = inst.get_output_clock_by_number(clk_idx)
            if clk is None:
                invalid_clk_mapping[inst] = clk_idx
            else:
                valid_clk_list.append(inst)

        # No resource assigned
        if not is_res_valid:
            self.error(f"PLL instance with resource {'/'.join(valid_pll_res_list)}"\
                " is required to be configured when reference clock is not from on-board crystal")
            return

        # No valid output clock
        elif len(valid_clk_list) <= 0 and len(invalid_clk_mapping) > 0:
            msg = "Either one of the following PLL instance(s) required to enable output clock "\
                "when reference clock is not from on-board crystal:"
            for inst, clk_idx in sorted(invalid_clk_mapping.items(),
                                                 key=lambda k_v: k_v[0].name):
                str_clk = f"CLKOUT_{clk_idx}"
                msg += f"\n- {inst.name}: {str_clk}"
            self.error(msg)
            return

        # Check if ref clk setting valid
        exp_refclk_src_list = pcie_ins.get_expected_pll_ref_clk_src(design.device_db)
        is_valid, valid_refclk_res_inst_list = self.is_ref_clock_src_valid(
            valid_clk_list, checker.pll_dev_service, design, exp_refclk_src_list)

        if not is_valid:
            msg = "Either one of the following PLL instance(s) required to use "\
                f"{'/'.join(exp_refclk_src_list)} as external reference clock source "\
                    "when PCIe reference clock is not from on-board crystal:"
            for inst in valid_clk_list:
                msg += f"\n- {inst.name}"
            self.error(msg)
            return

        # Feedback Mode
        is_valid_fb_mode = any(
            [inst.fb_mode == inst.FeedbackModeType.local for inst in valid_refclk_res_inst_list])

        if not is_valid_fb_mode:
            msg = "Either one of the following PLL instance(s) required to use "\
                "local feedback mode when reference clock is not from on-board crystal:"
            for valid_inst in valid_refclk_res_inst_list:
                msg += f"\n- {valid_inst.name}"
            self.error(msg)
            return


class QuadPCIERulePERSTnPin(base_rule.Rule):
    """
    Check that expected PERSTN IO has been configured
    """

    def __init__(self):
        super().__init__()
        self.name = "pcie_rule_perstn"

    def check(self, checker: QuadPCIEChecker, design_block: QuadPCIE):

        pcie = design_block

        if pcie.get_device() != "":

            # Find the resource associated to the PERSTN
            assert checker.design is not None

            gpio_reg = checker.design.gpio_reg
            assert gpio_reg is not None
            assert checker.dev_service is not None

            # We want to check the resource associated to the chosen perstn resource
            if checker.design.device_db is not None:

                # Don't care about the ref pin name since
                # it is a GPIO in input mode is all we care
                pres_name = checker.dev_service.get_valid_perstn_gpio_resource(pcie.get_device())

                assert pres_name is not None and pres_name != ""

                assert checker.design.gpio_reg is not None
                gpio_obj = checker.design.gpio_reg.get_gpio_by_device_name(
                    pres_name)

                if gpio_obj is not None:
                    # Check that it is in correct GPIO input connection type
                    if gpio_obj.mode == gpio_obj.PadModeType.input:
                        in_cfg = gpio_obj.input
                        assert in_cfg is not None
                        # TODO: change type when available in_cfg.ConnType.perstn_conn
                        if in_cfg.conn_type != in_cfg.ConnType.pcie_perstn_conn: # type: ignore
                            self.error('PERSTN resource {} is not configured'
                                       ' as perstn connection'.format(pres_name))
                            return False
                        elif in_cfg.name == "":
                            self.error(
                                'PERSTN resource {} input name is empty'.format(pres_name))
                            return False

                    else:
                        self.error(
                            "PERSTN resource {} is not configured as input".format(pres_name))

                else:
                    self.error(
                        "The PERSTN resource {} has not been configured".format(pres_name))


class QuadPCIERuleAXIClock(base_rule.Rule):
    """
    Check that AXI clock sourced from PLL is configured correctly.
    """

    def __init__(self):
        super().__init__()
        self.name = "pcie_rule_axi_clock"

    def check(self, checker: QuadPCIEChecker, design_block: QuadPCIE):

        pcie = design_block

        # Find the AXI_CLK
        gen_pin = pcie.gen_pin
        assert gen_pin is not None

        axi_clk_pin = gen_pin.get_pin_by_type_name("AXI_CLK")
        if axi_clk_pin is not None:
            if axi_clk_pin.name != "":
                # Check that if it is from any of the configured PLL output
                # clock name
                _, pll_outclk = checker.find_pll_output_clock(axi_clk_pin.name)

                # Clock from PLL
                if pll_outclk is not None and pll_outclk.out_clock_freq is not None:
                    # PT-2637
                    if pll_outclk.out_clock_freq < 125 or \
                            pll_outclk.out_clock_freq > 250:
                        self.error("PCIe AXI Clock from PLL frequency {}MHz is out of range. Min=125MHz Max=250MHz".format(
                            pll_outclk.get_output_freq_str()))

            else:
                self.error('AXI Clock pin name has not been configured')


class QuadPCIERuleAPBClock(base_rule.Rule):
    """
    Check that APB clock sourced from PLL is configured correctly.
    """

    def __init__(self):
        super().__init__()
        self.name = "pcie_rule_apb_clock"

    def check(self, checker: QuadPCIEChecker, design_block: QuadPCIE):

        pcie = design_block

        param_service = GenericParamService(
            pcie.get_param_group(), pcie.get_param_info())

        apb_en = param_service.get_param_value(
            PCIEParamInfo.Id.apb_en)

        if not apb_en:
            self.error('APB enable is compulsory')
            return

        # Check the APB Clock
        gen_pin = pcie.gen_pin
        assert gen_pin is not None

        apb_clk_pin = gen_pin.get_pin_by_type_name("USER_APB_CLK")

        if apb_clk_pin is None:
            self.error("Internal Error: APB Clock is not found in PCIe instance")
            return

        if apb_clk_pin.name != "":
            # Check that if it is from any of the configured PLL output
            # clock name
            pll_inst, pll_outclk = checker.find_pll_output_clock(
                apb_clk_pin.name)

            # Clock source for APB clock is not compulsory to be sourced from PLL
            if pll_inst is None or pll_outclk is None:
                return

            # PT-2543 must be configured as local feedback mode with
            # reference clock from external
            if pll_inst.fb_mode != pll_inst.FeedbackModeType.local:
                self.error(f"PLL {pll_inst.name} with output clock driving APB clock must be configured as local feedback mode")
                return

            elif pll_inst.ref_clock_mode != pll_inst.RefClockModeType.external:
                self.error(
                    f"PLL {pll_inst.name} with output clock driving APB clock must have its reference clock mode set to external")
                return

            # PLL Output clock
            if pll_outclk.out_clock_freq is not None:
                if pll_outclk.out_clock_freq > 200:
                    self.error("APB Clock from PLL frequency {}MHz is out of range. Max=200MHz".format(
                        pll_outclk.get_output_freq_str()))

        else:
            self.error(
                'APB Clock pin cannot be empty')


class QuadPCIERulePMClockName(base_rule.Rule):
    """
    Check that the pin name is not empty for PM_CLK
    """

    def __init__(self):
        super().__init__()
        self.name = "pcie_rule_pm_clock"

    def check(self, checker, design_block: QuadPCIE):

        pcie = design_block

        param_service = GenericParamService(
            pcie.get_param_group(), pcie.get_param_info())

        pwr_mgmt_en = param_service.get_param_value(
            PCIEParamInfo.Id.pwr_mgmt_en)

        if pwr_mgmt_en:
            pin_name = design_block.gen_pin.get_pin_name_by_type("PM_CLK")
            if pin_name == "":
                self.error("Empty power management clock pin name")


class QuadPCIERuleOSCClock(QuadRuleOSCClock):
    """
    Check that the oscillator has been configured. Now not associated
    to Power Management.
    """

    def __init__(self):
        super().__init__()
        self.name = "pcie_rule_osc_clock"

class QuadPCIERuleHW(QuadRuleHW):
    '''
    Run the DRC from ICD script
    '''
    def __init__(self):
        super().__init__()
        self.name = "pcie_rule_hw_drc"

    def get_param_id_enum(self):
        return PCIEConfigParamInfo.Id

class QuadPCIERuleInvalidHexValue(QuadRuleInvalidHexValue):
    def __init__(self):
        super().__init__()
        self.name = "pcie_rule_invalid_hex_value"

class QuadPCIeRuleExternalClock(base_rule.Rule):
    def __init__(self):
        super().__init__()
        self.name = "pcie_rule_external_clock"

    def check(self, checker: QuadPCIEChecker, design_block: QuadPCIE):
        blk = design_block
        param_service = GenericParamService(blk.param_group, blk.param_info)
        ref_clk_source = param_service.get_param_value(PCIEConfigParamInfo.Id.PIPE_CONFIG_CMN__config_reg_2__pma_cmn_refclk_sel)

        if ref_clk_source != "External":
            return

        ref_clk_id = param_service.get_param_value(PCIEConfigParamInfo.Id.PMA_CMN__cmn_plllc_gen_preg__cmn_plllc_pfdclk1_sel_preg)

        if ref_clk_id != "Refclk 0":
            msg = f"PCIE only allows external clock 0 to be configured."
            self.error(msg)

class QuadPCIeRuleGenerationLimit(base_rule.Rule):
    # DEVINFRA-919, PT-2492
    def __init__(self):
        super().__init__()
        self.name = "pcie_rule_generation"

    def check(self, checker: QuadPCIEChecker, design_block: QuadPCIE):

        if checker.design is not None and checker.design.device_db is not None:
            device_db = checker.design.device_db

            # If N484 and topaz device, then we limit to Gen3
            if not checker.is_gen4_supported:
                blk = design_block
                param_service = GenericParamService(blk.param_group, blk.param_info)

                gen_value = param_service.get_param_value(PCIEConfigParamInfo.Id.PIPE_CONFIG_CMN__config_reg_0__pcie_generation_sel)
                if gen_value == "Gen4":
                    msg = "PCIe Gen4 is not supported in device {}".format(device_db.get_device_name())
                    self.error(msg)
