from __future__ import annotations

import os
from enum import Enum
from typing import List, Tuple, Callable, Optional, TYPE_CHECKING

import pt_version

from util.widget_formatter import WidgetFormatter

from common_device.property import PropertyMetaData
from tx375_device.quad_pcie.design import QuadPCIERegistry, QuadPCIE
from tx375_device.quad_pcie.design_param_info import QuadPCIEDesignParamInfo
from tx375_device.quad_pcie.quad_pcie_prop_id import QuadPCIEConfigParamInfo as PCIEConfigParamInfo

from tx375_device.quad_pcie.quad_pcie_param_info import get_hidden_parameters, get_removed_parameters
from tx375_device.quad_pcie.quad_pcie_pin_dep_graph import get_hidden_ports
from tx375_device.quad_pcie.gui.builder import QuadPCIETabBuilder

from design.db import PeriDesign
from design.db_item import GenericParam, GenericClockPin, GenericParamGroup

from common_gui.base_viewer import BaseViewer, BaseViewerModel, GenericParam, GenericParamGroup, bool2str

if TYPE_CHECKING:
    from tx375_device.quad_pcie.gui.config import QuadPCIEConfig

IS_SHOW_HIDDEN = True if pt_version.PT_DEBUG_VERSION == True else False


class QuadPCIEViewer(BaseViewer):

    # Overload
    def build(self, block_name: str, design: PeriDesign):
        self.block_name = block_name
        self.design = design

        block_reg = self.design.get_block_reg(PeriDesign.BlockType.quad_pcie)
        assert isinstance(block_reg, QuadPCIERegistry)

        block = block_reg.get_inst_by_name(block_name)
        self.block_config: QuadPCIEConfig
        self.model = QuadPCIEViewerModel(self.block_config)

        # Remarks: This function called even block is not created
        if block is None:
            self.logger.warning(f"PCIe {block_name} does not exists")
            self.model.unload()

        elif isinstance(block, QuadPCIE):
            self.model.load(self.design, block_name=block_name)

        self.tv_viewer.setModel(self.model)

        formatter = WidgetFormatter()
        formatter.refresh_table_view_format(self.tv_viewer)


class QuadPCIEViewerModel(BaseViewerModel):
    def __init__(self, config: QuadPCIEConfig):
        super().__init__()
        self.config = config

    def load(self, design: PeriDesign, block_name: str):
        self.design = design
        block_reg = self.design.get_block_reg(PeriDesign.BlockType.quad_pcie)
        assert isinstance(block_reg, QuadPCIERegistry)

        block = block_reg.get_inst_by_name(block_name)

        # Block name can be from block folder (e.g. quad_pcie_info)
        if block is None:
            return

        assert isinstance(block, QuadPCIE), f"failed to find quad_pcie instance for block {block_name}"

        self.add_row_item("Instance Name", block.name)
        self.add_row_item("PCIe Resource", block.get_device())

        param_info = block.get_param_info()
        param_group = block.param_group

        def add_row_group_by_prop_category(category: str, group_display_name: str,
                                           custom_order: Optional[List[Enum]]=None,
                                           add_extra_info_func: Optional[Callable[[QuadPCIE], None]]=None):
            """
            Add Mutliple rows in the viewer about the parameters in specific cateogory

            :param category: The category of the parameters to be added
            :param group_display_name: The display name of the highlighted row at the top of the group
            :param custom_order: Arrangement orders of the parameters row, Default will be sorted by parameter's display name if not provided
            :param add_extra_info_func: Hook function to add extra rows at the bottom of the group
            """
            prop_list: List[PropertyMetaData.PropData] = []

            if category == "" and custom_order is not None:
                for param_id in custom_order:
                    prop = param_info.get_prop_info_by_name(param_id.value)
                    if prop is not None:
                        prop_list.append(prop)
            else:
                prop_list = param_info.get_prop_by_category(category)

            if len(prop_list) == 0:
                return

            available_prop_names = block.get_available_params()

            hidden_params = []
            if not IS_SHOW_HIDDEN:
                hidden_params = get_hidden_parameters()

            prop_id_list: List[QuadPCIEDesignParamInfo.Id] = [x.id for x in prop_list \
                if x.name in available_prop_names and x.name not in hidden_params and \
                    x.name not in get_removed_parameters()]

            if custom_order:
                order = {item:idx for idx, item in enumerate(custom_order)}
                prop_id_list.sort(key=lambda id: order.get(id, len(prop_id_list)))

            if len(prop_id_list) <= 0:
                return

            if group_display_name != "":
                self.add_row_item(group_display_name, "", highlight=True)

            self.add_generic_param_by_type(param_group, param_info, include_type=prop_id_list, is_sort=custom_order is None)

            # additional info
            if add_extra_info_func is not None:
                add_extra_info_func(block)

        mode = param_group.get_param_value(
            PCIEConfigParamInfo.Id.PIPE_CONFIG_CMN__config_reg_0__mode_select.value)

        hidden_tabs = QuadPCIETabBuilder.hidden_categories

        add_row_group_by_prop_category("Base", "Base")
        add_row_group_by_prop_category("Base (Root Port only)", "")
        add_row_group_by_prop_category("", "Reference Clock", custom_order=[
            QuadPCIEDesignParamInfo.Id.ref_clk_frequency,
            PCIEConfigParamInfo.Id.PIPE_CONFIG_CMN__config_reg_2__pma_cmn_refclk_sel,
            QuadPCIEDesignParamInfo.Id.ref_clk_pin_name,
            QuadPCIEDesignParamInfo.Id.pll_ref_clk0,
            QuadPCIEDesignParamInfo.Id.pll_ref_clk1,
        ])

        ref_clk = param_group.get_param_value(
            PCIEConfigParamInfo.Id.PIPE_CONFIG_CMN__config_reg_2__pma_cmn_refclk_sel.value)

        if ref_clk == "External":
            # Special case for external clock
            prop_id = PCIEConfigParamInfo.Id.PMA_CMN__cmn_plllc_gen_preg__cmn_plllc_pfdclk1_sel_preg
            ext_ref_clk: str = param_group.get_param_value(prop_id.value)
            ext_ref_clk = ext_ref_clk.replace("Refclk ", "CMN_REFCLK")
            disp_name = param_info.get_display_name(prop_id)
            self.add_row_item(disp_name, ext_ref_clk)

            # Other params for external clock
            add_row_group_by_prop_category("", "",
                custom_order=self.config.get_external_ref_clk_param_order())

        add_row_group_by_prop_category("reset", "Reset")

        # Load generated tab param category
        for category in QuadPCIETabBuilder.param_categories:
            if not IS_SHOW_HIDDEN and category in hidden_tabs:
                continue
            desc = QuadPCIETabBuilder.category2display_name.get(category, category)
            add_row_group_by_prop_category(category, desc)

            for child in QuadPCIETabBuilder.parent_category2children.get(category, []):
                desc = QuadPCIETabBuilder.category2display_name.get(child, child)
                add_row_group_by_prop_category(child, desc)

                # Maximum 2 layers of children
                for sec_child in QuadPCIETabBuilder.parent_category2children.get(child, []):
                    sec_desc = QuadPCIETabBuilder.category2display_name.get(sec_child, sec_child)
                    if "Physical Function" in child:
                        sec_desc = f"{child} - {sec_desc}"
                    add_row_group_by_prop_category(sec_child, sec_desc)

        # AXI tab
        add_row_group_by_prop_category("AXI", "AXI")

        if param_group.get_param_value(QuadPCIEDesignParamInfo.Id.axi_master_en.value):
            add_row_group_by_prop_category("AXI Master", "AXI Master", custom_order=[
                QuadPCIEDesignParamInfo.Id.axi_master_address,
                QuadPCIEDesignParamInfo.Id.axi_master_data_width,
            ])
        if param_group.get_param_value(QuadPCIEDesignParamInfo.Id.axi_slave_en.value):
            add_row_group_by_prop_category("AXI Slave", "AXI Slave", custom_order=[
                QuadPCIEDesignParamInfo.Id.axi_slave_address,
                QuadPCIEDesignParamInfo.Id.axi_slave_data_width,
            ])

        # Load if any parameters in pin tab
        categories = QuadPCIETabBuilder.pin_categories[1:] # Skip AXI tab
        for category in categories:
            if not IS_SHOW_HIDDEN and category in hidden_tabs:
                continue
            display_name = QuadPCIETabBuilder.category2display_name.get(category, category)
            add_row_group_by_prop_category(category, display_name)

            # Child tab
            for child in QuadPCIETabBuilder.parent_category2children.get(category, []):
                desc = QuadPCIETabBuilder.category2display_name.get(child, child)
                add_row_group_by_prop_category(child, desc)

        ## Load Pin information
        self.add_row_item("Pins", "", highlight=True)
        available_pin_type_names = set(block.get_available_pins())

        pin_section_color = BaseViewerModel.get_secondary_highlight_colour()

        hidden_ports = []
        if not IS_SHOW_HIDDEN:
            hidden_ports = get_hidden_ports()

        def _is_pin_available(pin_type: str):
            return pin_type in available_pin_type_names and pin_type not in hidden_ports

        def _add_pin_view(block: QuadPCIE, pin_class: str, filter_func: Callable):
            if mode == "Root Port" and pin_class in QuadPCIETabBuilder.root_port_exclude_category:
                return

            pin_type_name_list = block.get_pin_type_by_class(pin_class)

            pin_type_name_list = list(filter(filter_func, pin_type_name_list))

            desc = QuadPCIETabBuilder.category2display_name.get(pin_class, pin_class)
            if desc.endswith(("Channel", "Sideband")):
                # AXI related
                desc_list = desc.split(":")
                desc = f"{desc_list[1]} Pin ({desc_list[0]})"
            else:
                desc = desc.split(":")[-1] + " Pin"

            if len(pin_type_name_list) > 0:
                self.add_row_item(desc, "", highlight=True, highlight_colour=pin_section_color)
                self.add_generic_pin_by_type(block, include_type=pin_type_name_list, is_sort=False)

        _add_pin_view(block, "reset", filter_func=_is_pin_available)
        for pin_class in QuadPCIETabBuilder.pin_categories:
            _add_pin_view(block, pin_class, filter_func=_is_pin_available)

            for child in QuadPCIETabBuilder.parent_category2children.get(pin_class, []):
                _add_pin_view(block, child, filter_func=_is_pin_available)

                # Maximum 2 layers of children
                for sec_child in QuadPCIETabBuilder.parent_category2children.get(child, []):
                    _add_pin_view(block, sec_child, filter_func=_is_pin_available)

    def add_generic_param_by_type(self, gen_param_group: GenericParamGroup, param_info: PropertyMetaData, enumtype2str_func=None, include_type=[], is_sort=True):
        assert self.config.presenter is not None

        filtered_list: List[Tuple[GenericParam, PropertyMetaData.PropData]] = []

        param_list: List[GenericParam] = gen_param_group.get_all_param()
        for param in param_list:
            prop_info = param_info.get_prop_info_by_name(param.name)
            if prop_info is None or prop_info.id not in include_type:
                continue
            filtered_list.append((param, prop_info))

        if is_sort:
            quad_pcie_param_list = sorted(filtered_list,
                                    key=lambda i: i[1].disp_name)
        else:
            # By Default, Use the order in include_type
            quad_pcie_param_list = sorted(filtered_list,
                                    key=lambda i: include_type.index(i[1].id))

        for param, prop_info in quad_pcie_param_list:
            disp_name = prop_info.disp_name if prop_info.disp_name != "" else param.name
            display_value = self.config.presenter.convert_db2display_value(param.name, param.value)
            self._add_single_param(prop_info.id, prop_info.data_type, disp_name, display_value, enumtype2str_func)

    def add_generic_pin_by_type(self, pin_container, include_type=[], is_sort=True, is_show_inv_clock=False):

        pin_group = pin_container.gen_pin
        pin_list = pin_group.get_all_pin()
        if len(pin_list) == 0:
            return

        # Default sorting by name
        if is_sort:
            txf_pin_list = sorted(pin_list,
                                     key=lambda this_pin: pin_container.get_pin_property_name(this_pin.type_name))
        else:
            txf_pin_list = pin_list

        for pin in txf_pin_list:
            if pin.type_name not in include_type:
                continue

            prop_name = pin_container.get_pin_property_name(pin.type_name)

            if prop_name == "":
                prop_name = pin.type_name

            self.add_row_item(prop_name, pin.name)

            if is_show_inv_clock and isinstance(pin, GenericClockPin):
                prop_name = "Invert {}".format(prop_name)
                self.add_row_item(prop_name, bool2str(pin.is_inverted))


if __name__ == "__main__":
    pass
