"""

Copyright (C) 2017-2023 Efinix Inc. All rights reserved.

No portion of this code may be reused, modified or
distributed in any way without the expressed written
consent of Efinix Inc.

Created on Aug 18, 2023

@author: Shirley Chan
"""
from __future__ import annotations

from PyQt5.QtWidgets import QComboBox, QLabel, QLineEdit, QBoxLayout, QWidget
from design.db_item import GenericParamService, PeriDesignItem, GenericClockPin
from tx375_device.quad_pcie.design import QuadPCIE
from tx375_device.quad_pcie.design_param_info import QuadPCIEDesignParamInfo, get_pipe_freq_range
from tx375_device.quad_pcie.quad_pcie_param_info import get_hidden_param_values, get_display_values, get_hidden_parameters
from tx375_device.quad_pcie.quad_pcie_prop_id import QuadPCIEConfigParamInfo as PCIEConfigParamInfo

import util.gui_util as gui_util
from typing import TYPE_CHECKING, Dict, Optional, List, Any
from device.db_interface import DeviceDBService

from tx375_device.quad_pcie.device_service import QuadPCIEDeviceService

from common_gui.base_presenter import BasePresenter
from common_gui.builder.base import WidgetGroup
from common_gui.builder.primitive_widget import ChoiceWgtSpec

if TYPE_CHECKING:
    from design.db import PeriDesign


class QuadPCIEConfigUIPresenter(BasePresenter):

    def __init__(self, parent=None):
        super().__init__(parent=parent)
        self._quad_pcie_dev_service: Optional[QuadPCIEDeviceService] = None

    def setup_design(self, design: PeriDesign):
        self.design = design
        dev_db_service = DeviceDBService(design.device_db)
        self._quad_pcie_dev_service = dev_db_service.get_block_service(DeviceDBService.BlockType.QUAD_PCIE)

    def get_resource_names(self) -> List[str]:
        assert self._quad_pcie_dev_service is not None
        return self._quad_pcie_dev_service.get_usable_instance_names()

    def setup_ref_clk_option(self, block: QuadPCIE, cb_pll_clk_0: QComboBox, cb_pll_clk_1: QComboBox,
                             cb_ext_clk: QComboBox, is_show_hidden):
        cb_pll_clk_0.clear()
        dev_list = block.get_all_ref_clk0_pll_options(block.get_device())
        cb_pll_clk_0.addItems(dev_list)

        cb_pll_clk_1.clear()
        dev_list = block.get_all_ref_clk1_pll_options(block.get_device())
        cb_pll_clk_1.addItems(dev_list)

        cb_ext_clk.clear()
        options = self.get_displayed_ext_clk_option(block, is_show_hidden)
        cb_ext_clk.addItems(options)

    def get_displayed_ext_clk_option(self, block: QuadPCIE, is_show_hidden: bool):
        display_options = []
        param_id = PCIEConfigParamInfo.Id.PMA_CMN__cmn_plllc_gen_preg__cmn_plllc_pfdclk1_sel_preg
        options = block.param_info.get_valid_setting(param_id)
        hidden_options = get_hidden_param_values().get(param_id.value, [])
        assert isinstance(options, list)

        for option in options:
            assert isinstance(option, str)
            if not is_show_hidden and option in hidden_options:
                continue
            display_options.append(self.get_displayed_ext_clk_name(option))

        return display_options

    def get_displayed_ext_clk_name(self, ref_clk: str):
        return ref_clk.replace("Refclk ", "CMN_REFCLK")

    def get_ext_clk_name(self, ref_clk: str):
        return ref_clk.replace("CMN_REFCLK", "Refclk ")

    def load_ref_clk_res(self, block: QuadPCIE, cb_pll_clk_0: QComboBox, cb_pll_clk_1: QComboBox,
                         cb_ext_clk: QComboBox, is_show_hidden: bool):
        assert block is not None
        self.setup_ref_clk_option(block, cb_pll_clk_0, cb_pll_clk_1, cb_ext_clk, is_show_hidden)

        param_service = GenericParamService(block.get_param_group(), block.get_param_info())
        ref_clk_src0: str = param_service.get_param_value(QuadPCIEDesignParamInfo.Id.pll_ref_clk0)
        ref_clk_src1: str = param_service.get_param_value(QuadPCIEDesignParamInfo.Id.pll_ref_clk1)
        ext_ref_clk: str = param_service.get_param_value(PCIEConfigParamInfo.Id.PMA_CMN__cmn_plllc_gen_preg__cmn_plllc_pfdclk1_sel_preg)
        ext_ref_clk = self.get_displayed_ext_clk_name(ext_ref_clk)

        gui_util.set_config_widget(widget=cb_pll_clk_0, value=ref_clk_src0, tooltip=ref_clk_src0)
        gui_util.set_config_widget(widget=cb_pll_clk_1, value=ref_clk_src1, tooltip=ref_clk_src1)
        gui_util.set_config_widget(widget=cb_ext_clk, value=ext_ref_clk, tooltip=ext_ref_clk)

    def updated_param_option(self, block: QuadPCIE, param_id: PCIEConfigParamInfo.Id , cb_param: QComboBox):
        cb_param.clear()
        settings = block.param_info.get_valid_setting(param_id)
        assert isinstance(settings, list)

        # Get hidden options
        hidden_vals = []
        if param_id.value in get_hidden_param_values():
            hidden_vals = get_hidden_param_values().get(param_id.value, [])

        display_settings = set(settings) - set(hidden_vals)
        cb_param.addItems(display_settings)

        val = block.param_group.get_param_value(param_id.value)
        gui_util.set_config_widget(widget=cb_param, value=val, tooltip=val)

    def update_all_option_display(self, param_widget_map: Dict[str, WidgetGroup], is_show_hidden: bool):
        for param_name in self.get_update_param_display_list():
            if not is_show_hidden and param_name in get_hidden_parameters():
                continue

            widget_group = param_widget_map.get(param_name, None)
            assert widget_group is not None, f"cannot find parameter {param_name}"
            widget = widget_group.get_widget()
            assert isinstance(widget, QComboBox)
            spec = widget_group.spec
            assert isinstance(spec, ChoiceWgtSpec)
            self.update_option_display(widget, param_name, spec)

    def update_option_display(self, widget: QComboBox, param_name: str, spec: ChoiceWgtSpec):
        ui_options = []
        db_options = [widget.itemText(idx) for idx in range(widget.count())]

        widget.clear()

        for idx, db_option in enumerate(db_options):
            ui_value = self.convert_db2display_value(param_name, db_option)
            assert isinstance(ui_value, str)
            ui_options.append(ui_value)

            widget.addItem(ui_value)
            widget.setItemData(idx, db_option)

        spec.choices = ui_options

    def convert_db2display_value(self, param_name: str, db_value: Any) -> Any:
        ui_value = db_value

        if param_name in get_display_values():
            display_mapping = get_display_values().get(param_name)
            assert display_mapping is not None and isinstance(ui_value, str)

            for db_str, display_str in display_mapping.items():
                ui_value = ui_value.replace(db_str, display_str)

        elif param_name == PCIEConfigParamInfo.Id.PIPE_CONFIG_CMN__config_reg_0__pcie_generation_sel.value:
            min_freq, max_freq = get_pipe_freq_range(db_value)

            if db_value == "Gen4":
                ui_value = "{} (max {:.1f} Gbps)".format(db_value, min_freq)
            else:
                ui_value = "{} ({:.1f} Gbps)".format(db_value, min_freq)

        return ui_value

    def convert_display_value2db_value(self, param_name: str, display_value: str):
        db_value = display_value

        if param_name in get_display_values():
            display_mapping = get_display_values().get(param_name)
            assert display_mapping is not None and isinstance(db_value, str)

            for db_str, display_str in display_mapping.items():
                db_value = db_value.replace(display_str, db_str)

        elif param_name == PCIEConfigParamInfo.Id.PIPE_CONFIG_CMN__config_reg_0__pcie_generation_sel.value:
            db_value = display_value.split(" (")[0]

        return db_value

    def load_property(self, block: QuadPCIE, le_name: QLineEdit, cb_device: QComboBox):
        assert block is not None
        assert le_name is not None
        assert cb_device is not None

        gui_util.set_config_widget(widget=le_name, value=block.name, tooltip=block.name)

        self.setup_device_option(cb_device=cb_device)
        gui_util.set_config_widget(widget=cb_device, value=block.get_device(), tooltip=block.get_device())

    def load_generic_pin_property(self, db_item: PeriDesignItem, widget_map: Dict):
        """
        Load generic pin data from design to widget

        :param db_item: Design db item with generic pin
        :param widget_map: A map of widget name to its auto generated widget container/group
        """

        if db_item is None or widget_map is None:
            return

        assert isinstance(db_item, QuadPCIE)
        gen_pin = db_item.gen_pin
        for widget_name, widget_grp in widget_map.items():
            # widget name is pin type name
            pin = gen_pin.get_pin_by_type_name(widget_name)
            if pin is not None:
                gui_util.set_config_widget(widget_grp.get_widget(), pin.name, pin.name)

                # If this is a clock pin widget, we also load the invert
                if widget_grp.widget_type == WidgetGroup.WidgetType.clock_pin and \
                    isinstance(pin, GenericClockPin):
                    gui_util.set_config_widget(widget_grp.cb_is_inverted,
                                               pin.is_inverted, "{}".format(pin.is_inverted))

    def load_ref_clk_property(self, db_item: QuadPCIE, ref_clk_inst: Optional[QLabel],
                                  ref_clk_res: QLabel, param_id: PCIEConfigParamInfo.Id | QuadPCIEDesignParamInfo.Id):
        assert db_item is not None
        assert isinstance(db_item, QuadPCIE)
        assert ref_clk_res is not None
        assert self._quad_pcie_dev_service is not None
        assert self.design is not None
        assert self.design.device_db is not None

        quad_pcie_res_name = db_item.get_device()
        if quad_pcie_res_name == "":
            if ref_clk_inst is not None:
                ref_clk_inst.setText("Unknown")

            ref_clk_res.setText("Unknown")
            return

        pin, res_name_list = db_item.get_ref_clk_info(self._quad_pcie_dev_service, self.design, param_id)

        if ref_clk_inst is not None:
            ref_clk_inst.setText(pin)

        ref_clk_res.setText(", ".join(res_name_list))

    def load_perstn_res_property(self, db_item: QuadPCIE, lbl_perstn_gpio_res_list: QLabel,
                                 lbl_perstn_inst_list: QLabel):
        assert self._quad_pcie_dev_service is not None

        quad_pcie_res_name = db_item.get_device()
        if quad_pcie_res_name == "":
            lbl_perstn_gpio_res_list.setText("Unknown")
            lbl_perstn_inst_list.setText("Unknown")
            return

        pin, res_name_list = db_item.get_perstn_info(self._quad_pcie_dev_service, self.design)

        lbl_perstn_inst_list.setText(pin)
        lbl_perstn_gpio_res_list.setText(", ".join(res_name_list))

    def get_param_info(self, param_widget: QWidget, db_item: QuadPCIE, str2enumtype_funcs, user_data: bool = False):
        param_name = None
        param_value = None
        widget_name = param_widget.objectName()

        if isinstance(param_widget, QComboBox):
            param_name = widget_name[3:]

            if param_name == PCIEConfigParamInfo.Id.PIPE_CONFIG_CMN__config_reg_0__pcie_generation_sel.value:
                param_value = param_widget.currentText()[:4]

            elif param_name in get_display_values():
                param_value = param_widget.itemData(param_widget.currentIndex())

        if param_name is None or param_value is None:
            return super().get_param_info(param_widget, db_item, str2enumtype_funcs, user_data)

        return param_name, param_value

    def get_update_param_display_list(self):
        return list(get_display_values().keys()) + [
            PCIEConfigParamInfo.Id.PIPE_CONFIG_CMN__config_reg_0__pcie_generation_sel.value
        ]

    def get_display_param_value(self, param_grp, param_name):
        ui_value = param_grp.get_param_value(param_name)

        if param_name in get_display_values() or \
            param_name == PCIEConfigParamInfo.Id.PIPE_CONFIG_CMN__config_reg_0__pcie_generation_sel.value:
            ui_value = self.convert_db2display_value(param_name, ui_value)

        return ui_value
