"""

Copyright (C) 2017-2023 Efinix Inc. All rights reserved.

No portion of this code may be reused, modified or
distributed in any way without the expressed written
consent of Efinix Inc.

Created on Aug 18, 2023

@author: Shirley Chan
"""

from __future__ import annotations
from functools import partial
import os
from typing import Callable, Dict, List, Tuple, Union, TYPE_CHECKING, Optional
from PyQt5.QtCore import Qt, pyqtSlot

from PyQt5.QtWidgets import (QHBoxLayout, QSizePolicy,
                             QVBoxLayout, QWidget,
                             QComboBox, QLineEdit,
                             QMessageBox, QTabWidget, QSpacerItem)

import pt_version

import util.gui_util as gui_util

from design.db import PeriDesign
from design.db_item import GenericParamService, GenericPin
from common_device.property import PropertyMetaData
from common_device.quad.res_service import QuadResService

from common_gui.spec_builder import SpecBuilder
from common_gui.builder.primitive_widget import PrimitiveWidgetBuilder, TextWgtSpec, StaticTextWgtSpec
from common_gui.builder.base import WidgetGroup
from common_gui.base_config import BaseConfig
from common_gui.builder.smart_widget import PinWidgetGroup, GenericPinWidgetBuilder

from tx375_device.quad_pcie.design import QuadPCIERegistry, QuadPCIE
from tx375_device.quad_pcie.design_param_info import QuadPCIEDesignParamInfo as QuadPCIEParamInfo
from tx375_device.quad_pcie.quad_pcie_param_info import get_hidden_parameters, get_hidden_param_values, get_removed_parameters
from tx375_device.quad_pcie.quad_pcie_pin_dep_graph import get_hidden_ports
from tx375_device.quad_pcie.quad_pcie_prop_id import QuadPCIEConfigParamInfo as PCIEConfigParamInfo
from tx375_device.quad_pcie.gui.presenter import QuadPCIEConfigUIPresenter
from tx375_device.quad_pcie.gui.graph_observer import QuadPCIEGraphObserver
from tx375_device.quad_pcie.gui.builder import QuadPCIETabBuilder

if TYPE_CHECKING:
    from tx375_device.quad_pcie.gui.main_window import QuadPCIEWindow


FILTER_PROP_FUNC = Callable[[PropertyMetaData], List[PropertyMetaData.PropData]]
FILTER_PIN_FUNC = Callable[[QuadPCIE], List[GenericPin]]

IS_SHOW_HIDDEN = True if pt_version.PT_DEBUG_VERSION == True else False


def build_param_widgets(block_inst: QuadPCIE, container, category: Optional[str] = None,
                        filter_func: Optional[FILTER_PROP_FUNC] = None, keep_filter_order: bool = False,
                        is_last_spacer=False) -> Dict[str, WidgetGroup]:
    assert (category is None and filter_func is not None) or (category is not None)

    spec_builder = SpecBuilder()
    widget_builder = PrimitiveWidgetBuilder()
    param_info = block_inst.get_param_info()
    param_group = block_inst.get_param_group()

    if category is not None and filter_func is None:
        filter_func = lambda p,c =category: p.get_prop_by_category(c) # type: ignore

    assert filter_func is not None

    for idx, prop_info in enumerate(filter_func(param_info)):
        # Skip building hidden param
        if not IS_SHOW_HIDDEN and prop_info.name in get_hidden_parameters():
            continue

        if prop_info.name in get_removed_parameters():
            continue

        param = param_group.get_param_by_name(prop_info.name)
        assert param is not None
        widget_spec = spec_builder.build_single_param_widget_spec(param, param_info)
        assert widget_spec is not None, f"{prop_info.name} is None"
        if keep_filter_order:
            widget_spec.seqno = idx
        widget_builder.add_widget_spec(widget_spec)

    return widget_builder.build(container, is_sort=True,
                                is_sort_seqno=keep_filter_order,
                                is_last_spacer=is_last_spacer)

def build_pin_widgets(block_inst: QuadPCIE, container: Union[QVBoxLayout, QHBoxLayout],
                      category: Optional[str] = None, filter_func: Optional[FILTER_PIN_FUNC] = None,
                      is_laster_spacer: bool = False) -> Dict[str, PinWidgetGroup]:
    assert hasattr(block_inst, 'gen_pin')
    assert (category is None and filter_func is not None) or (category is not None)

    pin_builder = GenericPinWidgetBuilder()
    spec_builder = SpecBuilder()

    if category is not None and filter_func is None:
        filter_func = lambda b, c=category: b.get_pin_by_class(c) # type: ignore

    assert filter_func is not None
    pins = filter_func(block_inst)
    hidden_ports = []
    if not IS_SHOW_HIDDEN:
        hidden_ports = get_hidden_ports()
    pin_name_list = [p.type_name for p in pins if p.type_name not in hidden_ports]
    spec_builder.add_pin_widget_spec(block_inst, pin_name_list, pin_builder)

    return pin_builder.build(container, is_sort=True, is_last_spacer=is_laster_spacer)

def build_mixed_widget(block_inst: QuadPCIE,
                        container: QVBoxLayout| QHBoxLayout,
                        pins: List[GenericPin],
                        params: List[PropertyMetaData.PropData],
                        is_last_spacer=False) -> Tuple[
                            Dict[str, PinWidgetGroup],
                            Dict[str, WidgetGroup]
                        ] :
    """
    Build A Grouped Widget where pin and parameters widget put together
    """
    assert len(pins) > 0
    assert len(params) > 0

    group_widget = QWidget()
    group_widget.setSizePolicy(QSizePolicy.MinimumExpanding, QSizePolicy.Minimum)
    vbox_layout = QVBoxLayout(group_widget)
    vbox_layout.setContentsMargins(0, 0, 0, 0)

    pin_widgets_map = build_pin_widgets(block_inst,
                                        container,
                                        filter_func=lambda block_inst: pins
                                        )

    params_widgets_map = build_param_widgets(block_inst,
                                             container,
                                             filter_func=lambda param_info: params
                                             )

    if is_last_spacer:
        builder = PrimitiveWidgetBuilder()
        builder.add_spacer(container)

    return pin_widgets_map, params_widgets_map

def build_readonly_param_widget(obj2_display_name: Dict[str, str], container: QHBoxLayout| QVBoxLayout):
    widget_builder = PrimitiveWidgetBuilder()

    for obj_name, display_name in obj2_display_name.items():
        spec = StaticTextWgtSpec(obj_name, display_name)
        widget_builder.add_widget_spec(spec)

    # Build param
    return widget_builder.build(container)


class QuadPCIEConfig(BaseConfig):
    """
    Controller for specific instance configuration
    """

    def __init__(self, parent: QuadPCIEWindow):
        super().__init__(parent)
        self.parent: QWidget # type: ignore

        self.res_svc = QuadResService()

        # Qt widget
        self.tabw_config = parent.tabw_config
        self.tabw_config.tabBar().setUsesScrollButtons(False)
        self.tab_base = parent.tab_base
        self.tab_rst = parent.tab_rst
        self.tabw_pin = parent.tabw_pin

        self.vl_prop_param = parent.vl_prop_param
        self.vl_ref_clk_param = parent.vl_ref_clk_param

        self.vl_rst = parent.vl_rst

        # Ref clock related
        self.sw_ref_clk = parent.sw_ref_clk
        self.pg_ext_clk = parent.pg_ext_clk
        self.pg_pll = parent.pg_pll

        self.gb_ext_refclk = parent.gb_ext_refclk
        self.gb_pll_clk = parent.gb_pll_clk
        self.cb_pll_clk_0 = parent.cb_pll_clk_0
        self.cb_pll_clk_1 = parent.cb_pll_clk_1
        self.cb_ext_clk = parent.cb_ext_clk
        self.widget_ext_ref_clk_sel = parent.widget_ext_ref_clk_sel
        self.widget_ref_clk_res = parent.widget_ref_clk_res
        self.widget_ref_clk_inst = parent.widget_ref_clk_inst
        self.lbl_refclk_res = parent.lbl_refclk_res
        self.lbl_pll_res_0 = parent.lbl_pll_res_0
        self.lbl_pll_res_1 = parent.lbl_pll_res_1
        self.lbl_pll_inst_0 = parent.lbl_pll_inst_0
        self.lbl_pll_inst_1 = parent.lbl_pll_inst_1

        # PERSTn resource
        self.lbl_perstn_gpio_res_list = parent.lbl_perstn_gpio_res_list
        self.lbl_perstn_inst_list = parent.lbl_perstn_inst_list

        self.le_name: QLineEdit = parent.le_name
        self.cb_device: QComboBox = parent.cb_device
        self.gb_ref_clk = parent.gb_refclk

        self.param_widget_map: Dict[str, WidgetGroup] = {}
        self.pin_widget_map: Dict[str, PinWidgetGroup] = {}

        self.graph_observer = QuadPCIEGraphObserver(self)
        self.presenter: Optional[QuadPCIEConfigUIPresenter]
        self.block_inst: Optional[QuadPCIE]
        self.block_reg: QuadPCIERegistry
        self.is_monitor_dep_graph = False

        self.is_built_widget = False
        self.category2tab_page: Dict[str, QWidget] = {}
        self.vl_ext_clk = parent.vl_ext_clk
        self.category2vl: Dict[str, QVBoxLayout] = {
            'reset': self.vl_rst,
            'External Clock': self.vl_ext_clk
        }
        self.parent_category2sub_tabw: Dict[str, QTabWidget] = {}

        self.pf_spacer: Optional[QSpacerItem] = None

        self.read_only_widget_map: Dict[str, WidgetGroup] = {}

    def build_param_widgets(self) -> None:
        quad_pcie = self.block_inst
        assert quad_pcie is not None and self.presenter is not None

        # reference clock
        container = self.vl_ref_clk_param
        self.param_widget_map.update(
            build_param_widgets(quad_pcie,
                                container,
                                filter_func=lambda param_info: [
                                        param_info.get_prop_info(QuadPCIEParamInfo.Id.ref_clk_frequency), # type: ignore
                                        param_info.get_prop_info(PCIEConfigParamInfo.Id.PIPE_CONFIG_CMN__config_reg_2__pma_cmn_refclk_sel), # type: ignore
                                    ],
                                keep_filter_order=True
                                )
        )
        # base
        container = self.vl_prop_param
        mode_id = PCIEConfigParamInfo.Id.PIPE_CONFIG_CMN__config_reg_0__mode_select

        def build_base_param(filter_func: FILTER_PROP_FUNC):
            self.param_widget_map.update(
                build_param_widgets(quad_pcie,
                                    container,
                                    keep_filter_order=True,
                                    filter_func=filter_func,
                                    )
            )

        build_base_param(lambda param_info: [param_info.get_prop_info(mode_id)]) # type: ignore
        self.build_rootport_warning_label()
        build_base_param(lambda param_info: [param for param in param_info.get_prop_by_category('Base')\
            if param.id != mode_id])

        param_only_categories = [
            "Device Capability",
        ]
        def get_ext_clk_params(param_info: PropertyMetaData):
            param_id_list = self.get_external_ref_clk_param_order()
            return self.get_prop_list_with_order(param_info, param_id_list)

        for category, container in self.category2vl.items():
            is_last_spacer = False
            filter_func = None

            if category in param_only_categories:
                is_last_spacer = True
            elif category in QuadPCIETabBuilder.parent_category2children.keys():
                is_last_spacer = False

            if category in ["Debug", "Power Management"]:
                # build later
                continue
            elif category in ["AXI Master", "AXI Slave", "reset"]: # Read-only param
                continue
            elif category == "External Clock":
                filter_func = get_ext_clk_params

            self.param_widget_map.update(
                build_param_widgets(quad_pcie,
                                    container,
                                    category=category,
                                    filter_func=filter_func,
                                    keep_filter_order=(filter_func is not None),
                                    is_last_spacer=is_last_spacer
                                    )
            )

        widget_builder = PrimitiveWidgetBuilder()
        prop_info = quad_pcie.get_param_info().get_prop_info(QuadPCIEParamInfo.Id.ref_clk_pin_name)
        assert isinstance(prop_info, PropertyMetaData.PropData)
        spec = TextWgtSpec(widget_name=prop_info.name, display_name=prop_info.disp_name)
        widget_builder.add_widget_spec(spec)
        self.param_widget_map.update(
            widget_builder.build(self.vl_ref_clk_param, is_sort=False, is_last_spacer=False)
        )

        # Function
        category = "Function"
        container = self.category2vl.get(category)
        assert container is not None

        builder = PrimitiveWidgetBuilder()
        self.pf_spacer = builder.add_spacer(container)

    def get_external_ref_clk_param_order(self) -> List:
        return [
                PCIEConfigParamInfo.Id.PIPE_CONFIG_CMN__config_reg_2__pma_cmn_refclk_termen,
                PCIEConfigParamInfo.Id.PIPE_CONFIG_CMN__config_reg_2__pma_cmn_refclk1_termen,
                PCIEConfigParamInfo.Id.ss_refclk_onboard_osc,
        ]

    def get_prop_list_with_order(self, param_info: PropertyMetaData,
                                 param_id_list: List,
                                 category: Optional[str] = None):
        prop_list = []

        for param_id in param_id_list:
            prop = param_info.get_prop_info(param_id)
            if prop is not None:
                prop_list.append(prop)

        if category is not None:
            prop_list += [prop for prop in param_info.get_prop_by_category(category) \
                if prop not in prop_list]

        return prop_list

    def build_rootport_warning_label(self):
        container = self.vl_prop_param
        obj2display_name = {
            "rootport_limit": \
                "** limited features, only to be used for devkit to test / "\
                "evaluate Efinix device’s PCIe EP functionality"
        }
        self.read_only_widget_map.update(build_readonly_param_widget(obj2display_name, container))

    def build_tabs(self):
        blk_inst = self.block_inst
        assert blk_inst is not None

        # Build all tabs
        builder = QuadPCIETabBuilder()
        builder.build(IS_SHOW_HIDDEN)

        self.category2tab_page.update(builder.category2tabs)
        self.category2vl.update(builder.category2vl)
        self.parent_category2sub_tabw = builder.parent_category2sub_tabw

        # print("Categories not in builder: ", set(blk_inst.categories)-set(self.category2tab_page.keys()))

        idx = self.tabw_config.count() - 1 # Ensure Pin should be the last tab
        # Add param tab
        for category in builder.param_categories:
            tab = builder.category2tabs.get(category, None)
            if tab is None or tab.parent() is not None:
                continue

            builder.insert_tab(idx, tab, self.tabw_config, category)
            idx += 1

        builder.format_tab_widget(self.tabw_config)

        # Add pin tab
        for category in builder.pin_categories:
            tab = builder.category2tabs.get(category, None)
            if tab is None or tab.parent() is not None:
                continue

            builder.add_tab(tab, self.tabw_pin, category)

    def build(self, block_name: str, design: PeriDesign):
        """
        Update editor content based on currently selected block instance
        """
        self.design = design
        self.disable_mouse_wheel_scroll()

        quad_pcie_reg = self.design.get_block_reg(PeriDesign.BlockType.quad_pcie)
        assert quad_pcie_reg is not None
        assert isinstance(quad_pcie_reg, QuadPCIERegistry)

        # Setup the resource service for Quad PCIe
        self.res_svc.build(design)

        quad_pcie = quad_pcie_reg.get_inst_by_name(block_name)

        if quad_pcie is None:
            return

        assert isinstance(quad_pcie, QuadPCIE)

        if self.is_monitor:
            self.stop_monitor_changes()

        self.block_reg = quad_pcie_reg
        self.block_inst = quad_pcie

        if self.presenter is None:
            self.presenter = QuadPCIEConfigUIPresenter(self)
            self.presenter.setup_design(self.design)

        if not self.is_built_widget:
            self.clear_all_layout_widget()
            self.build_all_widgets()

            self.is_built_widget = True

        self.tabw_config.setCurrentIndex(0)

        self.presenter.load_base_property(self.block_inst, self.le_name, self.cb_device)
        self.presenter.load_ref_clk_res(self.block_inst,
                                        self.cb_pll_clk_0, self.cb_pll_clk_1,
                                        self.cb_ext_clk, IS_SHOW_HIDDEN)
        # Trigger the graph observer to synchronize the param / pin availablity
        self.graph_observer.on_graph_updated(self.block_inst.dependency_graph,
                                                updated_nodes=list(self.param_widget_map.keys()) + list(self.pin_widget_map.keys()))

        # Not using update_perstn_res_property to prevent viewer loaded twice
        self.load_perstn_res_property()

        gui_util.set_child_combo_spinbox_filter(self)
        self.monitor_changes()

    def clear_all_layout_widget(self):
        vl_list = [
            self.vl_ref_clk_param,
            self.vl_prop_param,
        ]
        new_vl = list(self.category2vl.values())
        vl_list += new_vl
        for vl in vl_list:
            gui_util.remove_layout_children(vl)

        self.param_widget_map.clear()
        self.pin_widget_map.clear()

    def build_all_widgets(self):
        self.build_tabs()

        self.build_param_widgets()
        self.build_pin_widgets()

        if not IS_SHOW_HIDDEN:
            self.hide_unsupported_param()

        self.update_option_display()

    def build_pin_widgets(self):
        assert self.block_inst is not None
        apb_top_pin = [
            "USER_APB_CLK"
        ]
        pin_class = "APB"
        container = self.category2vl.get(pin_class, None)

        # PT-2169 convenient user’s input to add source of this clock
        def apb_filter_pin(blk: QuadPCIE):
            return [pin for pin in blk.get_pin_by_class("APB") if pin.type_name not in apb_top_pin]

        if container is not None:
            filter_func = lambda b: [b.gen_pin.get_pin_by_type_name(pin) for pin in apb_top_pin]

            self.pin_widget_map.update(
                build_pin_widgets(self.block_inst,
                                    container,
                                    pin_class,
                                    is_laster_spacer=False,
                                    filter_func=filter_func,
            ))

        for pin_class, container in self.category2vl.items():
            is_last_spacer = True
            filter_func = None

            if pin_class in QuadPCIETabBuilder.parent_category2children.keys():
                is_last_spacer = False
            if pin_class in ["Debug", "Power Management"]:
                continue
            if pin_class == "APB":
                filter_func = apb_filter_pin

            self.pin_widget_map.update(
                build_pin_widgets(self.block_inst,
                                  container,
                                  pin_class,
                                  is_laster_spacer=is_last_spacer,
                                  filter_func=filter_func,
            ))

        # Debug tab
        if IS_SHOW_HIDDEN:
            self.build_debug_tab()
        self.build_pwr_manage_tab()

    def build_debug_tab(self):
        assert self.block_inst is not None
        container = self.category2vl.get("Debug", None)
        assert container is not None

        quad_pcie = self.block_inst

        self.param_widget_map.update(
            build_param_widgets(quad_pcie,
                                container,
                                filter_func=lambda param_info: [
                                    param_info.get_prop_info(QuadPCIEParamInfo.Id.debug_en), # type: ignore
                                ],
                                keep_filter_order=True,
                                is_last_spacer = False
                                )
        )

        def debug_filter(blk: QuadPCIE):
            return [pin for pin in blk.get_pin_by_class("Debug") if pin.type_name != "FORWARDED_DIV2_CLK"]

        self.pin_widget_map.update(build_pin_widgets(quad_pcie, container, filter_func=debug_filter))

        # FPGA divider
        fpga_conn_type = quad_pcie.param_info.get_prop_info(QuadPCIEParamInfo.Id.fpga_div2_clk_conn_type)
        params = [fpga_conn_type]
        pins = [quad_pcie.gen_pin.get_pin_by_type_name('FORWARDED_DIV2_CLK')]
        pin_widgets_map, params_widgets_map = build_mixed_widget(quad_pcie,container,
                                                                 pins=pins, params=params, # type: ignore
                                                                 is_last_spacer=True)
        self.pin_widget_map.update(pin_widgets_map)
        self.param_widget_map.update(params_widgets_map)

    def build_pwr_manage_tab(self):
        assert self.block_inst is not None
        category = "Power Management"
        container = self.category2vl.get(category, None)
        assert container is not None

        quad_pcie = self.block_inst

        set_top_params = [
            QuadPCIEParamInfo.Id.pwr_mgmt_en,
            PCIEConfigParamInfo.Id.ss_pcie_aspm,
            PCIEConfigParamInfo.Id.i_client_PF0__i_L1_PM_ctrl_1__L1ASPML11EN,
            PCIEConfigParamInfo.Id.i_client_PF0__i_L1_PM_ctrl_1__L1ASPML12EN,
            PCIEConfigParamInfo.Id.i_client_PF0__i_L1_PM_ctrl_1__L1PML11EN,
            PCIEConfigParamInfo.Id.i_client_PF0__i_L1_PM_ctrl_1__L1PML12EN,

            PCIEConfigParamInfo.Id.i_client_RC__i_link_ctrl_status__ASPMC,
            PCIEConfigParamInfo.Id.i_client_RC__i_L1_PM_ctrl_1__L1ASPML11EN,
            PCIEConfigParamInfo.Id.i_client_RC__i_L1_PM_ctrl_1__L1ASPML12EN,
            PCIEConfigParamInfo.Id.i_client_RC__i_L1_PM_ctrl_1__L1PML11EN,
            PCIEConfigParamInfo.Id.i_client_RC__i_L1_PM_ctrl_1__L1PML12EN,
        ]

        # Special case
        skip_param = [
            QuadPCIEParamInfo.Id.pmclk_conn_type, # Display with clock pin
        ] + set_top_params

        self.param_widget_map.update(
            build_param_widgets(quad_pcie,
                                container,
                                filter_func=partial(self.get_prop_list_with_order,
                                                    param_id_list=set_top_params),
                                keep_filter_order=True,
                                is_last_spacer = False
                                )
        )
        def get_pwr_manage_param_info(param_info: PropertyMetaData):
            return [
                param for param in param_info.get_prop_by_category(category) \
                    if param.id not in skip_param
            ]

        self.param_widget_map.update(
            build_param_widgets(quad_pcie,
                                container,
                                filter_func=get_pwr_manage_param_info,
                                keep_filter_order=True,
                                is_last_spacer = False
                                )
        )

        # conn type
        conn_type = quad_pcie.param_info.get_prop_info(QuadPCIEParamInfo.Id.pmclk_conn_type)
        params = [conn_type]
        pins = [quad_pcie.gen_pin.get_pin_by_type_name('PM_CLK')]
        pin_widgets_map, params_widgets_map = build_mixed_widget(quad_pcie,container,
                                                                 pins=pins, params=params, # type: ignore
                                                                 is_last_spacer=False)

        def pwr_filter(blk: QuadPCIE):
            return [pin for pin in blk.get_pin_by_class("Power Management") if pin.type_name != "PM_CLK"]

        self.pin_widget_map.update(build_pin_widgets(quad_pcie, container, filter_func=pwr_filter))

        self.pin_widget_map.update(pin_widgets_map)
        self.param_widget_map.update(params_widgets_map)

    def hide_unsupported_param(self):
        assert self.presenter is not None and self.block_inst is not None

        for param_name in get_hidden_param_values().keys():
            param_id = PCIEConfigParamInfo.Id(param_name)

            # reference clock related parameters
            if param_id in [PCIEConfigParamInfo.Id.PMA_CMN__cmn_plllc_gen_preg__cmn_plllc_pfdclk1_sel_preg]:
                continue

            widget_group = self.param_widget_map.get(param_id.value, None)
            assert widget_group is not None, f"cannot find parameter {param_id.value}"
            widget = widget_group.get_widget()
            assert isinstance(widget, QComboBox)
            self.presenter.updated_param_option(self.block_inst, param_id, widget)

    def update_option_display(self):
        assert self.presenter is not None
        self.presenter.update_all_option_display(self.param_widget_map, IS_SHOW_HIDDEN)

    @pyqtSlot()
    def on_param_changed(self):
        assert self.block_inst is not None
        assert self.presenter is not None

        widget = self.sender()
        assert isinstance(widget, QWidget)

        if self.is_monitor:
            # Keep dependency graph monitor changes
            self.is_monitor_dep_graph = False
            self.stop_monitor_changes()
            self.is_monitor_dep_graph = True

        if self.presenter.save_generic_param_property(self.block_inst, widget):
            self.sig_prop_updated.emit()

        self.monitor_changes()

    @pyqtSlot()
    def on_pin_changed(self):
        assert self.block_inst is not None
        assert self.presenter is not None

        widget = self.sender()
        if self.presenter.save_generic_pin_property(self.block_inst, widget):
            self.sig_prop_updated.emit()

    @pyqtSlot(int)
    def on_clkpin_invert_changed(self, state):
        assert self.block_inst is not None
        assert self.presenter is not None

        widget = self.sender()

        if self.presenter.save_generic_pin_clk_invert_property(self.block_inst, widget):
            self.sig_prop_updated.emit()

    @pyqtSlot(int)
    def on_pll_clk_changed(self, index: int):
        assert self.block_inst is not None
        assert self.presenter is not None

        sender = self.sender()
        assert isinstance(sender, QComboBox)
        pll_clk = sender.itemText(index)
        param_id = None

        if self.is_monitor:
            self.stop_monitor_changes()

        if sender == self.cb_pll_clk_0:
            param_id = QuadPCIEParamInfo.Id.pll_ref_clk0
        elif sender == self.cb_pll_clk_1:
            param_id = QuadPCIEParamInfo.Id.pll_ref_clk1

        assert param_id is not None
        param_service = GenericParamService(self.block_inst.get_param_group(),
                                            self.block_inst.get_param_info())
        param_service.set_param_value(param_id, pll_clk)
        self.graph_observer.load_pll_ref_clk(self.block_inst)
        self.sig_prop_updated.emit()

        self.monitor_changes()

    @pyqtSlot(int)
    def on_ext_clk_changed(self, index: int):
        assert self.block_inst is not None
        assert self.presenter is not None

        sender = self.sender()
        assert isinstance(sender, QComboBox)
        ext_clk = sender.itemText(index)
        param_service = GenericParamService(self.block_inst.get_param_group(),
                                            self.block_inst.get_param_info())

        db_ext_clk = self.presenter.get_ext_clk_name(ext_clk)
        param_id = PCIEConfigParamInfo.Id.PMA_CMN__cmn_plllc_gen_preg__cmn_plllc_pfdclk1_sel_preg

        param_service.set_param_value(param_id, db_ext_clk)
        self.graph_observer.load_ext_ref_clk(self.block_inst)
        self.sig_prop_updated.emit()

        self.monitor_changes()

    @pyqtSlot()
    def on_device_changed(self):
        if self.block_inst is None:
            return

        assert self.presenter is not None
        cb_device = self.sender()

        assert isinstance(cb_device, QComboBox)
        self.stop_monitor_changes()
        new_device = cb_device.currentText()
        if new_device != "":
            if new_device == self.block_inst.get_device():
                self.monitor_changes()
                return

            # If the quad_pcie device name is not in the valid list, then retain
            # current assignment (No change)
            res_label = self.lbl_refclk_res

            if new_device == "None":
                self.block_reg.reset_inst_device(self.block_inst)
                res_label.setText("Unknown")
            else:
                cur_device = self.block_inst.get_device()
                users_list = self.res_svc.find_resource_user(new_device)

                # PT-2303 Update based on UI review
                if len(users_list) > 0:
                    # Check if the resource is already used by other protocal
                    is_other_protocal = False
                    for inst in users_list:
                        if not isinstance(inst, QuadPCIE):
                            is_other_protocal = True
                            break

                    if is_other_protocal:
                        msg = f"You are already using {new_device} for a different "\
                            "transceiver protocol. Please choose another resource."
                    else:
                        msg = "Resource {} is already used. Please choose another.".format(new_device)

                    QMessageBox.warning(self.parent, "Update resource", msg) # type: ignore
                    self.logger.error("Fail to assign resource, it has been used")

                    # Revert to original name before edit
                    cb_device.setCurrentText(cur_device if cur_device != "" else "None")
                    self.monitor_changes()
                    return

                self.block_reg.assign_inst_device(self.block_inst, new_device)

            # Reload options
            self.presenter.load_ref_clk_res(
                self.block_inst,
                self.cb_pll_clk_0, self.cb_pll_clk_1,
                self.cb_ext_clk,
                IS_SHOW_HIDDEN)

            # Reload information
            self.graph_observer.on_graph_updated(
                self.block_inst.dependency_graph,
                updated_nodes=[PCIEConfigParamInfo.Id.PIPE_CONFIG_CMN__config_reg_2__pma_cmn_refclk_sel.value])
            self.update_perstn_res_property()

            self.sig_prop_updated.emit()
            self.sig_resource_updated.emit(new_device)
            self.monitor_changes()

    def update_ref_clk_property(self):
        if self.block_inst is None:
            return

        assert isinstance(self.block_inst, QuadPCIE)
        res_label = self.lbl_refclk_res
        assert self.presenter is not None

        self.presenter.load_ref_clk_property(self.block_inst, None, res_label,
                                             PCIEConfigParamInfo.Id.PMA_CMN__cmn_plllc_gen_preg__cmn_plllc_pfdclk1_sel_preg)
        self.sig_prop_updated.emit()

    def load_perstn_res_property(self):
        if self.block_inst is None:
            return

        assert isinstance(self.block_inst, QuadPCIE)
        inst_label = self.lbl_perstn_inst_list
        res_label = self.lbl_perstn_gpio_res_list
        assert self.presenter is not None

        self.presenter.load_perstn_res_property(self.block_inst, res_label, inst_label)

    def update_perstn_res_property(self):
        self.load_perstn_res_property()
        self.sig_prop_updated.emit()

    def monitor_changes(self):
        assert self.block_inst is not None

        if not self.is_monitor_dep_graph:
            self.is_monitor_dep_graph = True
            self.block_inst.dependency_graph.register_observer(self.graph_observer)

        self.le_name.editingFinished.connect(self.on_name_changed)
        self.cb_device.currentIndexChanged.connect(self.on_device_changed)

        self.monitor_param_widget(self.param_widget_map, True, self.on_param_changed)
        self.monitor_pin_widget(self.pin_widget_map, True, self.on_pin_changed)
        self.monitor_clk_pin_inversion_widget(self.pin_widget_map, True, self.on_clkpin_invert_changed)
        self.monitor_ref_clk_widget(True)

        self.is_monitor = True

    def monitor_ref_clk_widget(self, is_monitor: bool):
        if is_monitor:
            self.cb_pll_clk_0.currentIndexChanged.connect(self.on_pll_clk_changed)
            self.cb_pll_clk_1.currentIndexChanged.connect(self.on_pll_clk_changed)
            self.cb_ext_clk.currentIndexChanged.connect(self.on_ext_clk_changed)
        else:
            self.cb_pll_clk_0.currentIndexChanged.disconnect(self.on_pll_clk_changed)
            self.cb_pll_clk_1.currentIndexChanged.disconnect(self.on_pll_clk_changed)
            self.cb_ext_clk.currentIndexChanged.disconnect(self.on_ext_clk_changed)

    def stop_monitor_changes(self):
        if self.block_inst is None:
            return

        if self.is_monitor_dep_graph:
            self.is_monitor_dep_graph = False
            self.block_inst.dependency_graph.unregister_observer(self.graph_observer)

        self.le_name.editingFinished.disconnect(self.on_name_changed)
        self.cb_device.currentIndexChanged.disconnect(self.on_device_changed)

        self.monitor_param_widget(self.param_widget_map, False, self.on_param_changed)
        self.monitor_pin_widget(self.pin_widget_map, False, self.on_pin_changed)
        self.monitor_clk_pin_inversion_widget(self.pin_widget_map, False, self.on_clkpin_invert_changed)
        self.monitor_ref_clk_widget(False)

        self.is_monitor = False

    def sync_pin_display(self):
        """
        Refresh any widget that identifies pin name with design data
        """
        if self.block_inst is not None:

            if isinstance(self.block_inst, QuadPCIE):
                pin_widget_map = self.pin_widget_map
            else:
                self.logger.error(f"Unknown block instance: {type(self.block_inst)}")
                return

            gen_pin = self.block_inst.gen_pin

            all_pin_list = gen_pin.get_all_pin()
            for pin in all_pin_list:
                if isinstance(pin, GenericPin):
                    pin_widget = pin_widget_map.get(pin.type_name, None)
                    if pin_widget is not None:
                        gui_util.set_config_widget(pin_widget.le_pin, pin.name, pin.name)

    def get_inst_resource(self):
        """
        Get resource name of instance
        """
        assert self.block_inst is not None
        return self.block_inst.get_device()

