from __future__ import annotations
from enum import Enum, IntEnum, unique, auto
from typing import TYPE_CHECKING, Dict, List, Optional, Tuple

import util.excp as app_excp
import device.excp as dev_excp
from device.block_definition import Port, PortDir, ModeInterface
from device.block_instance import PeripheryBlockInstance, PinConnection

from common_device.device_service_interface import BlockService
from tx375_device.quad_pcie.design import QuadPCIE

if TYPE_CHECKING:
    from design.db import PeriDesign
    from design.db_item import PeriDesignItem


@unique
class PCRDefnType(Enum):
    PCR_UNSUPPORTED0 = "PCR_UNSUPPORTED0"
    PCR_UNSUPPORTED1 = "PCR_UNSUPPORTED1"
    PCR_UNSUPPORTED2 = "PCR_UNSUPPORTED2"
    PCR_UNSUPPORTED3 = "PCR_UNSUPPORTED3"
    PCR_UNSUPPORTED4 = "PCR_UNSUPPORTED4"
    PCR_UNSUPPORTED5 = "PCR_UNSUPPORTED5"
    PCR_UNSUPPORTED6 = "PCR_UNSUPPORTED6"
    PCR_UNSUPPORTED7 = "PCR_UNSUPPORTED7"
    PCR_UNSUPPORTED8 = "PCR_UNSUPPORTED8"
    PCR_UNSUPPORTED9 = "PCR_UNSUPPORTED9"
    PCR_UNSUPPORTED10 = "PCR_UNSUPPORTED10"
    PCR_UNSUPPORTED11 = "PCR_UNSUPPORTED11"


class QuadPCIEDeviceService(BlockService):

    @unique
    class QuadPCIEModeType(Enum):
        pcie = "pcie"
        ethernet_10g = "ethernet_10g"
        ethernet_1g = "ethernet_1g"
        sgmii = "sgmii"

    mode2usagerate_map = \
        {
            QuadPCIEModeType.pcie: 1,
            QuadPCIEModeType.ethernet_10g: 0.25,
            QuadPCIEModeType.ethernet_1g: 0.25,
            QuadPCIEModeType.sgmii: 0.25
        }

    @property
    def str2pcrdefn_map(self) -> Dict[str, PCRDefnType]:
        return {member.value: member for member in PCRDefnType}

    @property
    def str2mode_map(self) -> Dict[str, QuadPCIEModeType]:
        return {member.value: member for member in self.QuadPCIEModeType}

    def is_in_pcie_mode(self, hier_ins_name):
        if hier_ins_name.endswith(self.QuadPCIEModeType.pcie.value):
            return True

        return False

    def get_user_device_instance_names(self, design: PeriDesign) -> Dict[str, str]:
        dev2user_ins = {}

        all_ins_list = self.get_all_instances(design)

        for ins in all_ins_list:
            if ins.get_device() not in dev2user_ins:
                dev2user_ins[ins.get_device()] = ins.name  # type: ignore
            else:
                msg = 'Duplicated device instance name ' \
                      '{} in pcie design'.format(
                          ins.get_device())
                raise dev_excp.BlockInstanceDesignDuplicateException(
                    msg, app_excp.MsgLevel.warning)

        return dev2user_ins

    def get_configured_instances(self, design: PeriDesign, active_instances) -> Dict[str, str]:
        all_ins_map = self.get_all_instances(design)
        ins2blk = {}

        for ins in all_ins_map:

            if ins.get_device() in active_instances:
                ins2blk[ins.get_device()] = self.get_name()

            else:
                msg = "Resource {} assigned to {} does not exists in device".format(
                    ins.get_device(), ins.name) # type: ignore

                raise dev_excp.InstanceInvalidResourceException(
                    msg, app_excp.MsgLevel.error)

        return ins2blk

    def get_all_instances(self, design: PeriDesign) -> List[PeriDesignItem]:
        if design.quad_pcie_reg is not None:
            return design.quad_pcie_reg.get_all_inst()

        return []

    def is_pcr_defn_match(self, pcr_name: str, pcr_type: PCRDefnType) -> bool:
        try:
            pcr_def = PCRDefnType(pcr_name)
        except ValueError:
            # pcr_name cannot be mapped to PCRDefnType
            return False
        return pcr_def == pcr_type

    """
    def get_blk_port_to_gen_pins(self, des_ins, dev_ins,
                                 gen_pin, skip_pad=True, included_dir=[]):
        '''
        Iterates through the design pins to get the
        names and the corresponding device pin name.

        This is only applicable for device that supports
        generic pin ONLY.

        :param des_ins: The Design instance object
        :param dev_ins: The device instance object
        :param gen_pin: The gen_pin of the design object

        :return a map of bit-blasted device pin name to the design name
        '''

        gen_pins = {}

        # Do nothing if the block does not have generic pins
        if gen_pin is None:
            return gen_pins

        ref_blk = dev_ins.get_block_definition()
        if ref_blk is None:
            msg = "Missing block definition {} for instance {}".format(
                dev_ins.get_ref_name(), des_ins.name)
            raise dev_excp.ConfigurationInvalidException(
                msg, app_excp.MsgLevel.error)

        all_pins = gen_pin.get_all_pin()

        quad_mode = self.get_design_instance_mode(
            dev_ins.get_name(), self._device_db, des_ins)

        blk_def = self._device_db.find_block(self._name)

        if quad_mode != "" and blk_def is not None:
            # Get the Block Mode associated to the current instance mode
            ins_blk_mode = blk_def.get_mode(quad_mode)

            if ins_blk_mode is not None:

                inf2mode_map = ins_blk_mode.get_interface_to_ports_map()

                for cur_pin in all_pins:
                    if cur_pin.type_name == "":
                        msg = "Missing reference port in instance {}".format(
                            des_ins.name)
                        raise dev_excp.ConfigurationInvalidException(
                            msg, app_excp.MsgLevel.error)

                    # Find the interface associated to the type name based
                    # on the mode associated to the instance
                    if cur_pin.type_name in inf2mode_map:
                        inf_obj = inf2mode_map[cur_pin.type_name]

                        if inf_obj is None:
                            continue

                        # Skip pad
                        if skip_pad and inf_obj.get_type() == Port.TYPE_PAD:
                            continue

                        # The map of bitblasted interface name to the raw block port name
                        # Example: HS_IN[7:0] -> P_OUT[7:0] with type_name =
                        # HS_IN
                        inf2port_map = inf_obj.get_interface_to_ports_map()

                        for inf_name, pname in inf2port_map.items():
                            # If the interface is a bus, we need to add the index
                            # that matches the interface name index to the user
                            # name
                            if cur_pin.is_bus:
                                # Copy the interface name and replace it with the
                                # user pin name
                                tmp_name = inf_name
                                user_pin_name = tmp_name.replace(
                                    inf_name, cur_pin.name)

                            else:
                                user_pin_name = cur_pin.name

                            # We don't expect that there will be overlapped
                            # in raw port name. Meaning different interface will
                            # use different raw port
                            gen_pins[pname] = user_pin_name

        return gen_pins
    """

    def get_design_instance_mode(self, dev_ins_name, device_db, ins_obj):
        '''
        Based on the device instance name and the design instance, return the Quad PCIE
        block mode type that it represents (since the name is flattened)

        :param dev_ins_name: The device instance name stored to the
                design object
        :param device_db: The device DB
        :param des_ins: The design instance
        :return the QuadPCIEModeType identified. None if not identifiable
        '''
        quad_mode = ""

        if ins_obj is not None and dev_ins_name != "":
            dev_ins = device_db.find_instance(dev_ins_name)

            if isinstance(ins_obj, QuadPCIE):
                quad_mode = self._determine_pcie_mode(
                    dev_ins_name, dev_ins, ins_obj)

        return quad_mode

    def _determine_pcie_mode(self, dev_ins_name: str,
                             dev_ins: PeripheryBlockInstance,
                             ins_obj: QuadPCIE):
        quad_mode = ""

        if ins_obj.get_device() != "":
            res_name = ins_obj.get_device()
        else:
            # Check the dev_ins_name
            res_name = dev_ins_name

        mode2resource_map = dev_ins.get_mode_to_resource_map()
        for mode, resource in mode2resource_map.items():
            if resource == res_name:

                quad_mode = mode
                break

        return quad_mode

    def get_non_clock_output_from_core_pins(self, des_ins, dev_ins) -> List[str]:
        """
        This is to return a map of the output interface name that
        is found associated to the block or limited to the mode name (if mode_name is non-empty)

        :param mode: The mode type. If it is None, it means that
            caller wants all output interface (input port) regardless
            of mode
        :return a map of output pin name to user name
        """
        # this is a map of the port name to the user name
        # the value can be an empty string since we are taking the
        # value without context

        # This means that it wants to look at all ports defined
        # in the block
        ports_list = self.get_block_ports_based_on_direction(
            included_dir=[PortDir.input], skip_clockout=True)
        return ports_list

    def get_resource_on_mode_internal_pin(self, ins_name: str, inf_name: str,
                                          index: Optional[int], mode_name: str) -> Tuple:
        '''
        Get the resource if any for the given device instance name
        and reference clock index ONLY.
        :param ins_name: PCIE block device instance name
        :param inf_name: The name used in the interface. It could be
                the same name as port name if it was not renamed
        :param index: The port index
        :return 1) the resource name, if applicable. It is also
                possible a string of names delimited by ','.
                If not applicable, a None is returned. This could also
                be the pad name in the case that it is controlled by
                the package pad name
                2) the pin name: This is a single string which points to the
                reference pin name. If not applicable it is set to None.
        '''
        # Find the device instance
        ins_obj = self.get_device().find_instance(ins_name)

        if ins_obj is not None:
            # Use the mode interface name to find the actual
            # block hw port name
            if index is not None:
                inf_name = "{}[{}]".format(inf_name, index)
            else:
                inf_name = inf_name

            # Get the mode
            peri_blk = self.get_device().find_block(self._name)
            assert peri_blk is not None

            mode_obj = peri_blk.get_mode(mode_name)
            assert mode_obj is not None

            # Find the port associated to the inf
            ports_list: List[str] = mode_obj.get_ports_on_interface(
                inf_name)

            # Get the actual pin mapped to the mode inf
            assert len(ports_list) == 1

            # It could be none when it is driven by a external pad
            # which in this case has no resource
            clkin_pin = ins_obj.get_connection_pin(ports_list[0])
            if clkin_pin is not None:
                # We're not going to separate the index from base
                # since function can take it as it is
                final_dep_name, dep_ref_pin_name = self.get_resource_on_ins_pin(
                    ins_name, ports_list[0], None)

                # Only if the data is valid do we return 2 data
                if final_dep_name is not None:
                    return final_dep_name, dep_ref_pin_name

            else:
                # this could be a pad name so we return the port name
                return ports_list[0], None

        return None, None

    def get_external_refclk_pins(self, ins_name: str, option_name: str):
        '''
        The external refclk is a differential signal with a P/N pins.
        We expose the selection as a single element (ie. REFCLK0, REFCLK1).
        However, it is referring to both of those pins:
        REFCLK0 - REFCLK0_P, REFCLK0_N
        REFCLK1 - REFCLK1_P, REFCLK1_N

        This function checks that the necessary pin is bonded out to make sure
        that they can be used.

        :param ins_name: The device instance name
        :param option_name: Either REFCLK0, REFCLK1 indicates the pin to check
        :return the list of package pin name associated to the
                option_name.  An empty list is returned if the input is invalid
                or the pin isn't bonded out in the device
        '''
        refclk_pins = []

        if option_name in ["REFCLK0", "REFCLK1"]:

            # Get the io and pkg device
            io_pads = self._device_db.get_io_pad()
            dev_pkg = self._device_db.get_package()

            if io_pads is not None and dev_pkg is not None:

                quad_prefix = "Q{}".format(ins_name[-1])

                # TODO: REFCLK is actually REFCLK0. but not sure
                # when the name will be updated in ICD files. So, we're
                # making it flexible here
                p_pad_name = "{}_{}_P".format(quad_prefix, option_name)
                n_pad_name = "{}_{}_N".format(quad_prefix, option_name)

                is_found_pads = False

                if io_pads.find_pad(p_pad_name) is not None and\
                        io_pads.find_pad(n_pad_name) is not None:
                    is_found_pads = True
                """
                elif option_name == "REFCLK0":
                    # check without the '0'
                    p_pad_name = "{}_CMN_REFCLK_P".format(quad_prefix)
                    n_pad_name = "{}_CMN_REFCLK_N".format(quad_prefix)

                    if io_pads.find_pad(p_pad_name) is not None and\
                            io_pads.find_pad(n_pad_name) is not None:
                        is_found_pads = True
                """

                if is_found_pads:
                    # Check if it is bonded in package
                    if len(dev_pkg.find_package_pad(p_pad_name)) > 0 and\
                            len(dev_pkg.find_package_pad(n_pad_name)) > 0:
                        refclk_pins = [p_pad_name, n_pad_name]

        return refclk_pins

    def get_mode_interface_port_mapping(self, mode_name: str,
                                        inf_name: str, is_port_to_inf=False):
        '''
        Returns a mapping of interface name to the actual port name in the
        block.

        :param mode_name: The mode name being searched
        :param inf_name: The interface name and NOT the port name

        :return a map of interface name to the port name and also
                the reverse (caller to use  the one needed). If the interface
                is a bus parent, then the entries in this map is bitblasted
                since it could be made up of port of different source.
        '''
        inf2port_map = {}
        port2inf_map = {}   # Filled in only if is_port_to_inf set to True

        interface_map: Dict[str,
                            ModeInterface] = self.get_interface_by_mode(mode_name)

        inf_obj = interface_map.get(inf_name)
        if inf_obj is not None:
            inf2port_map = inf_obj.get_interface_to_ports_map()

            # Only reverse map if needed since O(n)
            if is_port_to_inf:
                port2inf_map = {v: k for k, v in inf2port_map.items()}

        return inf2port_map, port2inf_map

    def get_usable_instance_names(self) -> List:
        """
        Get resource name that is available for use in design

        :return a list of instance names (PeripheryBlockInstance)
        """
        # TODO: Interface to the Quad ResService when ready
        ins_names, _ = self._device_db.get_resources_by_type(self._name)

        # This returns all the flatten resource name as well
        # we want to filter out to take in only full quad. This is
        # temporary until the quad res service is ready
        # Remove instance name that doesn't start with "QUAD_*"
        final_ins_names = []

        for iname in ins_names:
            if iname.startswith("QUAD_"):
                final_ins_names.append(iname)

        return final_ins_names

    def get_usable_instances(self):
        '''
        Returns a list of instance that is used
        in the device

        :return a list of instance objects (PeripheryBlockInstance)
        '''
        # TODO: Interface to the Quad ResService when ready
        final_ins_obj = []

        _, ins_obj_list = self._device_db.get_resources_by_type(self._name)


        for ins in ins_obj_list:
            if ins is not None and ins.get_name().startswith("QUAD_"):
                final_ins_obj.append(ins)

        return final_ins_obj

    def get_valid_perstn_gpio_resource(self, quad_dev_name: str) -> str:
        assert self._device_db is not None, f"Device DB is None"
        res_name_list, _ = self.get_all_resource_on_ins_pin(
            quad_dev_name, "PERSTN", None)

        return res_name_list[0] if len(res_name_list) > 0 else ""

    def is_gen4_supported(self):
        is_supported = True

        if self._device_db is not None:
            if self._device_db.get_device_mapped_name() == "opx_334x484_b40_d20_bn484":
                is_supported = False
            else:
                family_name, _ = self._device_db.get_device_family_die_name(is_exact=True)

                if family_name == "Topaz":
                    is_supported = False

        return is_supported