from __future__ import annotations
from typing import List, Dict, Tuple, Optional, TYPE_CHECKING
from enum import Enum, unique
import copy

from device.block_definition import ModeInterface, Port
from device.db import PeripheryDevice
from common_device.device_service_interface import BlockService
from common_device.property import PropertyMetaData, RangeValidator
from common_device.quad.res_service import QuadResService, QuadType
from tx375_device.quad_pcie.gui.dep_graph import QuadPCIEDependencyGraph
from tx375_device.quad_pcie.design_param_info import QuadPCIEDesignParamInfo as PCIEParamInfo, get_pipe_freq_range
from tx375_device.quad_pcie.quad_pcie_dep_graph import get_params_with_hidden_conditions
from tx375_device.quad_pcie.quad_pcie_param_info import build_param_info, get_removed_parameters
from tx375_device.quad_pcie.quad_pcie_prop_id import QuadPCIEConfigParamInfo as PCIEConfigParamInfo
from tx375_device.quad_pcie.quad_pcie_pin_dep_graph import get_additional_params
from tx375_device.quad_pcie.quad_pcie_pin_dep_graph import get_hidden_ports

from design.db_item import (
    PeriDesignRegistry,
    PeriDesignGenPinItem,
    AbstractParamObserver,
    GenericParamGroup,
    GenericParam,
    GenericPin,
    GenericParamService
)

if TYPE_CHECKING:
    from common_gui.base_dep_graph import DependencyGraph
    from tx375_device.quad_pcie.device_service import QuadPCIEDeviceService
    from tx375_device.fpll.design import EfxFpllV1
    from design.db import PeriDesign

class ParamGroupMonitor(AbstractParamObserver):
    """
    This class act as a bridge between ParamGroup and Depenedency Graph
    """

    def __init__(self, graph: DependencyGraph) -> None:
        super().__init__()
        self._graph = graph

    def on_param_changed(self, param_group: GenericParamGroup, param_name: str):
        self._graph.propgrate_param(param_group=param_group, param_name=param_name)


class QuadPCIE(PeriDesignGenPinItem):

    _device_port_map: Dict = {}
    dev_service = None # type: Optional[QuadPCIEDeviceService]

    # TODO: Remove this class and use the one from SCF
    @unique
    class RefClockModeType(Enum):
        core = "core"
        pll = "pll"
        external = "external"

        @classmethod
        def has_member(cls, item: str) -> bool:
            return item in cls._value2member_map_

    def __init__(self, name: str, block_def: str = "", apply_default: bool = True):
        super().__init__()
        self.__name = name
        self.__block_def = block_def
        self._categories = set()
        self.quad_type = QuadType.quad_pcie

        self._param_info = self.build_param_info()
        self._param_group = self.build_param()
        self.build_generic_pin()

        self._dep_graph = QuadPCIEDependencyGraph(self._param_info, self.gen_pin, self.get_pin_type_by_class)
        self._param_group.register_param_observer(ParamGroupMonitor(self._dep_graph))

        if apply_default:
            self.set_default_setting()

    @property
    def name(self):
        return self.__name

    @name.setter
    def name(self, value: str):
        self.__name = value

    @property
    def dependency_graph(self):
        assert self._dep_graph is not None
        return self._dep_graph

    def get_device(self):
        return self.__block_def

    def set_device(self, device_name: str):
        self.__block_def = device_name

    def get_param_info(self):
        return self._param_info

    def get_param_group(self):
        return self._param_group

    @property
    def param_group(self):
        return self._param_group

    @property
    def param_info(self):
        return self._param_info

    def create_chksum(self):
        pass

    @staticmethod
    def str2enumtype(prop_id: PCIEParamInfo.Id, value: str):
        return None

    @staticmethod
    def enumtype2str(prop_id: PCIEParamInfo.Id, val):
        return None

    def is_gen4_supported(self, device_db):
        is_supported = True

        if device_db is not None:
            dev_pcie = QuadPCIERegistry.get_blk_service(device_db)
            is_supported = dev_pcie.is_gen4_supported()

        return is_supported
    
    def set_default_setting(self, device_db=None):
        param_service = GenericParamService(self._param_group, self._param_info)
        for param_id in PCIEParamInfo.Id:
            default_val = self._param_info.get_default(param_id)
            param_service.set_param_value(param_id, default_val)

        for param_id in PCIEConfigParamInfo.Id:
            default_val = self._param_info.get_default(param_id)
            param_service.set_param_value(param_id, default_val)

        for param_name in get_additional_params():
            default_val = self._param_info.get_default(param_name)
            param_service.set_param_value(param_name, default_val)

        # Override based on device
        if device_db is not None and not self.is_gen4_supported(device_db):
            default_val = "Gen3"
            param_service.set_param_value(
                PCIEConfigParamInfo.Id.PIPE_CONFIG_CMN__config_reg_0__pcie_generation_sel, default_val)

    def build_param_info(self):
        param_info = PCIEParamInfo()

        prop_data_list: List[PropertyMetaData.PropData] = [
            PropertyMetaData.PropData(id=PCIEParamInfo.Id.ref_clk_frequency,
                name=PCIEParamInfo.Id.ref_clk_frequency.value,
                data_type=GenericParam.DataType.dflo,
                default=100.0,
                valid_setting=RangeValidator(100.0, 100.0),
                disp_name='Reference Clock Frequency',
                category='ref_clk'),
            PropertyMetaData.PropData(id=PCIEParamInfo.Id.ref_clk_pin_name,
                name=PCIEParamInfo.Id.ref_clk_pin_name.value,
                data_type=GenericParam.DataType.dstr,
                default='',
                disp_name='Reference Clock Pin Name',
                category='ref_clk'),
            PropertyMetaData.PropData(id=PCIEParamInfo.Id.pll_ref_clk0,
                name=PCIEParamInfo.Id.pll_ref_clk0.value,
                data_type=GenericParam.DataType.dstr,
                default='PMA_CMN_REFCLK_PLL_1',
                valid_setting=self.get_all_ref_clk0_pll_options(self.__block_def),
                disp_name='PLL Reference Clock 0',
                category='ref_clk'),
            PropertyMetaData.PropData(id=PCIEParamInfo.Id.pll_ref_clk1,
                name=PCIEParamInfo.Id.pll_ref_clk1.value,
                data_type=GenericParam.DataType.dstr,
                default='PMA_CMN_REFCLK1_PLL_1',
                valid_setting=self.get_all_ref_clk1_pll_options(self.__block_def),
                disp_name='PLL Reference Clock 1',
                category='ref_clk'),
            PropertyMetaData.PropData(id=PCIEParamInfo.Id.fpga_div2_clk_conn_type,
                name=PCIEParamInfo.Id.fpga_div2_clk_conn_type.value,
                data_type=GenericParam.DataType.dstr,
                default="rclk",
                valid_setting=["gclk", "rclk"],
                disp_name='FPGA Divider to Clock Connection Type',
                category='Debug'),

            # This parameter is deprecated, leave it there to support older design
            PropertyMetaData.PropData(id=PCIEParamInfo.Id.perstn_source,
                name=PCIEParamInfo.Id.perstn_source.value,
                data_type=GenericParam.DataType.dstr,
                default="PERSTN0",
                valid_setting=["PERSTN0", "PERSTN1"],
                disp_name='PERSTn Resource',
                category='reset_hide'),

            PropertyMetaData.PropData(id=PCIEParamInfo.Id.pmclk_source,
                name=PCIEParamInfo.Id.pmclk_source.value,
                data_type=GenericParam.DataType.dstr,
                default="osc",
                valid_setting=["osc", "pll"],
                disp_name='PM Clock Resource',
                category='hidden'),
            PropertyMetaData.PropData(id=PCIEParamInfo.Id.pmclk_conn_type,
                name=PCIEParamInfo.Id.pmclk_conn_type.value,
                data_type=GenericParam.DataType.dstr,
                default="rclk",
                valid_setting=["gclk", "rclk"],
                disp_name='Power Management Clock Connection Type',
                category='Power Management'),
            PropertyMetaData.PropData(id=PCIEParamInfo.Id.axi_master_en,
                name=PCIEParamInfo.Id.axi_master_en.value,
                data_type=GenericParam.DataType.dbool,
                default=True,
                disp_name='Enable AXI Master Interface',
                category='AXI'),
            PropertyMetaData.PropData(id=PCIEParamInfo.Id.axi_slave_en,
                name=PCIEParamInfo.Id.axi_slave_en.value,
                data_type=GenericParam.DataType.dbool,
                default=True,
                disp_name='Enable AXI Slave Interface',
                category='AXI'),
            PropertyMetaData.PropData(id=PCIEParamInfo.Id.apb_en,
                name=PCIEParamInfo.Id.apb_en.value,
                data_type=GenericParam.DataType.dbool,
                default=True,
                disp_name='Enable Advanced Peripheral Bus',
                category='APB-hidden'),
            PropertyMetaData.PropData(id=PCIEParamInfo.Id.hot_plug_en,
                name=PCIEParamInfo.Id.hot_plug_en.value,
                data_type=GenericParam.DataType.dbool,
                default=False,
                disp_name='Enable Hot Plug',
                category='Hot Plug'),
            PropertyMetaData.PropData(id=PCIEParamInfo.Id.cfg_snoop_en,
                name=PCIEParamInfo.Id.cfg_snoop_en.value,
                data_type=GenericParam.DataType.dbool,
                default=False,
                disp_name='Enable Configuration Snoop',
                category='Config Snoop'),
            PropertyMetaData.PropData(id=PCIEParamInfo.Id.pwr_mgmt_en,
                name=PCIEParamInfo.Id.pwr_mgmt_en.value,
                data_type=GenericParam.DataType.dbool,
                default=False,
                disp_name='Enable Power Management',
                category='Power Management'),
            PropertyMetaData.PropData(id=PCIEParamInfo.Id.interrupt_en,
                name=PCIEParamInfo.Id.interrupt_en.value,
                data_type=GenericParam.DataType.dbool,
                default=False,
                disp_name='Enable Interrupt',
                category='Interrupt'),
            PropertyMetaData.PropData(id=PCIEParamInfo.Id.status_en,
                name=PCIEParamInfo.Id.status_en.value,
                data_type=GenericParam.DataType.dbool,
                default=False,
                disp_name='Enable Status',
                category='Status'),
            PropertyMetaData.PropData(id=PCIEParamInfo.Id.vendor_en,
                name=PCIEParamInfo.Id.vendor_en.value,
                data_type=GenericParam.DataType.dbool,
                default=False,
                disp_name='Enable Vendor Specific',
                category='Vendor Specific'),
            PropertyMetaData.PropData(id=PCIEParamInfo.Id.debug_en,
                name=PCIEParamInfo.Id.debug_en.value,
                data_type=GenericParam.DataType.dbool,
                default=False,
                disp_name='Enable Debug',
                category='Debug'),
            PropertyMetaData.PropData(id=PCIEParamInfo.Id.axi_master_address,
                name=PCIEParamInfo.Id.axi_master_address.value,
                data_type=GenericParam.DataType.dint,
                default=64,
                disp_name='AXI Address',
                category='AXI Master'),
            PropertyMetaData.PropData(id=PCIEParamInfo.Id.axi_master_data_width,
                name=PCIEParamInfo.Id.axi_master_data_width.value,
                data_type=GenericParam.DataType.dint,
                default=512,
                disp_name='AXI Data Width',
                category='AXI Master'),
            PropertyMetaData.PropData(id=PCIEParamInfo.Id.axi_slave_address,
                name=PCIEParamInfo.Id.axi_slave_address.value,
                data_type=GenericParam.DataType.dint,
                default=64,
                disp_name='AXI Address',
                category='AXI Slave'),
            PropertyMetaData.PropData(id=PCIEParamInfo.Id.axi_slave_data_width,
                name=PCIEParamInfo.Id.axi_slave_data_width.value,
                data_type=GenericParam.DataType.dint,
                default=512,
                disp_name='AXI Data Width',
                category='AXI Slave'),
            PropertyMetaData.PropData(id=PCIEParamInfo.Id.msi_en,
                name=PCIEParamInfo.Id.msi_en.value,
                data_type=GenericParam.DataType.dbool,
                default=False,
                disp_name='Enable MSI',
                category='Interrupt:MSI'),
            ]

        for prop_data in prop_data_list:
            param_info.add_prop_by_data(prop_data.id, prop_data)

        # Add the additional parameter that is from ICD
        more_param_info = build_param_info()

        self.build_additional_param_info(more_param_info)

        # Combined the Enum and the ParamInfo
        param_info.concat_param_info(more_param_info)

        return param_info

    def build_additional_param_info(self, param_info):
        prop_data_list: List[PropertyMetaData.PropData] = []
        for param_name in get_additional_params():
            prop_data_list.append(
                PropertyMetaData.PropData(id=param_name,
                name=param_name,
                data_type=GenericParam.DataType.dstr,
                default="0",
                category="PCR")
            )

        for prop_data in prop_data_list:
            param_info.add_prop_by_data(prop_data.id, prop_data)

    def build_param(self):
        assert self._param_info is not None
        param_group = GenericParamGroup()
        for param_info in self._param_info.get_all_prop():
            param_group.add_param(param_info.name, param_info.default, param_info.data_type)
            self._categories.add(param_info.category)

        return param_group

    def get_pin_with_input_and_clkout_name(self):
        return ["CH1_TX_FWD_CLK", "FORWARDED_DIV2_CLK"]

    def build_generic_pin(self):
        from device.block_definition import Port as DevicePort, PortDir
        self.gen_pin.clear()
        for device_port in self._device_port_map.values():
            self.dp_service.device_port = device_port
            if self.dp_service.get_type() == DevicePort.TYPE_CLOCK or \
                device_port.get_name() in self.get_pin_with_input_and_clkout_name():
                self.gen_pin.add_clock_pin(self.dp_service.get_name(), "", self.dp_service.is_bus_port())
            else:
                direction = BlockService.get_port_direction_from_core(device_port)
                if direction == PortDir.input:
                    self.gen_pin.add_input_pin(self.dp_service.get_name(), "", self.dp_service.is_bus_port())
                elif direction == PortDir.output:
                    self.gen_pin.add_output_pin(self.dp_service.get_name(), "", self.dp_service.is_bus_port())
                else:
                    self.gen_pin.add_pin(self.dp_service.get_name(), "", self.dp_service.is_bus_port())

            self._categories.add(self.dp_service.get_class())

    def generate_pin_name(self):
        self.generate_pin_name_from_inst(self.__name)

    @property
    def categories(self):
        return self._categories

    @staticmethod
    def get_all_ref_clk0_pll_options(ins_name: str):
        if QuadPCIE.dev_service is None or ins_name == "":
            return [
                "PMA_CMN_REFCLK_PLL_1",
                "PMA_CMN_REFCLK_PLL_2",
                "PMA_CMN_REFCLK1_PLL_1",
                "PMA_CMN_REFCLK1_PLL_2",
            ]

        from tx375_device.quad_pcie.device_service import QuadPCIEDeviceService

        options = []
        dev_service = QuadPCIE.dev_service
        mode_name = QuadPCIEDeviceService.QuadPCIEModeType.pcie.value
        device_port_map: Dict[str, ModeInterface] = dev_service.get_interface_by_mode(mode_name)
        mode_inf_obj = device_port_map.get("REFCLK", None)
        assert mode_inf_obj is not None

        inf2ports = mode_inf_obj.get_interface_to_ports_map()
        for name in inf2ports.values():
            res_name_list, ref_pin = dev_service.get_all_resource_on_ins_pin(ins_name, name, None)
            if len(res_name_list) <= 0:
                continue
            elif "PLL" not in name:
                continue
            options.append(name)

        return options

    @staticmethod
    def get_hidden_params():
        return [
            PCIEParamInfo.Id.fpga_div2_clk_conn_type.name,
            PCIEParamInfo.Id.pmclk_source.name,
            PCIEParamInfo.Id.hot_plug_en.name,
            PCIEParamInfo.Id.vendor_en.name,
            PCIEParamInfo.Id.debug_en.name,
        ]

    @staticmethod
    def get_all_ref_clk1_pll_options(ins_name: str):
        return QuadPCIE.get_all_ref_clk0_pll_options(ins_name)

    @staticmethod
    def get_pipe_freq_range_validator(lane_rate: str):
        min_freq, max_freq = get_pipe_freq_range(lane_rate)

        return RangeValidator(min_freq, max_freq)

    # Overload
    def generate_pin_name_from_inst(self, inst_name: str):
        assert self.gen_pin is not None

        for pin in self.gen_pin.get_all_pin():
            is_available = self._dep_graph.get_pin_attributes(pin.type_name)['is_available']
            if is_available:
                pin.generate_pin_name(inst_name)

        # Clockout interface and clock input need user to key in name.
        # Skip auto-generating hidden pins
        self.update_default_pin_name()

    def update_default_pin_name(self):
        pin_list: List[GenericPin] = self.gen_pin.get_all_pin()
        hidden_ports = get_hidden_ports()
        
        for pin in pin_list:
            # No auto-generate pin name for CLKOUT and input clock pins
            # Same thing for the hidden pins
            if self.__is_clkout_pin(pin.type_name) or \
                self.__is_input_clk_pin(pin.type_name) or \
                pin.type_name in hidden_ports:
                pin.name = ""

    def __is_clkout_pin(self, pin_type_name: str) -> bool:
        return pin_type_name in {
                                 'AXI_CLK',
                                 'FPGA_DIV2_CLK',
                                 'PMA_CMN_REFCLK1_CORE',
                                 'PMA_CMN_REFCLK_CORE',
                                 'USER_APB_CLK',
                                }

    def __is_input_clk_pin(self, pin_type_name: str) -> bool:
        return pin_type_name in {
                                 'CH0_TX_FWD_CLK',
                                 'CH1_TX_FWD_CLKL',
                                 'PM_CLK',
                                 'FORWARDED_DIV2_CLK'
                                }

    def get_clock_input_pin_info(self, port_name: str):
        clk_port2inf_map = {
            "CH0_TX_FWD_CLK": ("PM_CLK",  PCIEParamInfo.Id.pmclk_conn_type),
            "CH1_TX_FWD_CLK": ("FORWARDED_DIV2_CLK", PCIEParamInfo.Id.fpga_div2_clk_conn_type)
            # Not listing customized gen pin for the REFCLK since that
            # is not handled as part of gen pin
        }

        return clk_port2inf_map.get(port_name, None)

    # Overload this since we have some hw name mapped to interface
    # name and it may be searched by other non-pcie itself
    def get_inf_pin_by_type_name(self, type_name: str):
        translated_name = type_name
        tup_info = self.get_clock_input_pin_info(type_name)

        if tup_info is not None:
            translated_name, _ = tup_info

        return self.gen_pin.get_pin_by_type_name(translated_name)

    def get_user_clock_input_names_used(self):
        '''
        :return the user clock input name used within the context only.
            PM_CLK: When power mgmt is enabled
            FORWARDED_DIV2_CLK: When Debug is enabled
        '''
        user_clock_input = []

        if self.gen_pin is None:
            return  user_clock_input
        
        param_service = GenericParamService(
            self.get_param_group(), self.get_param_info())

        pwr_mgmt_en = param_service.get_param_value(
            PCIEParamInfo.Id.pwr_mgmt_en)
        debug_en = param_service.get_param_value(
            PCIEParamInfo.Id.debug_en)
        
        if pwr_mgmt_en:
            pm_clk_name = self.gen_pin.get_pin_name_by_type("PM_CLK")
            if pm_clk_name != "":
                user_clock_input.append(pm_clk_name)

        if debug_en:
            debug_clk_name = self.gen_pin.get_pin_name_by_type("FORWARDED_DIV2_CLK")
            if debug_clk_name != "":
                user_clock_input.append(debug_clk_name)

        return user_clock_input

    def get_clock_source_usage(self, port_name: str):
        '''
        The instance pin name in gen pin is a translated
        name per mode. Other block (ie. clockmux) only save
        based on the block port name. So, we need to get
        the mapped interface name and its info through this function

        :return the gen pin interface name and the connection type of
                the clock assigned
        '''

        inf_name = ""
        conn_type = None

        tup_info = self.get_clock_input_pin_info(port_name)
        if tup_info is not None:
            inf_name, param_id = tup_info

            param_service = GenericParamService(
                self.get_param_group(), self.get_param_info())

            # PM_CLK is dependent on the PWR MGMT enable setting.
            # So we return info provided that it's enabled
            conn_type = param_service.get_param_value(param_id)

            # This is PM_clock which depends on contex usage
            # (PM enabled)
            if inf_name == "PM_CLK" and not param_service.get_param_value(PCIEParamInfo.Id.pwr_mgmt_en):
                # return an empty interface name to indicate
                # that the pin name is not used
                inf_name = ""
                conn_type = None
            # The debug clock input is valid when debug is enabled
            elif inf_name == "FORWARDED_DIV2_CLK" and not param_service.get_param_value(PCIEParamInfo.Id.debug_en):
                inf_name = ""
                conn_type = None

        return inf_name, conn_type

    def generate_specific_pin_name_from_inst(self, pin_name, is_only_empty_name: bool = False):
        '''

        :param pin_name: The pin type_name to have its pin name generated
        :param is_empty_name: IF true, only regenerate if the pin name is empty.
        :return:
        '''
        assert self.gen_pin is not None

        if self.__is_clkout_pin(pin_name) or self.__is_input_clk_pin(pin_name):
            return

        if pin_name in get_hidden_ports():
            return

        pin = self.gen_pin.get_pin_by_type_name(pin_name)
        if pin is not None and (not is_only_empty_name or \
                                    (pin.name == "" and is_only_empty_name)):
            pin.generate_pin_name(self.name)

    def get_available_params(self) -> List[str]:
        param_name_list: List[str] = []
        for param in self._param_group.get_all_param():
            is_available = self.is_param_available(param.name)
            if is_available:
                param_name_list.append(param.name)
        return param_name_list

    def get_available_pins(self) -> List[str]:
        assert self._dep_graph is not None
        pin_type_name_list: List[str] = []
        for pin in self.gen_pin.get_all_pin():
            is_available = self.is_pin_available(pin.type_name)
            if is_available:
                pin_type_name_list.append(pin.type_name)
        return pin_type_name_list

    def is_param_available(self, param: str | PCIEParamInfo.Id | PCIEConfigParamInfo.Id) -> bool:
        assert self._dep_graph is not None

        if isinstance(param, str):
            param_name = param
        else:
            param_name = param.name

        is_available = self._dep_graph.get_param_attributes(param_name)['is_available']
        return is_available

    def is_pin_available(self, pin_type_name: str) -> bool:
        if self.gen_pin.get_pin_by_type_name(pin_type_name) is None:
            return False

        assert self._dep_graph is not None

        is_available = self._dep_graph.get_pin_attributes(pin_type_name)['is_available']
        return is_available

    @staticmethod
    def build_port_info(device_db: Optional[PeripheryDevice] = None, is_mockup: bool = False):
        """
        Build port info for PCIE which is mode based.
        This function need to be call only once since it is static shared
        between class.

        :param device_db: Device db instance
        :param is_mockup: True, build a mockup data, else build from device db
        """
        assert is_mockup is False
        assert isinstance(device_db, PeripheryDevice)

        from tx375_device.quad_pcie.device_service import QuadPCIEDeviceService

        dev_service = QuadPCIERegistry.get_blk_service(device_db)
        assert isinstance(dev_service, QuadPCIEDeviceService)

        QuadPCIE.dev_service = dev_service

        mode_name = QuadPCIEDeviceService.QuadPCIEModeType.pcie.value
        device_port_map: Dict[str, ModeInterface] = dev_service.get_interface_by_mode(mode_name)

        QuadPCIE._device_port_map = {}

        skip_port = ["FPGA_DIV2_CLK", "REFCLK", "PERSTN"]
        for name, mode_port in device_port_map.items():
            if name in skip_port:
                continue

            if mode_port.get_type() != Port.TYPE_PAD:

                # All the interface have description except some
                # which are in exlucded list
                assert mode_port.get_description() not in ("", None)
                QuadPCIE._device_port_map[name] = mode_port

    def get_pll_ref_clk_info(self, quad_pcie_dev_service: QuadPCIEDeviceService,
                         design: PeriDesign, param_id: PCIEConfigParamInfo.Id| PCIEParamInfo.Id) -> Tuple[str, List[str]]:
        ref_clk_inst = ""
        res_name_list: List[str] = []

        param_service = GenericParamService(self.get_param_group(), self.get_param_info())
        ref_clk_src = param_service.get_param_value(PCIEConfigParamInfo.Id.PIPE_CONFIG_CMN__config_reg_2__pma_cmn_refclk_sel)
        pll_ref_clk = param_service.get_param_value(param_id)

        if ref_clk_src !=  "PLL":
            return ref_clk_inst, res_name_list

        ref_clk_inst, res_name_list, _ = self.get_pll_ref_clk_info_by_refclk_name(
            pll_ref_clk, quad_pcie_dev_service, design)

        return ref_clk_inst, res_name_list

    def get_pll_ref_clk_info_by_refclk_name(self, refclk_name: str,
            quad_pcie_dev_service: QuadPCIEDeviceService, design: PeriDesign):
        ref_clk_inst = ""
        res_name_list: List[str] = []
        ref_pll: Optional[EfxFpllV1] = None

        quad_pcie_res_name = self.get_device()
        if quad_pcie_res_name == "":
            return ref_clk_inst, res_name_list, ref_pll

        res_name_list, ref_pin = quad_pcie_dev_service.get_all_resource_on_ins_pin(
            quad_pcie_res_name, refclk_name, None)

        pin = ""

        if design.pll_reg is None or ref_pin in ("", None):
            return ref_clk_inst, res_name_list, ref_pll

        assert isinstance(ref_pin, str)
        for res_name in res_name_list:
            pll_inst = design.pll_reg.get_inst_by_device_name(res_name)
            if pll_inst is not None:
                ref_pll = pll_inst
                break

        if ref_pll is not None:
            assert isinstance(ref_pin, str)
            pin_clk_no = self._convert_str2pll_outclk_num(ref_pin)
            assert pin_clk_no != -1

            pll_out = ref_pll.get_output_clock_by_number(pin_clk_no)
            if pll_out is not None:
                pin = pll_out.name

        ref_clk_inst = pin

        return ref_clk_inst, res_name_list, ref_pll

    @staticmethod
    def _convert_str2pll_outclk_num(ref_pin_str: str) -> int:
        return {
                "CLKOUT0": 0,
                "CLKOUT1": 1,
                "CLKOUT2": 2,
                "CLKOUT3": 3,
                "CLKOUT4": 4,
        }.get(ref_pin_str, -1)

    def get_ref_clk_info(self, quad_pcie_dev_service: QuadPCIEDeviceService,
                         design: PeriDesign, param_id: PCIEConfigParamInfo.Id| PCIEParamInfo.Id) -> Tuple[str, List[str]]:
        """
        Get ref clock information, return ref clk instance name and list of resource name

        :param quad_pcie_dev_service: Quad PCIe device service
        :type quad_pcie_dev_service: QuadPCIEDeviceService
        :param design: PeriDesign
        :type design: PeriDesign
        :return: ref clk instance name and list of resource name
        :rtype: Tuple[str, List[str]]
        """
        ref_clk_inst = ""
        res_name_list: List[str] = []

        param_service = GenericParamService(self.get_param_group(), self.get_param_info())
        ref_clk_src = param_service.get_param_value(PCIEConfigParamInfo.Id.PIPE_CONFIG_CMN__config_reg_2__pma_cmn_refclk_sel)

        match ref_clk_src:
            case "Internal":
                assert param_id == PCIEParamInfo.Id.ref_clk_pin_name
                ref_clk_inst = param_service.get_param_value(PCIEParamInfo.Id.ref_clk_pin_name)
                assert isinstance(ref_clk_inst, str)

            case "PLL":
                ref_clk_inst, res_name_list = self.get_pll_ref_clk_info(
                    quad_pcie_dev_service, design, param_id)

            case "External":
                ext_ref_clk = param_service.get_param_value(param_id)
                assert isinstance(ext_ref_clk, str)

                quad_pcie_res_name = self.get_device()
                if quad_pcie_res_name == "":
                    return ref_clk_inst, res_name_list

                res_name_list = quad_pcie_dev_service.get_external_refclk_pins(
                    quad_pcie_res_name, f"REFCLK{ext_ref_clk[-1]}")
                pin = ""

                ref_clk_inst = pin

        return ref_clk_inst, res_name_list

    def get_perstn_info(self, quad_pcie_dev_service: QuadPCIEDeviceService,
                         design: Optional[PeriDesign]) -> Tuple[str, List[str]]:
        """
        Get perstn information, return perstn instance name and list of resource name

        :param quad_pcie_dev_service: Quad PCIe device service
        :type quad_pcie_dev_service: QuadPCIEDeviceService
        :param design: PeriDesign
        :type design: PeriDesign
        :return: perstn instance name and list of resource name
        :rtype: Tuple[str, List[str]]
        """
        perstn_inst = ""
        res_name_list: List[str] = []

        # Get resource name
        quad_pcie_res_name = self.get_device()
        res_name = quad_pcie_dev_service.get_valid_perstn_gpio_resource(quad_pcie_res_name)
        assert res_name != ""
        res_name_list.append(res_name)

        # Get instance
        if design is not None:
            gpio_reg = design.gpio_reg
            assert gpio_reg is not None

            inst_list = []
            for res_name in res_name_list:
                inst = gpio_reg.get_inst_by_device_name(res_name)
                if inst is not None:
                    inst_list.append(inst.name)

            perstn_inst = ",".join(inst_list)

        return perstn_inst, res_name_list

    def get_parameters_for_lpf_gen(self, common_reg):
        '''
        Create a dictionary of paramter settings targetted
        for bitstream generation (input to hw script):
        user_registers = {
            "q#": {
                param_name (str): param_value (str),
            }
        }
        All parameters need to be written out eventhough not used
        in context
        '''
        ins_param_map = {}

        res2hwname_map = {
            "QUAD_0": "q0",
            "QUAD_1": "q1",
            "QUAD_2": "q2",
            "QUAD_3": "q3"
        }
        hw_name = res2hwname_map.get(self.get_device(), "")
        if hw_name != "":
            param_map = {}
            # we will need to populate all the parameter
            # that is only defined in the quad_pcie_param_info.py
            param_service = GenericParamService(self._param_group, self._param_info)
            for param_id in PCIEConfigParamInfo.Id:
                pval = param_service.get_param_value(param_id)
                pname = self._param_info.get_prop_name(param_id)
                assert pname not in param_map

                # PT-2235 removed some parameters
                if pname in get_removed_parameters():
                    continue

                # Skip if related to mode (e.g. Endpoint/ Root port)
                elif pname in get_params_with_hidden_conditions() and\
                    not self.is_param_available(pname):
                        continue

                # If it was a boolean, we need to convert it to 0,1
                prop_data = self._param_info.get_prop_info_by_name(pname)
                assert prop_data is not None
                if prop_data.data_type == GenericParam.DataType.dbool:
                    if pval:
                        pval = "1"
                    else:
                        pval = "0"
                elif prop_data.data_type == GenericParam.DataType.dhex:
                    assert isinstance(pval, str)
                    if not pval.startswith("0x"):
                        pval = "0x" + pval

                # convert to string
                param_map[pname] = str(pval)

            ins_param_map[hw_name] = param_map

        return ins_param_map

    @staticmethod
    def get_quad_idx2_pll_refclk_source(dev_die_name: str):
        match dev_die_name:
            case "Ti375":
                return {
                    0: ["GPIOR_140", "GPIOR_145"],
                    2: ["GPIOR_140", "GPIOR_145"],
                }
            case "Ti135":
                return {
                    0: ["GPIOR_129", "GPIOR_130"],
                }
            case _:
                return {}

    def get_external_refclk_related_pll(self, design: PeriDesign,
                    dev_service: QuadPCIEDeviceService) -> Tuple[Optional[EfxFpllV1], str]:
        target_inst: Optional[EfxFpllV1] = None
        refclk_name = ""

        param_id = PCIEConfigParamInfo.Id.ss_refclk_onboard_osc
        is_enable = self.param_group.get_param_value(param_id.value)
        is_param_available = self.is_param_available(param_id)

        # Nothing to do if not available
        if is_enable or not is_param_available or\
            design.device_db is None or self.get_device() == "":
            return target_inst, refclk_name

        # PLL device
        from device.db_interface import DeviceDBService
        dbi = DeviceDBService(design.device_db)
        pll_dev_service = dbi.get_block_service(design.device_db.get_pll_type())
        if pll_dev_service is None:
            return target_inst, refclk_name

        # Expected resource
        exp_refclk_src_list = self.get_expected_pll_ref_clk_src(design.device_db)

        if len(exp_refclk_src_list) <= 0 or \
            design.lvds_reg is None or design.gpio_reg is None:
            return target_inst, refclk_name

        # Check PLL settings
        pll_ref_clk_list = ["PMA_CMN_REFCLK_PLL_1", "PMA_CMN_REFCLK_PLL_2"]
        for pll_ref_clk in pll_ref_clk_list:
            # :PLL resource + output clock
            ref_clk_inst, _, inst = self.get_pll_ref_clk_info_by_refclk_name(
                pll_ref_clk, dev_service, design)

            if inst is None or ref_clk_inst == "":
                continue

            # : Ref clock mode + feedback mode
            elif inst.ref_clock_mode != inst.RefClockModeType.external or\
                    inst.fb_mode != inst.FeedbackModeType.local:
                continue

            # Ref clock resource
            ref_clk_info = inst.find_external_clock_info(inst.ext_ref_clock_id,
                        pll_dev_service, design.lvds_reg, design.gpio_reg, design.device_db)
            cur_src_list = [src for src, _, _ in ref_clk_info if src != ""]
            is_valid = any([exp_src in cur_src_list for exp_src in exp_refclk_src_list])

            if is_valid:
                target_inst = inst
                refclk_name = pll_ref_clk
                break

        return target_inst, refclk_name

    def get_expected_pll_ref_clk_src(self, device_db: PeripheryDevice):
        quad_idx, _ = QuadResService.break_res_name(self.get_device())

        if quad_idx == -1 or device_db is None:
            return []

        _, dev_die_name = device_db.get_device_family_die_name()
        quad_idx2ref_clk_src = self.get_quad_idx2_pll_refclk_source(
            dev_die_name)

        return quad_idx2ref_clk_src.get(quad_idx, [])


class QuadPCIERegistry(PeriDesignRegistry):
    def __init__(self):
        self.res_svc = None  #: Service for shared HSIO resource
        super().__init__()

    def set_resource_service(self, quad_svc: QuadResService):
        self.res_svc = quad_svc

    def apply_device_db(self, device_db: PeripheryDevice):
        assert device_db is not None
        self.device_db = device_db
        QuadPCIE.build_port_info(self.device_db, False)

        svc = self.get_blk_service(self.device_db)
        assert svc is not None

    def create_instance(self, name, apply_default=True, auto_pin=False) -> QuadPCIE:
        with self._write_lock:
            inst = QuadPCIE(name)
            if apply_default:
                inst.set_default_setting(self.device_db)
            if auto_pin:
                inst.generate_pin_name()

            self._register_new_instance(inst)
            return inst

    def is_device_valid(self, device_def: str) -> bool:
        #svc = self.get_blk_service(self.device_db)
        #assert svc is not None
        #return device_def in svc.get_usable_instance_names()
        assert self.res_svc is not None
        return self.res_svc.is_device_valid(device_def)

    @staticmethod
    def get_blk_service(device_db: PeripheryDevice):
        from device.db_interface import DeviceDBService
        dbi = DeviceDBService(device_db)
        return dbi.get_block_service(DeviceDBService.BlockType.QUAD_PCIE)

    def assign_inst_device(self, inst, new_dev):
        super().assign_inst_device(inst, new_dev)

        if self.device_db is None:
            return

        svc = self.get_blk_service(self.device_db)
        assert svc is not None

    def get_all_external_refclk_related_pll(self, design: PeriDesign):
        pll_inst_list = []
        if design.device_db is None:
            return pll_inst_list

        dev_service = self.get_blk_service(design.device_db)

        for inst in self.get_all_inst():
            inst: QuadPCIE
            pll_inst, _ = inst.get_external_refclk_related_pll(design, dev_service)
            if pll_inst is not None and pll_inst not in pll_inst_list:
                pll_inst_list.append(pll_inst)

        return pll_inst_list
