"""
Copyright (C) 2017-2023 Efinix Inc. All rights reserved.

No portion of this code may be reused, modified or
distributed in any way without the expressed written
consent of Efinix Inc.

Created on Sep 5, 2023

@author: maryam
"""
from typing import Dict, TYPE_CHECKING

import util.gen_util
import common_device.rule as base_rule

from tx60_device.osc.osc_rule_adv import OSCCheckerAdv
from tx60_device.osc.osc_design_adv import OSCAdvance


@util.gen_util.freeze_it
class OSCCheckerV3(OSCCheckerAdv):
    """
    Inherits Titanium oscillator with the addition of new
    rules for ti375.
    """

    def _build_rules(self):
        """
        Build oscillator rule database
        """
        super()._build_rules()
        self._add_rule(OSCRuleEnablePCIePinName())


class OSCRuleEnablePCIePinName(base_rule.Rule):
    """
    Check if oscillator enable pin name used while a PCIE
    is also configured.  There is a dedicated path from HPOSC
    to the Quad and it needs a free running clock. So,
    user cannot tristate the OSC if PCIe is configured.
    """

    def __init__(self):
        super().__init__()
        self.name = "osc_rule_ena_pin_transceiver"

    def is_serdes_quad_instance_exists(self, checker):
        is_quad_ins_found = False
        usage_str_list = []
        blk_names = ""

        if checker.design.quad_pcie_reg is not None and checker.design.quad_pcie_reg.get_inst_count() > 0:
            usage_str_list.append("PCIe")
        if checker.design.lane_10g_reg is not None and checker.design.lane_10g_reg.get_inst_count() > 0:
            usage_str_list.append("Ethernet XGMII")
        if checker.design.lane_1g_reg is not None and checker.design.lane_1g_reg.get_inst_count() > 0:
            usage_str_list.append("Ethernet SGMII")
        if checker.design.raw_serdes_reg is not None and checker.design.raw_serdes_reg.get_inst_count() > 0:
            usage_str_list.append("PMA Direct")

        if usage_str_list:
            blk_names = ",".join(usage_str_list)            
            is_quad_ins_found = True

        return is_quad_ins_found, blk_names
    
    def check(self, checker, design_block: OSCAdvance):

        util.gen_util.mark_unused(checker)

        is_quad_used, blk_name_msg = self.is_serdes_quad_instance_exists(checker)

        if is_quad_used:
            osc = design_block

            ena_pin = osc.gen_pin.get_pin_by_type_name("ENA")
            if ena_pin is not None and not self.is_name_empty(ena_pin.name):
                self.error(
                    f"The oscillator clock cannot be tri-state and has to be free running for {blk_name_msg} usage")
