from __future__ import annotations
from typing import TYPE_CHECKING
from decimal import Decimal

from util.signal_util import float_to_decimal

from common_device.quad.writer.timing import QuadSERDESTiming
from common_device.quad.res_service import QuadType
from tx375_device.lane1g.lane1g_prop_id import Lane1GConfigParamInfo as Lane1GParamInfo

if TYPE_CHECKING:
    from common_device.quad.lane_design import LaneBasedItem


class Lane1GTiming(QuadSERDESTiming):

    def get_quad_type(self):
        return QuadType.lane_1g
    
    def get_mode_name(self):
        return "LN0_1G"

    def get_user_blk_name(self):
        return "Ethernet SGMII"
    
    def is_pin_type_name_clkout(self, type_name, ins_obj):
        return ins_obj.is_skip_clk_port_name(type_name)

    def get_clk_period_precision(self):
        return 1

    def write(self, index, ins_to_block_map, clk2delay_map, blk_list):
        '''
        Write out the report and sdc constraint for this block.

        :param index: The index of this section. Used in printing to report
        :param ins_to_block_map: Map of device instance name to the block name
        :param clk2delay_map: Map of delay type to the corresponding delay value
        :param blk_list: List of PeripheryBlock that is of type quad
        '''
        ins_exists = False

        if self._design.lane_1g_reg is None:
            return ins_exists
        
        # Save the common registry to the writer
        self.common_quad_reg = self._design.lane_1g_reg.common_quad_reg

        try:

            # Get all instances of 10g
            ins_name2obj_map = self._design.lane_1g_reg.get_name2inst_map()

            if ins_name2obj_map:

                blk2mode_map = self.get_blk2mode_map(blk_list)
                
                # We're going to operate based on blocks: quad, quad_pcie
                # Since the timing data is saved per block definition
                def get_name(blk):
                    return blk.get_name()
                
                is_start = True

                for blk in sorted(blk2mode_map, key=get_name):                    
                    blk_mode = blk2mode_map[blk]

                    self.build_mode_based_arc(blk_mode)

                    # Then we need to separate the instance based on
                    # the block type (quad, quad_pcie)
                    blk_ins_name2obj_map = self.get_instance_based_on_blk(blk, ins_name2obj_map)    

                    # This is a map of the timign parameter name to
                    # the actual delay value. Ensure that the right
                    # block name is being passed since we want
                    # to get the delay based on the corresponding block (quad/quad_pcie)
                    max_param2delay_map = self._parse_simple_timing_table(
                        self._max_model, blk.get_name())
                    min_param2delay_map = self._parse_simple_timing_table(
                        self._min_model, blk.get_name()) 

                    # If any of the map is empty, then throw exception (unexpected)
                    if not max_param2delay_map or not min_param2delay_map:
                        raise ValueError('Unable to find the delay'
                                         ' parameter table for block {}'.format(blk.get_name()))

                    if blk_ins_name2obj_map:
                        if self.write_output(index, ins_to_block_map,
                                            clk2delay_map,
                                            max_param2delay_map,
                                            min_param2delay_map,
                                            blk_ins_name2obj_map, blk_mode, is_start):
                            ins_exists = True                    
                            is_start = False

        except Exception as excp:
            raise excp

        return ins_exists
    
    def write_create_clock(self, sdcfile, ins_obj):
        '''
        Writing out clock that are output from the block to core

        :param sdcfile: The FP for the SDC file
        :param ins_obj: The 1G design instance
        :return:
        '''
        if ins_obj is not None:
            # Write the clock source constraint
            clk_pin_type_names = ["1GBE_CLK", "1GBE_CLK_X2"]
            for type_name in clk_pin_type_names:
                gb_clk = ins_obj.gen_pin.get_pin_name_by_type(type_name)
                if gb_clk != "":
                    clk_period = self.get_clock_period(ins_obj, type_name)
                    sdc = "create_clock -period {} -name {} [get_ports {{{}}}]".format(clk_period, gb_clk, gb_clk)
                    sdcfile.write("{}\n".format(sdc))

    def get_clock_period(self, ins_obj: LaneBasedItem, pin_type_name: str) -> Decimal:
        precision = self.get_clk_period_precision()
        value = 0.0

        data_rate = ins_obj.param_group.get_param_value(
            Lane1GParamInfo.Id.ss_1gbe_data_rate_lane_NID.value)

        # PT-2647: CLK_X2 period should be halfed
        match data_rate:
            case "10/100/1000 Mbps":
                if pin_type_name == "1GBE_CLK_X2":
                    value = 8.0
                else:
                    value = 16.0
            case "2.5 Gbps":
                if pin_type_name == "1GBE_CLK_X2":
                    value = 3.2
                else:
                    value = 6.4

        return float_to_decimal(value, precision)
