from __future__ import annotations

from enum import Enum
from typing import List, Tuple, Callable, Optional, TYPE_CHECKING

import pt_version

from util.widget_formatter import WidgetFormatter

from common_device.property import PropertyMetaData
from common_device.quad.res_service import QuadType

from tx375_device.common_quad.design import QuadLaneCommon
from tx375_device.common_quad.design_param_info import QuadDesignParamInfo as QuadParamInfo
from tx375_device.common_quad.quad_prop_id import CommonQuadConfigParamInfo as QuadConfigParamInfo
from tx375_device.lane1g.design import Lane1GRegistry, Lane1G
from tx375_device.lane1g.design_param_info import Lane1GDesignParamInfo
from tx375_device.lane1g.lane1g_prop_id import Lane1GConfigParamInfo
from tx375_device.lane1g.quad_param_info import get_hidden_parameters
from tx375_device.lane1g.lane1g_pin_dep_graph import get_hidden_ports

from tx375_device.lane1g.gui.builder import Lane1GTabBuilder

from design.db import PeriDesign
from design.db_item import GenericParam, GenericParamGroup, GenericParamService

from common_gui.base_viewer import BaseViewer, BaseViewerModel, GenericParam, GenericParamGroup, bool2str

IS_SHOW_HIDDEN = True if pt_version.PT_DEBUG_VERSION == True else False

if TYPE_CHECKING:
    from tx375_device.lane1g.gui.config import Lane1GConfig


class Lane1GViewer(BaseViewer):

    # Overload
    def build(self, block_name: str, design: PeriDesign):
        self.block_name = block_name
        self.design = design
        self.block_config: Lane1GConfig

        block_reg = self.design.get_block_reg(PeriDesign.BlockType.lane_1g)
        assert isinstance(block_reg, Lane1GRegistry)

        block = block_reg.get_inst_by_name(block_name)
        self.model = Lane1GViewerModel(self.block_config)

        # Remarks: This function called even block is not created
        if block is None:
            self.logger.warning(f"1G {block_name} does not exists")
            self.model.unload()

        elif isinstance(block, Lane1G):
            self.model.load(self.design, block_name=block_name)

        self.tv_viewer.setModel(self.model)

        formatter = WidgetFormatter()
        formatter.refresh_table_view_format(self.tv_viewer)


class Lane1GViewerModel(BaseViewerModel):
    def __init__(self, config: Lane1GConfig):
        super().__init__()
        self.config = config
        self.presenter = config.presenter

    def load(self, design: PeriDesign, block_name: str):
        # Load design
        self.design = design
        block_reg = self.design.get_block_reg(PeriDesign.BlockType.lane_1g)
        assert isinstance(block_reg, Lane1GRegistry)

        # Get instance
        block = block_reg.get_inst_by_name(block_name)
        assert isinstance(block, Lane1G)
        common_inst = block_reg.get_cmn_inst_by_lane_name(block.name)
        assert isinstance(common_inst, QuadLaneCommon)

        # Load param + pin info
        self.add_param_info(block, common_inst)
        self.add_pin_info(block, common_inst)

    def add_param_info(self, block: Lane1G, common_inst: QuadLaneCommon):
        """
        Load param information (Lane-based instance + Common instance)
        """
        ##### Lane-based Param
        self.add_row_item("Instance Name", block.name)
        self.add_row_item("SGMII Resource", block.get_device())

        pins_categories_with_params = [
            "CLOCK and RESET",
        ]

        # Skip for Pin, Common Property (Add later)
        for category in Lane1GTabBuilder.param_categories[:-2] + pins_categories_with_params:
            desc = Lane1GTabBuilder.category2display_name.get(category, category)
            self.add_row_group_by_prop_category(block, category, desc)

        ##### Common Param
        self.add_row_item("Config", "", highlight=True)
        self.add_row_item("Common Instance Name", common_inst.name)
        self.add_cmn_param_info_by_list(common_inst, block, self.config.get_cmn_config_param())

        # Ref clock group
        self.add_ref_clk_info(common_inst, block)

        # Common categories
        for category in Lane1GTabBuilder.common_categories2display_name.keys():
            display_name = Lane1GTabBuilder.category2display_name.get(category, category)
            self.add_row_group_by_prop_category(common_inst, category, display_name,
                                                lane_name=block.name)

    def add_pin_info(self, block: Lane1G, common_inst: QuadLaneCommon):
        # Available & Hidden ports
        available_pin_type_names = set(block.get_available_pins())
        available_cmn_pins = common_inst.get_available_pins_by_quad_type(QuadType.lane_1g, block.name)

        hidden_ports = [] if IS_SHOW_HIDDEN else get_hidden_ports()
        hidden_ports += common_inst.get_filter_pins_by_quad_type(block.quad_type)

        # Filter function
        def _is_pin_available(pin_type: str):
            return pin_type in available_pin_type_names and pin_type not in hidden_ports

        def _is_common_pin_available(pin_type: str):
            return pin_type in available_cmn_pins and pin_type not in hidden_ports

        # Lane-based pins tabs
        self.add_row_item("Pin", "", highlight=True)
        for pin_class in Lane1GTabBuilder.pin_categories:
            self.add_pin_view(block, pin_class, filter_func=_is_pin_available)

            for child in Lane1GTabBuilder.parent_category2children.get(pin_class, []):
                self.add_pin_view(block, child, filter_func=_is_pin_available)

        # Common pin tab
        self.add_row_item("Common Pin", "", highlight=True)
        for pin_class in Lane1GTabBuilder.common_categories2display_name:
            self.add_pin_view(common_inst, pin_class, filter_func=_is_common_pin_available)

            for child in Lane1GTabBuilder.parent_category2children.get(pin_class, []):
                self.add_pin_view(common_inst, child, filter_func=_is_common_pin_available)

    def add_row_group_by_prop_category(self, block: Lane1G| QuadLaneCommon, category: str, group_display_name: str,
                                        lane_name: str = "",
                                        custom_order: Optional[List[Enum]]=None,
                                        add_extra_info_func: Optional[Callable[[Lane1G|QuadLaneCommon], None]]=None):
        """
        Add Mutliple rows in the viewer about the parameters in specific cateogory

        :param category: The category of the parameters to be added
        :param group_display_name: The display name of the highlighted row at the top of the group
        :param custom_order: Arrangement orders of the parameters row, Default will be sorted by parameter's display name if not provided
        :param add_extra_info_func: Hook function to add extra rows at the bottom of the group
        """
        if not IS_SHOW_HIDDEN and category in Lane1GTabBuilder.hidden_categories:
            return

        param_info = block.get_param_info()
        param_group = block.param_group
        prop_list: List[PropertyMetaData.PropData] = []

        if category == "" and custom_order is not None:
            for param_id in custom_order:
                prop = param_info.get_prop_info_by_name(param_id.value)
                if prop is not None:
                    prop_list.append(prop)
        else:
            prop_list = param_info.get_prop_by_category(category)

        if len(prop_list) == 0:
            return

        if isinstance(block, QuadLaneCommon):
            available_prop_names = block.get_available_params_by_quad_type(QuadType.lane_1g, lane_name)
        else:
            available_prop_names = block.get_available_params()

        hidden_params = []
        if not IS_SHOW_HIDDEN:
            hidden_params = get_hidden_parameters() + \
                [QuadParamInfo.Id.phy_reset_en.value]

        hidden_params += [
            Lane1GConfigParamInfo.Id.ss_topology_lane_NID.value,
        ]

        prop_id_list: List[Lane1GDesignParamInfo.Id] = [x.id for x in prop_list if x.name in available_prop_names and x.id.name not in hidden_params]

        if custom_order:
            order = {item:idx for idx, item in enumerate(custom_order)}
            prop_id_list.sort(key=lambda id: order.get(id, len(prop_id_list)))

        if len(prop_id_list) <= 0:
            return

        if group_display_name != "":
            self.add_row_item(group_display_name, "", highlight=True)
        self.add_generic_param_by_type(param_group, param_info, include_type=prop_id_list, is_sort=custom_order is None)

        # additional info
        if add_extra_info_func is not None:
            add_extra_info_func(block)

    def add_ref_clk_info(self, common_inst: QuadLaneCommon, block):
        """
        Load reference clock information
        """
        display_name = "Reference Clock"
        param_list: List[Enum] = []

        self.add_row_item(display_name, "", highlight=True)
        param_list = [
            QuadConfigParamInfo.Id.ss_1gbe_refclk_freq,
            QuadConfigParamInfo.Id.PIPE_CONFIG_CMN__config_reg_2__pma_cmn_refclk_sel,
        ]
        self.add_cmn_param_info_by_list(common_inst, block, param_list)

        # Reference clock source
        ref_clk = common_inst.param_group.get_param_value(
            QuadConfigParamInfo.Id.PIPE_CONFIG_CMN__config_reg_2__pma_cmn_refclk_sel.value)

        match ref_clk:
            case "External":
                param_list = [
                    QuadConfigParamInfo.Id.PMA_CMN__cmn_plllc_gen_preg__cmn_plllc_pfdclk1_sel_preg,
                    QuadConfigParamInfo.Id.PIPE_CONFIG_CMN__config_reg_2__pma_cmn_refclk_termen,
                    QuadConfigParamInfo.Id.PIPE_CONFIG_CMN__config_reg_2__pma_cmn_refclk1_termen,
                ]

            case "Internal":
                param_list = [
                    QuadParamInfo.Id.ref_clk_pin_name
                ]

        self.add_cmn_param_info_by_list(common_inst, block, param_list)

    def add_cmn_param_info_by_list(self, common_inst: QuadLaneCommon, block: Lane1G,
                                    param_list: List[Enum]):
        """
        Load common instance param information by a list of param id
        """
        cmn_param_svr = GenericParamService(common_inst.param_group, common_inst.param_info)
        assert self.presenter is not None

        for param in param_list:
            # Hidden param check
            if not IS_SHOW_HIDDEN and param.value in get_hidden_parameters():
                continue

            # If param is support in current protocol
            if not (param.value in block.get_supported_common_parameters() or\
                param in common_inst.get_sw_supported_cmn_parameters(block.quad_type)):
                continue

            # Show only if available
            if not common_inst.is_param_available(param.value, block.quad_type, block.name):
                continue

            value = cmn_param_svr.get_param_value(param)
            disp = cmn_param_svr.get_param_disp_name(param)

            if isinstance(value, bool):
                value = bool2str(value)
            elif param.name == \
                QuadConfigParamInfo.Id.PMA_CMN__cmn_plllc_gen_preg__cmn_plllc_pfdclk1_sel_preg.value:
                assert isinstance(value, str)
                value = self.presenter.get_displayed_ext_clk_name(value)

            self.add_row_item(disp, str(value))

    def add_pin_view(self, block: Lane1G| QuadLaneCommon, pin_class: str, filter_func: Callable):
        """
        Load lane/ common quad instance pin information by pin class.
        Can be filtered by adding filter function
        """
        if not IS_SHOW_HIDDEN and pin_class in Lane1GTabBuilder.hidden_categories:
            return
        pin_section_color = BaseViewerModel.get_secondary_highlight_colour()

        # Common class filter
        def common_filter_pin(blk: Lane1G| QuadLaneCommon, pin_class: str):
            pins = [pin.type_name for pin in blk.get_pin_by_class(f"{pin_class}:SW_COMMON")]
            pins += [pin.type_name for pin in blk.get_pin_by_class(pin_class)]
            return pins

        if pin_class in Lane1GTabBuilder.common_pin_categories:
            pin_type_name_list = common_filter_pin(block, pin_class)
        else:
            pin_type_name_list = block.get_pin_type_by_class(pin_class)

        pin_type_name_list = list(filter(filter_func, pin_type_name_list))

        cat_display_name = Lane1GTabBuilder.category2display_name.get(pin_class, pin_class)
        if len(pin_type_name_list) > 0:
            self.add_row_item(cat_display_name, "", highlight=True, highlight_colour=pin_section_color)
            self.add_generic_pin_by_type(block, include_type=pin_type_name_list, is_sort=False)

    def add_generic_param_by_type(self, gen_param_group: GenericParamGroup, param_info: PropertyMetaData, enumtype2str_func=None, include_type=[], is_sort=True):

        filtered_list: List[Tuple[GenericParam, PropertyMetaData.PropData]] = []

        param_list: List[GenericParam] = gen_param_group.get_all_param()
        for param in param_list:
            prop_info = param_info.get_prop_info_by_name(param.name)
            if prop_info is None or prop_info.id not in include_type:
                continue
            filtered_list.append((param, prop_info))

        if is_sort:
            lane_1g_param_list = sorted(filtered_list,
                                    key=lambda i: i[1].disp_name)
        else:
            # By Default, Use the order in include_type
            lane_1g_param_list = sorted(filtered_list,
                                    key=lambda i: include_type.index(i[1].id))

        for param, prop_info in lane_1g_param_list:
            disp_name = prop_info.disp_name if prop_info.disp_name != "" else param.name
            self._add_single_param(prop_info.id, prop_info.data_type, disp_name, param.value, enumtype2str_func)


if __name__ == "__main__":
    pass
