"""

Copyright (C) 2017-2024 Efinix Inc. All rights reserved.

No portion of this code may be reused, modified or
distributed in any way without the expressed written
consent of Efinix Inc.

Created on April 29, 2024

@author: Shirley Chan
"""

from __future__ import annotations
from typing import Callable, List, TYPE_CHECKING

import pt_version

from design.db import PeriDesign
from design.db_item import PeriDesignGenPinItem, GenericPin
from common_device.property import PropertyMetaData
from common_device.quad.gui.lane_config import BaseLaneConfig

from tx375_device.common_quad.design_param_info import QuadDesignParamInfo
from tx375_device.common_quad.quad_prop_id import CommonQuadConfigParamInfo as CommonQuadParamInfo

from tx375_device.lane1g.design import Lane1GRegistry, Lane1G
# from tx375_device.lane1g.lane1g_prop_id import Lane1GConfigParamInfo
from tx375_device.lane1g.quad_param_info import get_hidden_parameters

from tx375_device.lane1g.gui.presenter import Lane1GConfigUIPresenter
from tx375_device.lane1g.gui.graph_observer import Lane1GGraphObserver, CommonQuadGraphObserver
from tx375_device.lane1g.gui.builder import Lane1GTabBuilder

if TYPE_CHECKING:
    from tx375_device.lane1g.gui.main_window import Lane1GWindow


FILTER_PROP_FUNC = Callable[[PropertyMetaData], List[PropertyMetaData.PropData]]
FILTER_PIN_FUNC = Callable[[PeriDesignGenPinItem], List[GenericPin]]

IS_SHOW_HIDDEN = True if pt_version.PT_DEBUG_VERSION == True else False


class Lane1GConfig(BaseLaneConfig):
    """
    Controller for specific instance configuration
    """

    def __init__(self, parent: Lane1GWindow):
        super().__init__(parent)

        self.presenter: Lane1GConfigUIPresenter
        self.graph_observer: Lane1GGraphObserver
        self.common_graph_observer: CommonQuadGraphObserver
        self.block_inst: Lane1G
        self.block_reg: Lane1GRegistry

    def get_design_block_type(self) -> PeriDesign.BlockType:
        return PeriDesign.BlockType.lane_1g

    def get_ref_clk_cmn_params(self):
        return [
            CommonQuadParamInfo.Id.ss_1gbe_refclk_freq,
            CommonQuadParamInfo.Id.PIPE_CONFIG_CMN__config_reg_2__pma_cmn_refclk_sel,
            QuadDesignParamInfo.Id.ref_clk_pin_name
        ]

    def get_cmn_config_param(self) -> List:
        return [
            CommonQuadParamInfo.Id.ss_plllc1_refclk_sel,
            CommonQuadParamInfo.Id.ss_sim_enable,
            CommonQuadParamInfo.Id.sw_1gbe_remove_preamble,
        ]

    def get_hidden_parameters(self) -> List[str]:
        return get_hidden_parameters()

    def get_skip_parameters(self) -> List[str]:
        return [QuadDesignParamInfo.Id.phy_reset_en.value
                ] + super().get_skip_parameters()

    def build_blk_graph_observer(self):
        return Lane1GGraphObserver(self)

    def build_cmn_graph_observer(self):
        return CommonQuadGraphObserver(self)

    def build_presenter(self):
        return Lane1GConfigUIPresenter(self)

    def build_tab_builder(self):
        return Lane1GTabBuilder()

    def get_cmn_param_info(self, param_info: PropertyMetaData):
        return param_info.get_prop_info_by_id_list(self.get_cmn_config_param())
