import sys
import os

import util.gen_util

from design.db_patch import DesignVersion, DesignPatch
from design.db_item import GenericParamGroup
from tx375_device.lane1g.quad_param_info import Lane1GParamInfo as Lane1GParamInfo
from tx375_device.lane1g.design_param_info import Lane1GDesignParamInfo as DesignParamInfo
from tx375_device.lane1g.design import Lane1G, Lane1GRegistry
from tx375_device.common_quad.design import QuadLaneCommon
from tx375_device.common_quad.quad_prop_id import CommonQuadConfigParamInfo as CommonQuadParamInfo

sys.path.append(os.path.join(os.path.dirname(__file__), '..'))



@util.gen_util.freeze_it
class Lane1GPatch(DesignPatch):
    """
    Lane1G design backward compatibility patcher
    """

    def __init__(self, lane1g_reg: Lane1GRegistry, device_db):
        assert lane1g_reg.common_quad_reg is not None
        self.common_reg = lane1g_reg.common_quad_reg

        super().__init__(lane1g_reg, device_db)
        self.block_reg: Lane1GRegistry

        #
        # Patches sequence can imply dependency between patches. When the patch is applied,
        # it will be applied from the lowest version number to the highest.
        # This means it is safe for patch function to assume certain db states.
        #

        # PT-2453
        msg = "Ethernet SGMII: Remove common instance parameter"
        self.add_patch(DesignVersion.PATCH_VERSION_20242002,
                       Lane1GPatch.patch_remove_cmn_inst_raw_serdes_related_param, msg)

        # PT-2433
        msg = "Ethernet SGMII: Rename 1GBE X2 connection type parameter"
        self.add_patch(DesignVersion.PATCH_VERSION_20242006,
                       Lane1GPatch.patch_rename_inst_x2_conn_type_param, msg)
        
        # PT-2639
        msg = "Ethernet SGMII: Rename 1G_PCS_L_NID__an_advertised_abilities__sgmii_mode to 1G_PCS_L1__pcs_control__enable_auto_neg"
        self.add_patch(DesignVersion.PATCH_VERSION_20251004,
                       Lane1GPatch.patch_rename_auto_negotiation_param_name, msg)

    def patch_remove_cmn_inst_raw_serdes_related_param(self, lane_inst: Lane1G):
        """
        Remove some params in common instance due to Raw PLL settings

        Updated params:
        - ss_raw_data_rate, ss_raw_mode: common instance -> Raw Serdes
        - REF_CLK_FREQUENCY: raw serdes -> common instance

        :param lane_inst: Lane1G instance
        :return: True, if design is patched, else False
        """
        is_patch = False

        cmn_inst = self.block_reg.get_cmn_inst_by_lane_name(lane_inst.name)
        assert cmn_inst is not None

        cmn_param_group = cmn_inst.param_group

        cmn_param_list = [
            "ss_raw_data_rate",
            "ss_raw_mode"
        ]
        for cmn_param_name in cmn_param_list:
            cmn_param_group.remove_param(cmn_param_name)
            is_patch = True

        if is_patch:
            func_name = util.gen_util.get_function_name()
            self.logger.warning(
                "{}: Patch {} applied to {}".format(func_name, DesignVersion.PATCH_VERSION_20242002,
                                                    lane_inst.name))

        return is_patch

    def patch_rename_inst_x2_conn_type_param(self, lane_inst):
        is_patch = False

        param_group = lane_inst.get_param_group()

        rename_list = [
            ('ALT_FWD_CLK_CONN_TYPE', 'LN_1GBE_X2_CONN_TYPE'),
        ]
        is_patch = self.update_params(param_group, rename_list)

        if is_patch:
            func_name = util.gen_util.get_function_name()
            self.logger.warning(
                "{}: Patch {} applied to {}".format(func_name, DesignVersion.PATCH_VERSION_20242006, lane_inst.name))

        return is_patch
    
    def update_params(self, param_group: GenericParamGroup, rename_list):
        is_updated = False

        for ori_name, new_name in rename_list:
            ori_param = param_group.get_param_by_name(ori_name)
            new_param = param_group.get_param_by_name(new_name)

            if ori_param is None or new_param is None:
                continue

            param_group.set_param_value(new_name, ori_param.value)
            param_group.remove_param(ori_name)
            is_updated = True

        return is_updated
    
    def patch_rename_auto_negotiation_param_name(self, lane_inst):
        is_patch = False

        param_group = lane_inst.get_param_group()

        rename_list = [
            ('1G_PCS_L_NID__an_advertised_abilities__sgmii_mode', '1G_PCS_L_NID__pcs_control__enable_auto_neg'),
        ]
        is_patch = self.update_params(param_group, rename_list)

        if is_patch:
            func_name = util.gen_util.get_function_name()
            self.logger.warning(
                "{}: Patch {} applied to {}".format(func_name, DesignVersion.PATCH_VERSION_20251004, lane_inst.name))

        return is_patch        