from __future__ import annotations
from typing import List, Dict, Optional, TYPE_CHECKING, Tuple

from device.block_definition import ModeInterface, Port
from device.db import PeripheryDevice

from common_device.property import PropertyMetaData
from common_device.quad.res_service import QuadResService, QuadType
from common_device.quad.lane_design import LaneBasedItem, LaneBaseRegistry

from tx375_device.lane1g.gui.dep_graph import Lane1GDependencyGraph
from tx375_device.lane1g.design_param_info import Lane1GDesignParamInfo as Lane1GParamInfo
from tx375_device.lane1g.quad_param_info import build_param_info, get_supported_common_parameters
from tx375_device.lane1g.lane1g_prop_id import Lane1GConfigParamInfo
from tx375_device.lane1g.lane1g_pin_dep_graph import get_hidden_ports

from design.db_item import (
    GenericParam,
    GenericParamService,
)

if TYPE_CHECKING:
    from common_device.quad.res_service import QuadResService
    from tx375_device.quad_pcie.device_service import QuadPCIEDeviceService
    from tx375_device.common_quad.design import QuadLaneCommonRegistry


class Lane1G(LaneBasedItem):

    _device_port_map: Dict = {}
    dev_service = None # type: Optional[QuadPCIEDeviceService]

    def __init__(self, name: str, block_def: str = "", apply_default: bool = True):
        super().__init__(name, block_def=block_def)
        self.quad_type = QuadType.lane_1g
        self._dep_graph = Lane1GDependencyGraph(self)

        if apply_default:
            self.set_default_setting()

    @staticmethod
    def str2enumtype(prop_id: Lane1GParamInfo.Id, value: str):
        str2enum_map = None
        return str2enum_map

    @staticmethod
    def enumtype2str(prop_id: Lane1GParamInfo.Id, val):
        enum2str_map = None
        return enum2str_map

    def set_default_setting(self, device_db=None):
        param_service = GenericParamService(self._param_group, self._param_info)
        for param_id in Lane1GParamInfo.Id:
            default_val = self._param_info.get_default(param_id)
            param_service.set_param_value(param_id, default_val)

    def build_param_info(self):
        param_info = Lane1GParamInfo()

        prop_data_list: List[PropertyMetaData.PropData] = [
            PropertyMetaData.PropData(id=Lane1GParamInfo.Id.ln_1gbe_conn_type,
                name=Lane1GParamInfo.Id.ln_1gbe_conn_type.value,
                data_type=GenericParam.DataType.dstr,
                default="rclk",
                valid_setting=["gclk", "rclk"],
                disp_name='Interface Clock Input Connection Type',
                category='CLOCK and RESET'),
            PropertyMetaData.PropData(id=Lane1GParamInfo.Id.ln_1gbe_x2_conn_type,
                name=Lane1GParamInfo.Id.ln_1gbe_x2_conn_type.value,
                data_type=GenericParam.DataType.dstr,
                default="rclk",
                valid_setting=["gclk", "rclk"],
                disp_name='Interface Clock X2 Input Connection Type',
                category='CLOCK and RESET'),
            ]

        for prop_data in prop_data_list:
            param_info.add_prop_by_data(prop_data.id, prop_data)

        # Add the additional parameter that is from ICD
        more_param_info = build_param_info()

        # Combined the Enum and the ParamInfo
        param_info.concat_param_info(more_param_info)

        return param_info

    def _is_clkout_pin(self, pin_type_name: str) -> bool:
        return pin_type_name in {
                                 'PCS_CLK_RX',
                                 'PCS_CLK_TX',
                                }

    def get_input_clock_type_names(self) -> List[str]:
        return ['1GBE_CLK', '1GBE_CLK_X2']

    def get_used_pcs_clk_names(self, reg) -> Dict[str, Tuple[str, Lane1G]]:
        clk2user_name_ins_map = {}

        clk_in_name = self.gen_pin.get_pin_name_by_type('1GBE_CLK')
        if clk_in_name != "":
            # Get the lane index
            _, lane_idx = QuadResService.break_res_name(self.get_device())

            clkout_inf_names = ['PCS_CLK_TX', 'PCS_CLK_RX']

            for clkout in clkout_inf_names:
                port_name = '{}{}'.format(clkout, lane_idx)
                clk2user_name_ins_map[port_name] = (clk_in_name, self)

        return clk2user_name_ins_map

    def _is_input_clk_pin(self, pin_type_name: str) -> bool:
        return pin_type_name in self.get_input_clock_type_names()

    def is_skip_clk_port_name(self, port_name: str) -> bool:
        if port_name in ["PCS_CLK_RX", "PCS_CLK_TX"]:
            return True

        return False

    def get_clk_gpin_type_name(self, clkout_name: str) -> str:
        '''
        This is a very specific case to find the corresponding
        port that connects to the passed clockout name that is skip
        :param clkout_name: Skipped clock port name
        :return the corresponding created generic pin type name
        '''

        if self.is_skip_clk_port_name(clkout_name):
            return "1GBE_CLK"

        return ""
    
    @staticmethod
    def build_port_info(device_db: Optional[PeripheryDevice] = None, is_mockup: bool = False):
        """
        Build port info for 1G which is mode based.
        This function need to be call only once since it is static shared
        between class.

        :param device_db: Device db instance
        :param is_mockup: True, build a mockup data, else build from device db
        """
        assert is_mockup is False
        assert isinstance(device_db, PeripheryDevice)

        # We're using QUAD PCIE as a reference since they shsould
        # be the same list of interface for both QUAD and QUAD_PCIE
        from tx375_device.quad_pcie.device_service import QuadPCIEDeviceService

        dev_service = Lane1GRegistry.get_blk_service(device_db)
        assert isinstance(dev_service, QuadPCIEDeviceService)

        Lane1G.dev_service = dev_service

        mode_name_list = ['LN0_1G', 'LN1_1G', 'LN2_1G', 'LN3_1G']
        device_port_map: Dict[str, ModeInterface] = {}

        for mode_name in mode_name_list:
            device_port_map.update(dev_service.get_interface_by_mode(mode_name))

        Lane1G._device_port_map = {}

        skip_port = ["PCS_CLK_RX", "PCS_CLK_TX"]
        for name, mode_port in device_port_map.items():
            if name in skip_port:
                continue

            class_str = mode_port.get_class()
            if class_str is not None and class_str.endswith(":SW_COMMON"):
                continue

            if mode_port.get_type() != Port.TYPE_PAD:

                # All the interface have description except some
                # which are in exlucded list
                assert mode_port.get_description() not in ("", None)
                Lane1G._device_port_map[name] = mode_port

    def get_parameters_for_lpf_gen(self, common_reg: Optional[QuadLaneCommonRegistry]):
        '''
        Create a dictionary of paramter settings targetted
        for bitstream generation (input to hw script):
        user_registers = {
            "q#": {
                param_name (str): param_value (str),
            }
        }
        All parameters need to be written out eventhough not used
        in context
        '''
        ins_param_map: Dict[str, Dict[str, str]] = {}

        # Cannot determine when resource hasn't been assigned
        if self.get_device() == "":
            return ins_param_map

        def get_quad_name_lane_idx(res_name):
            quad_idx, lane_idx = QuadResService.break_res_name(res_name)
            if quad_idx != -1:
                return f'QUAD_{quad_idx}', lane_idx

            return "", lane_idx

        res2hwname_map = {
            "QUAD_0": "q0",
            "QUAD_1": "q1",
            "QUAD_2": "q2",
            "QUAD_3": "q3"
        }
        quad_ins_name, lane_idx = get_quad_name_lane_idx(self.get_device())

        hw_name = res2hwname_map.get(quad_ins_name, "")
        if hw_name != "":
            param_map = {}
            # we will need to populate all the parameter
            # that is only defined in the quad_pcie_param_info.py
            param_service = GenericParamService(self._param_group, self._param_info)
            for param_id in Lane1GConfigParamInfo.Id:

                # The following parameter is for SW purpose use only
                # and not meant for pcr generation (verified with ICD)
                if param_id.name.startswith('sw_'):
                    continue

                pval = param_service.get_param_value(param_id)
                pname = self._param_info.get_prop_name(param_id)
                assert pname not in param_map

                # If it was a boolean, we need to convert it to 0,1
                prop_data = self._param_info.get_prop_info_by_name(pname)
                assert prop_data is not None
                pval = self.translate_hw_param_value(pval, prop_data.data_type)

                # convert to string
                # also replace the pname
                # with the correct lane prefix based on the resource where
                # "L_NID" -> L#, with # the resource lane id

                revised_pname = self.translate_hw_param_name(pname, lane_idx)

                param_map[revised_pname] = str(pval)

            if common_reg is not None:
                param_map.update(self.get_cmn_parameters_from_cmn_reg(common_reg))

            ins_param_map[hw_name] = param_map

        return ins_param_map

    def get_clock_source_usage(self, port_name: str):
        '''
        The instance pin name in gen pin is a translated
        name per mode. Other block (ie. clockmux) only save
        based on the block port name. So, we need to get
        the mapped interface name and its info through this function

        :return the gen pin interface name and the connection type of
                the clock assigned
        '''

        inf_name = ""
        conn_type = None

        tup_info = self.get_clock_input_pin_info(port_name)
        if tup_info is not None:
            inf_name, param_id = tup_info

            param_service = GenericParamService(
                self.get_param_group(), self.get_param_info())

            # PM_CLK is dependent on the PWR MGMT enable setting.
            # So we return info provided that it's enabled
            conn_type = param_service.get_param_value(param_id)

        return inf_name, conn_type

    def get_clock_input_pin_info(self, port_name: str):
        clk_port2inf_map = {
            "CH_NID_TX_FWD_CLK": ("1GBE_CLK",  Lane1GParamInfo.Id.ln_1gbe_conn_type),
            "CH_NID_RX_FWD_CLK": ("1GBE_CLK_X2", Lane1GParamInfo.Id.ln_1gbe_x2_conn_type)
            # Not listing customized gen pin for the REFCLK since that
            # is not handled as part of gen pin
        }

        port_info = None

        if self.get_device() != "":
            # We can only tell which port it is if the resource has been set
            quad_idx, lane_idx = QuadResService.break_res_name(self.get_device())

            if quad_idx != -1 and lane_idx != -1:
                for temp_name in clk_port2inf_map:
                    pname = temp_name.replace("_NID", f'{lane_idx}')

                    if port_name == pname:
                        port_info = clk_port2inf_map[temp_name]
                        break

        return port_info

    # Overload this since we have some hw name mapped to interface
    # name and it may be searched by other non-pcie itself
    def get_inf_pin_by_type_name(self, type_name: str):
        translated_name = type_name
        tup_info = self.get_clock_input_pin_info(type_name)

        if tup_info is not None:
            mapped_name, _ = tup_info
            translated_name = mapped_name

        return self.gen_pin.get_pin_by_type_name(translated_name)

    def get_hidden_ports(self):
        return get_hidden_ports()

    @staticmethod
    def get_supported_common_parameters():
        return get_supported_common_parameters()


class Lane1GRegistry(LaneBaseRegistry):
    def apply_device_db(self, device_db: PeripheryDevice):
        assert device_db is not None
        self.device_db = device_db
        Lane1G.build_port_info(self.device_db, False)

    def get_instance_class(self):
        return Lane1G

    @staticmethod
    def get_blk_service(device_db: PeripheryDevice):
        from device.db_interface import DeviceDBService
        dbi = DeviceDBService(device_db)
        return dbi.get_block_service(DeviceDBService.BlockType.QUAD_PCIE)
