from __future__ import annotations
import os
import re
from decimal import Decimal, localcontext
from enum import Enum
from typing import Dict, TYPE_CHECKING, Optional, List

import util.gen_util
from util.app_setting import AppSetting
from util.signal_util import float_to_decimal

import device.db_interface as devdb_int
import common_device.rule as base_rule
from common_device.quad.rule import QuadLaneChecker, QuadRuleHW, QuadRuleOSCClock, QuadRuleInvalidHexValue, QuadRuleExternalClock
from common_device.quad.res_service import QuadResService

from design.db_item import GenericParamService, GenericParam, GenericPin

from tx375_device.common_quad.quad_prop_id import CommonQuadConfigParamInfo as CommonQuadParamInfo
from tx375_device.common_quad.design import QuadLaneCommon
from tx375_device.lane10g.design import Lane10G
from tx375_device.lane10g.lane10g_prop_id import Lane10GConfigParamInfo as Lane10GParamInfo

from efx_serdes.serdes_config import serdes_config

if TYPE_CHECKING:
    from tx375_device.lane10g.device_service import QuadLane10GDeviceService


@util.gen_util.freeze_it
class Lane10GChecker(QuadLaneChecker):

    def __init__(self, design):
        """
        Constructor
        """
        super().__init__(design)
        self.block_tag = "10gbase_kr"
        self.reg = design.lane_10g_reg
        self.common_quad_reg = design.common_quad_lane_reg
        self.dev_service = None # type: Optional[QuadLane10GDeviceService]

        if self.design is not None:
            self.reg = self.design.lane_10g_reg

            if self.design.device_db is not None:
                dbi = devdb_int.DeviceDBService(self.design.device_db)
                self.dev_service = dbi.get_block_service(
                    devdb_int.DeviceDBService.BlockType.QUAD_LANE_10G)

    def _build_rules(self):
        """
        Build lane10g rule database
        """
        self._add_rule(Lane10GRuleInstanceName())
        self._add_rule(Lane10GRuleResource())
        # Not activated and not yet removed in case it
        # is going to be used later with multiple lane-protocol support
        #self._add_rule(Lane10GRuleTopology())
        self._add_rule(Lane10GRuleExternalClock())
        self._add_rule(Lane10GRuleInvalidHexValue())
        self._add_rule(Lane10GRuleHW())
        self._add_rule(Lane10GRuleQuadResource())
        self._add_rule(Lane10GRuleOSCClock())
        self._add_rule(Lane10GRule10GBEClock())

class Lane10GRuleInstanceName(base_rule.Rule):
    """
    Check if the instance name is valid
    """

    def __init__(self):
        super().__init__()
        self.name = "10gbase_kr_rule_inst_name"

    def check(self, checker, design_block: Lane10G):
        util.gen_util.mark_unused(checker)

        lane10g = design_block

        if self.check_name_empty_or_invalid(lane10g.name,
                                            "Instance name is empty.",
                                            "Valid characters are alphanumeric characters with dash and underscore only."
                                            ):
            return

class Lane10GRuleResource(base_rule.Rule):
    """
    Check if the resource name is valid.
    """

    def __init__(self):
        super().__init__()
        self.name = "10gbase_kr_rule_resource"

    def check(self, checker: Lane10GChecker, design_block: Lane10G):

        lane10g = design_block

        if self.is_name_empty(lane10g.get_device()):
            self.error("Resource name is empty.")
            return

        # Check if resource name is a valid device instance name
        if checker.reg is not None and checker.dev_service is not None:
            lane10g_list = checker.dev_service.get_usable_instance_names()

            if lane10g.get_device() not in lane10g_list:
                self.error("Resource is not a valid Ethernet XGMII device instance")
                return

        else:
            checker.logger.warning(
                "{}: device db is empty, skip valid instance check.".format(self.name))

class Lane10GRuleTopology(base_rule.Rule):
    """
    Check if the 10G instance is part of a valid topology.
    """

    def __init__(self):
        super().__init__()
        self.name = "10gbase_kr_rule_topology"

    def check(self, checker: Lane10GChecker, design_block: Lane10G):

        lane10g = design_block

        # Resource is <quad>_<lane#>. THe following are valid topology
        # 1. all lanes used: LN0_LN1_LN2_LN3
        # 2. Only one lane per quad
        # 3. 2 lanes at LN0 and LN2 ONLY

        if checker.reg is not None and lane10g.get_device() != "":
            assert checker.reg.res_svc is not None

            quad_idx, _ = checker.reg.res_svc.break_res_name(lane10g.get_device())

            # Find instances using the same quad
            if quad_idx != -1:
                ins2reg_map = checker.reg.res_svc.find_instance_same_quad(f'Q{str(quad_idx)}')

                invalid_topology_msg = ""
                if len(ins2reg_map) != 4 and len(ins2reg_map) != 1:
                    if len(ins2reg_map) == 2:
                        res_used_list = ins2reg_map.keys()
                        ins_count = 0
                        for res_name in res_used_list:
                            if res_name.find("LN0") != -1 or\
                                res_name.find("LN2") != -1:
                                ins_count += 1

                        if ins_count != 2:
                            invalid_topology_msg = "Only Lanes 0 and 2 can be used together"

                    else:
                        invalid_topology_msg = "Allowed lane topology: 1, 2 or 4 lanes used only"

                if invalid_topology_msg != "":
                    self.error(invalid_topology_msg)

class Lane10GRuleExternalClock(QuadRuleExternalClock):
    def __init__(self):
        super().__init__()
        self.name = "10gbase_kr_rule_external_clock"
        self.protocol_name = "Ethernet XGMII"


class Lane10GRuleInvalidHexValue(QuadRuleInvalidHexValue):
    def __init__(self):
        super().__init__()
        self.name = "10gbase_kr_rule_invalid_hex_value"


class Lane10GRuleHW(QuadRuleHW):
    '''
    Run the DRC from ICD script
    '''
    def __init__(self):
        super().__init__()
        self.name = "10gbase_kr_rule_hw_drc"

    def get_param_id_enum(self):
        return Lane10GParamInfo.Id


class Lane10GRuleQuadResource(base_rule.Rule):
    '''
    Safety check to make sure that there is no conflict
    in resource usage
    '''
    def __init__(self):
        super().__init__()
        self.name = "10gbase_kr_rule_res_usage"

    def check(self, checker: Lane10GChecker, design_block: Lane10G):
        blk = design_block

        # only check if resource is set
        if blk.get_device() == "":
            return

        same_quad_ins_list, _ = checker.find_shared_instance_quad(blk)

        # Check against instance of the same quad
        if same_quad_ins_list:
            for inst in same_quad_ins_list:
                if inst.get_device() == blk.get_device():
                    self.error(f"Resource name conflict with instance {inst.name}")
                    return

        # Check if it conflict with other protocol
        if checker.design is not None and checker.design.quad_pcie_reg is not None:
            quad_idx, _ = QuadResService.break_res_name(blk.get_device())

            if quad_idx != -1 and \
                checker.design.quad_pcie_reg.is_device_used(f'QUAD_{quad_idx}'):
                self.error(f"Resource conflicts with PCI Express resource QUAD_{quad_idx}")
                return

class Lane10GRuleOSCClock(QuadRuleOSCClock):
    """
    Check that the oscillator has been configured.
    """

    def __init__(self):
        super().__init__()
        self.name = "10gbase_kr_rule_osc_clock"

class Lane10GRule10GBEClock(base_rule.Rule):
    '''
    Check that the 10GBE clock isn't empty
    '''
    def __init__(self):
        super().__init__()
        self.name = "10gbase_kr_rule_interface_clock"

    def check(self, checker: Lane10GChecker, design_block: Lane10G):
        blk = design_block

        if blk.gen_pin is not None:
            clk_name = blk.gen_pin.get_pin_name_by_type("10GBE_CLK")

            if clk_name == "":
                self.error(f"Interface Clock Input pin name cannot be empty")
