from __future__ import annotations

import os
from enum import Enum
from typing import List, Tuple, Callable, Optional, TYPE_CHECKING

import pt_version

from util.widget_formatter import WidgetFormatter

from common_device.property import PropertyMetaData
from common_device.quad.res_service import QuadType

from tx375_device.common_quad.design import QuadLaneCommon
from tx375_device.common_quad.design_param_info import QuadDesignParamInfo as QuadParamInfo
from tx375_device.common_quad.quad_prop_id import CommonQuadConfigParamInfo as QuadConfigParamInfo
from tx375_device.common_quad.design_param_info import QuadDesignParamInfo
from tx375_device.lane10g.design import Lane10GRegistry, Lane10G
from tx375_device.lane10g.design_param_info import Lane10GDesignParamInfo
from tx375_device.lane10g.lane10g_prop_id import Lane10GConfigParamInfo
from tx375_device.lane10g.quad_param_info import get_hidden_parameters
from tx375_device.lane10g.lane10g_pin_dep_graph import get_hidden_ports

from tx375_device.lane10g.gui.builder import Lane10GTabBuilder

from design.db import PeriDesign
from design.db_item import GenericParam, GenericClockPin, GenericParamGroup, GenericParamService

from common_gui.base_viewer import BaseViewer, BaseViewerModel, GenericParam, GenericParamGroup, bool2str

IS_SHOW_HIDDEN = True if pt_version.PT_DEBUG_VERSION == True else False

if TYPE_CHECKING:
    from tx375_device.lane10g.gui.config import Lane10GConfig

class Lane10GViewer(BaseViewer):

    # Overload
    def build(self, block_name: str, design: PeriDesign):
        self.block_name = block_name
        self.design = design
        self.block_config: Lane10GConfig

        block_reg = self.design.get_block_reg(PeriDesign.BlockType.lane_10g)
        assert isinstance(block_reg, Lane10GRegistry)

        block = block_reg.get_inst_by_name(block_name)
        self.model = Lane10GViewerModel(self.block_config)

        # Remarks: This function called even block is not created
        if block is None:
            self.logger.warning(f"10G {block_name} does not exists")
            self.model.unload()

        elif isinstance(block, Lane10G):
            self.model.load(self.design, block_name=block_name)

        self.tv_viewer.setModel(self.model)

        formatter = WidgetFormatter()
        formatter.refresh_table_view_format(self.tv_viewer)


class Lane10GViewerModel(BaseViewerModel):
    def __init__(self, config: Lane10GConfig):
        super().__init__()
        self.config = config
        self.presenter = config.presenter

    def load(self, design: PeriDesign, block_name: str):
        self.design = design
        block_reg = self.design.get_block_reg(PeriDesign.BlockType.lane_10g)
        common_quad_reg = self.design.get_block_reg(PeriDesign.BlockType.common_quad)
        assert isinstance(block_reg, Lane10GRegistry)
        assert common_quad_reg is not None

        block = block_reg.get_inst_by_name(block_name)
        assert isinstance(block, Lane10G)
        common_inst = block_reg.get_cmn_inst_by_lane_name(block.name)
        assert isinstance(common_inst, QuadLaneCommon)

        self.add_row_item("Instance Name", block.name)
        self.add_row_item("10G Resource", block.get_device())

        hidden_tabs = Lane10GTabBuilder.hidden_categories

        # Need to check the order with Lane10GTabBuilder
        pins_categories_with_params = [
            "CLOCK and RESET",
            "CONTROL"
        ]

        # Skip for Pin, Common Property (Add later)
        for category in Lane10GTabBuilder.param_categories[:-2] + pins_categories_with_params:
            if not IS_SHOW_HIDDEN and category in hidden_tabs:
                continue
            desc = Lane10GTabBuilder.category2display_name.get(category, category)
            self.add_row_group_by_prop_category(block, category, desc)

        display_name = "Config"
        self.add_row_item(display_name, "", highlight=True)
        self.add_row_item("Common Instance Name", common_inst.name)

        cmn_param_svr = GenericParamService(common_inst.param_group, common_inst.param_info)

        # Load Common Information
        def load_cmn_param_info_by_list(param_list: List[Enum]):
            assert self.presenter is not None

            for param in param_list:
                if not IS_SHOW_HIDDEN and param.value in get_hidden_parameters():
                    continue
                if not (param.value in block.get_supported_common_parameters() or\
                    param in common_inst.get_sw_supported_cmn_parameters(block.quad_type)):
                    continue

                if not common_inst.is_param_available(param.value, block.quad_type, block.name):
                    continue

                value = cmn_param_svr.get_param_value(param)
                disp = cmn_param_svr.get_param_disp_name(param)

                if isinstance(value, bool):
                    value = bool2str(value)
                elif param.name == \
                    QuadConfigParamInfo.Id.PMA_CMN__cmn_plllc_gen_preg__cmn_plllc_pfdclk1_sel_preg.value:
                    assert isinstance(value, str)
                    value = self.presenter.get_displayed_ext_clk_name(value)

                self.add_row_item(disp, str(value))

        param_list = [
            QuadConfigParamInfo.Id.ss_plllc1_refclk_sel,
            QuadConfigParamInfo.Id.ss_sim_enable
        ]
        load_cmn_param_info_by_list(param_list) # type: ignore

        # Reference clock
        self.add_row_group_by_prop_category(block, "ref_clk", 'Reference Clock', custom_order=[
            Lane10GDesignParamInfo.Id.ref_clk_frequency,
        ])

        param_list = [
            QuadConfigParamInfo.Id.PIPE_CONFIG_CMN__config_reg_2__pma_cmn_refclk_sel,
        ]
        load_cmn_param_info_by_list(param_list) # type: ignore

        # Reference Clock Source
        ref_clk = cmn_param_svr.get_param_value(QuadConfigParamInfo.Id.PIPE_CONFIG_CMN__config_reg_2__pma_cmn_refclk_sel)

        param_list = []
        match ref_clk:
            case "External":
                param_list = [
                    QuadConfigParamInfo.Id.PMA_CMN__cmn_plllc_gen_preg__cmn_plllc_pfdclk1_sel_preg,
                    QuadConfigParamInfo.Id.PIPE_CONFIG_CMN__config_reg_2__pma_cmn_refclk_termen,
                    QuadConfigParamInfo.Id.PIPE_CONFIG_CMN__config_reg_2__pma_cmn_refclk1_termen,
                ]

            case "Internal":
                param_list = [
                    QuadParamInfo.Id.ref_clk_pin_name
                ]

        load_cmn_param_info_by_list(param_list) # type: ignore

        # Common categories
        for category in Lane10GTabBuilder.common_categories2display_name.keys():
            if not IS_SHOW_HIDDEN and category in hidden_tabs:
                continue
            display_name = Lane10GTabBuilder.category2display_name.get(category, category)
            self.add_row_group_by_prop_category(common_inst, category, display_name,
                                                lane_name=block.name)

        ## Load Pin information
        self.add_row_item("Pins", "", highlight=True)
        available_pin_type_names = set(block.get_available_pins())

        hidden_ports = []

        if not IS_SHOW_HIDDEN:
            hidden_ports = get_hidden_ports()

        hidden_ports += common_inst.get_filter_pins_by_quad_type(block.quad_type)

        def _is_pin_available(pin_type: str):
            return pin_type in available_pin_type_names and pin_type not in hidden_ports

        # Common class
        def common_filter_pin(blk: Lane10G, pin_class: str):
            pins = [pin.type_name for pin in blk.get_pin_by_class(f"{pin_class}:SW_COMMON")]
            pins += [pin.type_name for pin in blk.get_pin_by_class(pin_class)]
            return pins

        pin_section_color = BaseViewerModel.get_secondary_highlight_colour()

        def _add_pin_view(block, pin_class: str, filter_func: Callable):
            if pin_class in Lane10GTabBuilder.common_pin_categories:
                pin_type_name_list = common_filter_pin(block, pin_class)
            else:
                pin_type_name_list = block.get_pin_type_by_class(pin_class)

            pin_type_name_list = list(filter(filter_func, pin_type_name_list))

            desc = Lane10GTabBuilder.category2display_name.get(pin_class, pin_class)

            if len(pin_type_name_list) > 0:
                self.add_row_item(desc, "", highlight=True, highlight_colour=pin_section_color)
                self.add_generic_pin_by_type(block, include_type=pin_type_name_list, is_sort=False)

        for pin_class in Lane10GTabBuilder.pin_categories:
            _add_pin_view(block, pin_class, filter_func=_is_pin_available)

            for child in Lane10GTabBuilder.parent_category2children.get(pin_class, []):
                _add_pin_view(block, child, filter_func=_is_pin_available)

        # Common pin
        self.add_row_item("Common Pin", "", highlight=True)
        available_cmn_pins = common_inst.get_available_pins_by_quad_type(QuadType.lane_10g, block.name)

        def _is_common_pin_available(pin_type: str):
            return pin_type not in hidden_ports and pin_type in available_cmn_pins

        for pin_class in Lane10GTabBuilder.common_categories2display_name:
            _add_pin_view(common_inst, pin_class, filter_func=_is_common_pin_available)

            for child in Lane10GTabBuilder.parent_category2children.get(pin_class, []):
                _add_pin_view(common_inst, child, filter_func=_is_common_pin_available)


    def add_row_group_by_prop_category(self, block: Lane10G| QuadLaneCommon, category: str, group_display_name: str,
                                        lane_name: str = "",
                                        custom_order: Optional[List[Enum]]=None,
                                        add_extra_info_func: Optional[Callable[[Lane10G|QuadLaneCommon], None]]=None):
        """
        Add Mutliple rows in the viewer about the parameters in specific cateogory

        :param category: The category of the parameters to be added
        :param group_display_name: The display name of the highlighted row at the top of the group
        :param custom_order: Arrangement orders of the parameters row, Default will be sorted by parameter's display name if not provided
        :param add_extra_info_func: Hook function to add extra rows at the bottom of the group
        """
        param_info = block.get_param_info()
        param_group = block.param_group

        prop_list: List[PropertyMetaData.PropData] = param_info.get_prop_by_category(category)

        if isinstance(block, QuadLaneCommon):
            available_prop_names = block.get_available_params_by_quad_type(QuadType.lane_10g, lane_name)
        else:
            available_prop_names = block.get_available_params()

        hidden_params = []
        if not IS_SHOW_HIDDEN:
            hidden_params = get_hidden_parameters() + \
                [Lane10GDesignParamInfo.Id.alt_fwd_conn_type.value,
                 QuadDesignParamInfo.Id.phy_reset_en.value]

        hidden_params += [
            Lane10GConfigParamInfo.Id.ss_topology_lane_NID.value,            
            QuadDesignParamInfo.Id.apb_en.value
        ]

        prop_id_list: List[Lane10GDesignParamInfo.Id] = [x.id for x in prop_list if x.name in available_prop_names and x.name not in hidden_params]

        if custom_order:
            order = {item:idx for idx, item in enumerate(custom_order)}
            prop_id_list.sort(key=lambda id: order.get(id, len(prop_id_list)))

        if len(prop_id_list) <= 0:
            return
        self.add_row_item(group_display_name, "", highlight=True)
        self.add_generic_param_by_type(param_group, param_info, include_type=prop_id_list, is_sort=custom_order is None)

        # additional info
        if add_extra_info_func is not None:
            add_extra_info_func(block)

    def add_generic_param_by_type(self, gen_param_group: GenericParamGroup, param_info: PropertyMetaData, enumtype2str_func=None, include_type=[], is_sort=True):
        assert self.presenter is not None

        filtered_list: List[Tuple[GenericParam, PropertyMetaData.PropData]] = []

        param_list: List[GenericParam] = gen_param_group.get_all_param()
        for param in param_list:
            prop_info = param_info.get_prop_info_by_name(param.name)
            if prop_info is None or prop_info.id not in include_type:
                continue
            filtered_list.append((param, prop_info))

        if is_sort:
            lane_10g_param_list = sorted(filtered_list,
                                    key=lambda i: i[1].disp_name)
        else:
            # By Default, Use the order in include_type
            lane_10g_param_list = sorted(filtered_list,
                                    key=lambda i: include_type.index(i[1].id))

        for param, prop_info in lane_10g_param_list:
            disp_name = prop_info.disp_name if prop_info.disp_name != "" else param.name
            display_value = self.presenter.convert_db2display_value(param.name, param.value)
            self._add_single_param(prop_info.id, prop_info.data_type, disp_name, display_value, enumtype2str_func)

    def add_generic_pin_by_type(self, pin_container, include_type=[], is_sort=True, is_show_inv_clock=False):

        pin_group = pin_container.gen_pin
        pin_list = pin_group.get_all_pin()
        if len(pin_list) == 0:
            return

        # Default sorting by name
        if is_sort:
            txf_pin_list = sorted(pin_list,
                                     key=lambda this_pin: pin_container.get_pin_property_name(this_pin.type_name))
        else:
            txf_pin_list = pin_list

        for pin in txf_pin_list:
            if pin.type_name not in include_type:
                continue

            prop_name = pin_container.get_pin_property_name(pin.type_name)

            if prop_name == "":
                prop_name = pin.type_name

            self.add_row_item(prop_name, pin.name)

            if is_show_inv_clock and isinstance(pin, GenericClockPin):
                prop_name = "Invert {}".format(prop_name)
                self.add_row_item(prop_name, bool2str(pin.is_inverted))


if __name__ == "__main__":
    pass
