"""

Copyright (C) 2017-2024 Efinix Inc. All rights reserved.

No portion of this code may be reused, modified or
distributed in any way without the expressed written
consent of Efinix Inc.

Created on Feb 05, 2024

@author: Shirley Chan
"""

from __future__ import annotations
from typing import Callable, List, TYPE_CHECKING

import pt_version

from design.db import PeriDesign
from design.db_item import PeriDesignGenPinItem, GenericPin

from common_device.property import PropertyMetaData
from common_device.quad.gui.lane_config import BaseLaneConfig
from common_device.property import PropertyMetaData

from tx375_device.common_quad.design_param_info import QuadDesignParamInfo
from tx375_device.common_quad.quad_prop_id import CommonQuadConfigParamInfo as CommonQuadParamInfo

from tx375_device.lane10g.design import Lane10GRegistry, Lane10G
from tx375_device.lane10g.design_param_info import Lane10GDesignParamInfo as Lane10GParamInfo
# from tx375_device.lane10g.lane10g_prop_id import Lane10GConfigParamInfo
from tx375_device.lane10g.quad_param_info import get_hidden_parameters
from tx375_device.lane10g.gui.presenter import Lane10GConfigUIPresenter
from tx375_device.lane10g.gui.graph_observer import Lane10GGraphObserver, CommonQuadGraphObserver
from tx375_device.lane10g.gui.builder import Lane10GTabBuilder

if TYPE_CHECKING:
    from tx375_device.lane10g.gui.main_window import Lane10GWindow


FILTER_PROP_FUNC = Callable[[PropertyMetaData], List[PropertyMetaData.PropData]]
FILTER_PIN_FUNC = Callable[[PeriDesignGenPinItem], List[GenericPin]]

IS_SHOW_HIDDEN = True if pt_version.PT_DEBUG_VERSION == True else False


class Lane10GConfig(BaseLaneConfig):
    """
    Controller for specific instance configuration
    """

    def __init__(self, parent: Lane10GWindow):
        super().__init__(parent)

        self.presenter: Lane10GConfigUIPresenter
        self.graph_observer: Lane10GGraphObserver
        self.common_graph_observer: CommonQuadGraphObserver
        self.block_inst: Lane10G
        self.block_reg: Lane10GRegistry

    def get_design_block_type(self) -> PeriDesign.BlockType:
        return PeriDesign.BlockType.lane_10g

    def get_ref_clk_cmn_params(self):
        return [
            CommonQuadParamInfo.Id.PIPE_CONFIG_CMN__config_reg_2__pma_cmn_refclk_sel,
            QuadDesignParamInfo.Id.ref_clk_pin_name
        ]

    def get_hidden_parameters(self) -> List[str]:
        return get_hidden_parameters()

    def get_skip_parameters(self) -> List[str]:
        return [
            Lane10GParamInfo.Id.alt_fwd_conn_type.value,
            QuadDesignParamInfo.Id.phy_reset_en.value
        ] + super().get_skip_parameters()

    def get_blk_ref_freq_param_info(self, param_info: PropertyMetaData):
        return [param_info.get_prop_info(Lane10GParamInfo.Id.ref_clk_frequency)]

    def get_cmn_param_info(self, param_info: PropertyMetaData):
        return [
            param_info.get_prop_info(
                CommonQuadParamInfo.Id.ss_plllc1_refclk_sel),
            param_info.get_prop_info(
                CommonQuadParamInfo.Id.ss_sim_enable),
            param_info.get_prop_info(
                QuadDesignParamInfo.Id.apb_en)
        ]

    def build_blk_graph_observer(self):
        return Lane10GGraphObserver(self)

    def build_cmn_graph_observer(self):
        return CommonQuadGraphObserver(self)

    def build_presenter(self):
        return Lane10GConfigUIPresenter(self)

    def build_tab_builder(self):
        return Lane10GTabBuilder()

    def build_pin_widgets(self):
        super().build_pin_widgets()

        # CONTROL (Enable KR build at last)
        category = "CONTROL"
        container = self.category2vl.get(category)
        assert container is not None
        self.param_widget_map.update(
            self.gen_param_widgets_by_builder(self.block_inst,
            container,
            category=category,
            filter_func=None,
            keep_filter_order=False,
            is_last_spacer=False)
        )
