import sys
import os

import util.gen_util

from design.db_patch import DesignVersion, DesignPatch
from design.db_item import GenericParamService, GenericClockPin
from tx375_device.lane10g.quad_param_info import Lane10GParamInfo as Lane10GParamInfo
from tx375_device.lane10g.design_param_info import Lane10GDesignParamInfo as DesignParamInfo
from tx375_device.lane10g.design import Lane10G, Lane10GRegistry
from tx375_device.common_quad.design import QuadLaneCommon
from tx375_device.common_quad.quad_prop_id import CommonQuadConfigParamInfo as CommonQuadParamInfo

sys.path.append(os.path.join(os.path.dirname(__file__), '..'))



@util.gen_util.freeze_it
class Lane10GPatch(DesignPatch):
    """
    Lane10G design backward compatibility patcher
    """

    def __init__(self, lane_10g_reg: Lane10GRegistry, device_db, unlinked_none_cmn_inst: QuadLaneCommon| None):
        assert lane_10g_reg.common_quad_reg is not None
        self.common_reg = lane_10g_reg.common_quad_reg
        self.unlinked_none_cmn_inst = unlinked_none_cmn_inst
        super().__init__(lane_10g_reg, device_db)
        self.block_reg: Lane10GRegistry

        #
        # Patches sequence can imply dependency between patches. When the patch is applied,
        # it will be applied from the lowest version number to the highest.
        # This means it is safe for patch function to assume certain db states.
        #

        # PT-2220
        msg = "Ethernet XGMII: Move common parameters and pins to common quad lane instances"
        self.add_patch(DesignVersion.PATCH_VERSION_20241004,
                       Lane10GPatch.patch_cmn_params_pins, msg)

        # PT-2250
        msg = "Ethernet XGMII: Rename some pin types"
        self.add_patch(DesignVersion.PATCH_VERSION_20241005,
                       Lane10GPatch.patch_rename_pin_types, msg)

        # PT-2321
        msg = "Ethernet XGMII: Rename pin types with LN_ prefix"
        self.add_patch(DesignVersion.PATCH_VERSION_20241007,
                       Lane10GPatch.patch_rename_pin_types_with_ln, msg)

        # PT-2325
        msg = "Ethernet XGMII: Allow 10G instance without resource to have own common quad lane instance"
        self.add_patch(DesignVersion.PATCH_VERSION_20241008,
                       Lane10GPatch.patch_cmn_inst_without_resource, msg)

        # PT-2453
        msg = "Ethernet XGMII: Remove common instance parameter"
        self.add_patch(DesignVersion.PATCH_VERSION_20242002,
                       Lane10GPatch.patch_remove_cmn_inst_raw_serdes_related_param, msg)

    def patch_cmn_params_pins(self, lane_inst: Lane10G):
        """
        Move common parameters and pins to common quad lane instances

        :param lane_inst: Lane10G instance
        :return: True, if design is patched, else False
        """

        is_patch = False

        cmn_inst = self.common_reg.get_inst_by_lane_name(lane_inst.quad_type, lane_inst.name)
        assert isinstance(cmn_inst, QuadLaneCommon)

        inst_list = sorted(self.block_reg.get_insts_by_quad_res(cmn_inst.get_device()),
                           key=lambda inst: inst.name)
        if len(inst_list) == 0:
            return is_patch

        is_first_inst = (inst_list[0] == lane_inst)

        # Param update
        cmn_parm_service = GenericParamService(cmn_inst.get_param_group(), cmn_inst.get_param_info())
        lane_param_group = lane_inst.param_group

        for param_id in CommonQuadParamInfo.Id:
            if lane_param_group.get_param_by_name(param_id.value) is None:
                continue

            # Save only for first instance
            if is_first_inst:
                val = lane_param_group.get_param_value(param_id.value)
                cmn_parm_service.set_param_value(param_id, val)

            lane_inst.param_group.remove_param(param_id.value)
            is_patch = True

        # Reference clock is not used, only support external clock
        lane_inst.param_group.remove_param("REF_CLK_NAME")

        # Pins update
        cmn_gen_pin = cmn_inst.gen_pin
        lane_gen_pin = lane_inst.gen_pin

        for cmn_pin in cmn_gen_pin.get_all_pin():
            lane_pin = lane_gen_pin.get_pin_by_type_name(cmn_pin.type_name)

            if lane_pin is None:
                continue

            # Save only for first instance
            if is_first_inst:
                cmn_pin.name = lane_pin.name
                if isinstance(lane_pin, GenericClockPin) and isinstance(cmn_pin, GenericClockPin):
                    cmn_pin.is_inverted = lane_pin.is_inverted

            lane_gen_pin.delete_pin_by_type(cmn_pin.type_name)
            is_patch = True

        if is_patch:
            func_name = util.gen_util.get_function_name()
            self.logger.warning(
                "{}: Patch {} applied to {}".format(func_name, DesignVersion.PATCH_VERSION_20241004,
                                                    lane_inst.name))

        return is_patch

    def patch_rename_pin_types(self, lane_inst: Lane10G):
        is_patch = False

        cmn_inst = self.common_reg.get_inst_by_lane_name(lane_inst.quad_type, lane_inst.name)
        assert isinstance(cmn_inst, QuadLaneCommon)

        renamed_port_map = {
            "PHY_INTERRUPT_P": "PHY_INTERRUPT",
            "PMA_XCVR_PLLCLK_EN_P": "PMA_XCVR_PLLCLK_EN",
            "PMA_XCVR_PLLCLK_EN_ACK_P": "PMA_XCVR_PLLCLK_EN_ACK",
            "PMA_XCVR_POWER_STATE_REQ_P": "PMA_XCVR_POWER_STATE_REQ",
            "PMA_XCVR_POWER_STATE_ACK_P": "PMA_XCVR_POWER_STATE_ACK",
            "ETH_EEE_ALERT_EN_LN": "ETH_EEE_ALERT_EN",
            "PMA_RX_SIGNAL_DETECT_LN": "PMA_RX_SIGNAL_DETECT",
            "PMA_TX_ELEC_IDLE_LN": "PMA_TX_ELEC_IDLE",
            "KR_FRAME_LOCK_LN": "KR_FRAME_LOCK",
            "KR_LOCAL_RX_TRAINED_LN": "KR_LOCAL_RX_TRAINED",
            "KR_RESTART_TRAINING_LN": "KR_RESTART_TRAINING",
            "KR_SIGNAL_DETECT_LN": "KR_SIGNAL_DETECT",
            "KR_TRAINING_ENABLE_LN": "KR_TRAINING_ENABLE",
            "KR_TRAINING_FAILURE_LN": "KR_TRAINING_FAILURE",
            "KR_TRAINING_LN": "KR_TRAINING",
            "PHY_P0_RESET_N": "PHY_RESET_N",
        }
        cmn_gen_pin = cmn_inst.gen_pin
        lane_gen_pin = lane_inst.gen_pin

        for old_type, new_type in renamed_port_map.items():
            gen_pin = lane_gen_pin
            pin = gen_pin.get_pin_by_type_name(old_type)

            if pin is None:
                gen_pin = cmn_gen_pin
                pin = gen_pin.get_pin_by_type_name(old_type)

            if pin is None:
                continue

            new_pin = gen_pin.get_pin_by_type_name(new_type)
            if new_pin is None:
                continue

            new_pin.name = pin.name
            gen_pin.delete_pin_by_type(old_type)

            is_patch = True

        if is_patch:
            func_name = util.gen_util.get_function_name()
            self.logger.warning(
                "{}: Patch {} applied to {}".format(func_name, DesignVersion.PATCH_VERSION_20241005,
                                                    lane_inst.name))

        return is_patch

    def patch_rename_pin_types_with_ln(self, lane_inst: Lane10G):
        is_patch = False

        gen_pin = lane_inst.gen_pin

        for old_type in gen_pin.get_all_pin_type_name():
            pin = gen_pin.get_pin_by_type_name(old_type)

            if pin is None or not old_type.startswith("LN_"):
                continue

            new_type = old_type.removeprefix("LN_")
            new_pin = gen_pin.get_pin_by_type_name(new_type)

            if new_pin is None:
                continue

            new_pin.name = pin.name
            gen_pin.delete_pin_by_type(old_type)

            is_patch = True

        if is_patch:
            func_name = util.gen_util.get_function_name()
            self.logger.warning(
                "{}: Patch {} applied to {}".format(func_name, DesignVersion.PATCH_VERSION_20241007,
                                                    lane_inst.name))

        return is_patch

    def patch_cmn_inst_without_resource(self, lane_inst: Lane10G):
        is_patch = False

        if lane_inst.get_device() != "" or self.unlinked_none_cmn_inst is None:
            return is_patch

        # Get the common instance with previous settings
        previous_cmn_inst = self.unlinked_none_cmn_inst

        if previous_cmn_inst is None:
            return is_patch

        target_cmn_inst = self.common_reg.get_inst_by_lane_name(lane_inst.quad_type, lane_inst.name)

        # All 10G instance should have a common instance after check_common_quad_lane_reg
        if target_cmn_inst is not None and target_cmn_inst != previous_cmn_inst:
            # Restore previous settings
            target_cmn_inst.update_settings_by_others(previous_cmn_inst)
            is_patch = True

        if is_patch:
            func_name = util.gen_util.get_function_name()
            self.logger.warning(
                "{}: Patch {} applied to {}".format(func_name, DesignVersion.PATCH_VERSION_20241008,
                                                    lane_inst.name))

        return is_patch

    def patch_remove_cmn_inst_raw_serdes_related_param(self, lane_inst: Lane10G):
        """
        Remove some params in common instance due to Raw PLL settings

        Updated params:
        - ss_raw_data_rate, ss_raw_mode: common instance -> Raw Serdes
        - REF_CLK_FREQUENCY: raw serdes -> common instance

        :param lane_inst: Lane1G instance
        :return: True, if design is patched, else False
        """
        is_patch = False

        cmn_inst = self.block_reg.get_cmn_inst_by_lane_name(lane_inst.name)
        assert cmn_inst is not None

        cmn_param_group = cmn_inst.param_group

        cmn_param_list = [
            "ss_raw_data_rate",
            "ss_raw_mode"
        ]
        for cmn_param_name in cmn_param_list:
            cmn_param_group.remove_param(cmn_param_name)
            is_patch = True

        if is_patch:
            func_name = util.gen_util.get_function_name()
            self.logger.warning(
                "{}: Patch {} applied to {}".format(func_name, DesignVersion.PATCH_VERSION_20242002,
                                                    lane_inst.name))

        return is_patch
