from __future__ import annotations
from decimal import Decimal
from pathlib import Path
from typing import List, TextIO, TYPE_CHECKING, Tuple, final

from prettytable import PrettyTable

from common_device.device_service_interface import BlockService
from common_device.writer import Writer
from common_device.pll.pll_design_param_info import PLLDesignParamInfo
from design.db_item import PeriDesignItem

from device.db import PeripheryDevice
import device.db_interface as devdb_int
import design.db as des_db
from device.excp import ConfigurationInvalidException
from tx375_device.fpll.calculator import PLLFreqCalculatorFractional
from tx375_device.fpll.device_service import EfxFpllV1DeviceService

import tx60_device.pll.writer.summary as pll_sum
from tx375_device.fpll.design import EfxFpllDynamicReconfigRecord, EfxFpllV1, EfxFpllV1OutputClock, EfxFpllV1OutputClockFrac
from util.excp import PTInteralAssertException, pt_assert
from util.gen_util import get_copyright_string, get_current_date_str, override
from util.signal_util import float_to_decimal, format_dec, format_frequency

from _version import __version__

if TYPE_CHECKING:
    from design.db import PeriDesign
    from tx375_device.fpll.writer.memory_initialization import MemoryInitalizationFileWriter


class EfxFpllV1Summary(pll_sum.PLLSummaryComplex):
    '''
    Class for Summary report generation.
    '''

    def _get_out_clk_names(self, pll_obj, clk_obj):
        clkout_names = []

        if clk_obj is not None and pll_obj is not None:
            if clk_obj.conn_type == "rclk":
                clkout_names.append(clk_obj.name)

                if clk_obj.clkmux_buf_name != "":
                    # Check that this is a valid connection although
                    # rule already check for it.
                    dbi = devdb_int.DeviceDBService(self.get_device())
                    plldev_service = dbi.get_block_service(self.get_device().get_pll_type())

                    clkout_pin_name = "CLKOUT{}".format(clk_obj.number)

                    if plldev_service is not None and \
                            plldev_service.is_pll_clkout_pin_support_mult_clkout_conn(pll_obj.pll_def, clkout_pin_name):
                        clkout_names.append(clk_obj.clkmux_buf_name)

            else:
                clkout_names.append(clk_obj.name)

        return clkout_names

    def _get_clkout_constraints(self, design, sdc_cmds, pll_obj, clk_obj, clkout_period):

        clkout_names = self._get_out_clk_names(pll_obj, clk_obj)

        if clk_obj is not None and pll_obj is not None:
            for clk_name in clkout_names:

                if not self.is_pll_clk_driving_ddr(design, pll_obj, clk_obj):
                    sdc_clkout_period = self.round_clk_period(
                        clkout_period * 1000, clk_obj)

                    period_in_ns = clkout_period * 1000
                    clkout_waveform = pll_obj.get_output_clock_sdc_waveform(
                        clk_obj, period_in_ns)

                    if clkout_waveform != "":
                        sdc = "create_clock -waveform {" + \
                            clkout_waveform + "} -period " + \
                            str(sdc_clkout_period) + \
                            " -name " + clk_name + " [get_ports {" + clk_name + "}]"
                    else:
                        sdc = "create_clock -period " + \
                            str(sdc_clkout_period) + \
                            " -name " + clk_name + " [get_ports {" + clk_name + "}]"

                    if sdc != "":
                        sdc_cmds.append(sdc)

    def is_pll_clk_driving_ddr(self, design, pll_obj, clk_obj):
        is_ddr_clk = False

        if design is not None and design.ddr_reg is not None:
            if design.ddr_reg.get_inst_count() > 0:

                dbi = devdb_int.DeviceDBService(design.device_db)
                ddr_service = dbi.get_block_service(devdb_int.DeviceDBService.BlockType.DDR_ADV)

                if pll_obj.is_pll_ddr_clock(design.pll_reg, ddr_service, design.ddr_reg, clk_obj):
                    is_ddr_clk = True

        return is_ddr_clk

    @override
    def _write_clkout_frequencies(self, outfile: TextIO,
                                  pll_obj: EfxFpllV1,
                                  freq_str: str,
                                  vco_freq: float,
                                  pll_freq: float,
                                  freq_calc: PLLFreqCalculatorFractional,
                                  sorted_out_clk_list: List[EfxFpllV1OutputClock]):
        # Write out the frequency calculation
        # VCO equation will be affected depending on the
        # feedback type.
        outfile.write("\nFrequency calculations:\n")
        disp_vco_freq = float_to_decimal(vco_freq, pll_obj.PLL_CLOCK_PRESCISION)
        disp_pll_freq = float_to_decimal(pll_freq, pll_obj.PLL_CLOCK_PRESCISION)
        disp_fb_mult = float_to_decimal(freq_calc.fb_mult, pll_obj.PLL_CLOCK_PRESCISION)
        disp_frac_div = float_to_decimal(freq_calc.fb_mult / freq_calc.post_divider, 3)
        outfile.write("\tVCO = REFCLK * ((M * FBK) /N)\n")
        outfile.write("\t    = {} * (({}*{}) /{})\n".format(
            freq_str,
            str(pll_obj.multiplier),
            str(disp_fb_mult),
            str(pll_obj.pre_divider)))
        outfile.write("\t    = {} MHz\n".format(
            disp_vco_freq))

        outfile.write("\tPLL = VCO / O\n")
        outfile.write("\t    = {} MHz / {}\n".format(
            disp_vco_freq,
            str(pll_obj.post_divider)))
        outfile.write("\t    = {} MHz\n\n".format(
            disp_pll_freq))

        is_frac_mode = pll_obj.param_group.get_param_value(PLLDesignParamInfo.Id.enable_fractional.value)

        for clk_obj in sorted_out_clk_list:
            if isinstance(clk_obj, EfxFpllV1OutputClockFrac) and is_frac_mode:
                outfile.write(
                '\tCLKOUT{} = PLL / (FBK / O)\n'.format(
                    str(clk_obj.number)))
                outfile.write(
                    '\t        = {} MHz / {}\n'.format(
                        disp_pll_freq,
                        str(disp_frac_div)))
                disp_frac_outclk = float_to_decimal(disp_pll_freq / disp_frac_div, clk_obj.PLL_CLOCK_PRESCISION)
                outfile.write(
                    '\t        = {} MHz\n'.format(
                        str(disp_frac_outclk)))
            else:
                outfile.write(
                    '\tCLKOUT{} = PLL / CLKOUT{}_DIV\n'.format(
                        str(clk_obj.number),
                        str(clk_obj.number)))

                outfile.write(
                    '\t        = {} MHz / {}\n'.format(
                        disp_pll_freq,
                        str(clk_obj.out_divider)))

                outfile.write(
                    '\t        = {} MHz\n'.format(
                        clk_obj.get_output_freq_str()))

    @override
    def _write_feedback(self, design: PeriDesign,
                        blk_service: EfxFpllV1DeviceService,
                        outfile: TextIO,
                        pll_obj: EfxFpllV1):
        outfile.write(
            '{:30s}: {}\n'.format(
                "Feedback Mode",
                pll_obj.fbmode2str_map[pll_obj.fb_mode]))

        # Prints out the resource if External
        if pll_obj.fb_mode == pll_obj.FeedbackModeType.external:
            fb_resource = pll_obj.get_ext_fb_res_name(design)
            if fb_resource is not None:
                outfile.write(
                    '{:30s}: {}\n'.format(
                        "Feedback Resource", fb_resource))

        if pll_obj.fb_clock_name != "":
            outfile.write(
                '{:30s}: {}\n'.format(
                    "Feedback Clock", pll_obj.fb_clock_name))

    @override
    def _write_pll_parameters(self, outfile: TextIO, pll_obj: EfxFpllV1) -> Tuple[str, str, PLLFreqCalculatorFractional]:
        # Print out the filled in data used
        # for calculating the output frequency
        # Changed to using similar method like UI, which is
        # using the freq_calc
        freq_calc = pll_obj.freq_calc

        vco_freq = freq_calc.get_freq_str(freq_calc.vco_freq)
        pll_freq = freq_calc.get_freq_str(freq_calc.pll_freq)

        outfile.write(
            '{:30s}: {}\n'.format(
                "Multiplier (M)", pll_obj.multiplier))

        disp_fb_mult = float_to_decimal(freq_calc.fb_mult, pll_obj.PLL_CLOCK_PRESCISION)
        outfile.write(
            '{:30s}: {}\n'.format(
                "Feedback Multiplier (FBK)", disp_fb_mult
            )
        )

        outfile.write(
            '{:30s}: {}\n'.format(
                "Pre-Divider (N)", pll_obj.pre_divider))

        outfile.write(
            '{:30s}: {}\n'.format(
                "VCO Frequency", vco_freq + " MHz"))

        outfile.write(
            '{:30s}: {}\n'.format(
                "Post-Divider (O)", pll_obj.post_divider))

        outfile.write(
            '{:30s}: {}\n'.format(
                "PLL Frequency", pll_freq + " MHz"))

        return vco_freq, pll_freq, freq_calc

    @override
    def _write_clock_output(self, design: PeriDesign,
                            outfile: TextIO,
                            pll_obj: EfxFpllV1,
                            out_clock_list: List[EfxFpllV1OutputClock]) -> Tuple[List[EfxFpllV1OutputClock], List[str]]:
        """
        Return a sorted list of output clocks and a list of SDC command
        """
        sdc_cmds = []

        # Function for sorting
        def get_number(out_clock):
            return out_clock.number

        sorted_out_clk_list = sorted(
            out_clock_list, key=get_number)

        freq_calc = pll_obj.freq_calc
        pll_freq = freq_calc.calc_pll_freq()
        clk2clkout_periods = self.get_all_clkout_periods(pll_freq, sorted_out_clk_list)

        for clk_obj in sorted_out_clk_list:

            clkout_freq = clk_obj.out_clock_freq

            # Check for denominator = 0
            if clkout_freq is not None and clkout_freq != 0:
                # PT-2358 keeping the integral multiple relationship of frequencies
                clkout_period = clk2clkout_periods.get(clk_obj)
                assert clkout_period is not None
            else:
                clkout_period = Decimal(str(0))

            if clkout_period < 1 or clkout_period == 0:
                # Convert to PS for printing
                tmp_period = self.round_clk_period(
                    clkout_period * 1000, clk_obj)
                period_str = tmp_period.__str__() + " ns"
            else:
                tmp_period = self.round_clk_period(
                    clkout_period, clk_obj)
                period_str = tmp_period.__str__() + " us"

            outfile.write(
                '\n{} {}\n'.format(
                    "Output Clock", str(clk_obj.number)))

            outfile.write(
                '{:30s}: {}\n'.format(
                    "Clock Pin Name", clk_obj.name))

            outfile.write(
                '{:30s}: {}\n'.format(
                    "Output Divider", clk_obj.out_divider))

            self._write_clkout_phase_shift(outfile, pll_obj, clk_obj)

            outfile.write(
                '{:30s}: {}\n'.format(
                    "Output Frequency",
                    clk_obj.get_output_freq_str() + " MHz"))

            outfile.write(
                '{:30s}: {}\n'.format(
                    "Output Period",
                    period_str))


            duty_cycle = 50
            is_enable_pdc = True

            if isinstance(clk_obj, EfxFpllV1OutputClockFrac):
                dc_param = clk_obj.duty_cycle_control_param
                if dc_param:
                    duty_cycle = dc_param.calculate_duty_cycle()

                is_enable_pdc = clk_obj.enable_pdc

            duty_cycle = 50 if not is_enable_pdc else duty_cycle
            disp_duty_cycle = float_to_decimal(duty_cycle, pll_obj.PLL_CLOCK_PRESCISION)
            outfile.write(
                '{:30s}: {} %\n'.format(
                    "Duty Cycle", disp_duty_cycle
                ))

            # clkout_period = float(
            #    "{0:.2f}".format(clkout_period))

            # Skip if this clock is not configured
            # (name is empty)
            if clk_obj.name != "":
                # SDC timing unit needs to be in ns
                self._get_clkout_constraints(design, sdc_cmds,
                    pll_obj, clk_obj, clkout_period)

        return sorted_out_clk_list, sdc_cmds

    @override
    def generate_inst_detail(self, design: PeriDesign, outfile: TextIO, blk_service: BlockService, inst: PeriDesignItem):
        """
        Generate the instance detail configurations to the outfile stream

        :param design: _description_
        :type design: des_db.PeriDesign
        :param outfile: _description_
        :type outfile: TextIO
        :param blk_service: _description_
        :type blk_service: BlockService
        :param inst: _description_
        :type inst: PeriDesignItem
        :return: _description_
        :rtype: _type_
        """
        pt_assert(isinstance(inst, EfxFpllV1), f"Internal error, unexpected type: {type(inst)}", PTInteralAssertException)
        def write_record(outfile: TextIO, field_name: str, value: str):
            outfile.write('{:30s}: {}\n'.format(field_name, value))

        assert isinstance(inst, EfxFpllV1)
        outfile.write("\n")
        write_record(outfile, 'Instance Name', inst.name)
        write_record(outfile, 'Resource', inst.pll_def)
        if inst.reset_name:
            write_record(outfile, "Reset Pin Name", inst.reset_name)

        if inst.locked_name != "":
            write_record(outfile, "Locked Pin Name", inst.locked_name)

        gen_pin = inst.gen_pin
        deskewed_name = gen_pin.get_pin_name_by_type("DESKEWED")
        if deskewed_name != "":
            write_record(outfile, inst.get_pin_property_name("DESKEWED"), deskewed_name)

        if inst.is_dynamic_phase_shift_used():
            for type_name in ["SHIFT_ENA", "SHIFT_SEL", "SHIFT"]:
                pin_name = gen_pin.get_pin_name_by_type(type_name)
                if pin_name != "":
                    write_record(outfile, inst.get_pin_property_name(type_name), gen_pin.get_pin_name_by_type(type_name))

        assert inst.ref_clock_mode is not None
        write_record(outfile, "Clock Source", inst.rcmode2str_map[inst.ref_clock_mode])

        # Write the resource for external or dynamic only
        self._write_ref_clock_resource(outfile, design, inst)

        # Common for all ref clock type
        ref_clocks = self._get_ref_clocks(design, inst)
        write_record(outfile, "Reference Clock", ref_clocks)

        self._write_feedback(design, blk_service, outfile, inst)
        outfile.write("\n")

        mode_str = "Fractional-N" if inst.fractional_mode else "Integer-N"
        write_record(outfile, "Mode", mode_str)

        freq_str = inst.get_refclock_freq_str() + " MHz"
        write_record(outfile, "Reference Clock Frequency", freq_str)

        self._write_ref_clk_info(outfile, inst)
        vco_freq, pll_freq, freq_calc = self._write_pll_parameters(outfile, inst)

        out_clock_list = inst.get_output_clock_list()
        if out_clock_list:
            sorted_out_clk_list, sdc_cmds = self._write_clock_output(design, outfile, inst, out_clock_list)
            self._write_clkout_frequencies(outfile, inst, freq_str, vco_freq, pll_freq, freq_calc, sorted_out_clk_list)

            outfile.write("\nSDC Constraints:\n")

            for cmd in sdc_cmds:
                outfile.write("\t{}\n".format(cmd))


@final
class EfxFpllV1DynamicReconfigurationSummaryWriter(Writer):
    """
    Responsible for write a report file for dyn reconfig settings
    """
    def __init__(self, device_db: PeripheryDevice, name: str, mem_writer: MemoryInitalizationFileWriter):
        self.mem_writer = mem_writer
        super().__init__(device_db, name)

    def format_record_summary(self, inst: EfxFpllV1, record: EfxFpllDynamicReconfigRecord) -> List[str]:
        results = []

        def add_field_value(field):

            value = record.get(field)
            display_name = EfxFpllDynamicReconfigRecord.get_display_name(field)
            results.append(f'{display_name:50}: {value}\n')

        add_field_value('REFCLK_SOURCE_SEL')
        add_field_value('N')
        add_field_value('M')
        add_field_value('O')
        add_field_value('FRACTIONAL_K')
        add_field_value('CLKOUT1_HALF_DC_SHIFT_EN')
        results.append("\n")
        for i in range(5):
            add_field_value(f'CLKOUT{i}_DIV')
            add_field_value(f'CLKOUT{i}_PHASE_SETTING')
            add_field_value(f'CLKOUT{i}_INVERT_EN')
            if i == 1:
                add_field_value(f'CLKOUT{i}_SWALLOWING_DIV')
            results.append("\n")

        results.append(self.get_hex_field_value(inst, record))
        results.append(self.get_hex_note())
        results.append("\n")

        return results

    def get_hex_note(self):
        return '(to be pasted in ip_configuration)\n'

    def get_hex_field_value(self, inst: EfxFpllV1, record: EfxFpllDynamicReconfigRecord):
        # Display Name
        display_name = "Configuration Setting"

        # Hex value
        hex_value = self.mem_writer.get_hex_value_per_reconfig_record(inst, record)

        return f'{display_name:50}: {hex_value}\n'

    def write(self, out_path: Path, inst: EfxFpllV1, design: PeriDesign):
        if len(inst.dyn_reconfig_params) == 0:
            return

        with open(out_path, encoding='utf-8', mode='w') as fp:
            # Header
            fp.write("\n")
            fp.write("Version: {}\n".format(__version__))
            fp.write("Date: {}\n".format(get_current_date_str()))
            fp.write("\n{}\n".format(get_copyright_string()))
            fp.write("\n\n")

            fp.write(f"Summary for all PLL: '{inst.name}' Dynamic Reconfigurations:\n")

            header = ["Address", "Ref. Freq", "CLKOUT0 Freq (Phase Shift)", "CLKOUT1 Freq (Phase Shift)", "CLKOUT2 Freq (Phase Shift)", "CLKOUT3 Freq (Phase Shift)", "CLKOUT4 Freq (Phase Shift)"]
            table = PrettyTable(header)

            clk_enable_status = [True if inst.get_output_clock_by_number(i) is not None else False for i in range(5)]
            no_summary = False
            for idx, record in enumerate(inst.dyn_reconfig_params):
                try:
                    summary_row = [idx]
                    ref_freq = record.requested.get('REFCLK_FREQ')
                    if ref_freq:
                        freq_str = format_frequency(ref_freq, inst.PLL_CLOCK_PRESCISION)
                        summary_row.append(freq_str)
                    else:
                        raise

                    act_values = record.calculate_act_freq()
                    for idx, enable in enumerate(clk_enable_status):
                        freq = act_values.get(f'ACT_CLKOUT{idx}_FREQ')
                        if enable and freq is not None:
                            freq_str = format_frequency(freq, inst.PLL_CLOCK_PRESCISION)
                            value = freq_str
                            ps_degree = act_values.get(f'ACT_CLKOUT{idx}_PHASE_SHIFT')
                            if ps_degree is not None and ps_degree > 0:
                                ps_str = format_dec(ps_degree, inst.PLL_CLOCK_PRESCISION) + "Deg"
                                value += f" ( {ps_str} )"
                            summary_row.append(value)
                        else:
                            summary_row.append("Disabled")

                    table.add_row(summary_row)
                except:
                    import traceback
                    traceback.print_exc()
                    no_summary = True
                    # Somewhere in the configuration resulting fail to calculate
                    break

            if not no_summary:
                fp.write(str(table))
                fp.write("\n\n\n")

            fp.write("Detailed register values for each configurations:\n\n"    )

            for idx, record in enumerate(inst.dyn_reconfig_params):
                fp.write(f"****************** Address {idx}: {record.name:25} ******************\n\n")
                lines = self.format_record_summary(inst, record)
                fp.writelines(lines)
                fp.write("\n")
