from __future__ import annotations
from decimal import Decimal
from typing import TYPE_CHECKING, Dict, Tuple, List

from device.db_interface import DeviceDBService
from util.gen_util import override
from util.excp import pt_assert
from device.excp import InvalidPCRException

from tx180_device.pll.writer.logical_periphery import PLLLogicalPeripheryV3Complex

from tx375_device.fpll.design import EfxFpllV1, EfxFpllV1OutputClockFrac
from tx375_device.fpll.device_service import DutyCycleControlParam, EfxFpllV1DeviceService

from tx375_device.quad_pcie.quad_pcie_prop_id import QuadPCIEConfigParamInfo as PCIeHWParamInfo
from tx375_device.quad_pcie.design import QuadPCIE

from common_device.quad.res_service import QuadResService
from common_device.writer import build_lookup_based_pcr_converter, build_offset_based_pcr_converter

if TYPE_CHECKING:
    from design.db import PeriDesign
    from device.pcr_device import BlockPCR, PCRMap


# TypeAlias, who want to type such a long name
PCRDefnType = EfxFpllV1DeviceService.PCRDefnType
PCRModeType = EfxFpllV1DeviceService.PCRModeType

multipler_to_pcr = build_lookup_based_pcr_converter({
    1: "1",
    2: "2",
    4: "4"
})

pre_div_to_pcr = build_lookup_based_pcr_converter({
    1: "1",
    2: "2",
    4: "4"
})

post_div_to_pcr = build_lookup_based_pcr_converter({
    1: '1',
    2: '2',
    4: '4',
    8: '8',
    16: '16',
    32: '32',
    64: '64',
    128: '128',
})

reset_pin_to_pcr = build_lookup_based_pcr_converter({
    "": "DISABLE"
}, default_val="ENABLE")


def fb_core_index(pll_obj: EfxFpllV1, dev_service: EfxFpllV1DeviceService, design: PeriDesign) -> int:
    assert pll_obj.fb_mode == pll_obj.FeedbackModeType.core
    core_index = 0
    core_ins2_index = dev_service.determine_core_fbk_pin_mapping(
        design, [], [pll_obj])

    if pll_obj in core_ins2_index:
        core_index = core_ins2_index[pll_obj]

    return core_index


def dev_fb_sel_values(pll_obj: EfxFpllV1, dev_service: EfxFpllV1DeviceService, design: PeriDesign) -> Tuple[str, str | None]:
    """
    Find the dev specific selector value for the two fb_sel mux (fb_sel / fb2_sel)
    """
    match pll_obj.fb_mode:
        case pll_obj.FeedbackModeType.local:
            if pll_obj.get_feedback_clock_no() == 1:
                return (PCRModeType.core1.value, PCRModeType.local.value)
            else:
                return (PCRModeType.local.value, None)
        case pll_obj.FeedbackModeType.external:
            return (PCRModeType.io.value, None)
        case pll_obj.FeedbackModeType.core:
            core_index = fb_core_index(pll_obj, dev_service, design)
            if core_index == 1:
                return (PCRModeType.core1.value, PCRModeType.core.value)
            else:
                return (PCRModeType.core0.value, None)
    raise NotImplementedError(f'Unhandled fb_mode = {pll_obj.fb_mode}')


ipfrz_to_pcr = build_lookup_based_pcr_converter({
    True: PCRModeType.pcr_ready.value,
    False: PCRModeType.unfrz.value
})


def dev_ref_clock_index(pll_obj: EfxFpllV1, dev_service: EfxFpllV1DeviceService) -> int:
    assert pll_obj.ref_clock_mode != pll_obj.RefClockModeType.dynamic
    if pll_obj.ref_clock_mode == pll_obj.RefClockModeType.core:
        return 3
    else:
        pt_assert(pll_obj.ext_ref_clock_id in {
            pll_obj.RefClockIDType.ext_clock_0,
            pll_obj.RefClockIDType.ext_clock_1,
            pll_obj.RefClockIDType.ext_clock_2
        }, "Invalid Reference Clock selection for external mode", InvalidPCRException)
        id_int = pll_obj.get_external_clkin_index(
                            dev_service, pll_obj.ext_ref_clock_id)
        pt_assert(id_int is not None, "Invalid Reference Clock index for external mode", InvalidPCRException)
        return id_int


dev_ref_clock_index_to_pcr = build_lookup_based_pcr_converter({
    0: "0",
    1: "1",
    2: "2",
    3: "3",
})

fractional_k_to_pcr = build_offset_based_pcr_converter(0)


def calculate_ssc_mod_period(ref_clk_mhz: Decimal, multiplier: int, ssc_khz: Decimal, pre_div: int) -> int:
    fref = Decimal(ref_clk_mhz) * 10 ** 6  # base was MHz
    num = fref * Decimal(multiplier)
    ssc_freq = Decimal(ssc_khz) * 10 ** 3  # base was Khz
    den = ssc_freq * 2 * Decimal(pre_div)
    mod_period = round(num / den)
    return mod_period


ssc_mod_period_to_pcr = build_offset_based_pcr_converter(-1)


def calculate_ssc_mod_step(ssc_amp: Decimal, out_divider: int, fractional_k: int, ssc_mod_period: int) -> int:
    mod_depth = ssc_amp / 100
    num = mod_depth * (out_divider + Decimal(fractional_k) / (2 ** 24))
    den = ssc_mod_period
    mod_step = round((num / den) * (2 ** 24))
    return mod_step


ssc_mod_step_to_pcr = build_offset_based_pcr_converter(0)


def ssc_mod_type_to_spread_pcr(mod_type: str) -> Tuple[str, str | None]:
    match mod_type:
        case "CENTER":
            return (PCRModeType.center.value, None)
        case "UP":
            return (PCRModeType.noncenter.value, PCRModeType.up.value)
        case "DOWN":
            return (PCRModeType.noncenter.value, PCRModeType.down.value)
    raise NotImplementedError(f'Unhandled mod_type = {mod_type}')


swallowing_div_to_pcr = build_offset_based_pcr_converter(-1)

dc_param_to_pcr_dcodd = build_lookup_based_pcr_converter({
    True: PCRModeType.enable.value,
    False: PCRModeType.disable.value
})

bool_to_enable_pcr = build_lookup_based_pcr_converter({
    True: PCRModeType.enable.value,
    False: PCRModeType.disable.value
})

clk_cascade_to_pcr = bool_to_enable_pcr

clk_invert_en_to_pcr = build_lookup_based_pcr_converter({
    True: PCRModeType.invert.value,
    False: PCRModeType.no_invert.value
})

clk_dyn_phase_en_to_pcr = build_lookup_based_pcr_converter({
    True: PCRModeType.core.value,
    False: PCRModeType.pcr.value
})

clk_phase_setting_to_pcr = build_lookup_based_pcr_converter({
    0: "00",
    1: "05",
    2: "10",
    3: "15",
    4: "20",
    5: "25",
    6: "30",
    7: "35",
})

clk_div_to_pcr = build_offset_based_pcr_converter(-1)

dyn_cfg_en_to_pcr = bool_to_enable_pcr


class EfxFpllV1LogicalPeriphery(PLLLogicalPeripheryV3Complex):
    def __init__(self, device_db, name):
        super().__init__(device_db, name)
        self.pcie_used_pll_list: List[EfxFpllV1] = []

    @override
    def _get_output_clock_related_value(self, pcr_name: str,
                                        pcr_obj: PCRMap,
                                        design: PeriDesign,
                                        pll_clock_info: Tuple,
                                        pll_obj: EfxFpllV1) -> str:

        if EfxFpllV1DeviceService.is_pcr_defn_match(pcr_name, EfxFpllV1DeviceService.PCRDefnType.pcr_div_sel1):
            clk = pll_obj.get_output_clock_by_number(1)
            if clk is not None:
                if clk.enable_pdc or pll_obj.fractional_mode:
                    dc_param = clk.duty_cycle_control_param
                    if dc_param is not None:
                        value = str(dc_param.pdiv - 1)
                        return value

        return super()._get_output_clock_related_value(pcr_name, pcr_obj, design, pll_clock_info, pll_obj)

    @override
    def generate_user_instance_parameters(self, pcr_block: BlockPCR, pll_obj: EfxFpllV1, design: PeriDesign) -> Dict[str, str]:
        """
        Generate a dictionary of PCR name string to PCR value string for a design instance
        """
        dbi = DeviceDBService(design.device_db)
        pll_dev_svc = dbi.get_block_service(design.device_db.get_pll_type())
        assert isinstance(pll_dev_svc, EfxFpllV1DeviceService)
        if design.quad_pcie_reg is not None:
            self.pcie_used_pll_list = design.quad_pcie_reg.get_all_external_refclk_related_pll(design)

        generated_values: Dict[str, str] = {}
        pcr_map = pcr_block.get_pcr_list()

        for pcr_name, pcr_obj in pcr_map.items():
            generated_values[pcr_name] = pcr_obj.get_default()

        generated_values[PCRDefnType.pcr_pll_bypass.value] = PCRModeType.disable.value
        generated_values[PCRDefnType.pcr_fb_div.value] = multipler_to_pcr(pll_obj.multiplier)
        generated_values[PCRDefnType.pcr_ref_div.value] = pre_div_to_pcr(pll_obj.pre_divider)
        generated_values[PCRDefnType.pcr_post_div.value] = post_div_to_pcr(pll_obj.post_divider)
        generated_values[PCRDefnType.pcr_user_en.value] = reset_pin_to_pcr(pll_obj.reset_name)

        fb_sel_pcr, fb2_sel_pcr = dev_fb_sel_values(pll_obj, pll_dev_svc, design)
        generated_values[PCRDefnType.pcr_fb_sel.value] = fb_sel_pcr
        if fb2_sel_pcr:
            generated_values[PCRDefnType.pcr_fb2_sel.value] = fb2_sel_pcr

        generated_values[PCRDefnType.pcr_ipen_frz.value] = ipfrz_to_pcr(self.is_ipfrz(pll_obj))
        generated_values[PCRDefnType.pcr_ipenable.value] = PCRModeType.enable.value
        generated_values[PCRDefnType.pcr_lock_bypass.value] = PCRModeType.enable.value
        if pll_obj.ref_clock_mode != pll_obj.RefClockModeType.dynamic:
            generated_values[PCRDefnType.pcr_dyn_ref.value] = PCRModeType.disable.value
            generated_values[PCRDefnType.pcr_ref_sel.value] = dev_ref_clock_index_to_pcr(dev_ref_clock_index(pll_obj, pll_dev_svc))
        else:
            generated_values[PCRDefnType.pcr_dyn_ref.value] = PCRModeType.enable.value

        if pll_obj.fractional_mode or pll_obj.ssc_mode != "DISABLE" or pll_obj.pdc_enable:
            clk1 = pll_obj.get_output_clock_by_number(1)
            pt_assert(clk1 is not None, "Clk1 should not None", InvalidPCRException)
            pt_assert(clk1.number == 1, "Only clk1 support PDC mode", InvalidPCRException)
            pt_assert(isinstance(clk1, EfxFpllV1OutputClockFrac), f"Internal type error, expect {EfxFpllV1OutputClockFrac} but got {type(clk1)}", InvalidPCRException)

        if pll_obj.fractional_mode:
            generated_values[PCRDefnType.pcr_dsm_en.value] = PCRModeType.enable.value
            generated_values[PCRDefnType.pcr_k.value] = fractional_k_to_pcr(pll_obj.fractional_coefficient)

            clk1 = pll_obj.get_output_clock_by_number(1)
            pt_assert(clk1 is not None, "Clk1 should not None", InvalidPCRException)
            pt_assert(clk1.number == 1, "Only clk1 support PDC mode", InvalidPCRException)

            generated_values[PCRDefnType.pcr_sdiv_sel1.value] = swallowing_div_to_pcr(clk1.swallowing_div)

        if pll_obj.ssc_mode != "DISABLE":
            generated_values[PCRDefnType.pcr_dsm_en.value] = PCRModeType.enable.value
            generated_values[PCRDefnType.pcr_ssc_en.value] = PCRModeType.enable.value
            mod_period = calculate_ssc_mod_period(pll_obj.ref_clock_freq,
                                                  pll_obj.multiplier,
                                                  pll_obj.ssc_frequency,
                                                  pll_obj.pre_divider)
            generated_values[PCRDefnType.pcr_ssc_modperiod.value] = ssc_mod_period_to_pcr(mod_period)
            mod_step = calculate_ssc_mod_step(pll_obj.ssc_amp,
                                              clk1.out_divider,
                                              pll_obj.fractional_coefficient,
                                              mod_period)
            generated_values[PCRDefnType.pcr_ssc_modstep.value] = ssc_mod_step_to_pcr(mod_step)

            spread_pcr, spread_dir_pcr = ssc_mod_type_to_spread_pcr(pll_obj.ssc_modulation_type)
            generated_values[PCRDefnType.pcr_ssc_spread.value] = spread_pcr
            if spread_dir_pcr:
                generated_values[PCRDefnType.pcr_ssc_spread_dir.value] = spread_dir_pcr

            if pll_obj.ssc_mode == "DYNAMIC":
                generated_values[PCRDefnType.pcr_user_ssc_en.value] = PCRModeType.enable.value

            generated_values[PCRDefnType.pcr_sdiv_sel1.value] = swallowing_div_to_pcr(clk1.swallowing_div)
        else:
            generated_values[PCRDefnType.pcr_ssc_en.value] = PCRModeType.disable.value

        if pll_obj.pdc_enable:
            generated_values[PCRDefnType.pcr_dcden.value] = PCRModeType.enable.value

            clk1 = pll_obj.get_output_clock_by_number(1)
            pt_assert(clk1 is not None, "Clk1 should not None", InvalidPCRException)
            pt_assert(clk1.number == 1, "Only clk1 support PDC mode", InvalidPCRException)
            pt_assert(isinstance(clk1, EfxFpllV1OutputClockFrac), f"Internal type error, expect {EfxFpllV1OutputClockFrac} but got {type(clk1)}", InvalidPCRException)
            generated_values[PCRDefnType.pcr_sdiv_sel1.value] = swallowing_div_to_pcr(clk1.swallowing_div)

            dc_param = clk1.duty_cycle_control_param
            pt_assert(dc_param is not None, f"Internal error, duty cycle control param is not set", InvalidPCRException)
            pt_assert(isinstance(dc_param, DutyCycleControlParam), f"Internal type error, expect {DutyCycleControlParam} but got {type(dc_param)}", InvalidPCRException)
            generated_values[PCRDefnType.pcr_dcodd.value] = dc_param_to_pcr_dcodd(dc_param.half_cycle_shift)

        generated_values[PCRDefnType.pcr_dyn_conf.value] = dyn_cfg_en_to_pcr(pll_obj.dyn_cfg_enable)

        for clk in pll_obj.get_output_clock_list():
            clk_number = clk.number

            pcr = PCRDefnType(f"PCR_EN{clk_number}")
            generated_values[pcr.value] = PCRModeType.enable.value

            pcr = PCRDefnType(f"PCR_PSHIFT_DYN{clk_number}")
            generated_values[pcr.value] = clk_dyn_phase_en_to_pcr(clk.is_dyn_phase)

            pcr = PCRDefnType(f"PCR_PSHIFT{clk_number}")
            generated_values[pcr.value] = clk_phase_setting_to_pcr(clk.phase_setting)

            pcr = PCRDefnType(f"PCR_CASEN{clk_number}")
            generated_values[pcr.value] = clk_cascade_to_pcr(clk.is_cascade)

            pcr = PCRDefnType(f"PCR_CLKOUT_INV{clk_number}")
            generated_values[pcr.value] = clk_invert_en_to_pcr(clk.is_inverted)

            clk_div = clk.out_divider
            if clk_number == 1 and (pll_obj.fractional_mode or pll_obj.pdc_enable or pll_obj.ssc_mode != "DISABLE"):
                dc_param = clk1.duty_cycle_control_param
                pt_assert(dc_param is not None, "Internal error, duty cycle control param is not set", InvalidPCRException)
                pt_assert(isinstance(dc_param, DutyCycleControlParam), f"Internal type error, expect {DutyCycleControlParam} but got {type(dc_param)}", InvalidPCRException)
                clk_div = dc_param.pdiv

            pcr = PCRDefnType(f"PCR_DIV_SEL{clk_number}")
            generated_values[pcr.value] = clk_div_to_pcr(clk_div)

        return generated_values

    def is_ipfrz(self, inst: EfxFpllV1):
        return inst.is_ipfrz or inst in self.pcie_used_pll_list
