from __future__ import annotations
from pprint import pprint
from typing import TYPE_CHECKING, Dict, List, Literal, Tuple, get_type_hints

from common_device.property import OptionValidator, RangeValidator
from common_device.rule import Checker, PinData, Rule, RuleEmptyName
import common_device.pll.pll_rule as an08_pllrule
import an20_device.pll.pll_rule_adv as an20_pllrule
from design.db import PeriDesign
from design.db_item import GenericParamService, GenericPin, GenericPinGroup, PeriDesignItem


from device.db_interface import DeviceDBService
import tx60_device.pll.pll_rule_comp as tx60_pllrule
import tx180_device.pll.pll_rule as tx180_pllrule

from util.gen_util import override
from common_device.pll.pll_design_param_info import PLLParamId
from tx375_device.fpll.design import EfxFpllDynamicReconfigRecord, EfxFpllV1OutputClockFrac, EfxFpllV1
from util.signal_util import format_frequency


if TYPE_CHECKING:
    from common_device.rule import Checker
    from tx375_device.fpll.device_service import EfxFpllV1DeviceService


class EfxFpllV1Checker(tx180_pllrule.PLLCheckerV3Complex):

    @override
    def _build_rules(self):
        """
        Build pll rule database
        """
        self._add_rule(an08_pllrule.PLLRuleInstanceName())
        self._add_rule(an08_pllrule.PLLRuleInputFreq())
        self._add_rule(tx60_pllrule.PLLRuleOutputDivider())
        self._add_rule(an08_pllrule.PLLRuleOutputClock())
        self._add_rule(an08_pllrule.PLLRuleOutputNumber())
        self._add_rule(an08_pllrule.PLLRuleResource())
        self._add_rule(an20_pllrule.PLLRuleVCOFreq())
        self._add_rule(tx60_pllrule.PLLPostVCOFrequency())
        self._add_rule(an08_pllrule.PLLRuleOutputClockFreq())
        self._add_rule(an08_pllrule.PLLRuleOutputName())
        self._add_rule(tx60_pllrule.PLLRuleRefClk())
        self._add_rule(an20_pllrule.PLLRuleFeedbackResource())
        self._add_rule(an08_pllrule.PLLRuleInputFreqLimit())
        self._add_rule(an20_pllrule.PLLRuleNonInternalFeedbackFreq())
        self._add_rule(tx60_pllrule.PLLRuleIOFeedbackStd())
        self._add_rule(tx60_pllrule.PLLRuleInternalFeedback())
        self._add_rule(tx60_pllrule.PLLRuleDynamicShiftPinEmptyName())
        self._add_rule(tx60_pllrule.PLLRuleDynamicShiftPinInvalidName())
        self._add_rule(tx60_pllrule.PLLRuleDynamicShiftFeedback())
        self._add_rule(tx60_pllrule.PLLRuleConfig())
        self._add_rule(tx60_pllrule.PLLRuleMultiplier())
        self._add_rule(tx60_pllrule.PLLRulePreDivider())
        self._add_rule(tx60_pllrule.PLLRulePostDivider())
        self._add_rule(tx60_pllrule.PLLRuleMIPIDPHYTxClock())
        self._add_rule(an20_pllrule.PLLRuleClockSelectorPinName())
        self._add_rule(an08_pllrule.PLLRuleDividerRelationWhenCascade())
        self._add_rule(tx180_pllrule.PLLConnTypeRule())
        self._add_rule(tx180_pllrule.PLLRuleOutputClockFreq())

        # New rule Ti375
        self._add_rule(PLLRuleFeedback())
        self._add_rule(PLLRuleSSCMode())
        self._add_rule(PLLRuleSSCFreq())
        self._add_rule(PLLRuleSSCAmp())
        self._add_rule(PLLRuleDycCfg())
        self._add_rule(PLLRuleFractionalEnable())
        self._add_rule(PLLRuleFractionalFeedback())
        self._add_rule(PLLRuleFractionalPDC())
        self._add_rule(PLLRuleFractionalInternalDivider())
        self._add_rule(PLLRulePDCInternalDivider())
        self._add_rule(PLLRuleFractionalSSC())

        #  Dynamic Reconfiguration Related
        self._add_rule(PLLRuleDynCfgPinsEmpty())
        self._add_rule(PLLRuleDynCfgRefClkMode())
        self._add_rule(PLLRuleDynCfgFbkClk())


class PLLRuleFeedback(tx60_pllrule.PLLRuleFeedback):
    """
    Check the feedback setting (clock name / index)
    """

    def check(self, checker, design_block: EfxFpllV1):

        pll = design_block

        if pll.fb_mode == pll.FeedbackModeType.core or\
                pll.fb_mode == pll.FeedbackModeType.external or \
                pll.fb_mode == pll.FeedbackModeType.local:

            if pll.fb_clock_name == "":
                self.error('Feedback clock name is required with non-internal feedback')
                return

            fb_clk_data = checker.reg.find_output_clock(pll.fb_clock_name)

            if fb_clk_data is not None:
                fb_pll_ins, fb_out_no = fb_clk_data

                # It should be of the same instance
                # TODO: remove this if we support clkout from other PLL
                if pll != fb_pll_ins:
                    self.error('Feedback clock name {} is not from the same PLL'.format(
                        pll.fb_clock_name))
                    return

                if pll.fb_mode == pll.FeedbackModeType.local:
                    if fb_out_no not in {0, 1}:
                        self.error('Feedback clock in local mode has to connect to output clock 0 or 1')
                        return

                out_clk = fb_pll_ins.get_output_clock_by_number(fb_out_no)
                if out_clk.phase_setting != 0:
                    self.info('Feedback clock phase shift is not 0-degree, check that the feedback clock is in-phase with the reference clock.')
                    return

            else:
                self.error('Feedback clock {} is not connected to pll clkout'.format(
                    pll.fb_clock_name))
                return


class PLLRuleSSCMode(Rule):

    def __init__(self):
        super().__init__()
        self.name = "pll_rule_ssc_mode"

    def check(self, checker: Checker, design_block: EfxFpllV1):
        assert isinstance(design_block, EfxFpllV1), type(design_block)

        param_svc = GenericParamService(design_block.param_group, design_block.param_info)
        pvalid, msg = param_svc.check_param_valid(PLLParamId.ssc_mode)
        if pvalid is False or msg != "":
            self.error(f'SSC Mode is invalid, {msg}')
            return


class PLLRuleSSCFreq(Rule):

    def __init__(self):
        super().__init__()
        self.name = "pll_rule_ssc_freq"

    def check(self, checker: Checker, design_block: EfxFpllV1):
        assert isinstance(design_block, EfxFpllV1), type(design_block)

        param_svc = GenericParamService(design_block.param_group, design_block.param_info)
        pvalid, msg = param_svc.check_param_valid(PLLParamId.ssc_mode)
        if pvalid is False or msg != "":
            # Checked by another rule
            self.bypass()
            return

        ssc_mode = param_svc.get_param_value(PLLParamId.ssc_mode)
        if ssc_mode == "DISABLE":
            self.bypass()
            return

        pvalid, msg = param_svc.check_param_valid(PLLParamId.ssc_frequency)
        if pvalid is False or msg != "":
            self.error(f'SSC Frequency is invalid, {msg}')
            return


class PLLRuleSSCAmp(Rule):

    def __init__(self):
        super().__init__()
        self.name = "pll_rule_ssc_amp"

    def check(self, checker: Checker, design_block: EfxFpllV1):
        assert isinstance(design_block, EfxFpllV1), type(design_block)

        param_svc = GenericParamService(design_block.param_group, design_block.param_info)
        pvalid, msg = param_svc.check_param_valid(PLLParamId.ssc_mode)
        if pvalid is False or msg != "":
            # Checked by another rule
            self.bypass()
            return

        ssc_mode = param_svc.get_param_value(PLLParamId.ssc_mode)
        if ssc_mode == "DISABLE":
            self.bypass()
            return

        pvalid, msg = param_svc.check_param_valid(PLLParamId.ssc_amplitude)
        if pvalid is False or msg != "":
            self.error(f'SSC Amplitude is invalid, {msg}')
            return


class PLLRuleDycCfg(Rule):

    def __init__(self):
        super().__init__()
        self.name = "pll_rule_dyn_cfg"

    def check(self, checker: Checker, design_block: EfxFpllV1):
        assert isinstance(design_block, EfxFpllV1), type(design_block)

        param_svc = GenericParamService(design_block.param_group, design_block.param_info)
        pvalid, msg = param_svc.check_param_valid(PLLParamId.dyn_cfg_en)
        if pvalid is False or msg != "":
            self.error(f'Dynamic reconfiguration status is invalid, {msg}')
            return


class PLLRuleFractionalEnable(Rule):

    def __init__(self):
        super().__init__()
        self.name = "pll_rule_frac_en"

    def check(self, checker: Checker, design_block: EfxFpllV1):
        assert isinstance(design_block, EfxFpllV1), type(design_block)

        param_svc = GenericParamService(design_block.param_group, design_block.param_info)
        pvalid, msg = param_svc.check_param_valid(PLLParamId.enable_fractional)
        if pvalid is False or msg != "":
            self.error(f'Fractional mode enable status is invalid, {msg}')
            return


class PLLRuleFractionalFeedback(Rule):

    def __init__(self):
        super().__init__()
        self.name = "pll_rule_frac_feedback"

    def check(self, checker: Checker, design_block: EfxFpllV1):
        assert isinstance(design_block, EfxFpllV1), type(design_block)

        if not design_block.fractional_mode:
            self.bypass()
            return

        if design_block.fb_mode != design_block.FeedbackModeType.local:
            self.error("Feedback mode should be local in fractional mode")
            return

        fb_clk = design_block.get_feedback_clock()
        if fb_clk is None:
            self.error("Feedback clock is not configured in fractional mode")
            return

        if fb_clk.number != 1:
            self.error("Feedback clock must be CLKOUT1 in fractional mode")
            return


class PLLRuleFractionalPDC(Rule):

    def __init__(self):
        super().__init__()
        self.name = "pll_rule_frac_pdc"

    def check(self, checker: Checker, design_block: EfxFpllV1):
        assert isinstance(design_block, EfxFpllV1), type(design_block)

        if design_block.fractional_mode is True and design_block.pdc_enable is True:
            self.error("Fractional mode shouldn't be used together with Programable duty cycle")
            return


class PLLRuleFractionalInternalDivider(Rule):
    """
    Check PDIV / SDIV / K are valid in fractional mode
    """

    def __init__(self):
        super().__init__()
        self.name = "pll_rule_frac_internal_div"

    def check(self, checker: Checker, design_block: EfxFpllV1):
        assert isinstance(design_block, EfxFpllV1), type(design_block)
        assert isinstance(checker, EfxFpllV1Checker), type(design_block)

        dev_svc:EfxFpllV1DeviceService = checker.plldev_service

        if not design_block.fractional_mode:
            self.bypass()
            return

        fb_clock = design_block.get_feedback_clock()
        if fb_clock is None:
            self.bypass()
            return

        if not isinstance(fb_clock, EfxFpllV1OutputClockFrac):
            self.bypass()
            return

        pll_param_svc = GenericParamService(design_block.param_group, design_block.param_info)
        pvalid, msg = pll_param_svc.check_param_valid(PLLParamId.fractional_coefficent)
        if pvalid is False or msg != "":
            self.error(f"Fractional coefficient is invalid, {msg}")
            return

        pvalid, msg = pll_param_svc.check_param_valid(PLLParamId.dsm_mode)
        if pvalid is False or msg != "":
            self.error(f"Delta sigma modulation (DSM) mode is invalid, {msg}")
            return

        param_svc = GenericParamService(fb_clock.param_group, design_block.param_info)
        pvalid, msg = param_svc.check_param_valid(PLLParamId.swallowing_div)
        if pvalid is False or msg != "":
            self.error(f"Fractional feedback clock {fb_clock.name}'s swallowing divider (SDIV) is invalid, {msg}")
            return

        pvalid, msg = param_svc.check_param_valid(PLLParamId.p_div)
        if pvalid is False or msg != "":
            self.error(f"Fractional feedback clock {fb_clock.name}'s divider (PDIV) is invalid, {msg}")
            return

        dsm_mode = design_block.dsm_mode
        assert dsm_mode in {"SDIV", "DIVQ"}

        pdiv = param_svc.get_param_value(PLLParamId.p_div)
        min_pdiv, max_pdiv = dev_svc.get_min_max_pdiv(dsm_mode)
        if pdiv < min_pdiv or pdiv > max_pdiv:
            self.error(f"Fractional feedback clock's PDIV is out of range. Value={pdiv} Min={min_pdiv} Max={max_pdiv}")
            return

        sdiv = param_svc.get_param_value(PLLParamId.swallowing_div)
        min_sdiv, max_sdiv = dev_svc.get_min_max_sdiv(dsm_mode)
        if sdiv < min_sdiv or sdiv > max_sdiv:
            self.error(f"Fractional feedback clock's SDIV is out of range. Value={sdiv} Min={min_sdiv} Max={max_sdiv}")
            return

        # Check pdiv + sdiv = out_divider
        sum_div = pdiv + sdiv
        if sum_div != fb_clock.out_divider:
            self.error(f"Fractional feedback clock's PDIV + SDIV should be same as its output divider")
            return


class PLLRulePDCInternalDivider(Rule):
    """
    Check PDIV / SDIV are valid in pdc mode
    """

    def __init__(self):
        super().__init__()
        self.name = "pll_rule_pdc_internal_div"

    def check(self, checker: Checker, design_block: EfxFpllV1):
        assert isinstance(design_block, EfxFpllV1), type(design_block)
        assert isinstance(checker, EfxFpllV1Checker), type(design_block)

        if not design_block.pdc_enable:
            self.bypass()
            return

        for clk in design_block.get_output_clock_list():
            if isinstance(clk, EfxFpllV1OutputClockFrac) and clk.enable_pdc:
                ctrl_param = clk.duty_cycle_control_param
                if ctrl_param is None:
                    self.error("Duty Cycle Control parameters are not set")
                    return

                high_cycles = ctrl_param.num_high_cycles
                full_cycles = ctrl_param.num_full_cycles
                min_cycles, max_cycles = (1, full_cycles - 0.5)
                if high_cycles < min_cycles or high_cycles > max_cycles:
                    self.error(f"Invalid number of vco cycles for high signal in PDC mode, value = {high_cycles}, Min={min_cycles}, Max={max_cycles}")
                    return


class PLLRuleFractionalSSC(Rule):
    """
    Check SSC feature only available when Fraction mode is on
    """

    def __init__(self):
        super().__init__()
        self.name = "pll_rule_frac_ssc"

    def check(self, checker: Checker, design_block: PeriDesignItem):
        assert isinstance(design_block, EfxFpllV1), type(design_block)

        param_svc = GenericParamService(design_block.param_group, design_block.param_info)
        ssc_mode = param_svc.get_param_value(PLLParamId.ssc_mode)
        if ssc_mode == "DISABLE":
            self.bypass()
            return

        if ssc_mode == "STATIC" or ssc_mode == "DYNAMIC":
            if not design_block.fractional_mode:
                self.error(f"SSC feature requires PLL operates in fractional mode")
                return

            if design_block.fb_mode != design_block.FeedbackModeType.local:
                self.error(f"SSC feature requires PLL in local feedback mode")
                return

            fb_clk = design_block.get_feedback_clock()
            if fb_clk is None:
                self.bypass()
                return

            if fb_clk.number != 1:
                self.error("SSC feature requires CLKOUT1 to be configured as feedback clock")
                return


class PLLRuleDynCfgPinsEmpty(RuleEmptyName):
    """
    Check dynamic reconfiguation pins are configured
    """

    def __init__(self):
        super().__init__()
        self.name = "pll_rule_dyn_cfg_empty_pins"

    def check(self, checker: Checker, design_block: PeriDesignItem):
        assert isinstance(design_block, EfxFpllV1)
        assert isinstance(checker, EfxFpllV1Checker)

        param_svc = GenericParamService(design_block.param_group, design_block.param_info)
        pvalid, msg = param_svc.check_param_valid(PLLParamId.dyn_cfg_en)
        if pvalid is False or msg != "":
            # Checked in PLLRuleDycCfg
            self.bypass()
            return

        is_enable = design_block.dyn_cfg_enable
        if not is_enable:
            self.bypass()
            return

        pins: List[GenericPin] = []
        gen_pin: GenericPinGroup = design_block.gen_pin
        pins = [
            gen_pin.get_pin_by_type_name('CFG_CLK'),
            gen_pin.get_pin_by_type_name('CFG_DATA_IN'),
            gen_pin.get_pin_by_type_name('CFG_SEL')
        ]

        pins_to_check: Dict[str, PinData] = {}
        for pin in pins:
            desc = design_block.get_pin_property_name(pin.type_name)
            assert desc not in pins_to_check, desc
            pins_to_check[desc] = PinData(pin.name)

        # UGLY: these are not GenericPin, TODO Refactor
        pins_to_check['Reference Clock Selector [1:0] Bus Name'] = PinData(design_block.clock_sel_name)
        pins_to_check['Reset Pin Name'] = PinData(design_block.reset_name)

        if not self.validate(pins_to_check):
            return


class PLLRuleDynCfgRefClkMode(Rule):
    """
    Check reference clock source must be dynamic when dynamic reconfiguration feature is on
    """

    def __init__(self):
        super().__init__()
        self.name = "pll_rule_dyn_cfg_ref_clk_source"

    def check(self, checker: Checker, design_block: PeriDesignItem):
        assert isinstance(design_block, EfxFpllV1)
        assert isinstance(checker, EfxFpllV1Checker)

        param_svc = GenericParamService(design_block.param_group, design_block.param_info)
        pvalid, msg = param_svc.check_param_valid(PLLParamId.dyn_cfg_en)
        if pvalid is False or msg != "":
            # Checked in PLLRuleDycCfg
            self.bypass()
            return

        is_enable = design_block.dyn_cfg_enable
        if not is_enable:
            self.bypass()
            return

        # Check that the PLL should not select dynamic reference clock mode
        if design_block.ref_clock_mode == design_block.RefClockModeType.dynamic:
            msg = "Dynamic Reference Clock mode is unsupported when dynamic reconfiguration enabled."
            self.error(msg)
            return

        gpio_reg = checker.design.get_block_reg(checker.design.gpio_type)
        lvds_reg = checker.design.get_block_reg(checker.design.lvds_type)
        pll_dev_service = DeviceDBService(checker.design.device_db).get_block_service(checker.design.device_db.pll_type)

        msg = ""
        for record in design_block.dyn_reconfig_params:
            refclk_sel = record.get('REFCLK_SOURCE_SEL')
            if refclk_sel is None:
                # Check in other Rule
                continue

            match refclk_sel:
                case 'CORE_0':
                    if design_block.ref_clock_name == "":
                        msg += f'Core refclk pin 0 has to be specified for dynamic reconfiguration record: {record.name}\n'
                case 'CORE_1':
                    if design_block.ref_clock1_name == "":
                        msg += f'Core refclk pin 1 has to be specified for dynamic reconfiguration record: {record.name}\n'
                case 'EXT_0':
                    clock_info = design_block.find_external_clock_info(design_block.RefClockIDType.ext_clock_0,
                                                                       pll_dev_service,
                                                                       lvds_reg,
                                                                       gpio_reg,
                                                                       checker.design.device_db)
                    if clock_info:
                        if design_block.get_refclk1_pin_name(checker.design) == "":
                            res_names = ",".join([n[0] for n in clock_info])
                            msg += f'Reference clock at {res_names} connected to external clock pin 0 has not been defined for dynamic reconfiguration record: {record.name}\n'
                    else:
                        msg += "Invalid external clock 0 resource selected: Resource Unbonded"
                case 'EXT_1':
                    clock_info = design_block.find_external_clock_info(design_block.RefClockIDType.ext_clock_1,
                                                                       pll_dev_service,
                                                                       lvds_reg,
                                                                       gpio_reg,
                                                                       checker.design.device_db)
                    if clock_info:
                        if design_block.get_refclk2_pin_name(checker.design) == "":
                            res_names = ",".join([n[0] for n in clock_info])
                            msg += f'Reference clock at {res_names} connected to external clock pin 1 has not been defined for dynamic reconfiguration record: {record.name}\n'
                    else:
                        msg += "Invalid external clock 1 resource selected: Resource Unbonded"
                case _:
                    msg += f"Unexpected reference clock type: {refclk_sel} for dynamic reconfiguration record: {record.name}\n"

        if msg != "":
            self.error(msg)
            return


class PLLRuleDynCfgFbkClk(Rule):
    """
    Check if the feedback match with all dynamic reconfigurations
    """

    def __init__(self):
        super().__init__()
        self.name = "pll_rule_dyn_cfg_fbk_clk"

    def check(self, checker: Checker, design_block: PeriDesignItem):
        assert isinstance(design_block, EfxFpllV1)
        assert isinstance(checker, EfxFpllV1Checker)

        param_svc = GenericParamService(design_block.param_group, design_block.param_info)
        pvalid, msg = param_svc.check_param_valid(PLLParamId.dyn_cfg_en)
        if pvalid is False or msg != "":
            # Checked in PLLRuleDycCfg
            self.bypass()
            return

        is_enable = design_block.dyn_cfg_enable
        if not is_enable:
            self.bypass()
            return

        msg = ""
        for record in design_block.dyn_reconfig_params:
            fbclk = record.requested.get('FEEDBACK_CLK')
            if fbclk is None:
                # ERROR: FEEDBACK CLK missing
                pass

            if fbclk != design_block.get_feedback_clock_no():
                msg += f"Feedback clock is CLKOUT{fbclk} in the record: {record.name} which mis-match with current PLL setting.\n"

        if msg != "":
            self.error(msg)
            return


def validate_recfg_record(record: EfxFpllDynamicReconfigRecord,
                          block_inst: EfxFpllV1,
                          design: PeriDesign) -> Tuple[bool, List[str]]:

    # FIXME: Should use PropertyMetaData
    meta = {
        'REFCLK_SOURCE_SEL': {
            'type': str,
            'display_name': 'Reference Clock Source',
            'checker': OptionValidator(['CORE_0', 'CORE_1', 'EXT_0', 'EXT_1'])
        },
        'M': {
            'type': int,
            'display_name': 'Multipler',
            'checker': OptionValidator([1, 2, 4])
        },
        'N': {
            'type': int,
            'display_name': 'Pre-Divider',
            'checker': OptionValidator([1, 2, 4])
        },
        'O': {
            'type': int,
            'display_name': 'Post-Divider',
            'checker': OptionValidator([1, 2, 4, 8, 16, 32, 64, 128])
        },
        'FRACTIONAL_K': {
            'type': int,
            'display_name': 'Fractional Coefficent',
            'checker': RangeValidator(0, (2 ** 24) - 1)
        },
    }

    for i in range(5):
        clk = block_inst.get_output_clock_by_number(i)
        if clk is None:
            continue

        meta[f'CLKOUT{i}_DIV'] = {
            'clk_idx': i,
            'type': int,
            'display_name': f'Output Clock {i} Divider',
            'checker': RangeValidator(1, 128)
        }
        meta[f'CLKOUT{i}_PHASE_SETTING'] = {
            'clk_idx': i,
            'type': int,
            'display_name': f'Output Clock {i} Phase Shift Setting',
            'checker': OptionValidator([0, 1, 2, 3, 4, 5, 6, 7])
        }
        meta[f'CLKOUT{i}_DELAY_SETTING'] = {
            'clk_idx': i,
            'type': int,
            'display_name': f'Output Clock {i} Delay',
            'checker': RangeValidator(0, (2 ** 4) - 1)
        }

        meta[f'CLKOUT{i}_INVERT_EN'] = {
            'clk_idx': i,
            'type': bool,
            'display_name': f'Output Clock {i} Inversion',
            'checker': OptionValidator([True, False])
        }

    clk1 = block_inst.get_output_clock_by_number(1)
    if clk1 is not None:
        if record.requested.get('FRACTIONAL_MODE') is True:
            meta['CLKOUT1_SWALLOWING_DIV'] = {
                'clk_idx': 1,
                'type': int,
                'display_name': 'Output Clock 1 Swallowing Divider',
                'checker': RangeValidator(0, 64)
            }

        if block_inst.pdc_enable is True:
            meta['CLKOUT1_HALF_DC_SHIFT_EN'] = {
                'clk_idx': 1,
                'type': bool,
                'display_name': 'Enable Half Duty Cycle Shift',
                'checker': OptionValidator([True, False])
            }

    msgs = []
    pass_status = False
    for field, m in meta.items():
        m = meta[field]
        value = record.get(field)

        clk_idx = m.get('clk_idx')
        if clk_idx is not None:
            # Skip validate that field if the clkout is disabled
            clk = block_inst.get_output_clock_by_number(clk_idx)
            if clk is None:
                continue

        if value is None:
            msgs.append(f'{record.name}: {m["display_name"]} is not set.')
            continue

        try:
            value_typed = m['type'](value)
            value = value_typed
        except Exception:
            # import traceback
            # msgs.append(traceback.format_exc())
            msgs.append(f'{record.name}: {m["display_name"]} has unexpected type: {type(value)}, expecting {m["type"]}')
            continue

        is_valid, msg = m['checker'].is_valid(value)
        if not is_valid:
            msgs.append(f'{record.name}: {m["display_name"]} {msg}')
            continue

    # Check PDIV / SDIV for frac / ssc mode
    if clk1 is not None:
        if block_inst.fractional_mode is True or block_inst.ssc_mode != 'DISABLE':
            # PDIV (5, 64)
            checker = RangeValidator(5, 64)
            value = record.get('CLKOUT1_DIV')
            if value is not None:
                is_valid, msg = checker.is_valid(value)
                if not is_valid:
                    msgs.append(f'{record.name}: Output Clock 1 Divider {msg} for fractional / ssc mode')

            # SDIV (5, 60)
            checker = RangeValidator(5, 60)
            value = record.get('CLKOUT1_SWALLOWING_DIV')
            if value is not None:
                is_valid, msg = checker.is_valid(value)
                if not is_valid:
                    msgs.append(f'{record.name}: Output Clock 1 Swallowing Divider {msg} for fractional / ssc mode')

    # Check Frequency
    try:
        pll_dev_service = DeviceDBService(design.device_db).get_block_service(design.device_db.pll_type)

        def format_out_of_range_msg(freq, min, max) -> str:
            freq_str = format_frequency(freq, block_inst.PLL_CLOCK_PRESCISION)
            min_str = format_frequency(min, block_inst.PLL_CLOCK_PRESCISION)
            max_str = format_frequency(max, block_inst.PLL_CLOCK_PRESCISION)
            return f'is out of range. Freq={freq_str} Min={min_str} Max={max_str}'

        act_freqs = record.calculate_act_freq()
        vco_freq = act_freqs['ACT_VCO_FREQ']
        min, max = pll_dev_service.get_min_max_vco_freq(design, block_inst)
        if vco_freq < min or vco_freq > max:
            msgs.append(f'{record.name}: VCO frequency {format_out_of_range_msg(vco_freq, min, max)}')

        pll_freq = act_freqs['ACT_PLL_FREQ']
        min, max = pll_dev_service.get_min_max_pll_freq(design, block_inst)
        if pll_freq < min or pll_freq > max:
            msgs.append(f'{record.name}: PLL frequency {format_out_of_range_msg(pll_freq, min, max)}')

        for i in range(5):
            if block_inst.get_output_clock_by_number(i) is None:
                continue
            clkout_freq = act_freqs[f'ACT_CLKOUT{i}_FREQ']
            min, max = pll_dev_service.get_min_max_output_clock_freq()
            min = 0 if min is None else min
            if clkout_freq < min or clkout_freq > max:
                msgs.append(f'{record.name}: CLKOUT{i} frequency {format_out_of_range_msg(clkout_freq, min, max)}')

    except Exception as exc:
        msgs.append(f'{record.name} {str(exc)}')

    # Check pll resource
    if block_inst.get_device() == "":
        msgs.append(f'PLL Instance {block_inst.name} does not have resource assigned')

    if msgs:
        pass_status = False
    else:
        pass_status = True

    return pass_status, msgs
