#  """
#  Copyright (C) 2017-2020 Efinix Inc. All rights reserved.
#
#  No portion of this code may be reused, modified or
#  distributed in any way without the expressed written
#  consent of Efinix Inc.
#
#  Created on October 19, 2021
#
#  @author: michael wu
#  """
#
from __future__ import annotations
from pathlib import Path
import sys
import os
from typing import TYPE_CHECKING, List, Tuple

from PyQt5 import QtCore
from PyQt5.QtCore import Qt
from PyQt5.QtGui import QCursor, QTextCharFormat, QColor, QIcon, QFont
from PyQt5.QtWidgets import QTableWidgetItem, QMainWindow, QDialog, \
                            QMessageBox, QFileDialog, QMenu, QAction, \
                            QAbstractItemView

from common_gui.builder.base import FilePathParamWidget
from tx375_device.fpll.design import EfxFpllV1
from util.gui_util import block_signal, WindowsFilenameValidator
from util.singleton_logger import Logger
from util.app_setting import AppSetting
from util.app_launcher import AppLauncher
from util.excp import PTException

sys.path.append(os.path.dirname(__file__))
from Ui_tx375_device_pll_setting import Ui_MainWindow as UIFORM_CLASS
from Ui_tx375_device_pll_dyn_recfg import Ui_PLLDynRecfgWizard

if TYPE_CHECKING:
    from tx375_device.fpll.design import EfxFpllDynamicReconfigRecord


class PLLWindow(QMainWindow, UIFORM_CLASS):
    """
    Main window for PLL configuration
    """
    _is_show_config = True

    # noinspection PyArgumentList
    def __init__(self, parent=None):
        """
        Main window for PLL, holds all the UI objects
        """

        self.design = None
        self.parent = parent
        self.logger = Logger

        QMainWindow.__init__(self, parent)
        self.setupUi(self)

        self.dw_config.setVisible(PLLWindow._is_show_config)
        self.actionShow_Hide_Config.triggered.connect(self.show_config_window)
        self.actionHelp.triggered.connect(self.on_show_help)
        self.statusBar().hide()

        self.dyn_recfg_wizard = PLLDynReconfigWizard(None)

        self.browser = AppLauncher()

    def update_design(self, design):
        self.design = design

    def show_config_window(self):

        if self.dw_config.isVisible():
            self.dw_config.hide()
            PLLWindow._is_show_config = False
        else:
            self.dw_config.show()
            PLLWindow._is_show_config = True

    # noinspection PyArgumentList
    @QtCore.pyqtSlot()
    def on_show_help(self):
        setting = AppSetting()
        help_file = setting.app_path[AppSetting.PathType.pll_comp_html_help]
        if os.path.exists(help_file):
            try:
                self.browser.open_in_browser(help_file)
            except PTException:
                msg = "Fail to open help for pll."
                QMessageBox.warning(self.parent, "Open pll help", msg)
                self.logger.warning(msg)

        else:
            msg = "Help file for pll is missing. Please contact support to fix your installation."
            QMessageBox.warning(self.parent, "Open pll help", msg)
            self.logger.warning(msg)


class PLLDynReconfigWizard(QMainWindow, Ui_PLLDynRecfgWizard):

    sig_record_add_requested = QtCore.pyqtSignal()
    sig_record_delete_requested = QtCore.pyqtSignal(int)
    sig_record_detail_load_requested = QtCore.pyqtSignal(int)
    sig_record_export_requested = QtCore.pyqtSignal(str)
    sig_record_detail_update_requested = QtCore.pyqtSignal()
    sig_record_rearranged = QtCore.pyqtSignal(int, int)
    sig_record_validation_request = QtCore.pyqtSignal()
    sig_record_detail_override_requested = QtCore.pyqtSignal(int)
    # Signal when the configurations require User's review
    sig_record_review_request = QtCore.pyqtSignal()
    sig_record_param_update_request = QtCore.pyqtSignal(object, tuple)

    def __init__(self, parent) -> None:
        super().__init__(parent)
        self.setupUi(self)
        self.detail.hide()

        self.design = None
        self.block_inst: EfxFpllV1 | None = None

        self.pb_add_record.clicked.connect(self.sig_record_add_requested.emit)
        self.table_dyn_record.itemSelectionChanged.connect(self.on_table_item_selected_changed)
        self.pb_export_record_file.clicked.connect(self.__on_export_button_clicked)
        self.pb_load_record_file.hide()
        self.pb_validate_all_records.clicked.connect(self.__on_verify_button_clicked)

        file_dialog = QFileDialog()
        file_dialog.setWindowTitle("Export MIF")
        file_dialog.setFileMode(QFileDialog.AnyFile)
        file_dialog.setViewMode(QFileDialog.Detail)
        file_dialog.setAcceptMode(QFileDialog.AcceptSave)
        file_dialog.setNameFilter("Hex File (*.hex)")
        file_dialog.setOption(QFileDialog.DontConfirmOverwrite)

        self.export_file_widget = FilePathParamWidget(file_dialog)
        self.export_file_widget.set_label_text("PLL Dynamic Reconfiguration File")
        self.vl_params.addWidget(self.export_file_widget)
        self.vl_params.addStretch()

        self.detail.sig_record_detail_updated.connect(self.sig_record_detail_update_requested.emit)
        self.table_dyn_record.cellChanged.connect(self.__on_record_name_changed)
        self.table_dyn_record.cellActivated.connect(self.__on_table_cell_activiated)

        self.table_menu = QMenu(self.table_dyn_record)
        self.table_dyn_record.setContextMenuPolicy(Qt.CustomContextMenu)
        self.table_dyn_record.customContextMenuRequested.connect(self.__on_menu_requested)

        self.table_dyn_record.verticalHeader().setSectionsMovable(True)
        self.table_dyn_record.verticalHeader().setDragEnabled(True)
        self.table_dyn_record.verticalHeader().setDragDropMode(QAbstractItemView.InternalMove)

        self.action_delete.triggered.connect(self.__on_delete_action_triggered)
        self.table_menu.addAction(self.action_delete)

        self.action_override.triggered.connect(self.__on_override_action_triggered)
        self.table_menu.addAction(self.action_override)

        self.action_move_up.triggered.connect(self.__on_move_up_action_triggered)
        self.table_menu.addAction(self.action_move_up)

        self.action_move_down.triggered.connect(self.__on_move_down_action_triggered)
        self.table_menu.addAction(self.action_move_down)

        self.table_dyn_record.verticalHeader().sectionMoved.connect(self.__on_section_moved)
        self.detail.sig_error.connect(self.__on_error)

        self.console.setStyleSheet("QWidget {{ font-family: {}; }}".format("Source Code Pro"))
        self.update_exportation_status(False)

    def write_coloured_text(self, msg: str, color: QColor, newline: bool=True):
        text_format = self.console.currentCharFormat()
        text_format.setForeground(color)
        self.console.setCurrentCharFormat(text_format)
        self.console.insertPlainText(msg)
        if newline:
            self.console.insertPlainText("\n")
        self.console.setCurrentCharFormat(QTextCharFormat())
        self.console.ensureCursorVisible()

    def write_error(self, msg: str, newline: bool = True):
        # TODO: Make console become standalone widget
        self.write_coloured_text(msg, QColor(Qt.red), newline)

    def write_info(self, msg: str, newline: bool=True):
        # TODO: Make console become standalone widget
        self.write_coloured_text(msg, QColor(Qt.black), newline)

    def write_warning(self, msg: str, newline: bool=True):
        # TODO: Make console become standalone widget
        self.write_coloured_text(msg, QColor(Qt.darkYellow), newline)

    def __on_error(self, msg: str):
        self.write_error(msg)

    def __on_section_moved(self, logical_idx: int, old_ui_idx: int, new_ui_idx: int):
        self.swap_record_index(logical_idx, new_ui_idx)

    def __on_table_cell_activiated(self, row: int, column: int):
        self.sig_record_detail_load_requested.emit(row)

    def __on_export_button_clicked(self):
        # Check Overwrite
        file_name = self.export_file_widget.editor_widget.text()
        validator = WindowsFilenameValidator()
        if file_name == "":
            self.write_error("Empty File Name, Unable to export file.")
            return
        elif validator.check_windows_reserved_keyword(file_name):
                QMessageBox.warning(
                        self, "Invalid Export Design File Path", f"Project path is not writable : {Path(file_name)}")
                return
        else:
            file_path = Path(file_name)
            if file_path.exists():
                confirm_overwrite_box = QMessageBox()
                confirm_overwrite_box.setWindowTitle("Export File Exists")
                msg = f"The specified path file name {file_path} already exist."
                confirm_overwrite_box.setText(msg)
                cancel_btn = confirm_overwrite_box.addButton("Cancel", QMessageBox.RejectRole)
                confirm_overwrite_box.addButton("Overwrite", QMessageBox.ApplyRole)

                confirm_overwrite_box.exec()
                btn = confirm_overwrite_box.clickedButton()
                if btn == cancel_btn:
                    return

        self.sig_record_export_requested.emit(str(file_name))

    def __on_record_name_changed(self, row: int, column: int):
        if row >= self.table_dyn_record.rowCount():
            return

        record = self.block_inst.dyn_reconfig_params[row]
        name_item = self.table_dyn_record.item(row, column)
        record.name = name_item.text()
        self.sig_record_detail_update_requested.emit()

    def __on_menu_requested(self, pos):
        idx = self.table_dyn_record.indexAt(pos)
        if idx.isValid():
            self.action_move_down.setDisabled(idx.row() == self.table_dyn_record.rowCount() - 1)
            self.action_move_up.setDisabled(idx.row() == 0)

        self.table_menu.exec_(self.table_dyn_record.viewport().mapToGlobal(pos))

    def __on_delete_action_triggered(self):
        selected_row = self.table_dyn_record.currentRow()
        if selected_row < 0 or selected_row >= self.table_dyn_record.rowCount():
            return

        self.detail.hide()
        self.table_dyn_record.removeRow(selected_row)
        for i in range(self.table_dyn_record.rowCount()):
            item = self.table_dyn_record.verticalHeaderItem(i)
            item.setText(str(i))

        self.sig_record_delete_requested.emit(selected_row)

    def __on_override_action_triggered(self):
        selected_row = self.table_dyn_record.currentRow()
        if selected_row < 0 or selected_row >= self.table_dyn_record.rowCount():
            return

        self.sig_record_detail_override_requested.emit(selected_row)

    def __on_verify_button_clicked(self):
        self.sig_record_validation_request.emit()

    def __on_move_up_action_triggered(self):
        selected_row = self.table_dyn_record.currentRow()
        if selected_row < 0 or selected_row >= self.table_dyn_record.rowCount():
            return
        self.swap_record_index(selected_row, selected_row - 1)

    def __on_move_down_action_triggered(self):
        print("__on_move_down_action_triggered")
        selected_row = self.table_dyn_record.currentRow()
        if selected_row < 0 or selected_row >= self.table_dyn_record.rowCount():
            return
        print(f"__on_move_down_action_triggered, selected_row = {selected_row}")
        self.swap_record_index(selected_row, selected_row + 1)

    def load_instance(self, design, inst: EfxFpllV1):
        with block_signal(self):
            self.console.clear()
            self.detail.hide()

            self.block_inst = inst
            self.design = design
            self.table_dyn_record.clearContents()
            self.table_dyn_record.setRowCount(0)

            self.table_dyn_record.setHorizontalHeaderLabels(["Record Name"])

            auto_actions: List[Tuple[str, EfxFpllDynamicReconfigRecord]] = []
            curr_fbk_idx = inst.get_feedback_clock_no()
            for index, record in enumerate(inst.dyn_reconfig_params):
                # Check PLL Feedback Clock
                record_fbk_idx = record.requested.get('FEEDBACK_CLK')
                if record_fbk_idx is None:
                    auto_actions.append(('SET_FB_CLK', index, record))
                elif record_fbk_idx != curr_fbk_idx:
                    auto_actions.append(('SET_FB_CLK', index, record))

                # TODO: Auto mode related check

                self.add_record(record)

            self.export_file_widget.file_dialog_widget.setDirectory(self.design.location)
            self.export_file_widget.editor_widget.setText(f"{design.location}/{inst.name}_dyn_cfg.hex")

        if auto_actions:
            for action, idx, record in auto_actions:
                # item.setIcon(QIcon(":/icons/error.svg"))
                match action:
                    case 'SET_FB_CLK':
                        old_fbk_idx = record.requested.get('FEEDBACK_CLK')
                        self.sig_record_param_update_request.emit(record, tuple([('FEEDBACK_CLK', old_fbk_idx, curr_fbk_idx)]))
                    case _:
                        raise NotImplementedError(f'Unhandled action: {action}')

                # Update icon to tell user
                item = self.table_dyn_record.item(idx, 0)
                if item is not None:
                    item.setIcon(":/icons/error.svg")
            self.sig_record_review_request.emit()

    def on_table_item_selected_changed(self):
        selected_indexs = self.table_dyn_record.selectedIndexes()
        if len(selected_indexs) <= 0:
            return

        row_idx = selected_indexs[0].row()
        self.sig_record_detail_load_requested.emit(row_idx)

    def load_detail(self, record: EfxFpllDynamicReconfigRecord):
        self.detail.show()
        self.detail.load(record)

    def add_record(self, record: EfxFpllDynamicReconfigRecord):
        row_idx = self.table_dyn_record.rowCount()
        self.table_dyn_record.insertRow(row_idx)
        name = QTableWidgetItem(str(record.name))
        name.setFlags(name.flags() | Qt.ItemIsDragEnabled)
        self.table_dyn_record.setItem(row_idx, 0, name)

        address = QTableWidgetItem(str(row_idx))
        self.table_dyn_record.setVerticalHeaderItem(row_idx, address)

    def swap_record_index(self, a_row: int, b_row: int):
        if a_row < 0 or a_row >= self.table_dyn_record.rowCount():
            return

        if b_row < 0 or b_row >= self.table_dyn_record.rowCount():
            return

        if a_row == b_row:
            return

        with block_signal(self.table_dyn_record):
            a_name = self.table_dyn_record.item(a_row, 0)
            a_text = a_name.text()
            b_name = self.table_dyn_record.item(b_row, 0)
            b_text = b_name.text()
            b_name.setText(a_text)
            a_name.setText(b_text)
            for row in range(self.table_dyn_record.rowCount()):
                address = self.table_dyn_record.verticalHeaderItem(row)
                address.setText(str(row))
            self.table_dyn_record.clearSelection()

        self.sig_record_rearranged.emit(a_row, b_row)

    def update_record_validation_result(self, idx: int, is_valid: bool,
                                        msg: str = ""):
        assert idx >= 0 and idx < self.table_dyn_record.rowCount()

        with block_signal(self):
            item = self.table_dyn_record.item(idx, 0)
            self.write_info(f"Record '{item.text()}: ", newline=False)
            if is_valid:
                self.write_coloured_text("PASS", QColor(Qt.darkGreen))
                item.setIcon(QIcon(":/icons/ok.svg"))
                if msg != "":
                    self.write_info(msg)
            else:
                self.write_coloured_text("FAIL", QColor(Qt.red))
                item.setIcon(QIcon(":/icons/error.svg"))
                self.write_error(msg)

            self.write_info("", newline=True)

    def update_exportation_status(self, enable: bool):
        self.pb_export_record_file.setEnabled(enable)
        if not enable:
            self.pb_export_record_file.setToolTip("Please run validation first.")
        else:
            self.pb_export_record_file.setToolTip("Export configuration to memory initailization file")
