'''
Copyright (C) 2017-2024 Efinix Inc. All rights reserved.

No portion of this code may be reused, modified or
distributed in any way without the expressed written
consent of Efinix Inc.

Created on March 08, 2024

@author: Shirley
'''
from __future__ import annotations

from typing import Dict, Optional, Set, Tuple, TYPE_CHECKING

import pt_version

import util.gen_util

import device.db_interface as devdb_int
import common_device.rule as base_rule
from common_device.quad.res_service import QuadResService

from design.db_item import GenericClockPin, GenericParamService


from tx375_device.common_quad.design import QuadLaneCommon, QuadLaneCommonRegistry
from tx375_device.common_quad.design_param_info import QuadDesignParamInfo
from tx375_device.common_quad.quad_prop_id import CommonQuadConfigParamInfo as CommonQuadParamInfo
from tx375_device.raw_serdes.raw_serdes_prop_id import RawSerdesConfigParamInfo as RawSerdesParamInfo
                    
if TYPE_CHECKING:
    from tx375_device.quad.device_service import QuadDeviceService

IS_SHOW_HIDDEN = True if pt_version.PT_DEBUG_VERSION == True else False

@util.gen_util.freeze_it
class QuadLaneCommonChecker(base_rule.Checker):

    def __init__(self, design):
        """
        Constructor
        """
        super().__init__(design)
        self.block_tag = "common_quad_lane"
        self.reg: Optional[QuadLaneCommonRegistry] = design.common_quad_lane_reg
        self.dev_service = None # type: Optional[QuadDeviceService]

        if self.design is not None:
            self.reg = self.design.common_quad_lane_reg

            if self.design.device_db is not None:
                dbi = devdb_int.DeviceDBService(self.design.device_db)
                self.dev_service = dbi.get_block_service(
                    devdb_int.DeviceDBService.BlockType.QUAD)

    def _build_rules(self):
        """
        Build common quad lan rule database
        """
        self._add_rule(QuadLaneCommonRuleInstanceName())
        self._add_reg_rule(QuadLaneCommonRulePinName())
        self._add_rule(QuadRuleAPBClock())
        self._add_rule(QuadLaneCommonTopology())
        self._add_rule(QuadLaneCommonPMADirectPLLConfig())
        self._add_rule(QuadLaneCommonPhyResetPin())

    def find_pll_output_clock(self, clk_name):
        if self.design is not None and self.design.pll_reg is not None:
            all_pll_inst = self.design.pll_reg.get_all_inst()

            for pll_inst in all_pll_inst:
                out_clk = pll_inst.get_output_clock(clk_name)

                # Return on first one found. They should be
                # unique as trapped later in interface writer
                if out_clk is not None:
                    return out_clk

        return None


class QuadLaneCommonRuleInstanceName(base_rule.Rule):
    """
    Check if the instance name is valid
    """

    def __init__(self):
        super().__init__()
        self.name = "common_quad_lane_rule_inst_name"

    def check(self, checker, design_block: QuadLaneCommon):
        util.gen_util.mark_unused(checker)

        inst = design_block

        if self.check_name_empty_or_invalid(inst.name,
                                            "Instance name is empty.",
                                            "Valid characters are alphanumeric characters with dash and underscore only."
                                            ):
            return


class QuadLaneCommonRulePinName(base_rule.Rule):
    """
    Check that if pin name is identical
    """

    def __init__(self):
        super().__init__()
        self.name = "common_quad_lane_rule_pin_name"

    def check(self, checker: QuadLaneCommonChecker, design_block: QuadLaneCommonRegistry):
        reg = design_block
        # (Pin type, pin_name): inst name list
        invalid_pin_types: Dict[Tuple[str, str], Set[str]] = {}

        # Pin type: dict(inst name: pin_name)
        pin_type2pin_info: Dict[str, Dict[str, str]] = {}

        # Description
        pin_type2desc: Dict[str, str] = {}

        for inst in reg.get_all_inst():
            gen_pin = inst.gen_pin
            if gen_pin is None:
                return

            for pin in gen_pin.get_all_pin():
                if isinstance(pin, GenericClockPin) or pin.name == "":
                    continue

                pin_name = pin.name
                pin_type = pin.type_name

                if pin_type not in pin_type2pin_info:
                    pin_type2pin_info[pin_type] = {inst.name: pin_name}
                    continue

                for inst_name, inst_pin_name in pin_type2pin_info[pin_type].items():
                    if pin_name == inst_pin_name:
                        key = (pin_type, pin_name)
                        invalid_inst_set = invalid_pin_types.get(key, {inst_name,})
                        invalid_inst_set.add(inst.name)
                        invalid_pin_types[key] = invalid_inst_set

                        if pin_type not in pin_type2desc:
                            pin_type2desc[pin_type] = inst.get_pin_property_name(pin_type)
                        break

                pin_info = pin_type2pin_info.get(pin_type, {})
                pin_info[inst.name] = pin_name
                pin_type2pin_info[pin_type] = pin_info

        if len(invalid_pin_types) <= 0:
            return

        msg = "The following pin(s) is/are identical across different QUAD:"
        # Description (pin_name): name list
        for key, inst_set in sorted(invalid_pin_types.items()):
            pin_type, pin_name = key
            desc = pin_type2desc.get(pin_type, "")
            device_list = []
            for inst_name in inst_set:
                inst = reg.get_inst_by_name(inst_name)
                assert inst is not None
                device_list.append(inst.get_device())

            quad_list = ", ".join(sorted(device_list))
            msg += f"\n- {desc} (pin name: {pin_name}): {quad_list}"

        self.error(msg)

class QuadRuleAPBClock(base_rule.Rule):
    """
    Check that APB clock is configured correctly.
    From PT-2350: APB pins are now always required for 10G and 1G. But we
        only check for the APB clock pin name since that's 
        empty by default (clkout pin).
    """

    def __init__(self):
        super().__init__()
        self.name = "common_quad_rule_apb_clock"

    def is_shared_with_non_raw_serdes(self, checker: QuadLaneCommonChecker,
                                      cmn_ins_name: str):
        is_shared  = False

        # iterate through all lane registry to find
        # lane instances associated to this common quad instance
        if checker.design is not None:
            lane_reg_list = [checker.design.lane_10g_reg,
                             checker.design.lane_1g_reg]
            
            for lane_reg in lane_reg_list:
                if lane_reg is None:
                    continue
                
                lane_ins = lane_reg.get_instance_associated_to_common_inst(cmn_ins_name)

                # If this common instance is used for 10G/1G, the apb
                # becomes compulsory
                if len(lane_ins) > 0:
                    is_shared = True
                    break

        return is_shared
    
    def check(self, checker: QuadLaneCommonChecker, design_block: QuadLaneCommon):

        cmn_ins = design_block

        # Check the APB Clock is non-empty
        gen_pin = cmn_ins.gen_pin
        assert gen_pin is not None

        apb_clk_pin = gen_pin.get_pin_by_type_name("USER_APB_CLK")
        if apb_clk_pin is not None:
            # Error only if this common instance has 10G/1G in 
            # the same quad
            if apb_clk_pin.name == "":
                
                if self.is_shared_with_non_raw_serdes(checker, cmn_ins.name):
                    self.error(
                        'APB Clock pin cannot be empty')
                else:
                    # This is if it was on raw serdes. So, we check the
                    # apb_en pin to error out or not
                    param_service = GenericParamService(cmn_ins.get_param_group(),
                                                        cmn_ins.get_param_info())
                    
                    apb_en = param_service.get_param_value(QuadDesignParamInfo.Id.apb_en)

                    if apb_en:
                        self.error("APB Clock pin cannot be empty when APB is enabled")    
                            
class QuadLaneCommonTopology(base_rule.Rule):
    """
    Check that a quad may not have multiple protocol types
    associated until we are ready.
    """

    def __init__(self):
        super().__init__()
        self.name = "common_quad_lane_rule_protocol"

    def check_common_shared_by_multiple_protocols(
            self, checker: QuadLaneCommonChecker,
            cmn_ins_name: str):
        
        protocol_cnt  = 0
        
        # iterate through all lane registry to find
        # lane instances associated to this common quad instance
        if checker.design is not None:
            lane_reg_list = checker.design.get_lane_based_reg()            

            for lane_reg in lane_reg_list:
                if lane_reg is None:
                    continue
                lane_ins = lane_reg.get_instance_associated_to_common_inst(cmn_ins_name)
                
                if len(lane_ins) > 0:
                    protocol_cnt += 1

        is_mult_protocol = False
        if protocol_cnt > 1:
            is_mult_protocol = True

        return is_mult_protocol
    
    def check(self, checker, design_block: QuadLaneCommon):

        cmn_ins = design_block
        if self.check_common_shared_by_multiple_protocols(checker, cmn_ins.name):
            self.error("Multiple lane-based protocol in the same quad is not supported")

class QuadLaneCommonPMADirectPLLConfig(base_rule.Rule):
    '''
    Check that the PLL config of all lanes in same quad is valid
    '''
    def __init__(self):
        super().__init__()
        self.name = "common_quad_rule_pma_direct_pll_config"

    def check(self, checker: QuadLaneCommonChecker, design_block: QuadLaneCommon):
        blk = design_block

        # Get all the lanes of the same quad
        if checker.design is not None and blk.get_device() != "" and\
            checker.design.raw_serdes_reg is not None:
            # This returns all raw serdes instances that belongs to the same quad
            same_quad_lane_inst = checker.design.raw_serdes_reg.get_instance_associated_to_common_inst(blk.name)

            if len(same_quad_lane_inst) > 1:
                quad_lane_config = {}
                ref_data_rate = None
                ref_refclk_freq = None
                ref_data_width = None
                invalid_comb_inst: Dict[str, tuple] = {}
                ref_ins_name = ""

                for inst in same_quad_lane_inst:
                    cmn_ins = checker.design.raw_serdes_reg.get_cmn_inst_by_lane_name(inst.name)
                    assert cmn_ins == blk
                    
                    # Get the value of the 3 key parameters        
                    param_service = GenericParamService(inst.get_param_group(), inst.get_param_info())
                    cmn_param_service = GenericParamService(cmn_ins.get_param_group(), cmn_ins.get_param_info())
                    
                    data_rate = param_service.get_param_value(RawSerdesParamInfo.Id.ss_raw_data_rate_lane_NID)
                    refclk_freq = cmn_param_service.get_param_value(CommonQuadParamInfo.Id.ss_raw_refclk_freq)
                    data_width = param_service.get_param_value(RawSerdesParamInfo.Id.ss_raw_serdes_width_lane_NID)

                    if inst.pll_cfg is not None:
                        cfg_map = inst.pll_cfg.get_pll_config_settings(data_rate, refclk_freq, data_width, IS_SHOW_HIDDEN)

                        # Don't need to check for None since that is handled
                        # by another rulel
                        if cfg_map is not None:
                            if not quad_lane_config:
                                # First insteance read
                                quad_lane_config = cfg_map
                                ref_data_rate = data_rate
                                ref_refclk_freq = refclk_freq
                                ref_data_width = data_width
                                ref_ins_name = inst.name

                            else:
                                if ref_data_rate != data_rate or \
                                    ref_refclk_freq != refclk_freq or \
                                    ref_data_width != data_width or \
                                    quad_lane_config != cfg_map:
                                    invalid_comb_inst[inst.name] = (data_rate, refclk_freq, data_width)
                                
                        else:
                            # We're finding that some instance has valid combination
                            # while some doesn't.
                            invalid_comb_inst[inst.name] = (data_rate, refclk_freq, data_width)

                if invalid_comb_inst:
                    # Add the ref ins name
                    invalid_comb_inst[ref_ins_name] = (ref_data_rate, ref_refclk_freq, ref_data_width)

                    invalid_ins_names = ",".join(sorted(list(invalid_comb_inst.keys())))
                    #quad_idx, _ = QuadResService.break_res_name(blk.get_device())
                    self.error(f'Found {invalid_ins_names} in the same {blk.get_device()} with different PLL configuration settings')

class QuadLaneCommonPhyResetPin(base_rule.Rule):
    '''
    PT-2559 Check if pin name is not empty if it is enabled
    '''
    def __init__(self):
        super().__init__()
        self.name = "common_quad_rule_phy_reset_pin"

    def check(self, checker: QuadLaneCommonChecker, design_block: QuadLaneCommon):
        raw_serdes_reg = checker.design.raw_serdes_reg

        # The pin is hidden in other protocol, skip checking
        if raw_serdes_reg is None or checker.reg is None:
            return

        # Check if Raw Serdes is using the current quad
        is_raw_serdes_used = False
        for inst in raw_serdes_reg.get_all_inst():
            if raw_serdes_reg.get_cmn_inst_by_lane_name(inst.name) == design_block:
                is_raw_serdes_used = True
                break

        if not is_raw_serdes_used:
            return

        is_enable = design_block.param_group.get_param_value(
            QuadDesignParamInfo.Id.phy_reset_en.value)

        if is_enable and design_block.gen_pin.get_pin_name_by_type("PHY_CMN_RESET_N") == "":
            self.error("PHY Quad Reset pin cannot be empty when PHY Quad Reset Pin is enabled")
