import sys
import os
import util.gen_util

from design.db_patch import DesignVersion, DesignPatch
from design.db_item import GenericParamGroup

from tx375_device.common_quad.design import QuadLaneCommon, QuadLaneCommonRegistry
from tx375_device.common_quad.design_param_info import QuadDesignParamInfo

sys.path.append(os.path.join(os.path.dirname(__file__), '..'))



@util.gen_util.freeze_it
class QuadLaneCommonPatch(DesignPatch):
    """
    QuadLaneCommon design backward compatibility patcher
    """

    def __init__(self, reg: QuadLaneCommonRegistry, device_db):
        super().__init__(reg, device_db)
        self.block_reg: QuadLaneCommonRegistry

        #
        # Patches sequence can imply dependency between patches. When the patch is applied,
        # it will be applied from the lowest version number to the highest.
        # This means it is safe for patch function to assume certain db states.
        #

        # PT-2453
        msg = "Common Quad Lane: Rename pin type USER_PHY_RESET_N to PHY_CMN_RESET_N"
        self.add_patch(DesignVersion.PATCH_VERSION_20251001,
                       QuadLaneCommonPatch.patch_rename_phy_reset_pin_type, msg)

        # PT-2595
        msg = "Common Quad Lane: Renaming param from sw_raw_refclk_freq to ss_raw_refclk_freq"
        self.add_patch(DesignVersion.PATCH_VERSION_20251002,
                       QuadLaneCommonPatch.patch_refclk_freq_param, msg)

        # PT-2598
        msg = "Common Quad Lane: Removing MHz from the ss_1gbe_refclk_freq options"
        self.add_patch(DesignVersion.PATCH_VERSION_20251003,
                       QuadLaneCommonPatch.patch_lane1g_refclk_freq_type_options, msg)

    def patch_rename_phy_reset_pin_type(self, inst: QuadLaneCommon):
        is_patch = False

        old_type = 'USER_PHY_RESET_N'
        new_type = 'PHY_CMN_RESET_N'

        pin = inst.gen_pin.get_pin_by_type_name(old_type)
        new_pin = inst.gen_pin.get_pin_by_type_name(new_type)

        if pin is not None and new_pin is not None:
            new_pin.name = pin.name
            inst.gen_pin.delete_pin_by_type(old_type)
            is_enable_reset = False if new_pin.name == "" else True
            inst.param_group.set_param_value(
                QuadDesignParamInfo.Id.phy_reset_en.value, is_enable_reset)

            is_patch = True

        if is_patch:
            func_name = util.gen_util.get_function_name()
            self.logger.warning(
                "{}: Patch {} applied to {}".format(func_name, DesignVersion.PATCH_VERSION_20251001,
                                                    inst.name))

        return is_patch

    def update_params(self, param_group: GenericParamGroup, rename_list):
        is_updated = False

        for ori_name, new_name in rename_list:
            ori_param = param_group.get_param_by_name(ori_name)
            new_param = param_group.get_param_by_name(new_name)

            if ori_param is None or new_param is None:
                continue

            param_group.set_param_value(new_name, ori_param.value)
            param_group.remove_param(ori_name)
            is_updated = True

        return is_updated  
    
    def patch_refclk_freq_param(self, inst: QuadLaneCommon):
        is_patch = False

        cmn_param_group = inst.param_group
        cmn_param_info = inst.param_info

        # Patch the value from sw_raw_refclk_freq to ss_raw_refclk_freq
        rename_list = [
            ("sw_raw_refclk_freq", "ss_raw_refclk_freq"),
        ]

        is_patch = self.update_params(cmn_param_group, rename_list)

        if is_patch:
            func_name = util.gen_util.get_function_name()
            self.logger.warning(
                "{}: Patch {} applied to {}".format(func_name, DesignVersion.PATCH_VERSION_20251002,
                                                    inst.name))
                    
        return is_patch

    def patch_lane1g_refclk_freq_type_options(self, inst: QuadLaneCommon):
        '''
        For this change, the param type has been changed from str to float

        '''
        is_patch = False

        cmn_param_group = inst.param_group
        cmn_param_info = inst.param_info

        param_name = "ss_1gbe_refclk_freq"
        prop_info = cmn_param_info.get_prop_info_by_name(param_name)

        if cmn_param_group.get_param_by_name(param_name) is not None and  \
            prop_info is not None:

            param_val = cmn_param_group.get_param_value(param_name)
            settings = prop_info.valid_setting

            if isinstance(settings, list) and param_val not in prop_info.valid_setting:
                # The option now no longer has the unit 'MHz' as part of the name
                unit_pos = param_val.find("MHz")
                if unit_pos != -1:
                    new_param = param_val.replace("MHz", "")

                    if util.gen_util.is_string_float(new_param):
                        param_to_set = float(new_param)
                    elif new_param in settings:
                        param_to_set = new_param
                    else:
                        # take default
                        param_to_set = prop_info.default                   
                    
                    cmn_param_group.set_param_value(param_name, param_to_set)

                    is_patch = True

            #if prop_info.data_type == GenericParam.DataType.dstr:
            #    prop_info.data_type = GenericParam.DataType.dflo

        if is_patch:
            func_name = util.gen_util.get_function_name()
            self.logger.warning(
                "{}: Patch {} applied to {}".format(func_name, DesignVersion.PATCH_VERSION_20251003,
                                                    inst.name))
                    
        return is_patch