from typing import List
from enum import Enum, unique

from design.db_item import GenericParam
from common_device.property import PropertyMetaData
from common_device.quad.res_service import QuadType

class QuadDesignParamInfo(PropertyMetaData):

    @unique
    class Id(Enum):
        apb_en = "APB_EN"

        ref_clk_pin_name = "REF_CLK_NAME" # When the ref clock comes from core
        pll_ref_clk0 = "PLL_REF_CLK0"
        pll_ref_clk1 = "PLL_REF_CLK1"

        # PMA Direct only
        phy_reset_en = "PHY_RESET_EN"

        @classmethod
        def has_member(cls, item: str) -> bool:
            return item in cls._value2member_map_

        @classmethod
        def has_key(cls, item: str) -> bool:
            return item in cls.__members__

def get_supported_params_by_quad_type(quad_type: QuadType):
    supported_params = []

    match quad_type:
        case QuadType.lane_10g| QuadType.lane_1g:
            supported_params = [
                QuadDesignParamInfo.Id.phy_reset_en,
                QuadDesignParamInfo.Id.ref_clk_pin_name,
                # QuadDesignParamInfo.Id.pll_ref_clk0,
                # QuadDesignParamInfo.Id.pll_ref_clk1,
            ]
        case QuadType.raw_serdes:
            supported_params = [
                QuadDesignParamInfo.Id.phy_reset_en,
                QuadDesignParamInfo.Id.apb_en,
                QuadDesignParamInfo.Id.ref_clk_pin_name,
                # QuadDesignParamInfo.Id.pll_ref_clk0,
                # QuadDesignParamInfo.Id.pll_ref_clk1,
            ]

    return supported_params

def get_deprecated_param_by_quad_type(quad_type: QuadType):
    if quad_type in [QuadType.lane_10g, QuadType.lane_1g]:
        return ["APB_EN"]

    return []

def build_design_param_info():
    param_info = QuadDesignParamInfo()

    prop_data_list: List[PropertyMetaData.PropData] = [
        PropertyMetaData.PropData(id=QuadDesignParamInfo.Id.apb_en,
                name=QuadDesignParamInfo.Id.apb_en.value,
                data_type=GenericParam.DataType.dbool,
                default=True,
                disp_name='Enable Advanced Peripheral Bus',
                category="APB:SW_COMMON"),
        PropertyMetaData.PropData(id=QuadDesignParamInfo.Id.ref_clk_pin_name,
                name=QuadDesignParamInfo.Id.ref_clk_pin_name.value,
                data_type=GenericParam.DataType.dstr,
                default='',
                disp_name='Reference Clock Pin Name',
                category='ref_clk'),
        PropertyMetaData.PropData(id=QuadDesignParamInfo.Id.pll_ref_clk0,
            name=QuadDesignParamInfo.Id.pll_ref_clk0.value,
            data_type=GenericParam.DataType.dstr,
            default='PMA_CMN_REFCLK_PLL_1',
            valid_setting=[
                "PMA_CMN_REFCLK_PLL_1",
                "PMA_CMN_REFCLK_PLL_2",
                "PMA_CMN_REFCLK1_PLL_1",
                "PMA_CMN_REFCLK1_PLL_2",
            ],
            disp_name='PLL Reference Clock 0',
            category='ref_clk'),
        PropertyMetaData.PropData(id=QuadDesignParamInfo.Id.pll_ref_clk1,
            name=QuadDesignParamInfo.Id.pll_ref_clk1.value,
            data_type=GenericParam.DataType.dstr,
            default='PMA_CMN_REFCLK1_PLL_1',
            valid_setting=[
                "PMA_CMN_REFCLK_PLL_1",
                "PMA_CMN_REFCLK_PLL_2",
                "PMA_CMN_REFCLK1_PLL_1",
                "PMA_CMN_REFCLK1_PLL_2",
            ],
            disp_name='PLL Reference Clock 1',
            category='ref_clk'),
        PropertyMetaData.PropData(id=QuadDesignParamInfo.Id.phy_reset_en,
                name=QuadDesignParamInfo.Id.phy_reset_en.value,
                data_type=GenericParam.DataType.dbool,
                default=False,
                disp_name='Enable PHY Quad Reset Pin',
                category="CLOCK and RESET:SW_COMMON"),
        ]

    for prop_data in prop_data_list:
        param_info.add_prop_by_data(prop_data.id, prop_data)

    return param_info
