'''
Copyright (C) 2017-2020 Efinix Inc. All rights reserved.

No portion of this code may be reused, modified or
distributed in any way without the expressed written
consent of Efinix Inc.

Created on Nov 1, 2020

@author: maryam
'''

import os
import sys

from util.gen_util import bool2str

import device.db_interface as devdb_int
import tx60_device.pll.writer.timing as tx60_timing
sys.path.append(os.path.join(os.path.dirname(__file__), '..'))


class PLLTimingV3Complex(tx60_timing.PLLTimingComplex):
    '''
    Class for writing out the timing file for Complex PLL
    '''

    def _get_out_clk_names(self, pll_obj, clk_obj):
        clkout_names = []
        if clk_obj.conn_type == "rclk":
            clkout_names.append(clk_obj.name)

            if clk_obj.clkmux_buf_name != "":
                # Check that this is a valid connection although
                # rule already check for it.
                dbi = devdb_int.DeviceDBService(self._device)
                plldev_service = dbi.get_block_service(self._device.get_pll_type())

                clkout_pin_name = "CLKOUT{}".format(clk_obj.number)

                if plldev_service is not None and \
                        plldev_service.is_pll_clkout_pin_support_mult_clkout_conn(pll_obj.pll_def, clkout_pin_name):
                    clkout_names.append(clk_obj.clkmux_buf_name)

        else:
            clkout_names.append(clk_obj.name)

        return clkout_names

    def _write_sdc_constraint(self, sdcfile, pll_obj, clk_obj, clkout_waveform, sdc_clkout_period):
        # If it is an output clock that connects to both regional and global, then we
        # need to write it out twice
        clkout_names = self._get_out_clk_names(pll_obj, clk_obj)

        for clk_name in clkout_names:
            if clkout_waveform != "":
                sdc = "create_clock -waveform {" + \
                        clkout_waveform + "} -period " + \
                        str(sdc_clkout_period) + " -name " + clk_name + " [get_ports {" + clk_name + "}]"
            else:
                sdc = "create_clock -period " + \
                        str(sdc_clkout_period) + " -name " + clk_name + " [get_ports {" + clk_name + "}]"

            sdcfile.write("{}\n".format(sdc))

    def _create_table_row_entries(self, table, pll_obj, clk_obj, sdc_clkout_period):
        clkout_names = self._get_out_clk_names(pll_obj, clk_obj)

        for clk_name in clkout_names:
            row_list = []

            phase_shift = round(pll_obj.phase_setting2degree(
                clk_obj.phase_setting, clk_obj.out_clock_freq,
                pll_obj.vco_freq, pll_obj.post_divider), 2)

            row_list.append(clk_name)
            row_list.append(str(sdc_clkout_period))
            row_list.append(clk_obj.is_dyn_phase)
            row_list.append(phase_shift)
            row_list.append(bool2str(clk_obj.is_inverted))

            table.add_row(row_list)

    def is_output_clk_internal_only(self, pll_obj, outclk_name: str) -> bool:
        # Find the output clock
        
        if self._design is not None and self._design.ddr_reg is not None and\
            self._design.device_db is not None:

            dbi = devdb_int.DeviceDBService(self._design.device_db)
            ddr_service = dbi.get_block_service(devdb_int.DeviceDBService.BlockType.DDR_ADV)

            clk_obj = pll_obj.get_output_clock(outclk_name)
            if clk_obj is not None and \
                pll_obj.is_pll_ddr_clock(self._design.pll_reg, ddr_service,
                                        self._design.ddr_reg, clk_obj):
                return True

        return False