'''
Copyright (C) 2017-2021 Efinix Inc. All rights reserved.

No portion of this code may be reused, modified or
distributed in any way without the expressed written
consent of Efinix Inc.

Created on Oct 6, 2021

@author: maryam
'''

from __future__ import annotations
import sys
import os
import enum
from typing import Callable, List, Optional, TYPE_CHECKING

from util.singleton_logger import Logger
import util.excp as app_excp
import util.gen_util as gen_util

try:
    from enum import auto
except ImportError:
    auto = gen_util.auto_builder()

from device.pcr_device import BlockPCRLookup as bpcrl
import device.excp as dev_excp
import device.block_instance as blk_ins
from device.device_config import DeviceBlockConfig

import common_device.device_service_interface as bsi
from common_device.ddr.ddr_device_service import DDRServiceBase
from common_device.ddr.ddr_design import DDRAdvConfig
from tx180_device.ddr.ddr_design_adv import DDRParamInfo, DDRAdvance

from design.db_item import GenericParamService

if TYPE_CHECKING:
    from device.io_info import IOPadInfo

sys.path.append(os.path.join(os.path.dirname(__file__), '..'))


class DDRServiceAdvance(DDRServiceBase):
    '''
    Class that is interfaced to the design
    to get ddr related information
    '''

    class MemoryType(enum.IntEnum):
        lpddr4 = 0
        lpddr4x = 1

    memory_type2str_map = {
        MemoryType.lpddr4: "LPDDR4",
        MemoryType.lpddr4x: "LPDDR4x"
    }

    memory_str2type_map = {
        "LPDDR4": MemoryType.lpddr4,
        "LPDDR4x": MemoryType.lpddr4x
    }

    # The following is the names used as the column
    # header in the database

    basic_tbl_header_name_list = \
        [
            DDRServiceBase.HEADER_COMB_ID_NAME,
            DDRServiceBase.HEADER_CTRL_WIDTH_NAME,
            DDRServiceBase.HEADER_MEMORY_TYPE_NAME,
            DDRServiceBase.HEADER_DRAM_DENSITY_NAME,
            DDRServiceBase.HEADER_PHYSICAL_RANK_NAME,
            DDRServiceBase.HEADER_COMMON_NAME,
            DDRServiceBase.HEADER_VERIFIED_NAME,
            DDRServiceBase.HEADER_DESCRIPTION_NAME
        ]

    class PortClassGroup(enum.IntEnum):
        axi_target_0 = auto()
        axi_target_1 = auto()
        controller = auto()
        ctrl_reg_inf = auto()
        cfg_control = auto()

    class PCRDefnType(enum.Enum):
        '''
        Enum that represents the allowed LVDS
        PCR names.
        '''
        pcr_bypass_cal = auto()
        pcr_clk_div = auto()
        pcr_ddr_clksel = auto()
        pcr_ddr_en = auto()
        pcr_ipready_bypass = auto()
        unused_0 = auto()
        pcr_unsupported1 = auto()
        pcr_unsupported2 = auto()
        pcr_unsupported3 = auto()
        pcr_unsupported4 = auto()
        pcr_unsupported5 = auto()
        pcr_unsupported6 = auto()
        pcr_unsupported7 = auto()
        pcr_unsupported8 = auto()
        pcr_axi0_256 = auto()
        pcr_cfg_ctrl = auto()
        pcr_rstn_ctrl = auto()
        pcr_axi1_256 = auto()
        pcr_unused = auto()
        pcr_ddr_dbg = auto()
        pcr_test_en = auto()
        pcr_jtag_dbg_en = auto()
        pcr_ddr_nisoen = auto()
        pcr_unsupported = auto()


    pcrdefn2str_map = \
        {
            PCRDefnType.pcr_bypass_cal: "PCR_BYPASS_CAL",
            PCRDefnType.pcr_clk_div: "PCR_CLK_DIV",
            PCRDefnType.pcr_ddr_clksel: "PCR_DDR_CLKSEL",
            PCRDefnType.pcr_ddr_en: "PCR_DDR_EN",
            PCRDefnType.pcr_ipready_bypass: "PCR_IPREADY_BYPASS",
            PCRDefnType.unused_0: "UNUSED_0",
            PCRDefnType.pcr_unsupported1: "PCR_UNSUPPORTED1",
            PCRDefnType.pcr_unsupported2: "PCR_UNSUPPORTED2",
            PCRDefnType.pcr_unsupported3: "PCR_UNSUPPORTED3",
            PCRDefnType.pcr_unsupported4: "PCR_UNSUPPORTED4",
            PCRDefnType.pcr_unsupported5: "PCR_UNSUPPORTED5",
            PCRDefnType.pcr_unsupported6: "PCR_UNSUPPORTED6",
            PCRDefnType.pcr_unsupported7: "PCR_UNSUPPORTED7",
            PCRDefnType.pcr_unsupported8: "PCR_UNSUPPORTED8",
            PCRDefnType.pcr_axi0_256: "PCR_AXI0_256",
            PCRDefnType.pcr_cfg_ctrl: "PCR_CFG_CTRL",
            PCRDefnType.pcr_rstn_ctrl: "PCR_RSTN_CTRL",
            PCRDefnType.pcr_axi1_256: "PCR_AXI1_256",
            PCRDefnType.pcr_unused: "UNUSED_0",
            PCRDefnType.pcr_ddr_dbg: "PCR_DDR_DBG",
            PCRDefnType.pcr_test_en: "PCR_TEST_EN",
            PCRDefnType.pcr_jtag_dbg_en: "PCR_JTAG_DBG_EN",
            PCRDefnType.pcr_ddr_nisoen: "PCR_DDR_NISOEN",
            PCRDefnType.pcr_unsupported: "PCR_UNSUPPORTED"
        }

    str2pcrdefn_map = \
        {
            "PCR_BYPASS_CAL": PCRDefnType.pcr_bypass_cal,
            "PCR_CLK_DIV": PCRDefnType.pcr_clk_div,
            "PCR_DDR_CLKSEL": PCRDefnType.pcr_ddr_clksel,
            "PCR_DDR_EN": PCRDefnType.pcr_ddr_en,
            "PCR_IPREADY_BYPASS": PCRDefnType.pcr_ipready_bypass,
            "UNUSED_0": PCRDefnType.unused_0,
            "PCR_UNSUPPORTED1": PCRDefnType.pcr_unsupported1,
            "PCR_UNSUPPORTED2": PCRDefnType.pcr_unsupported2,
            "PCR_UNSUPPORTED3": PCRDefnType.pcr_unsupported3,
            "PCR_UNSUPPORTED4": PCRDefnType.pcr_unsupported4,
            "PCR_UNSUPPORTED5": PCRDefnType.pcr_unsupported5,
            "PCR_UNSUPPORTED6": PCRDefnType.pcr_unsupported6,
            "PCR_UNSUPPORTED7": PCRDefnType.pcr_unsupported7,
            "PCR_UNSUPPORTED8": PCRDefnType.pcr_unsupported8,
            "PCR_AXI0_256": PCRDefnType.pcr_axi0_256,
            "PCR_CFG_CTRL": PCRDefnType.pcr_cfg_ctrl,
            "PCR_RSTN_CTRL": PCRDefnType.pcr_rstn_ctrl,
            "PCR_AXI1_256": PCRDefnType.pcr_axi1_256,
            "UNUSED_0": PCRDefnType.pcr_unused,
            "PCR_DDR_DBG": PCRDefnType.pcr_ddr_dbg,
            "PCR_TEST_EN": PCRDefnType.pcr_test_en,
            "PCR_JTAG_DBG_EN": PCRDefnType.pcr_jtag_dbg_en,
            "PCR_DDR_NISOEN": PCRDefnType.pcr_ddr_nisoen,
            "PCR_UNSUPPORTED": PCRDefnType.pcr_unsupported
        }

    class PCRModeType(enum.Enum):
        '''
        ENum that represents the allowed modes
        for the PCR definition (superset and
        not an exact map. i.e. ENABLE can be
        applicable to some of the list of the PCR
        definition.
        '''
        disable = auto()
        enable = auto()
        div2 = auto()
        div4 = auto()
        div8 = auto()
        div16 = auto()
        pll_tl0 = auto()
        pll_tl1 = auto()
        pll_tl2 = auto()
        vss = auto()

    pcrmode2str_map = \
        {
            PCRModeType.disable: "DISABLE",
            PCRModeType.enable: "ENABLE",
            PCRModeType.div2: "DIV2",
            PCRModeType.div4: "DIV4",
            PCRModeType.div8: "DIV8",
            PCRModeType.div16: "DIV16",
            PCRModeType.pll_tl0: "PLL_TL0",
            PCRModeType.pll_tl1: "PLL_TL1",
            PCRModeType.pll_tl2: "PLL_TL2",
            PCRModeType.vss: "VSS"
        }

    str2pcrmode_map = \
        {
            "DISABLE": PCRModeType.disable,
            "ENABLE": PCRModeType.enable,
            "DIV2": PCRModeType.div2,
            "DIV4": PCRModeType.div4,
            "DIV8": PCRModeType.div8,
            "DIV16": PCRModeType.div16,
            "PLL_TL0": PCRModeType.pll_tl0,
            "PLL_TL1": PCRModeType.pll_tl1,
            "PLL_TL2": PCRModeType.pll_tl2,
            "VSS": PCRModeType.vss
        }

    # Empty so it will be overriden by derived class

    def __init__(self, device_db, name):
        '''
        constructor
        '''

        super().__init__(device_db, name)

        # populate the advance parameter settings

        # FPGA setting
        self.lpddr4_fpga_vref_range0_list = []
        self.lpddr4_fpga_vref_range1_list = []

        self.lpddr4x_fpga_vref_range0_list = []
        self.lpddr4x_fpga_vref_range1_list = []

        # Memory Mode Register setting: They are common for CA VREF and DQ VREF
        self.mem_setting_range0_list = []
        self.mem_setting_range1_list = []

        self._populate_vref_valid_values()

    def _populate_vref_valid_values(self):
        '''
        Populate the correct list of values associated to all the advance
        setting parameters that accepts float of specific range and
        step value. This can be used to be checked against since the widget doesn't
        really control entering numerical value that is out of the range + step value.
        :return:
        '''
        #TODO: Retreive it from the DB. Ran out of time and hardcode them here for now
        
        # Generate the Memory mode setting
        # Range 0: [10,30,0.4]
        self.mem_setting_range0_list = ["{:.1f}".format(x * 0.1) for x in range(100,301,4)]
        # Range 1: [22,42,0.4]
        self.mem_setting_range1_list = ["{:.1f}".format(x * 0.1) for x in range(220,421,4)]

        # Generate the FPGA VREF setting
        # LPDDR4
        # Range0: [5.40,38.42,0.26]
        # Range1: [11.900,48.222,0.286]
        self.lpddr4_fpga_vref_range0_list = ["{:.2f}".format(x * 0.01) for x in range(540,3843,26)]
        self.lpddr4_fpga_vref_range1_list = ["{:.3f}".format(x * 0.001) for x in range(11900,48223,286)]

        # LPDDR4x
        # Range0: [11.60,49.70,0.3]
        # Range1: [21.20,59.30,0.3]
        self.lpddr4x_fpga_vref_range0_list = ["{:.2f}".format(x * 0.01) for x in range(1160,4971,30)]
        self.lpddr4x_fpga_vref_range1_list = ["{:.2f}".format(x * 0.01) for x in range(2120,5931,30)]

    def get_ddr_pad_name_extension(self, pad_info_obj: Optional[IOPadInfo], ddr_obj: DDRAdvance):
        '''
        PT-568: Indicates how the pin should be named

        Get the extension of the name (used in the pinout)
        if pad information match with ddr design instance/ always being used.

        :param pad_info_obj: The pad object we're looking at
        :param ddr_obj: The design instance of DDR
        :return The extension of the name to be used in the pinout
        '''

        final_ext_name = ""

        if pad_info_obj is None:
            return final_ext_name

        param_service = GenericParamService(
            ddr_obj.get_param_group(), ddr_obj.get_param_info())

        # Pins that are dependent on DQ Width
        dqwidth_pins = ["DQ"]
        dqwidth_byte_pins = ["DM", "DQS_N", "DQS"]

        # Remaining pins
        #ddr_pins = ["CAL", "A", "CKE", "CK", "CK_N",
        #             "CS_N", "RST_N"]

        #TODO: If we have different memory type with different
        #pin name support, then we need to distinguish
        ori_pin_name = pad_info_obj.get_pin_name()

        if ori_pin_name is None:
            return final_ext_name

        # Get the pin name without the bit blasting
        pos = ori_pin_name.find("[")
        if pos != -1:
            pin_name = ori_pin_name[:pos]
        else:
            pin_name = ori_pin_name

        # For the rest, we take it as it is
        if pin_name in dqwidth_pins or \
                pin_name in dqwidth_byte_pins:

            # Check the DQ Width and DDR Type
            width_num_int: int = param_service.get_param_value( # type: ignore
                DDRParamInfo.Id.data_width)

            # Get the pin index
            right_pos: int = ori_pin_name.find("]")
            if right_pos != -1 and width_num_int != -1:
                pin_index_str = ori_pin_name[pos +
                                                1: right_pos]
                try:
                    pin_index_int = int(pin_index_str)
                except ValueError as verr:
                    pin_index_int = -1

                # Take 1 bit less: x8 -> 0-7, x16 -> 0-15, x32
                # -> 0-31
                if pin_index_int != -1:
                    if pin_name in dqwidth_pins:
                        if pin_index_int < width_num_int:
                            final_ext_name = ori_pin_name
                    else:
                        # The following is how it is determined:
                        # a) DQ Width x8 -> [0]
                        # b) DQ Width x16 -> [1:0]
                        # c) DQ Width x32 -> [3:0]
                        # We don't need to check for the DDR
                        # type (16/32)

                        if width_num_int == 16 and pin_index_int <= 1:
                            final_ext_name = ori_pin_name
                        elif width_num_int == 32:
                            final_ext_name = ori_pin_name
        else:
            final_ext_name = ori_pin_name

        return final_ext_name

    def get_fpga_vref_list(self, ddr_ins):
        if ddr_ins is not None:
            adv_config = ddr_ins.adv_config

            if adv_config is not None:
                # Get the range
                fpga_param_group = adv_config.get_setting(DDRAdvConfig.CatType.fpga)

                if fpga_param_group is not None:
                    vref_parent_param = fpga_param_group.get_param_by_name("MEM_FPGA_VREF_RANGE")

                    if ddr_ins.get_memory_type() == "LPDDR4x":
                        if vref_parent_param.value == "Range 0":
                            return "min=11.60,max=49.70,step=0.3", self.lpddr4x_fpga_vref_range0_list

                        else:
                            return "min=21.20,max=59.30,step=0.3", self.lpddr4x_fpga_vref_range1_list
                    else:
                        if vref_parent_param.value == "Range 0":
                            return "min=5.40,max=38.42,step=0.26", self.lpddr4_fpga_vref_range0_list

                        else:
                            return "min=11.900,max=48.222,step=0.286", self.lpddr4_fpga_vref_range1_list

        return "", None

    def get_mem_vref_list(self, ddr_ins, is_ca):
        if ddr_ins is not None:
            adv_config = ddr_ins.adv_config

            if adv_config is not None:
                # Get the range
                mem_param_group = adv_config.get_setting(DDRAdvConfig.CatType.mem)

                if mem_param_group is not None:
                    if is_ca:
                        vref_parent_param = mem_param_group.get_param_by_name("MEM_CA_RANGE")
                    else:
                        vref_parent_param = mem_param_group.get_param_by_name("MEM_DQ_RANGE")

                    if vref_parent_param is not None:
                        if vref_parent_param.value == "RANGE[0]":
                            return "min=10.0,max=30.0,step=0.4", self.mem_setting_range0_list
                        else:
                            return "min=22.0,max=42.0,step=0.4", self.mem_setting_range1_list

        return None

    def get_debug_test_ports(self):
        '''

        :return: a map of port name to the port object that has the type
            set to "debug"
        '''
        from device.block_definition import Port as DevicePort

        debug_ports = {}

        block_ports = self.get_ports()
        for port_name, port_obj in block_ports.items():
            if port_obj is not None and port_obj.get_type() == DevicePort.TYPE_DEBUG:
                debug_ports[port_name] = port_obj

        return debug_ports

    def get_ports_by_class_group(self, class_group):
        '''
        :param class_group: PortClassGroup enum
        :return a map of port name to the port object that is associated to
                the specified class type
        '''
        from device.block_definition import Port as DevicePort

        class_ports = {}

        block_ports = self.get_ports()

        for port_name, port_obj in block_ports.items():
            if port_obj is not None and port_obj.get_class() is not None:
                port_class_name = port_obj.get_class()

                if class_group == self.PortClassGroup.axi_target_0:
                    if port_name.endswith("0") and \
                            self.is_axi_port_class(port_class_name):
                        class_ports[port_name] = port_obj
                elif class_group == self.PortClassGroup.axi_target_1:
                    if port_name.endswith("1") and \
                            self.is_axi_port_class(port_class_name):
                        class_ports[port_name] = port_obj
                elif class_group == self.PortClassGroup.controller:
                    if port_class_name == DevicePort.CLASS_DDR_CONTROLLER:
                        class_ports[port_name] = port_obj
                elif class_group == self.PortClassGroup.ctrl_reg_inf:
                    if port_class_name == DevicePort.CLASS_DDR_CTRL_REG_INF:
                        class_ports[port_name] = port_obj
                elif class_group == self.PortClassGroup.cfg_control:
                    if port_class_name == DevicePort.CLASS_DDR_CFG_CONTROL:
                        class_ports[port_name] = port_obj

        return class_ports

    def is_axi_port_class(self, port_class_name):
        from device.block_definition import Port as DevicePort

        if port_class_name == DevicePort.CLASS_DDR_AXI_GLB_SIG or\
                port_class_name == DevicePort.CLASS_DDR_AXI_RD_ADDR_CH or \
                port_class_name == DevicePort.CLASS_DDR_AXI_WR_ADDR_CH or \
                port_class_name == DevicePort.CLASS_DDR_AXI_WR_RESP_CH or \
                port_class_name == DevicePort.CLASS_DDR_AXI_RD_DATA_CH or \
                port_class_name == DevicePort.CLASS_DDR_AXI_WR_DATA_CH:
            return True

        return False

    def get_resource_on_clkin_pin(self, ins_name: str, index: int):
        '''
        Get the resource if any for the given device instance name
        and reference clock index ONLY.
        :param ins_name: DDR device instance name
        :param index: The clock index
        :return 1) the resource name, if applicable. It is also
                possible a string of names delimited by ','.
                If not applicable, a None is returned
                2) the pin name: This is a single string. If not applicable it is
                set to None
        '''
        # Find the instance
        ins_obj = self.get_device().find_instance(ins_name)

        # PT-1630 Only allow PLL_TL2 used for Ti180 DDR
        # TODO: revert if the spec change later
        #index = 2

        if ins_obj is not None:
            # Find the pin
            pin_name = "CLKIN[{}]".format(index)
            clkin_pin = ins_obj.get_connection_pin(pin_name)

            if clkin_pin is not None:
                if clkin_pin.get_type() == blk_ins.PinConnection.InterfaceType.internal:
                    dep_ins_name = clkin_pin.get_connecting_instance_name(
                        self.get_device())

                    dep_ref_pin_name = clkin_pin.get_connecting_ref_pin_name()

                    dep_list = []
                    #self.logger.debug("find phy clk resource on ddr {} pin {} res {} pin: {}".format(
                    #    ins_name, pin_name, dep_ins_name, dep_ref_pin_name))

                    if dep_ins_name is not None:
                        if dep_ins_name.find(",") != -1:
                            dep_list = dep_ins_name.split(",")
                        else:
                            dep_list.append(dep_ins_name)

                    # Check if the name is availabe in resource map
                    final_dep_name = None
                    for dname in dep_list:
                        if self.get_device().is_instance_in_resource_map(dname):
                            if final_dep_name is not None:
                                final_dep_name += "," + dname
                            else:
                                final_dep_name = dname

                    return final_dep_name, dep_ref_pin_name

        return None, None

    def get_phy_clkin_sel_resource(self, ins_name: str):
        """
        Find all the list of resources associated to the ddr instance
        :param ins_name: Device resource name
        :return: a map of resource name to the clkin sel index
        """
        res2index_map = {}

        clkin_index_list = self.get_clkin_sel_list()

        for index in clkin_index_list:
            res_name, _ = self.get_resource_on_clkin_pin(ins_name, index)
            if res_name is not None:
                res2index_map[res_name] = index

        return res2index_map

    def get_all_resource_on_phy_clkin_pin(self, ins_name: str, index: int):
        '''
        There is a scenario where one clock pin has a choice of resources
        and not fixed to one particular resource.  Hence, in this
        case a list of possible options is returned.  The differences
        is that we break up the string into multiple strings.

        :param ins_name: PLL device instance name
        :param index: The clock index integer that matches the hardware block indexing
        :return a list of resource names. Single element if there's only
                one resource,
                reference pin string associated to all those resources. Empty string if there's
                no resource found
        '''
        res_name, ref_pin_name = self.get_resource_on_clkin_pin(ins_name, index)
        self.logger.debug("ddr phy clk: ins_name {} clkin {} res {} ref_pin: {}".format(
            ins_name, index, res_name, ref_pin_name))

        res_list = []

        if res_name is not None:
            if res_name.find(",") != -1:
                res_list = res_name.split(",")
            else:
                res_list.append(res_name)
        # Reset the ref_pin
        else:
            ref_pin_name = None

        return res_list, ref_pin_name

    def get_max_ref_clk_freq(self):
        '''
        :return the allowed maximum frequency:
                PLL output freq (in unit MHz)
        '''
        return 933.25

    def get_min_ref_clk_freq(self):
        '''
        :return the minimum frequency:
                PLL output freq (in unit MHz)
        '''
        return 133.0

    def get_valid_options(self, bprop_type: DeviceBlockConfig.BlockPropertyType, type_cast: Callable) -> List:
        valid_opt = []
        if self._device_db is not None:
            block_config = self._device_db.get_block_configuration()
            if block_config is not None:
                # Find if there's any config of this block type
                block_prop = block_config.get_block_property(self._name)
                if block_prop is not None:
                    prop_name_str = DeviceBlockConfig.bprop2str_map[bprop_type]
                    bprop = block_prop.get_prop_value(prop_name_str)
                    if bprop is not None:
                        # It means there's an exception for this device. Titanium
                        # width is an integer option. So, convert this to integer
                        try:
                            valid_opt.append(type_cast(bprop))
                        except ValueError:
                            valid_opt = []
        return valid_opt

    def get_data_width_options(self):
        '''
        Based on the current device, check if the data width supported is
        a subset of the DDR data width list.
        This is applicable to Ti*M361 which only supported 16 instead of both
        with 32.  The info is saved at the device level

        :return: a list of options for data width.
        '''
        valid_opt = self.get_valid_options(DeviceBlockConfig.BlockPropertyType.bprop_ddr_config_width, int)

        if not valid_opt or len(valid_opt) == 0:
            # hardcode which should be the same as get_data_width_options() in DDRAdvance
            valid_opt = [16, 32]

        return valid_opt
    
    def get_memory_type_options(self):
        valid_opt = self.get_valid_options(DeviceBlockConfig.BlockPropertyType.bprop_ddr_memory_type, lambda x: x)
        if not valid_opt or len(valid_opt) == 0:
            valid_opt = ['LPDDR4', 'LPDDR4x']
        return valid_opt

    def get_memory_density_specific_default(self) -> str:
        '''
        Check if the device has specific Memory Density default. If not,
        return empty string.
        : return Memory Density default string or empty if not applicable.
        '''
        specific_default = self.get_valid_options(DeviceBlockConfig.BlockPropertyType.bprop_ddr_memory_density_default, lambda x: x)
        if len(specific_default) > 0:
            # takes the first one (assumed only one element)
            return specific_default[0]
        
        return ""
        
    def is_class_name_supported(self, class_name):
        if class_name == bpcrl.CLASS_DDR_FPGA_SETTINGS or \
                class_name == bpcrl.CLASS_DDR_MEMORY_SETTINGS or \
                class_name == bpcrl.CLASS_DDR_MEMORY_TIMING:
            return True

        return False

    def get_adv_step_value(self, mtype, param_name):
        '''
        Get the information on parameter with range that has step value

        :param mtype: The Memory type enum
        :param param_name: The parameter name of a ddr_memory_timing settings

        :return: The step value. If parameter is not found or is
                not a valid param or has no step value, return None
        '''
        step_val = None

        constraint_map = {self.HEADER_ADV_NAME: param_name}
        target_field_list = [self.HEADER_ADV_VALUE_NAME]

        config_tbl_name = self.get_adv_config_table_name(mtype)

        results = self.get_adv_config_data(config_tbl_name, constraint_map,
                                           target_field_list)

        range_list = self.get_results_column(results, 0)

        if len(range_list) == 1:
            range_str = range_list[0]

            data_split = range_str.split(":")
            if len(data_split) == 4:
                tmp_step_val = data_split[3]

                # Get the value type
                constraint_map = {self.HEADER_ADV_NAME: param_name}
                target_field_list = [self.HEADER_ADV_VALUE_TYPE_NAME]

                results = self.get_adv_config_data(config_tbl_name, constraint_map,
                                                   target_field_list)

                pval_str_list = self.get_results_column(results, 0)
                # TODO: Check for invalid (>1)
                if len(pval_str_list) == 1:
                    pval_str = pval_str_list[0]

                    # Don't do anything if it is a string type
                    if pval_str in self.vstr2type_map:
                        pval_type = self.vstr2type_map[pval_str]

                        if pval_type == self.AdvValueType.integer:
                            step_val = int(tmp_step_val)

                        elif pval_type == self.AdvValueType.float:
                            step_val = float(tmp_step_val)

        return step_val

    def is_ddr_rev_b(self):
        '''
        Check if the ddr is meant for the revB silicon

        :return: True if it is a ddr meant for rev B silicon
        '''
        is_revb_ddr = False

        pll_clkin_sel = self.get_clkin_sel_list()
        if len(pll_clkin_sel) > 1:
            is_revb_ddr = True

        return is_revb_ddr

    def get_clkin_sel_list(self):
        '''
        Based on the current device, get the list of clkin select that this device supports.

        :return: a list of options for data width.
        '''

        valid_opt = []

        if self._device_db is not None:

            block_config = self._device_db.get_block_configuration()

            if block_config is not None:
                # Find if there's any config of this block type
                block_prop = block_config.get_block_property(self._name)

                if block_prop is not None:
                    prop_name_str = DeviceBlockConfig.bprop2str_map[
                        DeviceBlockConfig.BlockPropertyType.bprop_ddr_pll_clkin_list]
                    bprop = block_prop.get_prop_value(prop_name_str)

                    if bprop is not None:
                        if block_prop.is_prop_value_list(prop_name_str):
                            # List of string. Convert to int
                            clkin_list_str = block_prop.get_prop_value_list(
                                prop_name_str)

                            try:
                                valid_opt = [int(i) for i in clkin_list_str]

                            except ValueError:
                                self.logger.debug(
                                    "Unexpected non-integer value in pll_clkin_list {}".format(
                                        ",".join(clkin_list_str)))
                                valid_opt = []

                        else:
                            bprop = block_prop.get_prop_value(prop_name_str)

                            if bprop is not None:
                                try:
                                    valid_opt.append(int(bprop))

                                except ValueError:
                                    valid_opt = []

        if not valid_opt or len(valid_opt) == 0:
            # hardcode which should be the same as get_clkin_options() in DDRAdvance
            valid_opt = [0, 1, 2]

        return valid_opt

    @staticmethod
    def is_pcr_defn_match(pcr_name, pcr_type):
        '''
        Check if the block has the PCR definition
        name defined.

        :param pcr_name: PCR definition string

        :return True if it exists
        '''

        if pcr_name in DDRServiceAdvance.str2pcrdefn_map and \
                DDRServiceAdvance.str2pcrdefn_map[pcr_name] == pcr_type:
            return True

        return False
